use std::io::{Error, Read, Write};
use std::net::{TcpStream, ToSocketAddrs};
use std::thread::sleep;
use std::time::Duration;

fn connect(remote: String, timeout: u64) -> Result<TcpStream, std::io::Error> {
    for sockaddr in remote.to_socket_addrs()? {
        let stream = TcpStream::connect_timeout(&sockaddr, Duration::from_secs(timeout));
        if stream.is_ok() {
            return stream;
        }
    }

    Err(Error::other("Could not connect"))
}

fn fetch_tcp(selector: &crate::selector::Selector) -> std::io::Result<Vec<u8>> {
    #[cfg(debug_assertions)]
    let timeout = 2;
    #[cfg(not(debug_assertions))]
    let timeout = 15;

    let mut stream = connect(format!("{}:{}", selector.host, selector.port), timeout)?;

    stream.set_read_timeout(Some(Duration::from_secs(timeout)))?;
    stream.set_write_timeout(Some(Duration::from_secs(timeout)))?;

    stream.write_all(format!("{}\r\n", selector.path).as_bytes())?;

    let mut content: Vec<u8> = Vec::new();
    let mut buf = [0; 1500];
    loop {
        let size_read = stream.read(&mut buf)?;
        if size_read == 0 {
            return Ok(content);
        }
        content.extend_from_slice(&buf[..size_read]);
    }
}

pub fn fetch(selector: &crate::selector::Selector) -> Result<String, String> {
    // Throttle everything.
    #[cfg(debug_assertions)]
    let throttle = 0;
    #[cfg(not(debug_assertions))]
    let throttle = 500;

    sleep(Duration::from_millis(throttle));

    match fetch_tcp(selector) {
        Ok(content) => Ok(String::from_utf8_lossy(&content).into_owned()),
        Err(reason) => Err(reason.to_string()),
    }
}
