// $Id: filter-dnsbl.go 66 2024-04-14 16:44:05Z umaxx $
// Copyright (c) 2018-2024 Joerg Jung <mail@umaxx.net>
//
// Permission to use, copy, modify, and distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

// filter-dnsbl - opensmtpd filter for DNSBL/DNSWL

package main

import (
	"bufio"
	"fmt"
	"log"
	"log/syslog"
	"net"
	"os"
	"strings"
)

const (
	v  = "1.0"
	yr = "2018-2024"
)

var (
	bl = []string{"bl.spamcop.net", "zen.spamhaus.org"}
	wl = []string{"list.dnswl.org"}
	l3 *syslog.Writer
)

func reverse(ip net.IP) string {
	rip := strings.Split(ip.String(), ".")
	for i, j := 0, len(rip)-1; i < len(rip)/2; i, j = i+1, j-1 {
		rip[i], rip[j] = rip[j], rip[i]
	}
	return strings.Join(rip, ".")
}

func lookup(ip net.IP, l string) bool {
	r, e := net.LookupHost(fmt.Sprintf("%s.%s", reverse(ip), l))
	if e != nil {
		l3.Debug(fmt.Sprintln(e))
		return false
	}
	return (len(r) > 0)
}

func check(sid string, tok string, ip net.IP) {
	if ip.To4() == nil { // todo: handle ipv6
		l3.Debug(fmt.Sprintln(sid, "non-ipv4 address", ip))
		fmt.Printf("filter-result|%s|%s|proceed\n", sid, tok)
		return
	}
	for _, v := range wl {
		if lookup(ip, v) {
			l3.Info(fmt.Sprintln(sid, "white listed", ip, v))
			fmt.Printf("filter-result|%s|%s|proceed\n", sid, tok)
			return
		}
	}
	for _, v := range bl {
		if lookup(ip, v) {
			l3.Info(fmt.Sprintln(sid, "black listed", ip, v))
			fmt.Printf("filter-result|%s|%s|disconnect|554 5.7.1 Address in DNSBL\n", sid, tok)
			return
		}
	}
	fmt.Printf("filter-result|%s|%s|proceed\n", sid, tok)
}

func register(in *bufio.Scanner) error {
	l3.Info("register")
	for in.Scan() { // skip config
		if in.Text() == "config|ready" {
			fmt.Println("register|filter|smtp-in|connect")
			fmt.Println("register|ready")
			return nil
		}
	}
	return in.Err()
}

func run() {
	l3.Info("start")
	defer l3.Info("exit")
	in := bufio.NewScanner(os.Stdin)
	if e := register(in); e != nil {
		l3.Err(fmt.Sprintln("register", e))
		return
	}
	for in.Scan() {
		f := strings.Split(in.Text(), "|")
		t, ver, ev, sid, tok, ip := f[0], f[1], f[4], f[5], f[6], net.ParseIP(strings.Trim(f[8], "[]"))
		if t != "filter" || ver != "0.7" || ev != "connect" || ip == nil {
			l3.Err(fmt.Sprintln(sid, "protocol", t, ver, ev, ip))
			return
		}
		go check(sid, tok, ip)
	}
	if e := in.Err(); e != nil {
		l3.Err(fmt.Sprintln("scanner", e))
		return
	}
}

func init() {
	log.SetFlags(log.Lshortfile)
}

func main() {
	var e error
	if len(os.Args) == 2 && os.Args[1] == "version" {
		fmt.Println("filter-dnsbl", v, "(c)", yr, "Joerg Jung")
		return
	}
	if len(os.Args) > 3 {
		log.Fatalf("usage: filter-dnsbl [<blacklist>,...] [<whitelist>,...]\n%28sfilter-dnsbl version\n", "")
	}
	if len(os.Args) >= 2 {
		bl = strings.Split(os.Args[1], ",")
		for i, v := range bl {
			bl[i] = strings.TrimSpace(v)
		}
	}
	if len(os.Args) == 3 {
		wl = strings.Split(os.Args[2], ",")
		for i, v := range wl {
			wl[i] = strings.TrimSpace(v)
		}
	}
	if l3, e = syslog.New(syslog.LOG_MAIL, "filter-dnsbl"); e != nil {
		log.Fatal(e)
	}
	defer l3.Close()
	run()
}
