// $Id: filter-delay.go 66 2024-04-14 16:44:05Z umaxx $
// Copyright (c) 2018-2024 Joerg Jung <mail@umaxx.net>
//
// Permission to use, copy, modify, and distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

// filter-delay - opensmtpd connect delay filter

package main

import (
	"bufio"
	"fmt"
	"log"
	"log/syslog"
	"os"
	"strings"
	"time"
)

const (
	v  = "0.9"
	yr = "2018-2024"
)

var (
	d  time.Duration = 5 // delay duration in seconds
	l3 *syslog.Writer
)

func delay(sid string, tok string) {
	time.Sleep(d * time.Second)
	fmt.Printf("filter-result|%s|%s|proceed\n", sid, tok)
}

func register(in *bufio.Scanner) error {
	l3.Info("register")
	for in.Scan() { // skip config
		if in.Text() == "config|ready" {
			fmt.Println("register|filter|smtp-in|connect")
			fmt.Println("register|ready")
			return nil
		}
	}
	return in.Err()
}

func run() {
	l3.Info("start")
	defer l3.Info("exit")
	in := bufio.NewScanner(os.Stdin)
	if e := register(in); e != nil {
		l3.Err(fmt.Sprintln("register", e))
		return
	}
	for in.Scan() {
		f := strings.Split(in.Text(), "|")
		t, ver, ev, sid, tok := f[0], f[1], f[4], f[5], f[6]
		if t != "filter" || ver != "0.7" || ev != "connect" {
			l3.Err(fmt.Sprintln(sid, "protocol", t, ver, ev))
			return
		}
		go delay(sid, tok)
	}
	if e := in.Err(); e != nil {
		l3.Err(fmt.Sprintln("scanner", e))
		return
	}
}

func init() {
	log.SetFlags(log.Lshortfile)
}

func main() {
	var e error
	if len(os.Args) == 2 && os.Args[1] == "version" {
		fmt.Println("filter-delay", v, "(c)", yr, "Joerg Jung")
		return
	}
	if len(os.Args) > 2 {
		log.Fatalf("usage: filter-delay [<duration>]\n%27sfilter-delay version\n", "")
	}
	if len(os.Args) == 2 {
		if d, e = time.ParseDuration(os.Args[1]); e != nil {
			log.Fatal(e)
		}
		if d.Seconds() == 0 || d.Seconds() > 300 {
			log.Fatal("duration not between 1s and 300s")
		}
	}
	if l3, e = syslog.New(syslog.LOG_MAIL, "filter-delay"); e != nil {
		log.Fatal(e)
	}
	defer l3.Close()
	run()
}
