;; run-mars: command-line utility for running redcode programs
;;

(import (chicken io)
        (chicken process-context)
        matchable
        mars visualizer parser)

(define (make-color-map progs colors)
  (let loop ((entries '())
             (progs-left progs)
             (colors-left colors))
    (if (null? progs-left)
        entries
        (if (null? colors-left)
            (error "Not enough colours in colour map!")
            (let ((this-prog (car progs-left))
                  (this-col (car colors-left)))
              (loop (cons (cons (prog-name this-prog) this-col) entries)
                    (cdr progs-left)
                    (cdr colors-left)))))))

(define (mars-runner files iters core-size visualization min-queue-count)
  (print "Iters: " iters ", core size: " core-size)
  (let* ((progs (map
                 (lambda (fname)
                   (string->prog (with-input-from-file fname read-string)))
                 files))
         (core (if visualization
                   (let* ((colors '("red" "blue" "green" "magenta" "cyan"))
                          (color-map (make-color-map progs colors))
                          (vis (make-vis 640 480 core-size color-map)))
                     (make-core core-size (lambda (i n)
                                            (vis 'update-owner i n))))
                   (make-core core-size)))
         (queues (run-mars core (install-progs core progs) iters min-queue-count)))
    (dump-queues queues core)
    (when visualization
      (print* "Press enter to finish...")
      (read-line))))

(define (print-usage)
  (print "Usage: run-mars [-h|--help]\n"
         "       run-mars [-c|--core size]\n"
         "                [-i|--iterations iters]\n"
         "                [-v|--visualization]\n"
         "                [-m|--min-queue-count]\n"
         "                warrior1.red [warrior2.red [...]]"))

(define (main)
  (let loop ((args (cdr (argv)))
             (iters 10000)
             (core-size 8000)
             (visualization #f)
             (min-queue-count 2))
    (match args
      ((or () ((or "-h" "--help")))
       (print-usage))
      (((or "-i" "--iterations") istr rest ...)
       (loop rest (string->number istr) core-size visualization min-queue-count))
      (((or "-c" "--core-size") cstr rest ...)
       (loop rest iters (string->number cstr) visualization min-queue-count))
      (((or "-v" "--visualization") rest ...)
       (loop rest iters core-size #t min-queue-count))
      (((or "-m" "--min-queue-count") mstr rest ...)
       (loop rest iters core-size visualization (string->number mstr)))
      ((files ...)
       (mars-runner files iters core-size visualization min-queue-count)))))

(main)
