/* 
 * xvgam.c - Gamma Correction box handling functions
 *
 * callable functions:
 *
 *   CreateGam(geom)        -  creates the ctrlW window.  Doesn't map it.
 *   GamBox(vis)            -  random processing based on value of 'vis'
 *                             maps/unmaps window, etc.
 *   RedrawGam(x,y,w,h)     -  called by 'expose' events
 *   RedrawGraph(x,y,w,h)   -  called by 'expose' events
 *   ClickGam(x,y)          -  called when B1 clicked in gamW 
 *   TrackGraph(x,y)        -  called when B1 clicked in graphW
 *   GenerateGamma()        -  called to generate/error-check 'ghand'
 *   GenerateFSGamma()      -  called to generate floyd steinberg correction
 *   GammifyColors()        -  does gamma correction of r[],g[],b[] arrays
 *   SetGPreset()           -  sets preset #n to supplied values
 */

/*
 * Copyright 1989, 1990 by the University of Pennsylvania
 *
 * Permission to use, copy, and distribute for non-commercial purposes,
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * The software may be modified for your own purposes, but modified versions
 * may not be distributed.
 *
 * This software is provided "as is" without any express or implied warranty.
 */


#include "xv.h"

#define MAXUNDO 8

#define BUTTW   80
#define BUTTW2 100
#define BUTTH   19

#define LINESTR "Lines"
#define CURVSTR "Spline"
#define HSVSTR  "HSV Mode"
#define RGBSTR  "RGB Mode"


XPoint presets[4][NUMHANDS];
XPoint undo[MAXUNDO][NUMHANDS];
XPoint defgam[NUMHANDS];
static int firsttime=1;


#ifdef __STDC__
static void DrawGVals(void);
static void DoGamCommand(int);
static void DoGammaApply(int);
static void SaveUndoState(void);
static void HSVgamma(void);
static void spline(int *, int *, int, float *);
#else
static void DoGamCommand(), DoGammaApply(), HSVgamma(), spline();
static void DrawGVals(), SaveUndoState();
#endif


/***************************************************/
void CreateGam(geom)
char *geom;
{
  int i,ptop;

  gamW = CreateWindow("xv gamma", geom, GAMWIDE,GAMHIGH,infofg,infobg);
  if (!gamW) FatalError("can't create gamma window!");

  graphW = XCreateSimpleWindow(theDisp,gamW, 10,40, 256,256, 1, infofg,infobg);
  if (!graphW) FatalError("can't create graph window!");
  XSelectInput(theDisp, graphW, ExposureMask | ButtonPressMask);

  /* call CreateCtrl first to create grayTile */
  XSetWindowBackgroundPixmap(theDisp, gamW, grayTile);

  BTCreate(&gbut[G_BUP_BR], gamW, 276, 40,
	   BUTTW, BUTTH, "Brighter", infofg, infobg);
  BTCreate(&gbut[G_BDN_BR], gamW, 276, 40 + BUTTH + 4,
	   BUTTW, BUTTH, "Dimmer", infofg, infobg);
  BTCreate(&gbut[G_BUP_CN], gamW, 276, 40 + 2*BUTTH + 8 + 16,
	   BUTTW, BUTTH, "Sharper", infofg, infobg);
  BTCreate(&gbut[G_BDN_CN], gamW, 276, 40 + 3*BUTTH+12 + 16,
	   BUTTW, BUTTH, "Duller", infofg, infobg);

  BTCreate(&gbut[G_BHSVRGB], gamW, 276, 40 + 4*BUTTH+16+32,
	   BUTTW, BUTTH, HSVSTR, infofg, infobg);

  BTCreate(&gbut[G_B1], gamW, 276, 40 + 5*BUTTH+20+48,
	   17, BUTTH, "1", infofg, infobg);
  BTCreate(&gbut[G_B2], gamW, 276 + ((BUTTW-17)*1)/3, 40 + 5*BUTTH+20+48,
	   17, BUTTH, "2", infofg, infobg);
  BTCreate(&gbut[G_B3], gamW, 276 + ((BUTTW-17)*2)/3, 40 + 5*BUTTH+20+48,
	   17, BUTTH, "3", infofg, infobg);
  BTCreate(&gbut[G_B4], gamW, 276 + ((BUTTW-17)*3/3), 40 + 5*BUTTH+20+48,
	   17, BUTTH, "4", infofg, infobg);

  BTCreate(&gbut[G_BSET], gamW, 276, 40 + 6*BUTTH+24+48,
	   38, BUTTH, "Set", infofg, infobg);
  gbut[G_BSET].toggle = 1;

  BTCreate(&gbut[G_BUNDO], gamW, 276 + 42, 40 + 6*BUTTH+24+48,
	   38, BUTTH, "Undo", infofg, infobg);

  ptop = GAMHIGH - (2*BUTTH + 3*8);
  BTCreate(&gbut[G_BAPPLY], gamW, 10, ptop+8, BUTTW2, BUTTH, 
	   "Apply", infofg, infobg);
  BTCreate(&gbut[G_BNOGAM],  gamW, 10, ptop + BUTTH + 2*8, BUTTW2, BUTTH,
	   "No Gamma", infofg, infobg);

  BTCreate(&gbut[G_BRESET], gamW, 10+(GAMWIDE-20-BUTTW2)/2, ptop+8, 
	   BUTTW2, BUTTH, "Linear", infofg, infobg);
  BTCreate(&gbut[G_BDEF],  gamW, 10+(GAMWIDE-20-BUTTW2)/2, ptop + BUTTH + 2*8,
	   BUTTW2, BUTTH, "Default", infofg, infobg);

  BTCreate(&gbut[G_BGTYPE], gamW, 10+(2*(GAMWIDE-20-BUTTW2))/2, ptop+8, 
	   BUTTW2, BUTTH, CURVSTR, infofg, infobg);
  BTCreate(&gbut[G_BCLOSE],gamW, 10+(2*(GAMWIDE-20-BUTTW2))/2, ptop+BUTTH+2*8,
	   BUTTW2, BUTTH, "Close", infofg, infobg);

  XMapSubwindows(theDisp, gamW);

  /* fill up the undo stack */
  for (i=0; i<MAXUNDO; i++)
    memcpy(undo[i], ghand, sizeof(ghand));
}
  

/***************************************************/
void GamBox(vis)
int vis;
{
  if (vis) XMapRaised(theDisp, gamW);
  else     XUnmapWindow(theDisp, gamW);

  gamUp = vis;
}


/***************************************************/
void RedrawGam(x,y,w,h)
int x,y,w,h;
{
  int i;
  XRectangle xr;

  xr.x = x;  xr.y = y;  xr.width = w;  xr.height = h;
  XSetClipRectangles(theDisp, theGC, 0,0, &xr, 1, Unsorted);

  XSetBackground(theDisp, theGC, infobg);

  XSetForeground(theDisp, theGC, infobg);
  XFillRectangle(theDisp,gamW, theGC, 10+1,10+1,256,19);
  XSetForeground(theDisp,theGC,infofg);
  XDrawRectangle(theDisp,gamW, theGC, 10,10,257,20);
  DrawGVals();

  for (i=0; i<G_NBUTTS; i++)
    BTRedraw(&gbut[i]);

  XSetClipMask(theDisp, theGC, None);
}


/***************************************************/
static void DrawGVals()
{
  int  w;
  char foo[40];

  XSetForeground(theDisp, theGC, infofg);
  XSetBackground(theDisp, theGC, infobg);

  sprintf(foo,"%3d   %3d %3d   %3d %3d   %3d", ghand[0].y, ghand[1].x,
	  ghand[1].y, ghand[2].x, ghand[2].y, ghand[3].y);

  w = XTextWidth(monofinfo, foo, strlen(foo));
  XSetFont(theDisp, theGC, monofont);
  XDrawImageString(theDisp, gamW, theGC, 10 + (256-w)/2, 
	      11+(20-(monofinfo->ascent+monofinfo->descent))/2
	      + monofinfo->ascent, foo, strlen(foo));
  XSetFont(theDisp, theGC, mfont);
}

/***************************************************/
void RedrawGraph(x,y,w,h)
int x,y,w,h;
{
  int i;
  XRectangle xr;
  XPoint  pts[256], *pt;

  xr.x = x;  xr.y = y;  xr.width = w;  xr.height = h;
  XSetClipRectangles(theDisp, theGC, 0,0, &xr, 1, Unsorted);

  XSetForeground(theDisp, theGC, infofg);
  XSetBackground(theDisp, theGC, infobg);

  XClearArea(theDisp, graphW, x,y,w,h, False);

  for (i=0, pt=pts; i<256; i++,pt++) {
    pt->x = i;  pt->y = 255-gamcr[i];
  }
  XDrawLines(theDisp, graphW, theGC, pts, 256, CoordModeOrigin);

  for (i=0; i<NUMHANDS; i++) {
    XSetForeground(theDisp, theGC, infobg);
    XFillRectangle(theDisp, graphW, theGC, 
		   ghand[i].x-2, 255-ghand[i].y-2,5,5);
    XSetForeground(theDisp,theGC,infofg);
    XDrawRectangle(theDisp, graphW, theGC,
		   ghand[i].x-3, 255-ghand[i].y-3,6,6);
    XDrawPoint(theDisp, graphW, theGC, ghand[i].x, 255-ghand[i].y);
  }

  XSetClipMask(theDisp, theGC, None);
}


/***************************************************/
void ClickGam(x,y)
int x,y;
{
  int i;
  BUTT *bp;

  for (i=0; i<G_NBUTTS; i++) {
    bp = &gbut[i];
    if (PTINRECT(x, y, bp->x, bp->y, bp->w, bp->h)) break;
  }

  /* if 'Set' is lit, and we didn't click 'set' or '1'..'4', turn it off */
  if (i!=G_BSET && i!=G_B1 && i!=G_B2 && i!=G_B3 && i!=G_B4 
      && gbut[G_BSET].lit) 
    { gbut[G_BSET].lit = 0;  BTRedraw(&gbut[G_BSET]); }

  if (i==G_NBUTTS) return;  /* didn't find one */

  if (BTTrack(bp)) DoGamCommand(i);
}



/**********************************************/
void TrackGraph(mx,my)
int mx,my;
{
  /* called when we've gotten a click inside graph window */

  Window       rW, cW;
  int          x, y, rx, ry;
  unsigned int mask;
  int          h, vertonly, offx, offy;

  if (gbut[G_BSET].lit) 
    { gbut[G_BSET].lit = 0;  BTRedraw(&gbut[G_BSET]); }

  my = 255 - my;   /* flip y axis */

  /* see if x,y is within any of the handles */
  for (h=0; h<NUMHANDS; h++) {
    if (PTINRECT(mx, my, ghand[h].x-5, ghand[h].y-5, 11,11)) break;
  }
  if (h==NUMHANDS) return;   /* didn't find one */

  offx = ghand[h].x - mx;  offy = ghand[h].y - my;

  vertonly = (h==0 || h==(NUMHANDS-1));

  SaveUndoState();

  while (XQueryPointer(theDisp,graphW,&rW,&cW,&rx,&ry,&x,&y,&mask)) {
    if (!(mask & Button1Mask)) break;    /* button released */
    y = 255 - y;   /* flip y-axis */

    /* keep this handle between other handles (x-axis) if nec. */
    if (!vertonly) {
      if (x+offx <= ghand[h-1].x) x = (ghand[h-1].x+1)-offx;
      if (x+offx >= ghand[h+1].x) x = (ghand[h+1].x-1)-offx;
    }
    else x = mx;   /* if vert only, ignore sidewards motion */

    if (mx != x || my != y) {   /* this handle has moved... */
      ghand[h].x = x+offx;  ghand[h].y = y+offy;
      RANGE(ghand[h].y,0,255);
      GenerateGamma();
      RedrawGraph(0,0,256,256);  /* open to better ideas... */
    }
    mx = x;  my = y;
  }

  if ((rwcolor && rwthistime)) DoGammaApply(1);
}


/*********************/
void GenerateGamma()
{
  /* this function generates a gamma correction curve (gamcr)

     This function generates a 4 point spline curve to be used as a 
     non-linear grey 'colormap'.  Two of the points are nailed down at 0,0
     and 255,255, and can't be changed.  You specify the other two.  If
     you specify points on the line (0,0 - 255,255), you'll get the normal
     linear reponse curve.  If you specify points of 50,0 and 200,255, you'll
     get grey values of 0-50 to map to black (0), and grey values of 200-255
     to map to white (255) (roughly).  Values between 50 and 200 will cover
     the output range 0-255.  The reponse curve will be slightly 's' shaped. */

  int i,j,k;
  static int x[NUMHANDS], y[NUMHANDS];
  float yf[NUMHANDS];
  static float splint();

  /* do some idiot-proofing (x-coords must be monotonically increasing)  */

  for (i=0; i<4; i++) { 
    RANGE(ghand[i].x, 0, 255); 
    RANGE(ghand[i].y, 0, 255);
  }

  ghand[0].x = 0;  ghand[3].x = 255;
  if (ghand[1].x < 1)  ghand[1].x = 1;
  if (ghand[1].x >253) ghand[1].x = 253;
  if (ghand[2].x < ghand[1].x) ghand[2].x = ghand[1].x + 1;
  if (ghand[2].x >254) ghand[2].x = 254;

  if (firsttime) {   /* if this is the first 'generate' save as 'default' */
    memcpy(defgam, ghand, sizeof(ghand));
    firsttime=0;
  }

  DrawGVals();

  if (!strcmp(gbut[G_BGTYPE].str,LINESTR)) {
    /* do linear interpolation */
    for (i=0; i<NUMHANDS-1; i++) {
      for (j=ghand[i].x,k=0; j<=ghand[i+1].x; j++,k++) {
	gamcr[j] = ghand[i].y + (k * (ghand[i+1].y - ghand[i].y)) / 
	                        (ghand[i+1].x - ghand[i].x);
      }
    }
    return;
  }



  for (i=0; i<NUMHANDS; i++) { x[i] = ghand[i].x;  y[i] = ghand[i].y; }
  spline(x, y, NUMHANDS, yf);
  
  for (i=0; i<256; i++) {
    j = (int) splint(x, y, yf, NUMHANDS, (float) i);
    if (j<0) j=0;
    else if (j>255) j=255;
    gamcr[i] = j;
  }
}


/*********************/
void GenerateFSGamma()
{
  /* this function generates the Floyd-Steinberg gamma curve (fsgamcr)

     This function generates a 4 point spline curve to be used as a 
     non-linear grey 'colormap'.  Two of the points are nailed down at 0,0
     and 255,255, and can't be changed.  You specify the other two.  If
     you specify points on the line (0,0 - 255,255), you'll get the normal
     linear reponse curve.  If you specify points of 50,0 and 200,255, you'll
     get grey values of 0-50 to map to black (0), and grey values of 200-255
     to map to white (255) (roughly).  Values between 50 and 200 will cover
     the output range 0-255.  The reponse curve will be slightly 's' shaped. */

  int i,j;
  static int x[4] = {0,32,224,255};
  static int y[4] = {0, 0,255,255};
  float yf[4];
  static float splint();

  spline(x, y, 4, yf);
  
  for (i=0; i<256; i++) {
    j = (int) splint(x, y, yf, 4, (float) i);
    if (j<0) j=0;
    else if (j>255) j=255;
    fsgamcr[i] = j;
  }
}


/*********************/
static void spline(x,y,n,y2)
     int *x, *y, n;
     float *y2;
{
  /* given arrays of data points x[0..n-1] and y[0..n-1], computes the
     values of the second derivative at each of the data points
     y2[0..n-1] for use in the splint function */

  int i,k;
  float p,qn,sig,un,u[NUMHANDS];

  y2[0] = u[0] = 0.0;

  for (i=1; i<n-1; i++) {
    sig = ((float) x[i]-x[i-1]) / ((float) x[i+1] - x[i-1]);
    p = sig * y2[i-1] + 2.0;
    y2[i] = (sig-1.0) / p;
    u[i] = (((float) y[i+1]-y[i]) / (x[i+1]-x[i])) - 
           (((float) y[i]-y[i-1]) / (x[i]-x[i-1]));
    u[i] = (6.0 * u[i]/(x[i+1]-x[i-1]) - sig*u[i-1]) / p;
  }
  qn = un = 0.0;

  y2[n-1] = (un-qn*u[n-2]) / (qn*y2[n-2]+1.0);
  for (k=n-2; k>=0; k--)
    y2[k] = y2[k]*y2[k+1]+u[k];
}



/*********************/
static float splint(xa,ya,y2a,n,x)
float y2a[],x;
int n,xa[],ya[];
{
  int klo,khi,k;
  float h,b,a;

  klo = 0;
  khi = n-1;
  while (khi-klo > 1) {
    k = (khi+klo) >> 1;
    if (xa[k] > x) khi = k;
    else klo = k;
  }
  h = xa[khi] - xa[klo];
  if (h==0.0) FatalError("bad xvalues in splint\n");
  a = (xa[khi]-x)/h;
  b = (x-xa[klo])/h;
  return (a*ya[klo] + b*ya[khi] + ((a*a*a-a)*y2a[klo] +(b*b*b-b)*y2a[khi]) 
	  * (h*h) / 6.0);
}
    

/*********************/
static void DoGamCommand(cmd)
int cmd;
{
  int i,p, app;

  app = 1;  p=0;

  switch (cmd) {
  case G_BAPPLY: DoGammaApply(1);  app = 0;  break;
  case G_BNOGAM: DoGammaApply(0);  app = 0;  break;

  case G_BRESET: SaveUndoState();
                 ghand[0].x =   0; ghand[0].y =   0;
                 ghand[1].x =  64; ghand[1].y =  64;
                 ghand[2].x = 192; ghand[2].y = 192;
                 ghand[3].x = 255; ghand[3].y = 255;

                 GenerateGamma();
                 RedrawGraph(0,0,256,256);
                 break;



  case G_BDEF:   SaveUndoState();
                 memcpy(ghand, defgam, sizeof(ghand));

                 GenerateGamma();
                 RedrawGraph(0,0,256,256);
                 break;

  case G_BGTYPE: if (!strcmp(gbut[G_BGTYPE].str, LINESTR))
                      gbut[G_BGTYPE].str = CURVSTR;
                 else gbut[G_BGTYPE].str = LINESTR;
                 BTRedraw(&gbut[G_BGTYPE]);
                 GenerateGamma();
                 RedrawGraph(0,0,256,256);
                 break;

  case G_BCLOSE: GamBox(0);  app=0;  break;

  case G_BUP_BR: SaveUndoState();
                 for (i=0; i<NUMHANDS; i++) {
                   ghand[i].y += 10;
		   RANGE(ghand[i].y,0,255);
		 }
                 
                 GenerateGamma();
                 RedrawGraph(0,0,256,256);
                 break;

  case G_BDN_BR: SaveUndoState();
                 for (i=0; i<NUMHANDS; i++) {
                   ghand[i].y -= 10;
		   RANGE(ghand[i].y,0,255);
		 }
                 
                 GenerateGamma();
                 RedrawGraph(0,0,256,256);
                 break;

  case G_BUP_CN: SaveUndoState();
                 ghand[0].y -= 10;  
                 ghand[1].y -= 10;  ghand[1].x += 5;
                 ghand[2].y += 10;  ghand[2].x -= 5;
                 ghand[3].y += 10;
                 if (ghand[1].x > 127) ghand[1].x = 127;
                 if (ghand[2].x <= ghand[1].x) ghand[2].x = ghand[1].x+1;
                 for (i=0; i<NUMHANDS; i++) RANGE(ghand[i].y,0,255);
                 
                 GenerateGamma();
                 RedrawGraph(0,0,256,256);
                 break;

  case G_BDN_CN: SaveUndoState();
                 if (abs(ghand[0].y - 127) < 10) ghand[0].y = 127;
                 else ghand[0].y += ((ghand[0].y>127) ? -10 : 10);

                 if (abs(ghand[1].y - 127) < 10) ghand[1].y = 127;
                 else ghand[1].y += ((ghand[1].y>127) ? -10 : 10);

                 if (abs(ghand[2].y - 127) < 10) ghand[2].y = 127;
                 else ghand[2].y += ((ghand[2].y>127) ? -10 : 10);

                 if (abs(ghand[3].y - 127) < 10) ghand[3].y = 127;
                 else ghand[3].y += ((ghand[3].y>127) ? -10 : 10);

                 for (i=0; i<NUMHANDS; i++) RANGE(ghand[i].y,0,255);
                 
                 GenerateGamma();
                 RedrawGraph(0,0,256,256);
                 break;

  case G_BHSVRGB:if (!strcmp(gbut[G_BHSVRGB].str, HSVSTR))
                      gbut[G_BHSVRGB].str = RGBSTR;
                 else gbut[G_BHSVRGB].str = HSVSTR;
                 BTRedraw(&gbut[G_BHSVRGB]);
                 break;

  case G_B1:
  case G_B2:
  case G_B3:
  case G_B4:     if      (cmd==G_B1) p = 0;
                 else if (cmd==G_B2) p = 1;
                 else if (cmd==G_B3) p = 2;
                 else if (cmd==G_B4) p = 3;
                
                 if (gbut[G_BSET].lit) {
                   memcpy(presets[p], ghand, sizeof(ghand));
		   gbut[G_BSET].lit = 0;
		   BTRedraw(&gbut[G_BSET]);
		   app=0;
		 }
                 else {
                   SaveUndoState();
		   memcpy(ghand, presets[p], sizeof(ghand));
		   GenerateGamma();
                   RedrawGraph(0,0,256,256);
		 }
                 break;

  case G_BSET:   app=0;  break;

#ifdef __COHERENT__
  case G_BUNDO:  memcpy(ghand, undo[0], sizeof(ghand));
                 memcpy(undo[0], undo[1], (MAXUNDO-1)*sizeof(ghand));
#else
  case G_BUNDO:  bcopy(undo[0], ghand, sizeof(ghand));
                 bcopy(undo[1], undo[0], (MAXUNDO-1)*sizeof(ghand));
#endif
                 GenerateGamma();
                 RedrawGraph(0,0,256,256);
                 break;
  }
  
  if (app && rwcolor && rwthistime) DoGammaApply(1);
}


/*********************/
static void DoGammaApply(app)
int app;
{
  int i,j;

  DoMonoAndRV();

  if (app) GammifyColors();

  /* special case: if using R/W color, just modify the colors and leave */
  if (rwcolor && rwthistime) {
    XColor ctab[256];
    for (i=0; i<nfcols; i++) {
      j = fc2pcol[i];
      ctab[i].pixel = freecols[i];
      ctab[i].red   = r[j]<<8;
      ctab[i].green = g[j]<<8;
      ctab[i].blue  = b[j]<<8;
      ctab[i].flags = DoRed | DoGreen | DoBlue;
      XStoreColor(theDisp, LocalCmap ? LocalCmap : theCmap, &ctab[i]);
    }
    return;
  }
    

  if (useroot && (LocalCmap || nfcols)) {
    XSetWindowBackgroundPixmap(theDisp, rootW, None);
  }


  if (LocalCmap) {
    XClearWindow(theDisp,mainW);
    XFreeColormap(theDisp,LocalCmap);
    LocalCmap = 0;
  }
  else {
    if (!brokeFreeCols) {
      for (i=0; i<nfcols; i++) 
	XFreeColors(theDisp, theCmap, &freecols[i], 1, 0L);
    }
    else {
      for (i=0; i<nfcols; i++) {
	int j;
	for (j=0; j<i; j++) {
	  if (freecols[i] == freecols[j])   /* already been freed once */
	    break;
	}
	if (j==i)      /* wasn't found in already-freed list */
	  XFreeColors(theDisp, theCmap, &freecols[i], 1, 0L);
      }
    }

    if (nfcols) {
      XFlush(theDisp);  /* just to make sure they're all freed right now... */

      /* clear window if we're on a non-static display */
      if (theVisual->class & 1 && ncols>0) XClearWindow(theDisp,mainW);
    }
  }


  AllocColors();
  CreateXImage();

  if (useroot) MakeRootPic();
          else DrawWindow(0,0,eWIDE,eHIGH);
  if (but[BCROP].active) InvCropRect();
  SetCursors(-1);
}


/*********************/
void GammifyColors()
{
  int i;

  if (!strcmp(gbut[G_BHSVRGB].str,RGBSTR)) {
    for (i=0; i<numcols; i++) {
      r[i] = gamcr[r[i]];
      g[i] = gamcr[g[i]];
      b[i] = gamcr[b[i]];
    }
  }
  else HSVgamma();
}


/*********************/
void SetGPreset(n,y0,x1,y1,x2,y2,y3)
int n,y0,x1,y1,x2,y2,y3;
{
  if (n<0 || n>3) return;
  
  presets[n][0].x = 0;     presets[n][0].y = y0;
  presets[n][1].x = x1;    presets[n][1].y = y1;
  presets[n][2].x = x2;    presets[n][2].y = y2;
  presets[n][3].x = 255;   presets[n][3].y = y3;
}

  
#define NOHUE -1

/*********************/
static void HSVgamma()
{
  int    i, vi, j;
  double rd, gd, bd, h, s, v, max, min, del, rc, gc, bc;
  double f, p, q, t;

  for (i=0; i<numcols; i++) {
    /* convert RGB to HSV */
    rd = r[i] / 255.0;            /* rd,gd,bd range 0-1 instead of 0-255 */
    gd = g[i] / 255.0;
    bd = b[i] / 255.0;

    /* compute maximum of rd,gd,bd */
    if (rd>=gd) { if (rd>=bd) max = rd;  else max = bd; }
           else { if (gd>=bd) max = gd;  else max = bd; }

    /* compute minimum of rd,gd,bd */
    if (rd<=gd) { if (rd<=bd) min = rd;  else min = bd; }
           else { if (gd<=bd) min = gd;  else min = bd; }

    del = max - min;
    v = max;
    if (max != 0.0) s = (del) / max;
               else s = 0.0;

    h = NOHUE;
    if (s != 0.0) {
      rc = (max - rd) / del;
      gc = (max - gd) / del;
      bc = (max - bd) / del;

      if      (rd==max) h = bc - gc;
      else if (gd==max) h = 2 + rc - bc;
      else if (bd==max) h = 4 + gc - rc;

      h = h * 60;
      if (h<0) h += 360;
    }

    /* map near-black to black to avoid weird effects */
    if (v <= .0625) s = 0.0;

    /* apply gamcr[] function to 'v' (the intensity) */
    vi = (int) floor(v * 255);
    v = gamcr[vi] / 255.0;

    /* convert HSV back to RGB */
    if (s==0.0) { rd = v;  gd = v;  bd = v; }
    else {
      if (h==360.0) h = 0.0;
      h = h / 60.0;
      j = (int) floor(h);
      f = h - j;
      p = v * (1-s);
      q = v * (1 - (s*f));
      t = v * (1 - (s*(1 - f)));

      switch (j) {
      case 0:  rd = v;  gd = t;  bd = p;  break;
      case 1:  rd = q;  gd = v;  bd = p;  break;
      case 2:  rd = p;  gd = v;  bd = t;  break;
      case 3:  rd = p;  gd = q;  bd = v;  break;
      case 4:  rd = t;  gd = p;  bd = v;  break;
      case 5:  rd = v;  gd = p;  bd = q;  break;
      }
    }

    r[i] = (int) floor(rd * 255);
    g[i] = (int) floor(gd * 255);
    b[i] = (int) floor(bd * 255);
  }
}


  

/*********************/
static void SaveUndoState()
{
  /* pushes the current 'ghand' values onto the undo stack */

  /* use bcopy 'cause it knows how to do overlapped copies */
#ifdef __COHERENT__
  memcpy(undo[1], undo[0], (MAXUNDO-1) * sizeof(ghand));
#else
  bcopy(undo[0], undo[1], (MAXUNDO-1) * sizeof(ghand));
#endif
  /* and put 'ghand' on the top of the stack */
#ifdef __COHERENT__
  memcpy(undo[0], ghand, sizeof(ghand));
#else
  bcopy(ghand, undo[0], sizeof(ghand));
#endif
}
