/* 
 * xvdir.c - Directory changin', file i/o dialog box
 *
 * callable functions:
 *
 *   CreateDirW(geom,bwidth)-  creates the dirW window.  Doesn't map it.
 *   DirBox(vis)            -  random processing based on value of 'vis'
 *                             maps/unmaps window, etc.
 *   RedrawDirW(x,y,w,h)    -  called by 'expose' events
 *   ClickDirW()            -  handles mouse clicks in DirW
 *   LoadCurrentDirectory() -  loads up current dir information for dirW
 *   DoSave()               -  calls appropriate save routines
 *   SetDirFName()          -  sets the 'save-as' filename 
 */

/*
 * Copyright 1989, 1990 by the University of Pennsylvania
 *
 * Permission to use, copy, and distribute for non-commercial purposes,
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * The software may be modified for your own purposes, but modified versions
 * may not be distributed.
 *
 * This software is provided "as is" without any express or implied warranty.
 */


#define NEEDSDIR
#include "xv.h"

#define NLINES 9                   /* # of lines in list control (keep odd) */
#define LISTW  200

#define BUTTW   60
#define BUTTH   19
#define DDWIDE  LISTW-80+15

#define MAXDEEP 30    /* maximum number of directories in cwd path */
#define MAXFNLEN 40   /* max length of filename being entered */

#define DEFFILENAME ""   /* default filename filled in when program starts */

#ifdef __STDC__
static void RedrawDList(void);
static int  dnamcmp(char **, char **);
#else
static void RedrawDList();
static int  dnamcmp();
#endif

static int    listh;
static char  *dirnames[MAXNAMES];
static int    numdirnames = 0, ndirs = 0;
static char   path[MAXPATHLEN+1];
static char  *dirs[MAXDEEP];            /* list of directory names */
static char  *lastdir;                  /* name of the directory we're in */
static char   filename[MAXFNLEN];       /* filename being entered */
static RBUTT *formatRB, *colorRB, *sizeRB;


/***************************************************/
void CreateDirW(geom)
char *geom;
{
  int y;

  listh = LINEHIGH * NLINES;

  dirW = CreateWindow("xv save dialog",geom,DIRWIDE, DIRHIGH, infofg, infobg);
  if (!dirW) FatalError("couldn't create 'save' window!");

  /* create doo-wah's */
  ddirW = XCreateSimpleWindow(theDisp, dirW, 40, 5, DDWIDE, LINEHIGH,
			      2, infofg, infobg);
  if (!ddirW) FatalError("can't create path window");
  XSelectInput(theDisp, ddirW, ExposureMask | ButtonPressMask);

  dnamW = XCreateSimpleWindow(theDisp, dirW, 80, listh+75-ASCENT-4, 
			      200, LINEHIGH+4, 1, infofg, infobg);
  if (!dnamW) FatalError("can't create name window");
  XSelectInput(theDisp, dnamW, ExposureMask);


  LSCreate(&dList, dirW, 10, 14+LINEHIGH, LISTW, listh, NLINES,
	   dirnames, numdirnames,
	   infofg, infobg, RedrawDList, 1, 0);

  BTCreate(&dbut[S_BOPEN], dirW, 233, dList.y-9+listh/5, 60, BUTTH, 
	   "Open", infofg, infobg);
  BTCreate(&dbut[S_BSAVE], dirW, 233, dList.y-9+(listh*2)/5, 60, BUTTH, 
	   "Save", infofg, infobg);
  BTCreate(&dbut[S_BCANC], dirW, 233, dList.y-9+(listh*3)/5, 60, BUTTH, 
	   "Cancel", infofg, infobg);
  BTCreate(&dbut[S_BQUIT], dirW, 233, dList.y-9+(listh*4)/5, 60, BUTTH, 
	   "Quit", infofg, infobg);

  y = listh + 110;
  formatRB = RBCreate(NULL, dirW, 26, y, "GIF", infofg, infobg);
  RBCreate(formatRB, dirW, 26, y+18, "PM", infofg, infobg);
  RBCreate(formatRB, dirW, 26, y+36, "PBM (raw)", infofg, infobg);
  RBCreate(formatRB, dirW, 26, y+54, "PBM (ascii)", infofg, infobg);
  RBCreate(formatRB, dirW, 26, y+72, "X11 Bitmap", infofg, infobg);

  colorRB = RBCreate(NULL, dirW, DIRWIDE/2, y, "Full Color", infofg, infobg);
  RBCreate(colorRB, dirW, DIRWIDE/2, y+18, "Greyscale", infofg, infobg);
  RBCreate(colorRB, dirW, DIRWIDE/2, y+36, "B/W Dithered", infofg, infobg);

  y = y + 115;
  sizeRB = RBCreate(NULL, dirW, 26, y, "Normal Size", infofg, infobg);
  RBCreate(sizeRB, dirW, 26, y+18, "At Current Expansion", infofg, infobg);

  SetDirFName(DEFFILENAME);

  LoadCurrentDirectory();

  XMapSubwindows(theDisp, dirW);
}
  

/***************************************************/
void DirBox(vis)
int vis;
{
  if (vis) XMapRaised(theDisp, dirW);
  else     XUnmapWindow(theDisp, dirW);

  BTSetActive(&but[BSAVE], !vis);

  dirUp = vis;
}


/***************************************************/
void RedrawDirW(x,y,w,h)
int x,y,w,h;
{
  int  i,ypos;
  char foo[30];
  XRectangle xr;

  xr.x = x;  xr.y = y;  xr.width = w;  xr.height = h;
  XSetClipRectangles(theDisp, theGC, 0,0, &xr, 1, Unsorted);

  if (dList.nstr==1) strcpy(foo,"1 file");
                else sprintf(foo,"%d files",dList.nstr);

  ypos = dList.y + dList.h + 5 + ASCENT;
  XSetForeground(theDisp, theGC, infobg);
  XFillRectangle(theDisp, dirW, theGC, 10, ypos-ASCENT, DIRWIDE, CHIGH);
  XSetForeground(theDisp, theGC, infofg);
  XDrawString(theDisp, dirW, theGC, 10, ypos, foo, strlen(foo));

  XDrawString(theDisp, dirW, theGC, 10, dList.h+75, "File name:",10);

  for (i=0; i<S_NBUTTS; i++) BTRedraw(&dbut[i]);

  RBRedraw(formatRB, -1);
  RBRedraw(colorRB, -1);
  RBRedraw(sizeRB, -1);

  ULineString(dirW, "Format", formatRB->x-16, formatRB->y-3-DESCENT);
  ULineString(dirW, "Colors", colorRB->x-16,  colorRB->y-3-DESCENT);
  ULineString(dirW, "Size",   sizeRB->x-16,   sizeRB->y-3-DESCENT);

  XSetClipMask(theDisp, theGC, None);
}


/***************************************************/
void RedrawDDirW()
{
  XSetForeground(theDisp, theGC, infofg);
  XSetBackground(theDisp, theGC, infobg);

  XClearWindow(theDisp, ddirW);
  XDrawString(theDisp, ddirW, theGC, 3, ASCENT + 1,
              lastdir, strlen(lastdir));
}


/***************************************************/
int ClickDirW(x,y)
int x,y;
{
  BUTT  *bp;
  int    bnum;


  /* check the RBUTTS first, since they don't DO anything */
  if ( (bnum=RBClick(formatRB, x,y)) >= 0) { 
    RBTrack(formatRB, bnum);
    if (RBWhich(formatRB)==4) {  /* turn off FULLCOLOR + GRAYSCALE */
      RBSetActive(colorRB,0,0);
      RBSetActive(colorRB,1,0);
      RBSelect(colorRB,2);
    }
    else {                       /* turn on FULLCOLOR + GRAYSCALE */
      RBSetActive(colorRB,0,1);
      RBSetActive(colorRB,1,1);
    }
    return -1;
  }

  if ( (bnum=RBClick(colorRB, x,y)) >= 0) 
    { RBTrack(colorRB, bnum);  return -1; }

  if ( (bnum=RBClick(sizeRB, x,y)) >= 0) 
    { RBTrack(sizeRB, bnum);  return -1; }

  for (bnum=0; bnum<S_NBUTTS; bnum++) {
    bp = &dbut[bnum];
    if (PTINRECT(x, y, bp->x, bp->y, bp->w, bp->h)) break;
  }

  if (bnum<S_NBUTTS) {   /* found one */
    if (BTTrack(bp)) return (bnum);
  }

  return -1;
}


/***************************************************/
void SelectDir(n)
int n;
{
  int pend;

  /* called when entry #n in the dir list was selected/double-clicked */

  if (dList.str[n][0] == C_DIR || 
      dList.str[n][0] == C_LNK) {  /* it's cool, it's (possibly) a directory */
    pend = strlen(path);
    strcat(path,dList.str[n]+1);   /* add to pathname */
    if (chdir(path)) {
      fprintf(stderr,"unable to cd to '%s'\n",path);
      path[pend] = '\0';           /* undo path modification */
    }
    else 
      LoadCurrentDirectory();
  }

  else {  /* not a directory */
    /* copy the clicked-on filename into the 'save-as' filename */
    SetDirFName(dList.str[n]+1);
  }
}


/***************************************************/
void TrackDDirW(x,y)
int x,y;
{
  Window        menuW, rW, cW;
  int           rx, ry, i,j, sel, lastsel;
  unsigned int  mask;

  XSetForeground(theDisp, theGC, infofg);
  XSetBackground(theDisp, theGC, infobg);

  menuW = XCreateSimpleWindow(theDisp, dirW, 40, 5, DDWIDE, ndirs*LINEHIGH,
			      2, infofg, infobg);
  if (!menuW) FatalError("can't create path window");

  XMapRaised(theDisp, menuW);

  for (i=ndirs-1, j=0; i>=0; i--,j++)
    XDrawString(theDisp, menuW, theGC, 3, j*LINEHIGH + ASCENT + 1,
		dirs[i], dirs[i+1]-dirs[i]);

  XFlush(theDisp);
  XSetFunction(theDisp, theGC, GXinvert);
  XSetPlaneMask(theDisp, theGC, infofg ^ infobg);
  lastsel = -1;  sel = 0;

  while (XQueryPointer(theDisp, menuW, &rW, &cW, &rx, &ry, &x, &y, &mask)) {
    if (!(mask & Button1Mask)) break;

    /* see if mouse has left window */

    sel = y / LINEHIGH;
    if (sel>=ndirs) sel = ndirs-1;
    if (sel<0) sel = 0;
    if (sel != lastsel) {
      XFillRectangle(theDisp,menuW,theGC,0,lastsel*LINEHIGH,DDWIDE,LINEHIGH);
      XFillRectangle(theDisp,menuW,theGC,0,sel*LINEHIGH,DDWIDE,LINEHIGH);
      lastsel = sel;
    }
  }

  XSetFunction(theDisp, theGC, GXcopy);
  XSetPlaneMask(theDisp, theGC, AllPlanes);
  XDestroyWindow(theDisp, menuW);

  if (sel!=0) { /* changed directories */
    /* end 'path' by changing trailing '/' (of dir name) to a '\0' */
    *(dirs[(ndirs-1)-sel + 1] - 1) = '\0';

    /* special case:  if cd to '/', fix path (it's currently "") */
    if (path[0] == '\0') strcpy(path,"/");

    if (chdir(path)) {
      fprintf(stderr,"unable to cd to '%s'\n",path);
    }
    else 
      LoadCurrentDirectory();   
  }
}


/***************************************************/
static void RedrawDList()
{
  LSRedraw(&dList);
}


/***************************************************/
void LoadCurrentDirectory()
{
  DIR           *dirp;
#ifdef DIRENT
  struct dirent *dp;
#else
  struct direct *dp;
#endif
  int            i, ftype;
  struct stat    st;
  char          *dbeg, *dend;

  /* get rid of previous file names */
  for (i=0; i<numdirnames; i++) free(dirnames[i]);

  numdirnames = 0;

#if defined(SYSV) || defined(__COHERENT__)
  getcwd(path, sizeof(path));
#else
  getwd(path);
#endif
  if (path[strlen(path)-1] != '/')
    strcat(path,"/");   /* tack on a trailing '/' to make path consistent */

  /* path will be something like: "/u3/bradley/src/weiner/whatever/" */
  /* parse path into individual directory names */
  dbeg = dend = path;
  for (i=0; i<MAXDEEP && dend; i++) {
    dend = strchr(dbeg,'/');  /* find next '/' char */
    dirs[i] = dbeg;
    dbeg = dend+1;
  }
  ndirs = i-1;

  lastdir = dirs[ndirs-1];
  RedrawDDirW();

  dirp = opendir(".");
  if (!dirp) {
    fprintf(stderr,"unable to open current directory");
    return;
  }

  i=0;
  while ( (dp = readdir(dirp)) != NULL) {
    if (strcmp(dp->d_name, ".")==0 || strcmp(dp->d_name, "..")==0) {
      /* skip over '.' and '..' */
    }
    else {
#ifdef DIRENT
#ifdef i386
      /* Not 100% sure d_reclen is correct, but it works...  MWS 10/18/90 */
      dirnames[i] = (char *) malloc(dp->d_reclen + 2); /* filetype + '\0'*/
#else
      dirnames[i] = (char *) malloc(strlen(dp->d_name) + 3);
#endif
#else
      dirnames[i] = (char *) malloc(dp->d_namlen + 2); /* +2=filetype + '\0'*/
#endif
      if (!dirnames[i]) FatalError("malloc error while reading directory");
      strcpy(dirnames[i]+1, dp->d_name);

      /* figure out what type of file the beastie is */
      dirnames[i][0] = C_REG;   /* default to normal file, if lstat fails */

#if defined(i386) || defined (SYSV) || defined(COHERENT)
      if (stat(dirnames[i]+1, &st)==0) {
#else
      if (lstat(dirnames[i]+1, &st)==0) {
#endif
	ftype = st.st_mode & S_IFMT;   /* mask off uninteresting bits */
	if      (ftype == S_IFDIR)  dirnames[i][0] = C_DIR;
	else if (ftype == S_IFCHR)  dirnames[i][0] = C_CHR;
	else if (ftype == S_IFBLK)  dirnames[i][0] = C_BLK;

#ifdef S_IFIFO
	else if (ftype == S_IFIFO)  dirnames[i][0] = C_FIFO;
#endif

#ifdef S_IFLNK
	else if (ftype == S_IFLNK)  dirnames[i][0] = C_LNK;
#endif

#ifdef S_IFSOCK
        else if (ftype == S_IFSOCK) dirnames[i][0] = C_SOCK;
#endif
      }
      else {
	/* fprintf(stderr,"problems 'stat-ing' files\n");*/
	dirnames[i][0] = C_REG;
      }
      i++;
    }
  }

  closedir(dirp);

  numdirnames = i;

  qsort((char *) dirnames, numdirnames, sizeof(char *), dnamcmp);
  LSNewData(&dList, dirnames, numdirnames);
  DirOpenActive();
  RedrawDirW(0,0,DIRWIDE,DIRHIGH);
}


/***************************************************/
static int dnamcmp(s1,s2)
char **s1, **s2;
{
  /* sort so that directories are at beginning of list */

  /* if both dirs or both not dirs, sort on name */
  if ( (**s1 == C_DIR && **s2 == C_DIR) || (**s1 != C_DIR && **s2 != C_DIR))
    return (strcmp((*s1)+1, (*s2)+1));

  else if (**s1==C_DIR) return -1;  /* s1 is first */
  else return 1;                    /* s2 is first */
}





/***************************************************/
int DirKey(c)
int c;
{
  /* got keypress in dirW.  stick on end of filename */
  int len;

  len = strlen(filename);
  
  if (c>' ' && c<'\177') {              /* printable characters */
    if (c=='/') return(-1);             /* no directories in filename */
    if (len >= MAXFNLEN-1) return(-1);  /* max length of string */
    filename[len]=c;  filename[len+1]='\0';
  }

  else if (c=='\010' || c=='\177') {    /* BS or DEL */
    if (len==0) return(-1);             /* string already empty */
    filename[len-1]='\0';
  }

  else if (c=='\025' || c=='\013') {    /* ^U or ^K clear line */
    filename[0] = '\0';
  }

  else if (c=='\012' || c=='\015') {    /* CR or LF */
    FakeButtonPress(&dbut[S_BSAVE]);
  }

  else return(-1);                      /* unhandled character */

  SetDirFName(filename);
  return(0);
}


/***************************************************/
void RedrawDNamW()
{
  int width, len;

  len = strlen(filename);
  XSetForeground(theDisp, theGC, infofg);
  XDrawString(theDisp, dnamW, theGC, 3, ASCENT+3, filename, len);
  width = StringWidth(filename);
  XDrawLine(theDisp, dnamW, theGC, 3+width+1, 3, 3+width+1, 
	    3+CHIGH);
}


/***************************************************/
void DirOpenActive()
{
  if (dList.selected>=dList.nstr || dList.selected<0) 
    BTSetActive(&dbut[S_BOPEN],0);

  else if (dList.str[dList.selected][0] == C_DIR ||
      dList.str[dList.selected][0] == C_LNK) 
    BTSetActive(&dbut[S_BOPEN],1);

  else
    BTSetActive(&dbut[S_BOPEN],0);

  XFlush(theDisp);
}



/***************************************************/
int DoSave()
{
  FILE *fp;
  byte *thepic, *bwpic;
  int   w,h,rv,i;

  /* opens file, does appropriate color pre-processing, calls save routine
     based on chosen format.  Returns '0' if successful */

  WaitCursor();

  bwpic = NULL;

  if (RBWhich(sizeRB)==1) { thepic = epic;  w = eWIDE;  h = eHIGH; }
                     else { thepic = cpic;  w = cWIDE;  h = cHIGH; }

  if (RBWhich(colorRB)==2) {
    /* generate a FSDithered 1-byte per pixel image */
    bwpic = (byte *) malloc(w*h);
    if (!bwpic) FatalError("unable to malloc dithered picture (DoSave)");
    FSDither(thepic, w, h, bwpic);
    thepic = bwpic;
  }

  /* open file */
  fp = fopen(filename, "w");
  if (!fp) {
    SetISTR(ISTR_INFO,"Can't create '%s' -  %s",filename,sys_errlist[errno]);
    Warning();
    if (bwpic) free(bwpic);
    SetCursors(-1);
    return -1;
  }

  if ((mono || ncols==0) && RBWhich(colorRB)==0) {
    /* if we're saving color, but we're viewing B/W we have to NOT do
       the 'monofication' of the colormap ... */
    for (i=0; i<numcols; i++) {
      r[i] = rorg[i];  g[i] = gorg[i];  b[i] = borg[i];  /* original */
      
      if (revvideo) {
	r[i] = 255-r[i];  g[i] = 255-g[i];  b[i] = 255-b[i];
      }
   } 

    GammifyColors();
  }
      
  rv = 0;
  i = RBWhich(formatRB);
  switch (i) {
  case 0:  rv = WriteGIF(fp,thepic,w, h, r, g, b, numcols, RBWhich(colorRB));
           break;
  case 1:  rv = WritePM (fp,thepic,w, h, r, g, b, numcols, RBWhich(colorRB)); 
           break;
  case 2:  rv = WritePBM(fp,thepic,w, h, r, g, b, numcols, RBWhich(colorRB),1);
           break;
  case 3:  rv = WritePBM(fp,thepic,w, h, r, g, b, numcols, RBWhich(colorRB),0);
           break;
  case 4:  rv = WriteXBM(fp,thepic,w, h, filename);
           break;
  }

  fclose(fp);
  if (rv) unlink(filename);   /* couldn't properly write file:  delete it */

  if (!rv) {
    SetISTR(ISTR_INFO,"Successfully wrote '%s'",filename);
    LoadCurrentDirectory();   /* wrote file: rescan directory */
  }

  if (bwpic) free(bwpic);

  if ((mono || ncols==0) && RBWhich(colorRB)==0) {
    /* restore normal colormap */
    DoMonoAndRV();
    GammifyColors();
  }

  SetCursors(-1);

  return rv;
}



/***************************************************/
void SetDirFName(st)
char *st;
{
  strncpy(filename, st, MAXFNLEN-1);
  filename[MAXFNLEN-1] = '\0';  /* make sure it's terminated */
  XClearWindow(theDisp, dnamW);
  RedrawDNamW();
  BTSetActive(&dbut[S_BSAVE], strlen(filename)!=0);
}

