/* 
 * xvctrl.c - Control box handling functions
 *
 * callable functions:
 *
 *   CreateCtrl(geom)       -  creates the ctrlW window.  Doesn't map it.
 *   CtrlBox(vis)           -  random processing based on value of 'vis'
 *                             maps/unmaps window, etc.
 *   RedrawCtrl(x,y,w,h)    -  called by 'expose' events
 *   ClickCtrl(x,y)
 *   DrawCtrlStr()          -  called to redraw 'ISTR_INFO' string in ctrlW
 *   ScrollToCurrent()      -  called when 'curname' is changed 
 *
 *   LSCreate()             -  creates a listbox
 *   LSRedraw()             -  redraws 'namelist' box
 *   LSClick()              -  operates list box
 *   LSNewData()            -  called when strings or number of them change
 *
 */

/*
 * Copyright 1989, 1990 by the University of Pennsylvania
 *
 * Permission to use, copy, and distribute for non-commercial purposes,
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * The software may be modified for your own purposes, but modified versions
 * may not be distributed.
 *
 * This software is provided "as is" without any express or implied warranty.
 */


#include "xv.h"
#include "bitmaps.h"

#define DBLCLKTIME 500             /* double-click speed in milliseconds */

#define INACTIVE(lptr, item) ((lptr)->filetypes && (lptr)->dirsonly && \
			      (item) >= 0 && (item) < (lptr)->nstr && \
			      (lptr)->str[(item)][0] != C_DIR && \
			      (lptr)->str[(item)][0] != C_LNK)

#define NLINES 9                   /* # of lines in list control (keep odd) */
#define LISTW   330

#define BUTTW   60
#define BUTTH   19

static int    listh;               /* height of list/scrl controls */
static int    ptop;                /* y-coord of top of button area in ctrlW */

static Pixmap fifoPix, chrPix, dirPix, blkPix, lnkPix, sockPix, regPix;

#ifdef __STDC__
static void drawSel(LIST *, int);
static void RedrawNList(void);
#else
static void drawSel(), RedrawNList();
#endif


/***************************************************/
void CreateCtrl(geom)
char *geom;
{
  int i,b2wide;

  ctrlW = CreateWindow("xv controls", geom, CTRLWIDE,CTRLHIGH,infofg,infobg);
  if (!ctrlW) FatalError("can't create controls window!");

  grayTile = XCreatePixmapFromBitmapData(theDisp, ctrlW, gray25_bits,
	     gray25_width, gray25_height, infofg, infobg, dispDEEP);

  grayStip = XCreatePixmapFromBitmapData(theDisp, ctrlW, gray50_bits,
	     gray50_width, gray50_height, 1, 0, 1);
  
  fifoPix  = XCreatePixmapFromBitmapData(theDisp, ctrlW, i_fifo_bits,
	     i_fifo_width, i_fifo_height, 1, 0, 1);
  
  chrPix  = XCreatePixmapFromBitmapData(theDisp, ctrlW, i_chr_bits,
	     i_chr_width, i_chr_height, 1,0,1);
  
  dirPix  = XCreatePixmapFromBitmapData(theDisp, ctrlW, i_dir_bits,
	     i_dir_width, i_dir_height, 1,0,1);
  
  blkPix  = XCreatePixmapFromBitmapData(theDisp, ctrlW, i_blk_bits,
	     i_blk_width, i_blk_height, 1,0,1);
  
  lnkPix  = XCreatePixmapFromBitmapData(theDisp, ctrlW, i_lnk_bits,
	     i_lnk_width, i_lnk_height, 1,0,1);
  
  sockPix  = XCreatePixmapFromBitmapData(theDisp, ctrlW, i_sock_bits,
	     i_sock_width, i_sock_height, 1,0,1);
  
  regPix  = XCreatePixmapFromBitmapData(theDisp, ctrlW, i_reg_bits,
	     i_reg_width, i_reg_height, 1,0,1);
  

  XSetWindowBackgroundPixmap(theDisp, ctrlW, grayTile);

  /* create doo-wahs */
  listh = LINEHIGH * NLINES;

  LSCreate(&nList, ctrlW, 10, 10+CHIGH+3, LISTW, listh, NLINES, dispnames, 
	   numnames, infofg, infobg, RedrawNList, 0, 0);

  ptop = CTRLHIGH - (3*BUTTH + 4*8);

  i = listh-BUTTH;

  BTCreate(&but[BNEXT], ctrlW, 368, nList.y+(i*0)/4, 60, 
	   BUTTH, "Next",     infofg, infobg);
  BTCreate(&but[BPREV], ctrlW, 368, nList.y+(i*1)/4, 60, 
	   BUTTH, "Previous", infofg, infobg);
  BTCreate(&but[BINFO], ctrlW, 368, nList.y+(i*2)/4, 60, 
	   BUTTH, "Info", infofg, infobg);
  BTCreate(&but[BSAVE], ctrlW, 368, nList.y+(i*3)/4, 60, 
	   BUTTH, "Save", infofg, infobg);
  BTCreate(&but[BQUIT], ctrlW, 368, nList.y+(i*4)/4, 60, 
	   BUTTH, "Quit", infofg, infobg);

  BTCreate(&but[BCROP], ctrlW, 10, ptop+8, 
	   BUTTW, BUTTH, "Crop", infofg, infobg);
  BTCreate(&but[BUNCROP], ctrlW, 10, ptop + BUTTH + 2*8, 
	   BUTTW, BUTTH, "UnCrop", infofg, infobg);

  BTCreate(&but[BMAX], ctrlW, 10+(CTRLWIDE-20-BUTTW)/5, ptop+8, 
	   BUTTW, BUTTH, "Max Size", infofg, infobg);
  BTCreate(&but[BNORM], ctrlW, 10+(CTRLWIDE-20-BUTTW)/5, ptop + BUTTH + 2*8, 
	   BUTTW, BUTTH, "Normal", infofg, infobg);

  BTCreate(&but[BUP2], ctrlW, 10+(2*(CTRLWIDE-20-BUTTW))/5, ptop+8, 
	   BUTTW, BUTTH, "Dbl Size", infofg, infobg);
  BTCreate(&but[BDN2], ctrlW, 10+(2*(CTRLWIDE-20-BUTTW))/5, ptop+BUTTH+2*8, 
	   BUTTW, BUTTH, "Half Size", infofg, infobg);

  BTCreate(&but[BUP10], ctrlW, 10+(3*(CTRLWIDE-20-BUTTW))/5, ptop+8, 
	   BUTTW, BUTTH, "+10%", infofg, infobg);
  BTCreate(&but[BDN10], ctrlW, 10+(3*(CTRLWIDE-20-BUTTW))/5, ptop+BUTTH+2*8, 
	   BUTTW, BUTTH, "-10%", infofg, infobg);

  BTCreate(&but[B4BY3], ctrlW, 10+(4*(CTRLWIDE-20-BUTTW))/5, ptop+8, 
	   BUTTW, BUTTH, "4x3", infofg, infobg);
  BTCreate(&but[BASPECT], ctrlW, 10+(4*(CTRLWIDE-20-BUTTW))/5, ptop+BUTTH+2*8, 
	   BUTTW, BUTTH, "Aspect", infofg, infobg);

  BTCreate(&but[BROTL],  ctrlW, 10+(5*(CTRLWIDE-20-BUTTW))/5, ptop+8,
	   BUTTW, BUTTH, "Turn L", infofg, infobg);
  BTCreate(&but[BROTR],  ctrlW, 10+(5*(CTRLWIDE-20-BUTTW))/5, ptop+BUTTH+2*8,
	   BUTTW, BUTTH, "Turn R", infofg, infobg);

  b2wide = (CTRLWIDE - 20 - 6*BUTTW)/5 + 2*BUTTW;
  BTCreate(&but[BACROP], ctrlW, 10 + (0*(CTRLWIDE-20-b2wide))/2, 
	   ptop + 2*BUTTH + 3*8, b2wide, BUTTH, "AutoCrop", infofg, infobg);
  BTCreate(&but[BMAXPECT],   ctrlW, 10 + (1*(CTRLWIDE-20-b2wide))/2, 
	   ptop + 2*BUTTH + 3*8, b2wide, BUTTH, "Maxpect", infofg, infobg);
  BTCreate(&but[BGAMMA],    ctrlW, 10 + (2*(CTRLWIDE-20-b2wide))/2, 
	   ptop + 2*BUTTH + 3*8, b2wide, BUTTH, "Gamma", infofg, infobg);

  XMapSubwindows(theDisp, ctrlW);
}
  

/***************************************************/
void CtrlBox(vis)
int vis;
{
  if (vis) XMapRaised(theDisp, ctrlW);
  else     XUnmapWindow(theDisp, ctrlW);

  ctrlUp = vis;
}


/***************************************************/
void RedrawCtrl(x,y,w,h)
int x,y,w,h;
{
  char foo[40];
  int i;
  XRectangle xr;

  xr.x = x;  xr.y = y;  xr.width = w;  xr.height = h;
  XSetClipRectangles(theDisp, theGC, 0,0, &xr, 1, Unsorted);

  XSetForeground(theDisp, theGC, infofg);
  XSetBackground(theDisp, theGC, infobg);

  XDrawLine(theDisp, ctrlW, theGC, 0, ptop, CTRLWIDE, ptop);

  if (numnames>1) sprintf(foo,"%d files",numnames);
  else strcpy(foo,"1 file");
    
  XSetForeground(theDisp, theGC, infobg);
  XFillRectangle(theDisp,ctrlW, theGC, 10+1,5+1,StringWidth(foo)+4,CHIGH+2);
  XSetForeground(theDisp,theGC,infofg);
  XDrawRectangle(theDisp,ctrlW, theGC, 10,5,StringWidth(foo)+5,CHIGH+3);
  XDrawString(theDisp, ctrlW, theGC, 10+3, 5+ASCENT+2,
	      foo, strlen(foo));
 
  for (i=0; i<NBUTTS; i++)
    BTRedraw(&but[i]);

  DrawCtrlStr();

  XSetClipMask(theDisp, theGC, None);
}


/***************************************************/
void DrawCtrlStr()
{
  int   y;
  char *st;

  y = ptop - (CHIGH + 2);
  st = GetISTR(ISTR_INFO);

  XSetForeground(theDisp, theGC, infobg);
  XFillRectangle(theDisp, ctrlW, theGC, 0, y-1, CTRLWIDE, CHIGH+3);

  XSetForeground(theDisp, theGC, infofg);
  XDrawLine(theDisp, ctrlW, theGC, 0, y-2, CTRLWIDE, y-2);

  XDrawString(theDisp, ctrlW, theGC, 10, y+ASCENT, st, strlen(st));
}


/***************************************************/
int ClickCtrl(x,y)
int x,y;
{
  BUTT *bp;
  int   i;

  for (i=0; i<NBUTTS; i++) {
    bp = &but[i];
    if (PTINRECT(x, y, bp->x, bp->y, bp->w, bp->h)) break;
  }

  if (i<NBUTTS) {                   /* found one */
    if (BTTrack(bp)) return (i);    /* and it was clicked */
  }

  return -1;
}



/***************************************************/
void ScrollToCurrent()
{
  /* called when 'curname' is changed by anything (next/prev buttons,
     wait timeout, whatever.  IF curname is already visible, just redraws
     list to reflect changed selection.  If not, trys to adjust 'liststart' 
     so that curname will appear in the center of the list window */

  int halfway;

  nList.selected = curname;
/*  if (nList.selected >= nList.scrl.val && 
      nList.selected <  nList.scrl.val + nList.nlines) LSRedraw(&nList); */

  if (nList.selected > nList.scrl.val && 
      nList.selected <  nList.scrl.val + nList.nlines-1) LSRedraw(&nList);
  else {
    halfway = (nList.nlines)/2;   /* offset to the halfway pt. of the list */
    SCSetVal(&nList.scrl, nList.selected - halfway);
  }
}


/***************************************************/
static void RedrawNList()
{
  LSRedraw(&nList);
}




/***************** LIST STUFF *********************/

/***************************************************/
void LSCreate(lp, win, x, y, w, h, nlines, strlist, nstr, fg, bg, fptr, 
	      typ, donly)
LIST         *lp;
Window        win;
int           x,y,w,h,nlines,nstr,typ,donly;
unsigned long fg, bg;
char        **strlist;    /* a pointer to a list of strings */
void        (*fptr)();
{
  lp->win = XCreateSimpleWindow(theDisp,win,x,y,w,h,1,infofg,infobg);
  if (!lp->win) FatalError("can't create list window!");

  lp->x = x;    lp->y = y;   
  lp->w = w;    lp->h = h;
  lp->fg = fg;  lp->bg = bg;
  lp->str      = strlist;
  lp->nstr     = nstr;
  lp->selected = 0;
  lp->nlines   = nlines;
  lp->filetypes= typ;
  lp->dirsonly = donly;

  XSelectInput(theDisp, lp->win, ExposureMask | ButtonPressMask);

  SCCreate(&lp->scrl, win, x+w, y, 1, h, 0, nstr-nlines, curname, nlines-1,
	   fg, bg, fptr);
}



/***************************************************/
void LSNewData(lp, strlist, nstr)
LIST         *lp;
char        **strlist;
int           nstr;
{
  lp->str = strlist;
  lp->nstr = nstr;
  lp->selected = 0;
  SCSetRange(&lp->scrl, 0, nstr - lp->nlines, 0, lp->nlines-1);
}


/***************************************************/
static void drawSel(lp,j)
LIST *lp;
int j;
{
  int i, inactive;
  unsigned long fg, bg;

  inactive = INACTIVE(lp,j);

  i = j - lp->scrl.val;
  if (i<0 || i>=lp->nlines) return;  /* off screen */

  if (j == lp->selected && !inactive && j<lp->nstr) 
       { fg = lp->bg;  bg = lp->fg; }  /* invert */
  else { fg = lp->fg;  bg = lp->bg; }

  XSetForeground(theDisp, theGC, bg);
  XFillRectangle(theDisp, lp->win, theGC, 0,i*LINEHIGH, lp->w, LINEHIGH);

  if (j>=0 && j<lp->nstr) {   /* only draw string if valid */
    /* make non-dirs inactive, if dirsonly and filetypes */
    XSetForeground(theDisp, theGC, fg);
    XSetBackground(theDisp, theGC, bg);

    if (!lp->filetypes) 
      XDrawString(theDisp, lp->win, theGC, 3, i*LINEHIGH + ASCENT + 1, 
		  lp->str[j], strlen(lp->str[j]));
    else {
      int ypos = i*LINEHIGH + (LINEHIGH - i_fifo_height)/2;

      if (lp->str[j][0] == C_FIFO) 
	XCopyPlane(theDisp, fifoPix, lp->win, theGC, 0, 0,
		   i_fifo_width, i_fifo_height, 3, ypos, 1L);

      else if (lp->str[j][0] == C_CHR) 
	XCopyPlane(theDisp, chrPix, lp->win, theGC, 0, 0,
		   i_chr_width, i_chr_height, 3, ypos, 1L);

      else if (lp->str[j][0] == C_DIR) 
	XCopyPlane(theDisp, dirPix, lp->win, theGC, 0, 0,
		   i_dir_width, i_dir_height, 3, ypos, 1L);

      else if (lp->str[j][0] == C_BLK) 
	XCopyPlane(theDisp, blkPix, lp->win, theGC, 0, 0,
		   i_blk_width, i_blk_height, 3, ypos, 1L);

      else if (lp->str[j][0] == C_LNK) 
	XCopyPlane(theDisp, lnkPix, lp->win, theGC, 0, 0,
		   i_lnk_width, i_lnk_height, 3, ypos, 1L);

      else if (lp->str[j][0] == C_SOCK) 
	XCopyPlane(theDisp, sockPix, lp->win, theGC, 0, 0,
		   i_sock_width, i_sock_height, 3, ypos, 1L);

      else  /* lp->str[j][0] == C_REG */
	XCopyPlane(theDisp, regPix, lp->win, theGC, 0, 0,
		   i_reg_width, i_reg_height, 3, ypos, 1L);


      XDrawString(theDisp, lp->win, theGC, 3 + i_fifo_width + 3, 
		  i*LINEHIGH + ASCENT + 1, 
		  lp->str[j]+1, strlen(lp->str[j]+1));

#ifdef STIPPLE
      if (inactive) { 
	/* stipple the icon by drawing 'bg' where there's 1's in the stipple */
	XSetFillStyle(theDisp, theGC, FillStippled);
	XSetStipple(theDisp, theGC, grayStip);
	XSetForeground(theDisp, theGC, bg);
	XSetBackground(theDisp, theGC, fg);
	XFillRectangle(theDisp,lp->win,theGC,3,i*LINEHIGH,lp->w,LINEHIGH);
	XSetForeground(theDisp, theGC, fg);
	XSetFillStyle(theDisp, theGC, FillSolid);
      }
#endif

    }
  }
}


/***************************************************/
void LSRedraw(lp)
LIST *lp;
{
  int  i;

  for (i = lp->scrl.val; i < lp->scrl.val + lp->nlines; i++) 
    drawSel(lp,i);
}


/***************************************************/
int LSClick(lp,ev)
LIST *lp;
XButtonEvent *ev;
{
  /* returns '-1' normally.  returns 0 -> numnames-1 for a goto */

  Window       rW, cW;
  int          rx, ry, x, y, sel, oldsel;
  unsigned int mask;
  static Time  lasttime=0;
  static int   lastsel = -1;

  x = ev->x;  y = ev->y;
  sel = lp->scrl.val + y/LINEHIGH;
  if (sel >= lp->nstr) sel = lp->selected;

  /* see if it's a double click */
  if (ev->time - lasttime < DBLCLKTIME && sel==lastsel 
      && (lp->scrl.val + y/LINEHIGH) < lp->nstr
      && !INACTIVE(lp,sel)) {
    return (sel);
  }

  lasttime = ev->time;  lastsel = sel;

  /* if not clicked on selected, turn off selected and select new one */
  if (sel != lp->selected) {
    oldsel = lp->selected;
    lp->selected = sel;
    drawSel(lp,sel);  drawSel(lp,oldsel);
    XFlush(theDisp);
  }

  while (XQueryPointer(theDisp,lp->win,&rW,&cW,&rx,&ry,&x,&y,&mask)) {
    if (!(mask & Button1Mask)) break;    /* button released */

    if (y<0) { /* scroll up in list */ 
      if (lp->scrl.val > lp->scrl.min) {
	lp->selected = lp->scrl.val - 1;
	SCSetVal(&lp->scrl, lp->scrl.val - 1);
	Timer(100);
      }
    }

    else if (y > lp->h) { /* scroll down in list */
      if (lp->scrl.val < lp->scrl.max) {
	lp->selected = lp->scrl.val + lp->nlines;
	if (lp->selected >= lp->nstr) lp->selected = lp->nstr - 1;
	SCSetVal(&lp->scrl, lp->scrl.val + 1);
	Timer(100);
      }
    }

    else {
      sel = lp->scrl.val + y/LINEHIGH;
      if (sel >= lp->nstr) sel = lp->nstr - 1;

      if (sel != lp->selected && sel >= lp->scrl.val &&
	  sel < lp->scrl.val + lp->nlines) {  
	/* dragged to another on current page */
	oldsel = lp->selected;
	lp->selected = sel;
	drawSel(lp, sel);  drawSel(lp, oldsel);
	XFlush(theDisp);
      }
    }
  }

  return(-1);
}



