/* 
 * xvbutt.c - regular and 'radio' pushbuttons
 *
 * callable functions:
 *
 *   BTCreate()             -  create a button
 *   BTSetActive()          -  change 'active' status of button
 *   BTRedraw()             -  redraw button
 *   BTTrack()              -  clicked in button.  track until mouse up
 *
 *   RBCreate()             -  create an RBUTT and append to supplied list
 *   RBRedraw()             -  redraw one or all RBUTTs in a list
 *   RBSelect()             -  change selected item in list of RBUTTs
 *   RBWhich()              -  returns index of selected RBUTT in list
 *   RBCount()              -  returns # of RBUTTs in list
 *   RBSetActive()          -  sets active status of an RBUTT
 *   RBClick()              -  finds clicked-on rb in a list
 *   RBTrack()              -  tracks rb after click, until release
 */

/*
 * Copyright 1989, 1990 by the University of Pennsylvania
 *
 * Permission to use, copy, and distribute for non-commercial purposes,
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * The software may be modified for your own purposes, but modified versions
 * may not be distributed.
 *
 * This software is provided "as is" without any express or implied warranty.
 */


#include "xv.h"
#include "bitmaps.h"

static int    rbpixmade = 0;
static Pixmap rbon, rboff, rbon1, rboff1;

#ifdef __STDC__
static void drawRB(RBUTT *);
#else
static void drawRB();
#endif

/**********************************************/
void BTCreate(bp,win,x,y,w,h,str,fg,bg)
BUTT         *bp;
Window        win;
int           x,y,w,h;
char         *str;
unsigned long fg,bg;
{
  bp->win = win;
  bp->x = x;  bp->y = y;  bp->w = w;  bp->h = h;
  bp->str = str;
  bp->fg = fg;  bp->bg = bg;
  bp->lit = 0;
  bp->active = 1;
  bp->toggle = 0;
}



/**********************************************/
void BTSetActive(bp,act)
BUTT         *bp;
int           act;
{
  if (bp->active != act) {
    bp->active = act;
    BTRedraw(bp);
  }
}



/**********************************************/
void BTRedraw(bp)
BUTT *bp;
{
  int x,y,w,h,r;
  XPoint poly[9];

  x = bp->x;  y=bp->y;  w=bp->w;  h=bp->h;  r=3;

  /* set up the polygon */
  poly[0].x = x;      poly[0].y = y+r;
  poly[1].x = x;      poly[1].y = y+h-r;
  poly[2].x = x+r;    poly[2].y = y+h;
  poly[3].x = x+w-r;  poly[3].y = y+h;
  poly[4].x = x+w;    poly[4].y = y+h-r;
  poly[5].x = x+w;    poly[5].y = y+r;
  poly[6].x = x+w-r;  poly[6].y = y;
  poly[7].x = x+r;    poly[7].y = y;
  poly[8].x = x;      poly[8].y = y+r;


  if (!bp->active) bp->lit = 0;   /* sanity assertion */

  if (bp->lit) XSetForeground(theDisp, theGC, bp->fg);
          else XSetForeground(theDisp, theGC, bp->bg);
  XFillPolygon(theDisp,bp->win,theGC,poly,9,Convex,CoordModeOrigin);

  XSetForeground(theDisp, theGC, bp->fg);
  XDrawLines(theDisp,bp->win,theGC,poly,9,CoordModeOrigin);

  if (!bp->active) {   /* stipple the text if not active */
    XSetFillStyle(theDisp, theGC, FillStippled);
    XSetStipple(theDisp, theGC, grayStip);
  }

  if (bp->lit) XSetForeground(theDisp, theGC, bp->bg);
          else XSetForeground(theDisp, theGC, bp->fg);
  XDrawString(theDisp, bp->win, theGC, CENTERX(mfinfo,x+w/2,bp->str),
	      CENTERY(mfinfo,y+h/2), bp->str, strlen(bp->str));

  if (!bp->active) XSetFillStyle(theDisp,theGC,FillSolid);  /* back to norm */
}



/**********************************************/
int BTTrack(bp)
BUTT *bp;
{
  /* called when we've gotten a click inside 'bp'.  returns 1 if button
     was still selected lit when mouse was released. */

  Window       rW, cW;
  int          x, y, rx, ry, rval, inval;
  unsigned int mask;

  if (!bp->active) return 0;   /* inactive button */

  bp->lit = !bp->lit;
  inval = bp->lit;

  BTRedraw(bp);  XFlush(theDisp);
  Timer(75);  /* long enough for turn on to be visible */

  while (XQueryPointer(theDisp,bp->win,&rW,&cW,&rx,&ry,&x,&y,&mask)) {
    if (!(mask & Button1Mask)) break;    /* button released */

    if (bp->lit!=inval && PTINRECT(x, y, bp->x, bp->y, bp->w, bp->h)) {
      bp->lit = inval;  BTRedraw(bp);  XFlush(theDisp);
    }
    
    if (bp->lit==inval && !PTINRECT(x, y, bp->x, bp->y, bp->w, bp->h)) {
      bp->lit = !inval;  BTRedraw(bp);  XFlush(theDisp);
    }
  }

  rval = (bp->lit == inval);
  
  if (bp->lit && !bp->toggle) 
    { bp->lit = 0;  BTRedraw(bp);  XFlush(theDisp); }

  return(rval);
}








/***********************************************/
RBUTT *RBCreate(rblist, win, x,y,str, fg, bg)
      RBUTT        *rblist;
      Window        win;
      int           x,y;
      char         *str;
      unsigned long fg,bg;
{
  /* mallocs an RBUTT, fills in the fields, and appends it to rblist
     if rblist is NULL, this is the first rb in the list.  It will
     be made the 'selected' one 

     Note: no need to check return status.  It'll fatal error if it 
     can't malloc */

  RBUTT *rb, *rbptr;

  rb = (RBUTT *) malloc(sizeof(RBUTT));
  if (!rb) FatalError("couldn't malloc RBUTT");

  /* fill in the fields of the structure */
  rb->win      = win;
  rb->x        = x;
  rb->y        = y;
  rb->str      = str;
  rb->selected = 0;
  rb->active   = 1;
  rb->next     = (RBUTT *) NULL;
  rb->fg       = fg;
  rb->bg       = bg;

  if (rblist) {            /* append to end of list */
    rbptr = rblist;
    while (rbptr->next) rbptr = rbptr->next;
    rbptr->next = rb;
  }
  else {                   /* this is the first one in the list.  select it */
    rb->selected = 1;
  }


  /* and, on an unrelated note, if the RB pixmaps haven't been created yet,
     do so.  We'll be needing them, y'see... */

  if (!rbpixmade) {
    rbon  = XCreatePixmapFromBitmapData(theDisp, rootW, rb_on_bits,
	     rb_on_width, rb_on_height, fg, bg, dispDEEP);

    rboff = XCreatePixmapFromBitmapData(theDisp, rootW, rb_off_bits,
	     rb_off_width, rb_off_height, fg, bg, dispDEEP);
    rbon1 = XCreatePixmapFromBitmapData(theDisp, rootW, rb_on1_bits,
	     rb_on1_width, rb_on1_height, fg, bg, dispDEEP);
    rboff1= XCreatePixmapFromBitmapData(theDisp, rootW, rb_off1_bits,
	     rb_off1_width, rb_off1_height, fg, bg, dispDEEP);

    rbpixmade = 1;
  }

  return(rb);
}
  



/***********************************************/
void RBRedraw(rblist, num)
RBUTT *rblist;
int    num;
{
  /* redraws the 'num-th' RB in the list.  if num < 0, redraws entire list */

  RBUTT *rb;
  int    i;

  /* point 'rb' at the appropriate RBUTT, *if* we're not drawing entire list */
  if (num>=0) {
    i=0;  rb=rblist;
    while (i!=num && rb) { rb = rb->next;  i++; }
    if (!rb) return;                     /* num is out of range.  do nothing */
    drawRB(rb);
  }

  else {                                 /* draw entire list */
    rb = rblist;
    while (rb) {
      drawRB(rb);
      rb = rb->next;
    }
  }
}


static void drawRB(rb)
RBUTT *rb;
{
  /* draws the rb being pointed at */

  if (!rb) return;  /* rb = NULL */

  XSetForeground(theDisp, theGC, rb->fg);
  XSetBackground(theDisp, theGC, rb->bg);

  if (rb->selected) 
    XCopyArea(theDisp, rbon, rb->win, theGC, 0, 0, rb_on_width, rb_on_height, 
	      rb->x, rb->y);
  else
    XCopyArea(theDisp, rboff, rb->win, theGC, 0, 0, rb_on_width, rb_on_height, 
	      rb->x, rb->y);

  XDrawString(theDisp, rb->win, theGC, rb->x+rb_on_width+4, 
	      rb->y+rb_on_height/2 - CHIGH/2 + ASCENT,rb->str,strlen(rb->str));

  /* if non-active, dim button and string */
  if (!rb->active) { 
    /* stipple the RB by drawing 'bg' where there's 1's in the stipple */
    XSetFillStyle(theDisp, theGC, FillStippled);
    XSetStipple(theDisp, theGC, grayStip);
    XSetForeground(theDisp, theGC, rb->bg);
    XFillRectangle(theDisp,rb->win,theGC,rb->x,rb->y,rb_on_width,rb_on_height);
    XFillRectangle(theDisp,rb->win,theGC, rb->x + rb_on_width+4,
		   rb->y+rb_on_height/2 - CHIGH/2, StringWidth(rb->str),CHIGH);
    XSetFillStyle(theDisp, theGC, FillSolid);
  }
}


/***********************************************/
void RBSelect(rblist, n)
RBUTT *rblist;
int    n;
{
  RBUTT *rbold, *rb;
  int    i;

  /* makes rb #n the selected rb in the list.  Does all redrawing.  Does
     nothing if rb already selected */

  /* get pointers to the currently selected rb and the desired rb */
  rbold = rblist;
  while (rbold && !rbold->selected) rbold = rbold->next;
  if (!rbold) return;    /* no currently selected item.  shouldn't happen */

  rb = rblist;  i=0;
  while (rb && i!=n) {rb = rb->next;  i++; }
  if (!rb) return;    /* 'n' is out of range */


  if (rb == rbold) return;   /* 'n' is already selected.  do nothing */

  rbold->selected = 0;
  rb->selected    = 1;
  drawRB(rbold);
  drawRB(rb);
}


	      
/***********************************************/
int RBWhich(rblist)
RBUTT *rblist;
{
  int i;

  /* returns index of currently selected rb.  if none, returns -1 */

  i = 0;
  while (rblist && !rblist->selected) { rblist = rblist->next;  i++; }

  if (!rblist) return -1;             /* didn't find one */
  return i;
}


/***********************************************/
int RBCount(rblist)
RBUTT *rblist;
{
  int i;

  /* returns # of rb's in the list */

  i = 0;
  while (rblist) { rblist = rblist->next; i++; }
  return i;
}


/***********************************************/
void RBSetActive(rblist, n, act)
RBUTT *rblist;
int n,act;
{
  RBUTT *rb;
  int    i;

  /* sets 'active' status of rb #n.  does redrawing */

  rb=rblist;  i=0;
  while (rb && i!=n) { rb = rb->next; i++; }
  if (!rb) return;                         /* n out of range.  do nothing */

  if (rb->active != act) {
    rb->active = act;
    drawRB(rb);
  }
}


/***********************************************/
int RBClick(rblist, mx, my)
RBUTT *rblist;
int    mx,my;
{
  int i;

  /* searches through rblist to see if mouse click at mx,my is in the
     clickable region of any of the rb's.  If it finds one, it returns 
     it's index in the list.  If not, returns -1 */

  i = 0;
  while (rblist) {
    if (PTINRECT(mx, my, rblist->x, rblist->y, rb_on_width, rb_on_height)) 
      break;
    rblist = rblist->next;
    i++;
  }

  if (!rblist) return -1;
  return(i);
}


/***********************************************/
void RBTrack(rblist, n)
RBUTT *rblist;
int    n;
{
  RBUTT       *rb;
  Window       rW, cW;
  int          i, x, y, rx, ry, lit;
  unsigned int mask;
  Pixmap litpix, darkpix;

  rb=rblist;  i=0;
  while (rb && i!=n) { rb = rb->next; i++; }
  if (!rb) return;                    /* n out of range */

  /* called once we've figured out that the mouse clicked in 'rb' */

  if (!rb->active) return;

  if (rb->selected) { litpix = rbon1;   darkpix = rbon; }
               else { litpix = rboff1;  darkpix = rboff; }

  lit = 1;
  XCopyArea(theDisp, litpix, rb->win, theGC, 0, 0, rb_on_width, rb_on_height, 
	      rb->x, rb->y);
  XFlush(theDisp);
  Timer(75);          /* give chance for 'turn on' to become visible */

  while (XQueryPointer(theDisp,rb->win,&rW,&cW,&rx,&ry,&x,&y,&mask)) {
    if (!(mask & Button1Mask)) break;    /* button released */

    if (!lit && PTINRECT(x, y, rb->x, rb->y, rb_on_width, rb_on_height)) {
      lit=1;
      XCopyArea(theDisp, litpix, rb->win, theGC, 0,0,rb_on_width,rb_on_height, 
	      rb->x, rb->y);
      XFlush(theDisp);
    }
    
    if (lit && !PTINRECT(x, y, rb->x, rb->y, rb_on_width, rb_on_height)) {
      lit=0;
      XCopyArea(theDisp, darkpix,rb->win,theGC, 0,0,rb_on_width,rb_on_height, 
	      rb->x, rb->y);
      XFlush(theDisp);
    }
  }

  if (lit) {
    XCopyArea(theDisp, darkpix, rb->win, theGC, 0, 0, 
	      rb_on_width, rb_on_height, rb->x, rb->y);
    RBSelect(rblist, n);
  }

  XFlush(theDisp);
}
