/* This program is Copyright (c) 1991 David Allen.  It may be freely
   distributed as long as you leave my name and copyright notice on it.
   I'd really like your comments and feedback; send e-mail to
   allen@viewlogic.com, or send us-mail to David Allen, 10 O'Moore Ave,
   Maynard, MA 01754. */

#include "const.h"
#include "tec.h"

/* These are all defined in tec1.c */
extern char m[2][MAXX][MAXY], r[MAXX][MAXY], e[MAXX][MAXY], kid[MAXFRAG],
   mm[MAXPLATE][MAXPLATE];
extern unsigned char t[2][MAXX][MAXY];
extern int step, phead, tarea[MAXPLATE], karea[MAXFRAG];
extern struct plate p [MAXPLATE];


makefrac (src, match) int src, match; {
   /* This function uses a very simple fractal technique to draw a blob.  Four
   one-dimensional fractals are created and stored in array x, then the
   fractals are superimposed on a square array, summed and thresholded to
   produce a binary blob.  Squares in the blob are set to the value in `match'.
   A one-dimensional fractal of length n is computed like this.  First,
   set x [n/2] to some height and set the endpoints (x[0] and x[n]) to 0.
   Then do log-n iterations.  The first iteration computes 2 more values:
   x[n/4] = average of x[0] and x[n/2], plus some random number, and
   x[3n/4] = average of x[n/2] and x[n], plus some random number.  The second
   iteration computes 4 more values (x[n/8], x[3n/8], ...) and so on.  The
   random number gets smaller by a factor of two each time also.
 
   Anyway, you wind up with a number sequence that looks like the cross-section
   of a mountain.  If you sum two fractals, one horizontal and one vertical,
   you get a 3-d mountain; but it looks too symmetric.  If you sum four,
   including the two 45 degree diagonals, you get much better results. */
 
   register int xy, dist, n, inc, i, j, k; char x[4][65];
   int xofs, yofs, xmin, ymin, xmax, ymax, bloblevel, hist[256];
 
   /* Compute offsets to put center of blob in center of the world, and
      compute array limits to clip blob to world size */
   xofs = (XSIZE - 64) >> 1; yofs = (YSIZE - 64) >> 1;
   if (xofs < 0) { xmin = -xofs; xmax = 64 + xofs; }
   else { xmin = 0; xmax = 64; }
   if (yofs < 0) { ymin = -yofs; ymax = 64 + yofs; }
   else { ymin = 0; ymax = 64; }
 
   for (xy=0; xy<4; xy++) {
      /* Initialize loop values and fractal endpoints */
      x [xy] [0] = 0; x [xy] [64] = 0; dist = 32;
      x [xy] [32] = 24 + rnd (16); n = 2; inc = 16;
 
      /* Loop log-n times, each time halving distance and doubling iterations */
      for (i=0; i<5; i++, dist>>=1, n<<=1, inc>>=1)
         for (j=0, k=0; j<n; j++, k+=dist)
            x[xy][k+inc] = ((x[xy][k]+x[xy][k+dist])>>1) + rnd (dist) - inc; }
 
   /* Superimpose fractals into the output array.  x[0] is horizontal, x[1] */
   /* vertical, x[2] diagonal from top left to bottom right, x[3] diagonal */
   /* from TR to BL.  While superimposing, create a histogram of the values.*/
   for (i=0; i<256; i++) hist[i] = 0;
   for (i=xmin; i<xmax; i++) for (j=ymin; j<ymax; j++) {
      k = x[0][i] + x[1][j] + x[2][(i - j + 64) >> 1] + x[3][(i + j) >> 1];
      if (k < 0) k = 0; if (k > 255) k = 255;
      hist[k]++; m[src][i+xofs][j+yofs] = k; }

   /* Pick a threshhold to get as close to the goal number of squares as */
   /* possible, then go back through the array and adjust it */
   bloblevel = XSIZE * YSIZE * (100 - HYDROPCT) / 100;
   for (k=255, i=0; k>=0; k--) if ((i += hist[k]) > bloblevel) break;
   for (i=xmin; i<xmax; i++) for (j=ymin; j<ymax; j++)
      m[src][i+xofs][j+yofs] = (m[src][i+xofs][j+yofs] > k) ? match : 0; }
 
 
singlefy (src, match) int src, match; {
   /* This is a subfunction of init() which is called twice to improve the
   fractal blob.  It calls segment() and then interprets the result.  If
   only one region was found, no improvement is needed; otherwise, the
   area of each region must be computed by summing the areas of all its
   fragments, and the index of the largest region is returned. */
 
   int i, reg, frag, besti, besta;
 
   segment (src, match, &frag, &reg);
   if (reg == 1) return (-1); /* No improvement needed */
 
   /* Initialize the areas to zero, then sum frag areas */
   for (i=1; i<=reg; i++) tarea[i] = 0;
   for (i=1; i<=frag; i++) tarea [kid[i]] += karea [i];
 
   /* Pick largest area of all regions and return it */
   for (i=1, besta=0, besti=0; i<=reg; i++)
      if (besta < tarea[i]) { besti = i; besta = tarea[i]; }
   return (besti); }
 
 
newdxy () {
   /* For each plate, compute how many squares it should move this step.
   Multiply the plate's basic movement vector odx,ody by the age modifier
   MR[], then add the remainders rx,ry from the last move to get some large
   integers.  Then turn the large integers into small ones by dividing by
   REALSCALE and putting the remainders back into rx,ry.  This function also
   increases the age of each plate, but doesn't let the age of any plate
   go above MAXLIFE.  This is done to make sure that MR[] does not need to
   be a long vector. */
 
   register int i, a;
 
   /* If there is only a single supercontinent, anchor it */
   for (i=phead, a=0; i; i=p[i].next, a++);
   if (a == 1) if (p[phead].odx || p[phead].ody) {
      p[phead].odx = 0; p[phead].ody = 0; }

   for (i=phead; i; i=p[i].next) {
      a = (p[i].odx * MR[p[i].age]) + p[i].rx;
      p[i].dx = a / REALSCALE; p[i].rx = a % REALSCALE;
      a = (p[i].ody * MR[p[i].age]) + p[i].ry;
      p[i].dy = a / REALSCALE; p[i].ry = a % REALSCALE;
      if (p[i].age < MAXLIFE-1) (p[i].age)++; } }
 
 
move (src, dest) int src, dest; {
   /* This function moves all the plates that are drifting.  The amount to
   move by is determined in newdxy().  The function simply steps through
   every square in the array; if there's a plate in a square, its new location
   is found and the topography is moved there.  Overlaps between plates are
   detected and recorded so that merge() can resolve the collision; mountain
   growing is performed.  If two land squares wind up on top of each other,
   folded mountains are produced.  If a land square winds up where ocean was
   previously, that square is the leading edge of a continent and grows a
   mountain by subsuming the ocean basin. */
 
   register int i, j; int a, b, c, x, y;
 
   /* Clear out the merge matrix and the destination arrays */
   for (i=1; i<MAXPLATE; i++) for (j=1; j<MAXPLATE; j++) mm[i][j] = 0;
   for (i=0; i<XSIZE; i++) for (j=0; j<YSIZE; j++) {
      m[dest][i][j] = 0; t[dest][i][j] = 0; }
 
   checkmouse ();
   /* Look at every square which belongs to a plate */
   for (i=0; i<XSIZE; i++) for (j=0; j<YSIZE; j++) if ((a = m[src][i][j]) > 0) {
 
      /* Add the plate's dx,dy to the position to get the square's new */
      /* location; if it is off the map, throw it away */
      x = p[a].dx + i; y = p[a].dy + j;
      if ((x >= XSIZE) || (x < 0) || (y >= YSIZE) || (y < 0)) p[a].area--;
      else { /* It IS on the map */

         /* If the destination is occupied, remove the other guy but */
         /* remember that the two plates overlapped; set the new height */
         /* to the larger height plus half the smaller. */
         if (c = m[dest][x][y]) {
            (mm[a][c])++; (p[c].area)--;
            b = t[src][i][j]; c = t[dest][x][y];
            t[dest][x][y] = (b > c) ? b + (c>>1) : c + (b>>1); }

         /* The destination isn't occupied.  Just copy the height. */
         else t[dest][x][y] = t[src][i][j];

         /* If this square is over ocean, increase its height. */
         if (t[src][x][y] < ZCOAST) t[dest][x][y] += ZSUBSUME;

         /* Plate A now owns this square */
         m[dest][x][y] = a; } } }
 
 
merge (dest) int dest; {
   /* Since move has set up the merge matrix, most of the work is done.  This
   function calls bump once for each pair of plates which are rubbing; note
   that a and b below loop through the lower diagonal part of the matrix.
   One subtle feature is that a plate can bump with several other plates in
   a step; suppose that the plate is merged with the first plate it bumped.
   The loop will try to bump the vanished plate with the other plates, which
   would be wrong.  To avoid this, the lookup table lut is used to provide
   a level of indirection.  When a plate is merged with another, its lut
   entry is changed to indicate that future merges with the vanished plate
   should be applied to the plate it has just been merged with. */
 
   char lut[MAXPLATE]; int a, aa, b, bb, i;
 
   for (a=1; a<MAXPLATE; a++) lut[a] = a;
   for (a=2; a<MAXPLATE; a++) for (b=1; b<a; b++) if (mm[a][b] || mm[b][a]) {
      aa = lut [a]; bb = lut[b];
      if (aa != bb) if (bump (dest, aa, bb)) {
         lut[aa] = bb;
         for (i=1; i<MAXPLATE; i++) if (lut[i] == aa) lut[i] = bb; } } }
 
 
bump (dest, a, b) int dest, a, b; {
   /* Plates a and b have been moved on top of each other by some amount;
   alter their movement rates for a slow collision, possibly merging them.
   The collision "strength" is a ratio of the area overlap (provided by
   move ()) to the total area of the plates involved.  A fraction of each
   plate's current movement vector is subtracted from the movement vector
   of the other plate.  If the two vectors are now within some tolerance
   of each other, they are almost at rest so merge them with each other. */
 
   double maa, mab, ta, tb, rat, area; register int i, j, x;
 
   checkmouse();
   /* Find a ratio describing how strong the collision is */
   x = mm[a][b] + mm[b][a]; area = p[a].area + p[b].area;
   rat = x / (MAXBUMP + (area / 20)); if (rat > 1.0) rat = 1.0;
 
   /* Do some math to update the move vectors.  This looks complicated */
   /* because a plate's actual movement vector must be multiplied by */
   /* MR[age], and because I have rewritten the equations to maximize */
   /* use of common factors.  Trust me, it's just inelastic collision. */
   maa = p[a].area * MR[p[a].age]; mab = p[b].area * MR[p[b].age];
   ta = MR[p[a].age] * area;
   p[a].odx = (p[a].odx * maa + p[b].odx * mab * rat) / ta;
   p[a].ody = (p[a].ody * maa + p[b].ody * mab * rat) / ta;
   tb = MR[p[b].age] * area;
   p[b].odx = (p[b].odx * mab + p[a].odx * maa * rat) / tb;
   p[b].ody = (p[b].ody * mab + p[a].ody * maa * rat) / tb;
 
   /* For each axis, compute the remaining relative velocity.  If it is */
   /* too large, return without merging the plates */
   if (ABS (p[a].odx*MR[p[a].age] - p[b].odx*MR[p[b].age]) > BUMPTOL) return(0);
   if (ABS (p[a].ody*MR[p[a].age] - p[b].ody*MR[p[b].age]) > BUMPTOL) return(0);
 
   /* The relative velocity is small enough, so merge the plates.  Replace */
   /* all references to a with b, free a, and tell merge() a was freed. */
   for (i=0; i<XSIZE; i++) for (j=0; j<YSIZE; j++)
      if (m[dest][i][j] == a) m[dest][i][j] = b;
   p[b].area += p[a].area; pfree (a);
   return (a); }
 
 
erode (dest) int dest; {
   /* This function takes the topography in t[dest] and smooths it, lowering
   mountains and raising lowlands and continental margins.  It does this by
   stepping across the entire array and doing a computation once for each
   pair of 8-connected pixels.  The computation is done by onerode(), below.
   The computation result for a pair is a small delta for each square, which
   is summed in the array e.  When the computation is finished, the delta
   is applied; if this pushes an ocean square high enough, it is added to
   an adjacent plate if one can be found.  Also, if a land square is eroded
   low enough, it is turned into ocean and removed from its plate. */
 
   register int i, j, x, z, xx;
 
   /* Zero out the array for the deltas first */
   for (i=0; i<XSIZE; i++) for (j=0; j<YSIZE; j++) e[i][j] = 0;
   checkmouse();

   /* Step across the entire array; each pixel is adjacent to 8 others, and */
   /* it turns out that if four pairs are considered for each pixel, each */
   /* pair is considered exactly once.  This is important for even erosion */
   for (i=1; i<XSIZE; i++) for (j=1; j<YSIZE; j++) {
      onerode (dest, i, j,  0, -1);
      onerode (dest, i, j, -1, -1);
      onerode (dest, i, j, -1,  0);
      if (i < XSIZE-1) onerode (dest, i, j, -1, 1); }
 
   /* Now go back across the array, applying the delta values from e[][] */
   for (i=0; i<XSIZE; i++) for (j=0; j<YSIZE; j++) {
      z = t[dest][i][j] + e[i][j]; if (z < 0) z = 0; if (z > 255) z = 255;
 
      /* If the square just rose above shelf level, look at the four */
      /* adjacent squares.  If one is a plate, add the square to that plate */
      if ((z >= ZSHELF) && (t[dest][i][j] < ZSHELF)) { xx = 0;
         if (i > 1)       if (x = m[dest][i-1][j]) xx = x;
         if (i < XSIZE-1) if (x = m[dest][i-1][j]) xx = x;
         if (j > 1)       if (x = m[dest][i][j-1]) xx = x;
         if (j < YSIZE-1) if (x = m[dest][i][j+1]) xx = x;
         if (xx) { p[xx].area++; m[dest][i][j] = xx; } }
 
      /* Add the increment to the old value; if the square is lower than */
      /* shelf level but belongs to a plate, remove it from the plate */
      t[dest][i][j] = z;
      if ((z < ZSHELF) && (x = m[dest][i][j])) {
         p[x].area--; m[dest][i][j] = 0; } } }


onerode (dest, i, j, di, dj) int dest, i, j, di, dj; {
   /* This function is called once per pair of squares in the array.  The
   amount each square loses to an adjacent but lower square in each step is
   one-eighth the difference in altitude.  This is coded as a shift right 3
   bits, but since -1 >> 3 is still -1, the code must be repeated to avoid
   shifting negative numbers.  Also, since erosion is reduced below the
   waterline, if an altitude is lower than ZERODE, ZERODE is used instead. */

   register int z, t1, t2;

   t1 = t[dest][i][j]; t2 = t[dest][i+di][j+dj]; z = 0;
   if ((t1 > t2) && (t1 > ZERODE)) {
      if (t2 < ZERODE) t2 = ZERODE;
      z = ((t1 - t2 + ERODERND) >> 3); }
   else if ((t2 > t1) && (t2 > ZERODE)) {
      if (t1 < ZERODE) t1 = ZERODE;
      z = -((t2 - t1 + ERODERND) >> 3); }
   if (z) { e[i][j] -= z; e[i+di][j+dj] += z; } }
