/* This program is Copyright (c) 1991 David Allen.  It may be freely
   distributed as long as you leave my name and copyright notice on it.
   I'd really like your comments and feedback; send e-mail to
   allen@viewlogic.com, or send us-mail to David Allen, 10 O'Moore
   Avenue, Maynard, MA 01754. */

/* This file contains the routines to compute global high and low
   pressure areas. */

#include "const.h"
#include "clim.h"

/* These are the data arrays required: l is an input array defining the
   ocean, land and mountain areas; ts is the temperature array computed
   by the functions in heat.c.  Array pr is filled by ocean(), land() and
   heateq(), below; PR_HIGH indicates a high pressure zone, PR_LOW indicates
   a low, and PR_HEQ indicates the heat equator, a low zone.  Array pm is
   the output array for this file, and it contains an edge map which has
   edges in color 1 surrounding lows and color 0 around highs.  Array r is
   temporary storage for local calls to range() in main.c. */

extern unsigned char l[MAXX][MAXY], ts[MAXB][MAXX][MAXY];
static char r[MAXX][MAXY];
static unsigned char pm[MAXB][MAXX][MAXY];
unsigned char pr[MAXB][MAXX][MAXY];


/* The externs below are declared in main.c; they represent global parameters.
   The ints are described in params.doc. */

extern int BSIZE, XSIZE, YSIZE, PRINTMODE;
int OOTHRESH = 5, OLTHRESH = 1, LOTHRESH = 3, LLTHRESH = 7;
int OHMIN = 130, OHMAX = 180, OLMIN =  40, OLMAX =  65;
int LHMIN =   0, LHMAX =  20, LLMIN = 220, LLMAX = 255;
int PRINTPR = 0;


presspar (s) char *s; {
   /* This function is called by mainpar() in main.c; it simply tests input
   parameters to see if they are defined in this file.  Each of the above
   ints is referred to in this function.  If an input string matches here,
   the function returns true. */
   if      (CMP ("OOTHRESH")) getlng  (&OOTHRESH, M_PRESS);
   else if (CMP ("OLTHRESH")) getlng  (&OLTHRESH, M_PRESS);
   else if (CMP ("OHMIN"))    getlng  (&OHMIN,    M_PRESS);
   else if (CMP ("OHMAX"))    getlng  (&OHMAX,    M_PRESS);
   else if (CMP ("OLMIN"))    getlng  (&OLMIN,    M_PRESS);
   else if (CMP ("OLMAX"))    getlng  (&OLMAX,    M_PRESS);

   else if (CMP ("LOTHRESH")) getlng  (&LOTHRESH, M_PRESS);
   else if (CMP ("LLTHRESH")) getlng  (&LLTHRESH, M_PRESS);
   else if (CMP ("LHMIN"))    getlng  (&LHMIN,    M_PRESS);
   else if (CMP ("LHMAX"))    getlng  (&LHMAX,    M_PRESS);
   else if (CMP ("LLMIN"))    getlng  (&LLMIN,    M_PRESS);
   else if (CMP ("LLMAX"))    getlng  (&LLMAX,    M_PRESS);

   else if (CMP ("PRINTPR"))  getlng  (&PRINTPR,  M_PRESS);
   else return (0);
   return (1); }


presscomp () {
   /* The main routine for this file.  It just calls the four routines
   which do all the work.  Ocean() finds pressure extremes on the ocean;
   land() does the same for land; heateq() defines the heat equator, and
   setpm() computes pm[][] from pr[][]. */

   int buf;

   for (buf=0; buf<BSIZE; buf++) {
      status (M_PRESS, buf); ocean (buf); land (buf);
      checkmouse (); findheq (buf); setpm (buf); }
   if (PRINTPR) for (buf=0; buf<BSIZE; buf++) {
      if (PRINTMODE == PRINTMODE_GREY)
         putmat ("PRESSURE", buf, PRINTMODE_SHORT, l, pm[buf]);
      else putmat ("PRESSURE", buf, PRINTMODE_SHORT, pr[buf], 0); } }


pressdraw (n) int n; { draw (DRAW_LAND, LINE_CORN, l, pm[n]); }
   /* This function calls draw with the right arguments to display pressure */


ocean (buf) int buf; {
   /* Determine ocean highs and lows.  An ocean high or low must occur over
   ocean, far away from major land masses.  Two calls to range() are made
   to find the qualifying ocean areas; then temperature criteria are used
   to select the actual pressure zones. */

   register int i, j; int x;

   /* Set r to the distance on land from the coast. */
   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++) r[i][j] = l[i][j] ? 0 : -1;
   range (r);

   /* Initialize r to contain blobs on land which are at least OLTHRESH squares
      away from the coast.  Then set r to the distance from these.  The result
      in r is the distance from the nearest big piece of land (ignoring
      islands). */
   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++)
      r[i][j] = (r[i][j] > OLTHRESH) ? -1 : 0;
   range (r);

   /* For each array element, if it is at least OOTHRESH squares from the
      nearest big piece of land, it might be the center of an ocean pressure
      zone.  The pressure zones are defined by temperature ranges; if the
      temperature in ts is between OLMIN and OLMAX, a low is recorded, while
      if the temperature is between OHMIN and OHMAX, a high is recorded. */
   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++) {
      pr[buf][i][j] = 0; x = ts[buf][i][j];
      if (r[i][j] > OOTHRESH) {
         if ((x >= OLMIN) && (x <= OLMAX)) pr[buf][i][j] = PR_LOW;
         if ((x >= OHMIN) && (x <= OHMAX)) pr[buf][i][j] = PR_HIGH; } } }


land (buf) int buf; {
   /* This function is simply the complement of ocean(): it finds land highs
   and lows.  A land high or low must occur over land, far from major oceans.
   Two calls to range() are made to find the qualifying land areas; then
   temperature criteria are used to select the actual pressure zones. */

   register int i, j; int x;

   /* Set r to distance on water from coast. */
   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++) r[i][j] = l[i][j] ? -1 : 0;
   range (r);

   /* Initialize r to contain blobs on ocean which are at least LOTHRESH
      squares away from the coast.  Then set r to the distance from these.  The
      result in r is the distance from the nearest ocean, ignoring lakes. */
   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++)
      r[i][j] = (r[i][j] > LOTHRESH) ? -1 : 0;
   range (r);

   /* For each array element, if it is at least LLTHRESH squares from the
      nearest large ocean, it might be the center of a land pressure zone.
      The pressure zones are defined by temperature ranges; if the temperature
      in ts is between LLMIN and LLMAX, a low is recorded, while if the
      temperature is between LHMIN and LHMAX, a high is recorded. */
   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++) {
      x = ts[buf][i][j];
      if (r[i][j] > LLTHRESH) {
         if ((x >= LLMIN) && (x <= LLMAX)) pr[buf][i][j] = PR_LOW;
         if ((x >= LHMIN) && (x <= LHMAX)) pr[buf][i][j] = PR_HIGH; } } }


findheq (buf) int buf; {
   /* This function finds the heat equator and marks it in pr.  For each
   vertical column of ts, the median position is found and marked.  To
   make the heat equator continuous, jlast is set to the position of the
   heat equator in the previous column; a connection is made in the present
   column to ensure continuity. */

   register int i, j; int sum, jlast = 0, jnext;

   for (i=0; i<XSIZE; i++) {
      /* Find the total of the temperatures in this column */
      for (sum=0, j=0; j<YSIZE; j++) sum += ts[buf][i][j];

      /* Step through the column again until the total so far is exactly
         half the total for the column.  This is the median position. */
      for (sum>>=1, j=0; j<YSIZE && sum>0; j++) sum -= ts[buf][i][j];

      /* Mark this position and remember it with jnext */
      pr[buf][i][j] = PR_HEQ; jnext = j;

      /* For each column except the first (where i = 0), if the last heat
         equator is above this one, move upwards to it, marking each square,
         to ensure continuity; if below this one, move downwards to it. */

      if (i && (j > jlast)) for (; j>=jlast; j--) pr[buf][i][j] = PR_HEQ;
      else if (i && (j < jlast)) for (; j<=jlast; j++) pr[buf][i][j] = PR_HEQ;

      /* Remember this position for the next column.  Note that no check is
         done to ensure continuity at the wraparound point; this is bad. */
      jlast = jnext; } }


setpm (buf) int buf; {
   /* Setpm() is called after the above three functions have filled pr with
   the codes for high, low and heat equator.  The purpose of this function
   is to create an edge map surrounding lows with color 1 and highs with
   color 0. */

   register int i, j, k; int col;

   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++) {
      k = pr[buf][i][j]; col = 0;

      /* If not at the top edge, and if the pressure status here is not equal
         to the pressure status one square up, then put a horizontal line in
         this square.  The color is zero if there is a high here; if a low or
         heat equator (a type of low) is here, the color is one. */
      if (j) if (k != pr[buf][i][j-1]) col =
         ((k == PR_HIGH) || (pr[buf][i][j-1] == PR_HIGH)) ? LINE_0H : LINE_1H;
      /* Similarly, if not at the left edge, put a vertical line in the right
         color.  Notice that this color is OR'ed with the previous color. */
      if (i) if (k != pr[buf][i-1][j]) col |=
         ((k == PR_HIGH) || (pr[buf][i-1][j] == PR_HIGH)) ? LINE_0V : LINE_1V;

      /* Set the square in the pm array to the resultant color */
      pm[buf][i][j] = col; } }

