/* This program is Copyright (c) 1991 David Allen.  It may be freely
   distributed as long as you leave my name and copyright notice on it.
   I'd really like your comments and feedback; send e-mail to
   allen@viewlogic.com, or send us-mail to David Allen, 10 O'Moore
   Avenue, Maynard, MA 01754. */

/* This file contains the routines that compute local climate */

#include "const.h"
#include "clim.h"


/* Input arrays are l from main.c, rn from rain.c, and t from heat.c.  Note
   that the unscaled temperatures are used.  The output array is cl.  Array
   lm (from main.c) is used by the drawing routine.*/

extern unsigned char l[MAXX][MAXY], lm[MAXX][MAXY], rn[MAXB][MAXX][MAXY];
extern int tt[MAXB][MAXX][MAXY];
unsigned char cl[MAXX][MAXY];


/* These parameters are defined in main.c or heat.c */
extern int BSIZE, XSIZE, YSIZE, TEMPSCALE, PRINTMODE;

/* These parameters are used here, and are described in params.doc */
int tempcut[] = { 0, 40, 90, 120 }, raincut[] = { 40, 60, 110, 160, 180 };
int PRINTCLIM = 0, MTDELTA = 20, TCSIZE = 4, RCSIZE = 5;
double ICEBERGK = 263.0;


/* This array is the heart of the climate routine; temperature increases
   going down the array, and rainfall increases going from left to right. */

static unsigned char climkey[4][5] = {
   { C_TUNDRA, C_TUNDRA, C_TUNDRA, C_TUNDRA, C_TUNDRA },
   { C_STEPPE, C_STEPPE, C_DECID,  C_DECID,  C_DECID  },
   { C_DESERT, C_SAVANA, C_DECID,  C_JUNGLE, C_SWAMP  },
   { C_DESERT, C_SAVANA, C_JUNGLE, C_SWAMP,  C_SWAMP  } };


climpar (s) char *s; {
   /* This function is called by mainpar() in main.c; it simply tests input
   parameter names to see if they are defined in this file.  Each of the
   above ints are defined in this file.  If the input string matches here,
   the function returns true. */
   if      (CMP ("ICEBERGK"))    getdbl  (&ICEBERGK,        M_CLIM);
   else if (CMP ("PRINTCLIM"))   getlng  (&PRINTCLIM,       M_CLIM);
   else if (CMP ("TEMPCUT"))     getlvec (&TCSIZE, tempcut, M_CLIM);
   else if (CMP ("RAINCUT"))     getlvec (&RCSIZE, raincut, M_CLIM);
   else if (CMP ("MTDELTA"))     getlng  (&MTDELTA,         M_CLIM);
   else if (CMP ("PRINTCLIM"))   getlng  (&PRINTCLIM,       M_CLIM);
   else return (0);
   return (1); }


climdraw () { draw (DRAW_CLIM, LINE_CORN, cl, lm); }
   /* This routine calls draw with the proper arguments to display climate */


climcomp () {
   /* The outer loop looks at each square.  If it is ocean, the climate will
   be ocean unless the temperature is below ICEBREGK degrees all year round.
   If it is land, then the average rainfall and temperature (in Farenheit) are
   computed for the square.  If the square is mountain, it is colder; the
   temperature is decreased.  These two figures are then turned into array
   indices by using the tempcut and raincut parameter vectors.  The climate
   for the square is then simply a table lookup.  Finally, the array is printed
   if desired. */

   register int i, j, buf;
   int noice, avetemp, averain, ttt, r;

   usermessage ("Computing climate"); checkmouse ();
   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++) {
      if (!l[i][j]) { /* ocean */
         for (noice=0, buf=0; buf<BSIZE; buf++)
            noice |= (tt[buf][i][j] > TEMPSCALE * ICEBERGK);
         cl[i][j] = noice ? C_OCEAN : C_OCEANICE; }
      else { /* land or mountain */
         for (averain=0, avetemp=0, buf=0; buf<BSIZE; buf++) {
            averain += rn[buf][i][j]; avetemp += tt[buf][i][j]; }
         averain /= BSIZE; avetemp /= BSIZE;
         avetemp = ((double) (avetemp / TEMPSCALE) - 273.0) * 1.8 + 32.0;
         if (l[i][j] == 2) avetemp -= MTDELTA;
         ttt = 0; while ((avetemp > tempcut[ttt]) && (ttt < TCSIZE-1)) ttt++;
         r  = 0; while ((averain > raincut[r])  && (r  < RCSIZE-1)) r++;
         cl[i][j] = climkey[ttt][r]; } }
   if (PRINTCLIM) {
      if (PRINTMODE == PRINTMODE_GREY)
         putmat ("CLIMATE", -1, PRINTMODE_CLIM, cl, lm);
      else putmat ("CLIMATE", -1, PRINTMODE_SHORT, cl, lm); } }
