/* RWDOCS */
/* Cohulip version, 32 bit strategy, optimized for speed
 * and cpu time, rather than space.
 */

/* Primitive mbuf allocate/free routines */
/* With version k29, adopted the pointer and count of mbufs duped with dup_p
 * to correct a problem where packets in ax.25 queue were garbaged when tcpin
 * freed acked data.  This was preferable to using copy_p in tcpout.c 0 K5JB
 */

#include "global.h"
#include "mbuf.h"

extern int disable();
extern void restore();

#undef MBUFDEBUG	/* See also alloc.c */

char *memcpy();

/* Allocate mbuf with associated buffer of 'size' bytes */
struct mbuf *
alloc_mbuf(size)
register int32 size;
{

/* RWDOCE */
	register struct mbuf *bp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif

	if((bp = (struct mbuf *)malloc((unsigned)(size + sizeof(struct mbuf)))) == NULLBUF)
		return NULLBUF;

	bp->next = bp->anext = NULLBUF;
	if(size != 0){
		bp->data = (char *)(bp + 1);
	} else {
		bp->data = NULLCHAR;
	}
	bp->size = size;
	bp->cnt = 0;
	bp->refcnt = 1;
	bp->dup = NULLBUF;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return bp;
}

/* RWDOCS */
/* Free all resources associated with mbuf
 * Return pointer to next mbuf in packet chain
 */
struct mbuf *
free_mbuf(bp)
register struct mbuf *bp;
{

/* RWDOCE */
	void free();
	struct mbuf *bpnext;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(bp == NULLBUF) {
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return NULLBUF;
		}

	bpnext = bp->next;

	if(bp->dup != NULLBUF) {	/* taken from NOS */
		free_mbuf(bp->dup);
		bp->dup = NULLBUF;
	}
	if(--bp->refcnt <= 0) {
		bp->next = NULLBUF;	/* detect attempts to use */
		bp->data = NULLCHAR;	/* a freed mbuf */
		free((char *)bp);
	}
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return bpnext;
}

/* RWDOCS */
/* Free packet (a chain of mbufs). Return pointer to next packet on queue,
 * if any
 */
struct mbuf *
free_p(bp)
register struct mbuf *bp;
{

/* RWDOCE */
	struct mbuf *abp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(bp == NULLBUF) {
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return NULLBUF;
		}
	abp = bp->anext;
	while(bp != NULLBUF)
		bp = free_mbuf(bp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return abp;
}

/* RWDOCS */
/* Free entire queue of packets (of mbufs) */
void 
free_q(q)
struct mbuf **q;
{

/* RWDOCE */
	register struct mbuf *bp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	while((bp = dequeue(q)) != NULLBUF)
		free_p(bp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
}

/* RWDOCS */
/* Count up the total number of bytes in an mbuf chain */
int32
len_mbuf(bp)
register struct mbuf *bp;
{

/* RWDOCE */
	int32 cnt;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	cnt = 0;
	while(bp != NULLBUF){
		cnt += bp->cnt;
		bp = bp->next;
	}
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return cnt;
}

/* RWDOCS */
/* Count up the number of packets in a queue */
int32
len_q(bp)
register struct mbuf *bp;
{

/* RWDOCE */
	register int32 cnt;
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif

	for(cnt=0;bp != NULLBUF;cnt++,bp = bp->anext)
		;
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return cnt;
}

/* RWDOCS */
/* Trim mbuf to specified length by lopping off end */
void
trim_mbuf(bpp,length)
struct mbuf **bpp;
int32 length;
{

/* RWDOCE */
	register struct mbuf *bp;
	register int32 tot = 0;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(bpp == NULLBUFP || *bpp == NULLBUF) {
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return;	/* Nothing to trim */
		}

	if(length == 0) {
		/* Toss the whole thing */
		free_p(*bpp);
		*bpp = NULLBUF;
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return;
	}
	/* Find the point at which to trim. If length is greater than
	 * the packet, we'll just fall through without doing anything
	 */
	for( bp = *bpp; bp != NULLBUF; bp = bp->next){
		if(tot + bp->cnt < length){
			tot += bp->cnt;
		} else {
			/* Cut here */
			bp->cnt = length - tot;
			free_p(bp->next);
			bp->next = NULLBUF;
			break;
		}
	}
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
}

/* RWDOCS */
/* Duplicate/enqueue/dequeue operations based on mbufs */

/* Duplicate first 'cnt' bytes of packet starting at 'offset'.
 * This is done without copying data; only the headers are duplicated,
 * but without data segments of their own. The pointers are set up to
 * share the data segments of the original copy. The return pointer is
 * passed back through the first argument, and the return value is the
 * number of bytes actually duplicated.
 */
int32
dup_p(hp,bp,offset,cnt)
struct mbuf **hp;
register struct mbuf *bp;
register int32 offset;
register int32 cnt;
{

/* RWDOCE */
	register struct mbuf *cp;
	int32 tot;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(cnt == 0 || bp == NULLBUF || hp == NULLBUFP) {
		if(hp != NULLBUFP)
			*hp = NULLBUF;
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return 0;
	}

	if((*hp = cp = alloc_mbuf(0)) == NULLBUF){
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return 0;
	}

	/* Skip over leading mbufs that are smaller than the offset */
	while(bp != NULLBUF && bp->cnt <= offset){
		offset -= bp->cnt;
		bp = bp->next;
	}

	if(bp == NULLBUF) {
		free_mbuf(cp);
		*hp = NULLBUF;
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return 0;	/* Offset was too big */
	}

	tot = 0;
	for(;;){
		/* Make sure we get the original, "real" buffer (i.e. handle the
		 * case of duping a dupe)
		 */
		if(bp->dup != NULLBUF)
			cp->dup = bp->dup;
		else
			cp->dup = bp;

		/* Increment the duplicated buffer's reference count */
		cp->dup->refcnt++;

		cp->data = bp->data + offset;
		cp->cnt = min(cnt,bp->cnt - offset);
		offset = 0;
		cnt -= cp->cnt;
		tot += cp->cnt;
		bp = bp->next;
		if(cnt == 0 || bp == NULLBUF || (cp->next = alloc_mbuf(0)) == NULLBUF)
			break;
		cp = cp->next;
	}
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return tot;
}

/* RWDOCS */
/* Copy first 'cnt' bytes of packet into a new, single mbuf */
struct mbuf *
copy_p(bp,cnt)
register struct mbuf *bp;
register int32 cnt;
{

/* RWDOCE */
	register struct mbuf *cp;
	register char *wp;
	register int32 n;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(bp == NULLBUF || cnt == 0 || (cp = alloc_mbuf(cnt)) == NULLBUF) {
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return NULLBUF;
		}
	wp = cp->data;
	while(cnt != 0 && bp != NULLBUF){
		n = min(cnt,bp->cnt);
		memcpy(wp,bp->data,n);
		wp += n;
		cp->cnt += n;
		cnt -= n;
		bp = bp->next;
	}
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return cp;
}

/* RWDOCS */
/* Copy and delete "cnt" bytes from beginning of packet. Return number of
 * bytes actually pulled off - dinked with this while looking for cause
 * of packets being garbaged by tcpin when they are in the ax25 transmit
 * queue.  Think I fixed it with a copy_p() in tcpout so restored this
 * to original - K5JB (k29)
 */
int32
pullup(bph,buf,cnt)
struct mbuf **bph;
char *buf;
int32 cnt;
{

/* RWDOCE */
	register struct mbuf *bp;
	int32 n,tot;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	tot = 0;
	if(bph == NULLBUFP)
		return 0;

	while(cnt != 0 && ((bp = *bph) != NULLBUF)) {
		n = min(cnt,bp->cnt);
		if(buf != NULLCHAR && n != 0){
			memcpy(buf,bp->data,n);
			buf += n;
		}
		tot += n;
		cnt -= n;
		bp->data += n;
		bp->cnt -= n;
		if(bp->cnt == 0)
			*bph = free_mbuf(bp);
	} /* while */
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return tot;
}

/* RWDOCS */
/* Append mbuf to end of mbuf chain */
void
append(bph,bp)
struct mbuf **bph;
struct mbuf *bp;
{

/* RWDOCE */
	register struct mbuf *p;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(bph == NULLBUFP || bp == NULLBUF) {
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return;
		}
	if(*bph == NULLBUF){
		/* First one on chain */
		*bph = bp;
	} else {
		for(p = *bph ; p->next != NULLBUF ; p = p->next)
			;
		p->next = bp;
	}
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
}

/* RWDOCS */
/* Insert specified amount of contiguous new space at the beginning of an
 * mbuf chain. If enough space is available in the first mbuf, no new space
 * is allocated. Otherwise a mbuf of the appropriate size is allocated and
 * tacked on the front of the chain.
 *
 * This operation is the logical inverse of pullup(), hence the name.
 */
struct mbuf *
pushdown(bp,size)
register struct mbuf *bp;
int32 size;
{

/* RWDOCE */
	register struct mbuf *nbp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	/* Check that bp is real and that there's data space associated with
	 * this buffer (i.e., this is not a buffer from dup_p) before
	 * checking to see if there's enough space at its front
	 */
	if(bp != NULLBUF && bp->refcnt == 1 && bp->dup == NULLBUF
		&& bp->data - (char *)(bp+1) >= size){
		/* No need to alloc new mbuf, just adjust this one */

		bp->data -= size;
		bp->cnt += size;
	} else {

		if((nbp = alloc_mbuf(size)) != NULLBUF){
			nbp->next = bp;
			nbp->cnt = size;
			bp = nbp;
		} else {

			bp = NULLBUF;
		}
	}
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return bp;
}

/* RWDOCS */
/* Append packet to end of packet queue */
void
enqueue(q,bp)
struct mbuf **q;
struct mbuf *bp;
{

/* RWDOCE */
	register struct mbuf *p;
	int32 i_state;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(q == NULLBUFP || bp == NULLBUF) {
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return;
		}
	i_state = disable();
	if(*q == NULLBUF){
		/* List is empty, stick at front */
		*q = bp;
	} else {
		for(p = *q ; p->anext != NULLBUF ; p = p->anext)
			;
		p->anext = bp;
	}
	restore(i_state);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif

}

/* RWDOCS */
/* Unlink a packet from the head of the queue */
struct mbuf *
dequeue(q)
register struct mbuf **q;
{

/* RWDOCE */
	register struct mbuf *bp;
	int32 i_state;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(q == NULLBUFP) {
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return NULLBUF;
		}

	i_state = disable();
	if((bp = *q) != NULLBUF){
		*q = bp->anext;
		bp->anext = NULLBUF;
	}
	restore(i_state);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return bp;
}

/* RWDOCS */
/* Copy user data into an mbuf */
struct mbuf *
qdata(data,cnt)
char *data;
int32 cnt;
{

/* RWDOCE */
	register struct mbuf *bp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if((bp = alloc_mbuf(cnt)) == NULLBUF) {
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return NULLBUF;
		}
	memcpy(bp->data,data,cnt);
	bp->cnt = cnt;
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return bp;
}

/* RWDOCS */
/* Copy mbuf data into user buffer */
int32
dqdata(bp,buf,cnt)
struct mbuf *bp;
char *buf;
int32 cnt;
{

/* RWDOCE */
	int32 n,tot;
	struct mbuf *bp1;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(buf == NULLCHAR) {
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return 0;
		}

	tot = 0;
	for(bp1 = bp;bp1 != NULLBUF; bp1 = bp1->next){
		n = min(bp1->cnt,cnt);
		memcpy(buf,bp1->data,n);
		cnt -= n;
		buf += n;
		tot += n;
	}
	free_p(bp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return tot;
}

/* RWDOCS */
/* Pull a 32-bit integer in host order from buffer in network byte order */
int32
pull32(bpp)
struct mbuf **bpp;
{

/* RWDOCE */
	int32 rval;
	char buf[4];
	register char *cp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(pullup(bpp,buf,4) != 4){
		/* Return zero if insufficient buffer */
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return 0;
	}
	cp = buf;

	/* Unwound for speed */
	rval = (*cp++ & 0xff);
	rval <<= 8;
	rval |= (*cp++ & 0xff);
	rval <<= 8;
	rval |= (*cp++ & 0xff);
	rval <<= 8;
	rval |= (*cp & 0xff);

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return rval;
}

/* RWDOCS */
/* Pull a 16-bit integer in host order from buffer in network byte order */
int32
pull16(bpp)
struct mbuf **bpp;
{

/* RWDOCE */
	int32 rval;
	char buf[2];
	register char *cp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	rval = 0;
	if(pullup(bpp,buf,2) != 2) {
		/* Return zero if insufficient buffer */
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return 0;
	}
	cp = buf;

	rval = (*cp++ & 0xff);
	rval <<= 8;
	rval |= (*cp & 0xff);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return (rval & 0xffff);
}

/* RWDOCS */
/* Pull single character from mbuf */
int32
pullchar(bpp)
struct mbuf **bpp;
{

/* RWDOCE */
	int32 c;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if(pullup(bpp,(char *)&c,1) != 1)
		/* Return zero if nothing left */
		c = 0;
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return (c & 0xff);
}

/* RWDOCS */
/* Put a long in host order into a char array in network order */
char *
put32(cp,x)
register char *cp;
int32 x;
{

/* RWDOCE */
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	*cp++ = (unsigned char) ((x >> 24) & 0xff);
	*cp++ = (unsigned char) ((x >> 16) & 0xff);
	*cp++ = (unsigned char) ((x >> 8) & 0xff);
	*cp++ = (unsigned char) (x & 0xff);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return cp;
}

/* RWDOCS */
/* Put a short in host order into a char array in network order */
char *
put16(cp,x)
register char *cp;
int32 x;
{

/* RWDOCE */
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	*cp++ = (unsigned char) ((x >> 8) & 0xff);
	*cp++ = (unsigned char) (x & 0xff);

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return cp;
}
