/*

 Copyright (C) 1990,1991 Mark Adler, Richard B. Wales, and Jean-loup Gailly.
 Permission is granted to any individual or institution to use, copy, or
 redistribute this software so long as all of the original files are included
 unmodified, that it is not sold for profit, and that this copyright notice
 is retained.

*/

/*
 *  zipcloak.c by Mark Adler.
 */

#define UTIL
#include "revision.h"
#include "zip.h"
#include <signal.h>

#define PWLEN 80        /* Input buffer size for reading encryption key */


/* Temporary zip file name and file pointer */
local char *tempzip;
local FILE *tempzf;


/* Local functions */
#ifdef PROTO
   local void err(int, char *);
   local void handler(int);
   local void license(void);
   local void help(void);
   void main(int, char **);
#endif /* PROTO */



local void err(c, h)
int c;                  /* error code from the ZE_ class */
char *h;                /* message about how it happened */
/* Issue a message for the error, clean up files and memory, and exit. */
{
  if (PERR(c))
    perror("zipcloak error");
  fprintf(stderr, "zipcloak error: %s (%s)\n", errors[c-1], h);
  if (tempzf != NULL)
    fclose(tempzf);
  if (tempzip != NULL)
  {
    destroy(tempzip);
    free((voidp *)tempzip);
  }
  if (zipfile != NULL)
    free((voidp *)zipfile);
#ifdef VMS
  exit(0);
#else /* !VMS */
  exit(c);
#endif /* ?VMS */
}


local void handler(s)
int s;                  /* signal number (ignored) */
/* Upon getting a user interrupt, turn echo back on for tty and abort
   cleanly using err(). */
{
#ifndef MSVMS
  echon();
  putc('\n', stderr);
#endif /* !MSVMS */
  err(ZE_ABORT, "aborting");
  s++;                                  /* keep some compilers happy */
}


void warn(a, b)
char *a, *b;            /* message strings juxtaposed in output */
/* Print a warning message to stderr and return. */
{
  fprintf(stderr, "zipcloak warning: %s%s\n", a, b);
}


local void license()
/* Print license information to stdout. */
{
  extent i;             /* counter for copyright array */

  for (i = 0; i < sizeof(copyright)/sizeof(char *); i++)
    puts(copyright[i]);
  for (i = 0; i < sizeof(disclaimer)/sizeof(char *); i++)
    puts(disclaimer[i]);
}


local void help()
/* Print help (along with license info) to stdout. */
{
  extent i;             /* counter for help array */

  /* help array */
  static char *text[] = {
"",
"ZipCloak %d.%d (%s)",
"Usage:  zipcloak [-d] [-b path] zipfile",
"  the default action is to encrypt all unencrypted entries in the zip file",
"  -d   decrypt--decrypt encrypted entries (copy if given wrong password)",
"  -b   use \"path\" for the temporary zip file",
"  -h   show this help               -l   show software license"
  };

  for (i = 0; i < sizeof(copyright)/sizeof(char *); i++)
    puts(copyright[i]);
  for (i = 0; i < sizeof(text)/sizeof(char *); i++)
  {
    printf(text[i], REVISION / 10, REVISION % 10, REVDATE);
    putchar('\n');
  }
}


void main(argc, argv)
int argc;               /* number of tokens in command line */
char **argv;            /* command line tokens */
/* Encrypt or decrypt all of the entries in a zip file.  See the command
   help in help() above. */
{
  int a;                /* attributes of zip file */
  ulg c;                /* start of central directory */
  int d;                /* decryption flag */
  int k;                /* next argument type */
  char p[PWLEN+1];      /* password for encryption or decryption */
  char *q;              /* steps through option arguments */
  int r;                /* arg counter, temporary variable */
  ulg s;                /* length of central directory */
  FILE *x, *y;          /* input and output zip files */
  struct zlist far *z;  /* steps through zfiles linked list */


  /* If no args, show help */
  if (argc == 1)
  {
    help();
    exit(0);
  }

  /* Go through args */
  zipfile = tempzip = NULL;
  tempzf = NULL;
  signal(SIGINT, handler);
  signal(SIGTERM, handler);
  k = d = 0;
  for (r = 1; r < argc; r++)
    if (*argv[r] == '-')
      if (argv[r][1])
        for (q = argv[r]+1; *q; q++)
          switch(*q)
          {
            case 'b':   /* Specify path for temporary file */
              if (k)
                err(ZE_PARMS, "use -b before zip file name");
              else
                k = 1;          /* Next non-option is path */
              break;
            case 'd':
              d = 1;  break;
            case 'h':   /* Show help */
              help();  exit(0);
            case 'l':   /* Show copyright and disclaimer */
              license();  exit(0);
            default:
              err(ZE_PARMS, "unknown option");
          }
      else
        err(ZE_PARMS, "zip file cannot be stdin");
    else
      if (k == 0)
        if (zipfile == NULL)
        {
          if ((zipfile = ziptyp(argv[r])) == NULL)
            err(ZE_MEM, "was processing arguments");
        }
        else
          err(ZE_PARMS, "can only specify one zip file");
      else
      {
        tempath = argv[r];
        k = 0;
      }
  if (zipfile == NULL)
    err(ZE_PARMS, "need to specify zip file");

  /* Read zip file */
  if ((r = readzipfile()) != ZE_OK)
    err(r, zipfile);
  if (zfiles == NULL)
    err(ZE_NAME, zipfile);

  /* Check for something to do */
  for (z = zfiles; z != NULL; z = z->nxt)
    if (d ? z->flg & 1 : !(z->flg & 1))
      break;
  if (z == NULL)
    err(ZE_NONE, d ? "no encrypted files" : "all files encrypted already");

  /* Before we get carried away, make sure zip file is writeable */
  if ((x = fopen(zipfile, "a")) == NULL)
    err(ZE_CREAT, zipfile);
  fclose(x);
  a = getfileattr(zipfile);

  /* Open output zip file for writing */
  if ((tempzf = y = fopen(tempzip = tempname('Z'), FOPW)) == NULL)
    err(ZE_TEMP, tempzip);

  /* Get password */
  if (getp("Enter password: ", p, PWLEN+1) == NULL)
    err(ZE_PARMS, "stderr is not a tty (you may never see this message!)");

  /* Open input zip file again, copy preamble if any */
  if ((x = fopen(zipfile, FOPR)) == NULL)
    err(ZE_NAME, zipfile);
  if (zipbeg && (r = fcopy(x, y, zipbeg)) != ZE_OK)
    err(r, r == ZE_TEMP ? tempzip : zipfile);

  /* Go through local entries, copying, encrypting, or decrypting */
  for (z = zfiles; z != NULL; z = z->nxt)
    if (d && (z->flg & 1))
    {
      printf("decrypting %s", z->zname);
      fflush(stdout);
      if ((r = zipbare(z, x, y, p)) != ZE_OK)
        if (r == ZE_MISS)
          printf(" (wrong password--just copying)");
        else
          err(r, "was decrypting an entry");
      putchar('\n');
    }
    else if ((!d) && !(z->flg & 1))
    {
      printf("encrypting %s\n", z->zname);
      fflush(stdout);
      if ((r = zipcloak(z, x, y, p)) != ZE_OK)
        err(r, "was encrypting an entry");
    }
    else
    {
      printf("copying %s\n", z->zname);
      fflush(stdout);
      if ((r = zipcopy(z, x, y)) != ZE_OK)
        err(r, "was copying an entry");
    }
  fclose(x);

  /* Write central directory and end of central directory */
  if ((c = ftell(y)) == -1L)    /* get start of central */
    err(ZE_TEMP, tempzip);
  for (z = zfiles; z != NULL; z = z->nxt)
    if ((r = putcentral(z, y)) != ZE_OK)
      err(r, tempzip);
  if ((s = ftell(y)) == -1L)    /* get end of central */
    err(ZE_TEMP, tempzip);
  s -= c;                       /* compute length of central */
  if ((r = putend((int)zcount, s, c, zcomlen, zcomment, y)) != ZE_OK)
    err(r, tempzip);
  tempzf = NULL;
  if (fclose(y))
    err(ZE_TEMP, tempzip);
  if ((r = replace(zipfile, tempzip)) != ZE_OK)
  {
    warn("new zip file left as: ", tempzip);
    free((voidp *)tempzip);
    tempzip = NULL;
    err(r, "was replacing the original zip file");
  }
  free((voidp *)tempzip);
  tempzip = NULL;
  setfileattr(zipfile, a);
  free((voidp *)zipfile);
  zipfile = NULL;

  /* Done! */
  exit(0);
}
