/*

 Copyright (C) 1990,1991 Mark Adler, Richard B. Wales, and Jean-loup Gailly.
 Permission is granted to any individual or institution to use, copy, or
 redistribute this software so long as all of the original files are included
 unmodified, that it is not sold for profit, and that this copyright notice
 is retained.

*/

/*
 *  crypt.c by Mark Adler.
 */

/* These encryption routines should not be distributed outside the U.S.A. */

/* The encryption algorithm is from Roger Schlafly as supplied to Phil Katz
   for the original PKZIP and PKUNZIP programs, documented in the file
   appnote.txt that comes with those programs. */

#include "zip.h"
#include <time.h>

void srand OF((unsigned int));
int rand OF((void));


/* Local functions */
#ifdef PROTO
   local int pseudo(ulg *);
   local void updkeys(ulg *, int);
   local void setkeys(ulg *, char *);
#endif /* PROTO */


local int pseudo(k)
ulg *k;                 /* pointer to three long keys */
/* Return the next byte in the psuedo-random sequence defined by the sequence
   of key (k[]) values. */
{
  ush t;                /* temporary variable */

  t = (ush)k[2] | 2;
  return ((ush)(t * (t ^ 1)) & 0xffff) >> 8;
}


local void updkeys(k, c)
ulg *k;                 /* pointer to three long keys */
int c;                  /* byte to update keys with */
/* Update the keys k[] with the byte c. */
{
  k[0] = crc32(k[0], c);
  k[1] += k[0] & 0xff;
  k[1] = k[1] * 134775813L + 1;
  k[2] = crc32(k[2], (int)(k[1] >> 24));
}


local void setkeys(k, p)
ulg *k;                 /* pointer to three long keys */
char *p;                /* password string to modify keys with */
/* Initialize the keys k[] with the password *p. */
{
  int c;                /* temporary variable */

  k[0] = 305419896L;
  k[1] = 591751049L;
  k[2] = 878082192L;
  while ((c = *p++) != 0)
    updkeys(k, c);
}


/* These macros encode and decode a single byte, b, using the keys k.
   The argument b is evaluated only once.  They both use the temporary
   int t, and ENCODE uses the temporary int s.  */
#define ENCODE(b,k) (t=pseudo(k),updkeys(k,s=(int)(b)&0xff),s^t)
#define DECODE(b,k) (updkeys(k,t=((b)&0xff)^pseudo(k)),t)


local ulg keys[3];

void crypthead(p, c, y)
char *p;                /* password string */
ulg c;                  /* crc of file being encrypted */
FILE *y;                /* where to write header to */
/* Start encryption process and write encryption header to file *y using
   the password *p and the file cyclic redundancy check c. */
{
  int n;                /* counter for 10 random bytes */
  int s, t;             /* used by ENCODE macro */

  /* Initialize keys with password */
  setkeys(keys, p);

  /* Encrypt random header (last two bytes is high word of crc) */
  srand((unsigned int)time(NULL));      /* initialize generator */
  for (n = 10; n; n--)
    putc(ENCODE(rand() >> 7, keys), y);
  putc(ENCODE(c >> 16, keys), y);
  putc(ENCODE(c >> 24, keys), y);
}


#ifdef UTIL

int zipcloak(z, x, y, p)
struct zlist far *z;    /* zip entry to encrypt */
FILE *x, *y;            /* source and destination files */
char *p;                /* password string */
/* Encrypt the zip entry described by *z from file *x to file *y using the
   password *p.  Return an error code in the ZE_ class. */
{
  int c;                /* temporary variable */
  ulg n;                /* holds offset and counts size */
  int s, t;             /* used by ENCODE macro */


  /* Set encrypted bit and write local header to output file */
  if ((n = ftell(y)) == -1L)
    return ZE_TEMP;
  z->off = n;
  z->flg |= 1;
  z->lflg |= 1;
  z->siz += 12;
  if ((c = putlocal(z, y)) != ZE_OK)
    return c;

  /* Initialize keys with password and write random header */
  crypthead(p, z->crc, y);

  /* Skip local header in input file */
  if (fseek(x, (long)(4 + LOCHEAD + (ulg)z->nam + (ulg)z->ext), SEEK_CUR))
    return ferror(x) ? ZE_READ : ZE_EOF;

  /* Encrypt data */
  for (n = z->siz - 12; n; n--)
    if ((c = getc(x)) != EOF)
      putc(ENCODE(c, keys), y);
    else
      return ferror(x) ? ZE_READ : ZE_EOF;
  if (fflush(y) == EOF)
    return ZE_TEMP;

  /* Done */
  return ZE_OK;
}


int zipbare(z, x, y, p)
struct zlist far *z;    /* zip entry to encrypt */
FILE *x, *y;            /* source and destination files */
char *p;                /* password string */
/* Decrypt the zip entry described by *z from file *x to file *y using the
   password *p.  Return an error code in the ZE_ class. */
{
  int c;                /* temporary variable */
  ush h;                /* last two bytes of decrypted header */
  ulg n;                /* holds offsets and counts size */
  int r;                /* temporary variable */
  int t;                /* used by DECODE macro */


  /* Save position and skip local header in input file */
  if ((n = ftell(x)) == -1L ||
      fseek(x, (long)(4 + LOCHEAD + (ulg)z->nam + (ulg)z->ext), SEEK_CUR))
    return ferror(x) ? ZE_READ : ZE_EOF;

  /* Initialize keys with password */
  setkeys(keys, p);

  /* Decrypt encryption header, save last two bytes */
  h = 0;
  for (r = 12; r; r--)
    if ((c = getc(x)) != EOF)
      h = (h >> 8) | (DECODE(c, keys) << 8);
    else
      return ferror(x) ? ZE_READ : ZE_EOF;

  /* If last two bytes of header don't match crc, back up and just copy */
  if (h != (ush)(z->crc >> 16))
  {
    if (fseek(x, n, SEEK_SET))
      return ferror(x) ? ZE_READ : ZE_EOF;
    if ((r = zipcopy(z, x, y)) != ZE_OK)
      return r;
    return ZE_MISS;
  }

  /* Clear encrypted bit and write local header to output file */
  if ((n = ftell(y)) == -1L)
    return ZE_TEMP;
  z->off = n;
  z->flg &= ~1;
  z->lflg &= ~1;
  z->siz -= 12;
  if ((r = putlocal(z, y)) != ZE_OK)
    return r;

  /* Decrypt data */
  for (n = z->siz; n; n--)
    if ((c = getc(x)) != EOF)
      putc(DECODE(c, keys), y);
    else
      return ferror(x) ? ZE_READ : ZE_EOF;
  if (fflush(y) == EOF)
    return ZE_TEMP;

  /* Done */
  return ZE_OK;
}


#else /* !UTIL */

int zfwrite(b, m, n, f)
voidp *b;               /* data buffer */
extent m;               /* size of each item in bytes */
extent n;               /* number of items */
FILE *f;                /* file to write to */
/* If requested, encrypt the data in b[], and in any case call fwrite() with
   the arguments to zfwrite().  Return what fwrite() returns. */
{
  if (key != NULL)
  {
    ulg k;              /* counts size */
    char *p;            /* steps through buffer */
    int s, t;           /* used by ENCODE macro */

    /* Encrypt data in buffer */
    for (p = (char *)b, k = m*(ulg)n; k; p++, k--)
      *p = (char)ENCODE(*p, keys);
  }

  /* Write the buffer out */
  return fwrite(b, m, n, f);
}


int zfputc(c, f)
int c;                  /* character to write */
FILE *f;                /* file to write it to */
/* Encrypt the byte c and then do a putc().  The macro zputc defined in
   crypt.h checks keys and calls zfputc if needed.  Return what putc()
   returns. */
{
  int s, t;             /* used by ENCODE macro */

  return putc(ENCODE(c, keys), f);
}

#endif /* ?UTIL */
