/* cohstuff.c - Coherent 386 specific routines            */
/* modified by shendi@saruman.stgt.sub.org from msstuff.c */

#include "xlisp.h"
#include <termio.h>

#define LBSIZE 200

/* external variables */
extern LVAL s_unbound,true;
extern FILE *tfp;
extern int errno;

/* extern functions */
extern char *getenv();

/* local variables */
static char lbuf[LBSIZE];
static int lpos[LBSIZE];
static int lindex;
static int lcount;
static int lposition;
static long rseed = 1L;

static struct termio oldtty, newtty;

/* osinit - initialize */
osinit(banner)
  char *banner;
{
    printf("%s\n",banner);
    lposition = 0;
    lindex = 0;
    lcount = 0;

    ioctl(0, TCGETA, &oldtty);
    newtty = oldtty;
    newtty.c_cc[VTIME] = 0;
    newtty.c_cc[VMIN] = 0;
    newtty.c_lflag &= ~(ISIG | ICANON | ECHO); 
    newtty.c_oflag &= ~(OPOST | OLCUC | ONLCR | OCRNL);
    newtty.c_iflag &= ~(ISTRIP | IUCLC | INPCK | PARMRK);
    newtty.c_iflag |= IXANY;
    ioctl(0, TCSETAF, &newtty);   
        
}

/* osfinish - clean up before returning to the operating system */
osfinish()
{
  ioctl(0, TCSETAF, &oldtty);
}

/* oserror - print an error message */
oserror(msg)
  char *msg;
{
    printf("error: %s\n",msg);
}

/* osrand - return a random number between 0 and n-1 */
int osrand(n)
  int n;
{
    long k1;

    /* make sure we don't get stuck at zero */
    if (rseed == 0L) rseed = 1L;

    /* algorithm taken from Dr. Dobbs Journal, November 1985, page 91 */
    k1 = rseed / 127773L;
    if ((rseed = 16807L * (rseed - k1 * 127773L) - k1 * 2836L) < 0L)
	rseed += 2147483647L;

    /* return a random number between 0 and n-1 */
    return ((int)(rseed % (long)n));
}

/* osaopen - open an ascii file */
FILE *osaopen(name,mode)
  char *name,*mode;
{
    return (fopen(name,mode));
}

/* osbopen - open a binary file */
FILE *osbopen(name,mode)
  char *name,*mode;
{
    char bmode[10];
    strcpy(bmode,mode); strcat(bmode,"b");
    return (fopen(name,bmode));
}

/* osclose - close a file */
int osclose(fp)
  FILE *fp;
{
    return (fclose(fp));
}

/* osagetc - get a character from an ascii file */
int osagetc(fp)
  FILE *fp;
{
    return (getc(fp));
}

/* osaputc - put a character to an ascii file */
int osaputc(ch,fp)
  int ch; FILE *fp;
{
    return (putc(ch,fp));
}

/* osbgetc - get a character from a binary file */
int osbgetc(fp)
  FILE *fp;
{
    return (getc(fp));
}

/* osbputc - put a character to a binary file */
int osbputc(ch,fp)
  int ch; FILE *fp;
{
    return (putc(ch,fp));
}

/* ostgetc - get a character from the terminal */
int ostgetc()
{
    int ch;

    /* check for a buffered character */
    if (lcount--)
	return (lbuf[lindex++]);

    /* get an input line */
    for (lcount = 0; ; )
	switch (ch = xgetc()) {
	case '\r':
        case '\n':
		lbuf[lcount++] = '\n';
		xputc('\r'); xputc('\n'); lposition = 0;
		if (tfp)
		    for (lindex = 0; lindex < lcount; ++lindex)
			osaputc(lbuf[lindex],tfp);
		lindex = 0; lcount--;
		return (lbuf[lindex++]);
	case '\010':
	case '\177':
		if (lcount) {
		    lcount--;
		    while (lposition > lpos[lcount]) {
			xputc('\010'); xputc(' '); xputc('\010');
			lposition--;
		    }
		}
		break;
	case '\004':
		xflush();
		return (EOF);
	default:
		if (ch == '\t' || (ch >= 0x20 && ch < 0x7F)) {
		    lbuf[lcount] = ch;
		    lpos[lcount] = lposition;
		    if (ch == '\t')
			do {
			    xputc(' ');
			} while (++lposition & 7);
		    else {
			xputc(ch); lposition++;
		    }
		    lcount++;
		}
		else {
		    xflush();
		    switch (ch) {
                    case '\002':	xlflush();      /* control-B */
                                        xlbreak("BREAK", s_unbound);
                                        break;
		    case '\003':	xltoplevel();	/* control-c */
                                        break;
		    case '\007':	xlcleanup();	/* control-g */
                                        break;
		    case '\020':	xlcontinue();	/* control-p */
                                        break;
                    case '\024':        xinfo();	/* control-T */
					break;
      		    case '\004':	return (EOF);	/* control-D */
                                        break;
		    default:		return (ch);
		    }
		}
	}
}

/* ostputc - put a character to the terminal */
ostputc(ch)
  int ch;
{
    /* check for control characters */
    oscheck();

    /* output the character */
    if (ch == '\n')
     { 
       lposition = 0;
       xputc('\n');
       xputc('\r');
    }
    else 
    {     
      lposition++;
      xputc(ch);
    }

   /* output the character to the transcript file */
   if (tfp)
	osaputc(ch,tfp);
}

/* osflush - flush the terminal input buffer */
osflush()
{
    lindex = lcount = lposition = 0;
}

/* oscheck - check for control characters during execution */
oscheck()
{
    int ch;
    if (ch = xcheck())
	switch (ch) {
	case '\002':	/* control-b */
	    xflush();
	    xlbreak("BREAK",s_unbound);
	    break;
	case '\003':	/* control-c */
	    xflush();
	    xltoplevel();
	    break;
	case '\024':	/* control-t */
	    xinfo();
	    break;
	}
}

/* xinfo - show information on control-t */
static xinfo()
{
    extern int nfree,gccalls;
    extern long total;
    char buf[80];
    sprintf(buf,"\n\r[ Free: %d, GC calls: %d, Total: %ld ]\n\r",
	    nfree,gccalls,total);
    errputstr(buf);
}

/* xflush - flush the input line buffer and start a new line */
static xflush()
{
    osflush();
    ostputc('\n');
}

/* xgetc - get a character from the terminal without echo */
static int xgetc()
{
    static char MyChar;
    int n;

    newtty.c_cc[VMIN] = 1;
    ioctl(0, TCSETAF, &newtty);

    read(0, &MyChar, 1);

    newtty.c_cc[VMIN] = 0;
    ioctl(0, TCSETAF, &newtty);

    return ((int) MyChar);
}

/* xputc - put a character to the terminal */
static xputc(ch)
  int ch;
{
    putchar(ch);
    fflush(stdout);
}

/* xcheck - check for a character */
static int xcheck()
{
    int nbytes;
    static char MyChar;

        
    nbytes = read(0, &MyChar, 1);
    return (nbytes ? (int) MyChar : 0);
}

/* xsystem - execute a system command */
LVAL xsystem()
{
    char *cmd="/bin/sh";
    int retval;

    if (moreargs())
	cmd = (char *)getstring(xlgastring());
    xllastarg();
    ioctl(0, TCSETAF, &oldtty);    /* restore TTY settings */
    retval = system(cmd);
    ioctl(0, TCSETAF, &newtty);    /* set TTY to raw mode */
    return (retval ? true : cvfixnum((FIXTYPE)errno));
}

/* xgetenv - get an environment variable */
LVAL xgetenv()
{
   char *EnvVar;
   char *value;
   
   EnvVar =(char *) getstring(xlgastring());
   xllastarg();
   return (((value = getenv(EnvVar))  == ((char *) NULL)) ? NIL: 
           cvstring(value));
}

/* xgetkey - get a key from the keyboard */
LVAL xgetkey()
{
    xllastarg();
    return (cvfixnum((FIXTYPE)xgetc()));
}

/* ossymbols - enter os specific symbols */
ossymbols()
{
}
