/*
 * W-NEWS       A simple NEWS processing package.
 *              Unbatch is used by the NEWS system to unbatch the news
 *              batches that were prepared and queued by the 'rnews'
 *              program.  Its task is to (optionally) uncompress a batch,
 *              and then to unpack all articles from the news batch, and
 *              to put those articles in the article spool directories.
 *
 * Usage:       unbatch [-d] [ batch_file ]
 *
 * Version:     @(#)unbatch.c	4.02	02/23/93
 *
 * Author:      Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *		Copyright 1988-1993 MicroWalt Corporation
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
#include "wnews.h"


static char *Release = VERSION;
static char *Version = "@(#) unbatch 4.02 (02/23/93)";


char *lgmonths[] = {                    /* for debugging purposes       */
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};
int debug = 0;                          /* debugging flag               */


_PRO( int unbatch, (char *batch)					);


/* Write something to the log files. */
#ifdef __STDC__
void logmsg(int err, char *fmt, ...)
#else
void logmsg(err, fmt)
int err;
char *fmt;
#endif
{
  char buff[1024];
  FILE *lfp, *efp;
  char lfn[128], efn[128];
  struct tm *mydate;
  va_list args;
  time_t now;

  sprintf(lfn, "%s/%s", CONFDIR, LOGFILE);
  sprintf(efn, "%s/%s", CONFDIR, ERRLOG);

  if ((efp = fopen(efn, "a")) == (FILE *)NULL) {
	fprintf(stderr, "unbatch: cannot append(%s)\n", efn);
	return;
  }

  if ((lfp = fopen(lfn, "a")) == (FILE *)NULL) {
	fprintf(efp, "unbatch: cannot append(%s)\n", lfn);
	(void) fclose(efp);
	return;
  }

  (void) time(&now);
  mydate = localtime(&now);
  sprintf(buff, "%s %02d %02d:%02d  ",
	lgmonths[mydate->tm_mon], mydate->tm_mday,
			mydate->tm_hour, mydate->tm_min);
  va_start(args, fmt);
  if (err == 1) {
	fprintf(efp, buff);
	vfprintf(efp, fmt, args);
  } else {
	fprintf(lfp, buff);
	vfprintf(lfp, fmt, args);
  }
  if (debug > 0) {
	fprintf(stderr, buff);
	vfprintf(stderr, fmt, args);
  }
  (void) fclose(lfp);
  (void) fclose(efp);
}


/*
 * Unpack the current batch of articles contained in 'batch'.
 * A batch consists of a header, followed by the actual articles
 * in a format like:
 *
 *      #! rnews 1234\n
 *      article-text
 *      #! rnews 5523\n
 *      article-text
 *      ...
 *
 * The number following the 'rnews' text is the size of the message
 * in bytes.
 *
 * Optionally, we may have a compressed batch file, which is essentially
 * a normal batch (see above), but now routed through the compress(1)
 * program, and prepended with a header like:
 *
 *      #! cunbatch\n
 *
 * So, we have to perform the following actions:
 *
 *      1) (optionally) decompress a compressed batch.
 *      2) Separate the messages.
 *      3) Write the separated messages into their dirs by calling the
 *         'inews' program for local posting.
 */
int unbatch(batch)
char *batch;
{
  char buff[1024], fname[128], tname[128];
  char cmd[128], artnam[128];
  char *args[8];
  int countr, countw;
  int pid, bfd, fd, s;
  register char c;
  register FILE *fp, *ap;
  char *bp;
  long count;

  logmsg(0, "unbatch: processing %s\n", batch);

  strcpy(fname, "/tmp/unb.XXXXXX");
  strcpy(artnam, "/tmp/unbart.XXXXXX");
  mktemp(fname); mktemp(artnam);

  /* First of all, open the batch file. */
  if ((fp = fopen(batch, "r")) == (FILE *)NULL) {
	logmsg(1, "unbatch: cannot open(%s)\n", batch);
	return(-1);
  }

  /* Now, check the type of this batch. */
  if (fgets(buff, 128, fp) == (char *)NULL) {
	logmsg(1, "unbatch: cannot read batch header\n");
	(void) fclose(fp);
	return(-1);
  }

  /* Is this a compressed batch? */
  if (strncmp(buff, COMPRHDR, strlen(COMPRHDR)) == 0) {
	/* Yes. Re-open the batch in file descriptor mode. */
	(void) fclose(fp);
	if ((fd = open(batch, O_RDONLY)) < 0) {
		logmsg(1, "unbatch: cannot re-open(%s)\n", batch);
		return(-1);
	}
	/* Create a temp file for decompression. */
	(void) unlink(fname);
	strcpy(tname, fname);
	strcat(tname, ".Z");

	if ((bfd = creat(tname, 0600)) < 0) {
		logmsg(1, "unbatch: cannot create(%s)\n", fname);
		(void) close(fd);
		return(-1);
	}
	/* Skip the 'compressed' mark header. */
	if (read(fd, buff, strlen(COMPRHDR)) != strlen(COMPRHDR)) {
		logmsg(1, "unbatch: cannot skip compress header\n");
		(void) close(fd);
		(void) close(bfd);
		(void) unlink(fname);
		return(-1);
	}

	/* Copy rest of fd to bfd. */
	while ((countr = read(fd, buff, 1024)) > 0)
			countw = write(bfd, buff, countr);
        (void) close(fd);		
	(void) close(bfd);

	/* Create a pipeline and decompress the temp file. */
	sprintf(buff,"%s -d %s 2>/dev/null\n", COMPRESS, tname);
	if (system(buff) != 0) {
              logmsg(1,"unbatch: cannot exec(%s)\n", COMPRESS);
              return(-1);
        }
 
  	/* We now have a decompressed file in /tmp. Open it! */
	if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
		logmsg(1, "unbatch: cannot open(%s)\n", fname);
		(void) unlink(fname);
		return(-1);
	}
  } else rewind(fp);

  /* Build a calling sequence for 'inews'. */
  sprintf(cmd, "%s/inews", LIBDIR);
  pid = 0;
  args[pid++] = cmd;
  if (debug == 1) args[pid++] = "-v";
  args[pid++] = "-p";
  args[pid++] = artnam;
  args[pid] = (char *)NULL;

  /*
   * We are now ready to post articles.
   * If this is a batch of files, then the first line will start
   * with the "!# rnews" header (RNEWSHDR) text.  Otherwise, the
   * text might be a single article.  Start reading the text and
   * post the generated articles by calling inews(8).
   */
  while(1) {
	if (fgets(buff, 1024, fp) == (char *)NULL) break;
	if (strncmp(buff, RNEWSHDR, strlen(RNEWSHDR))) {
		(void) rewind(fp);	/* single article, no header!	*/
		count = (long) 0;	/* all of the "batch" file..	*/
	} else {
		/* Normal article.  Determine its length. */
		if ((bp = strchr(buff, '\n')) != (char *)NULL) *bp = '\0';
		bp = buff + strlen(RNEWSHDR);
		count = atol(bp);
	}

	/* Create an article "temp" file. */
	if ((ap = fopen(artnam, "w")) == (FILE *)NULL) {
		logmsg(1, "unbatch: cannot create(%s)\n", artnam);
		(void) fclose(fp);
		(void) unlink(fname);
		return(-1);
	}

	/* Extract the article from the batch (or single article) file. */
	while(count > (long) 0) {
		c = fgetc(fp);
		if (c == EOF) break;
		fputc(c, ap);
		count--;
	}
	(void) fprintf(ap, "\n");
	(void) fflush(ap); (void) fclose(ap);

	/* Tell INEWS to post this article. */
	if ((pid = fork()) == 0) {
		s = execv(cmd, args);
		exit(s);
	} else {
		s = 0;
		if (pid > 0) while(wait(&s) != pid)
				;
	}

	/* Article done.  Should we continue? */
	(void) unlink(artnam);
	if (WEXITSTATUS(s) != 0) {
		logmsg(0, "unbatch: INEWS error %d, quitting\n",
							WEXITSTATUS(s));
		(void) fclose(fp);
		return(-1);
	}

  }

  /* Close batch file and remove (decompressed temporary) files. */
  (void) fclose(fp);
  (void) unlink(fname);
  (void) unlink(artnam);

  return(0);
}


void usage()
{
  fprintf(stderr, "Usage: unbatch [-d] [ batch_file ]\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  char temp[128], buff[128];
  char unblock[128];
  struct dirent *de;
  DIR *dp;
  int st;
  extern int getopt(), optind;
#ifndef COHERENT
  extern int opterr;
#endif

#ifndef COHERENT
  opterr = 0;
#endif
  while ((st = getopt(argc, argv, "d")) != EOF) switch(st) {
	case 'd':
		debug = 1;
		break;
	default:
		usage();
  }

  (void) umask(033);                    /* set protection to rw-r--r-- */
  signal(SIGPIPE, SIG_IGN);

  /* Check if we are already (or still?) running. */
  sprintf(unblock, "%s/%s", CONFDIR, UNBLOCK);
  if (access(unblock, 0) == 0) exit(0);

  /* No, we are not yet busy. */
  (void) creat(unblock, 0);

  /* At most one more argument allowed. */
  if (optind == (argc - 1)) {
	st = unbatch(argv[optind]);
	if (st == 0) (void) unlink(argv[optind]);
  } else if (optind != argc) usage();

  (void) chdir(SPOOLDIR);
  if ((dp = opendir(INCOMING)) == (DIR *)NULL) {
	logmsg(1, "unbatch: cannot opendir(%s/%s)\n", SPOOLDIR, INCOMING);
	(void) unlink(unblock);
	exit(-1);
  }

  (void) readdir(dp);
  (void) readdir(dp);
  while ((de = readdir(dp)) != (struct dirent *)0) {
	sprintf(buff, "%s/%s/%s", SPOOLDIR, INCOMING, de->d_name);
	if ((st = unbatch(buff)) != 0) {
		sprintf(temp, "%s/%s", SPOOLDIR, de->d_name);
		logmsg(1, "unbatch: inbound news garbled; saved in %s\n",
									temp);
		(void) link(buff, temp);
	}
	(void) unlink(buff);
  }
  (void) closedir(dp);

  /* Remove the lockfile. */
  (void) unlink(unblock);

  exit(0);
}
