/*
 * W-NEWS       A simple NEWS processing package.
 *              The 'sendbatch' program is used to create new batches that
 *              can be transported to other systems via the UUCP network.
 *              It gathers all files in the OUTGOING queue, and creates
 *              batches of up to LIM bytes.  Optionally, the generated batch
 *              is compressed using the 'compress(1)' program.  The final
 *              result will be handed to 'uux(1)' for transportation.
 *
 * Usage:       sendbatch [-c] [-d] [-l complevel] [-s maxsize] sysname
 *
 * Version:     @(#)sendbatch.c	4.02	02/23/93
 *
 * Author:      Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *		Copyright 1988-1993 MicroWalt Corporation
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
#include "wnews.h"


static char *Release = VERSION;
static char *Version = "@(#) sendbatch 4.02 (02/23/93)";


char *lgmonths[] = {                    /* for debugging purposes       */
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};
char remote[128];                       /* name of remote system        */
char batchname[128];                    /* path name of current batch   */
char complvl[8];                        /* how many bits to compress?	*/
int debug = 0;                          /* debugging flag               */
int opt_c = 0;                          /* compress after generation    */
long maxsize = (long) BATCHLIM;         /* maximum size of a batch      */
long cursize = (long) 0;                /* current size of batch        */
FILE *bfp = (FILE *)NULL;               /* FILE ptr of current batch    */


_PRO( int send_it, (char *name)						);
_PRO( int cr_batch, (void)						);
_PRO( int cl_batch, (void)						);
_PRO( int add_art, (char *name)						);
_PRO( int mk_batch, (char *who)						);

#ifdef MINIX
_PRO( FILE *popen, (char *cmd)						);
#endif


/* Write something to the log files. */
#ifdef __STDC__
void logmsg(int err, char *fmt, ...)
#else
void logmsg(err, fmt)
int err;
char *fmt;
#endif
{
  char buff[1024];
  FILE *lfp, *efp;
  char lfn[128], efn[128];
  struct tm *mydate;
  va_list args;
  time_t now;

  sprintf(lfn, "%s/%s", CONFDIR, LOGFILE);
  sprintf(efn, "%s/%s", CONFDIR, ERRLOG);

  if ((efp = fopen(efn, "a")) == (FILE *)NULL) {
	fprintf(stderr, "sendbatch: cannot append(%s)\n", efn);
	return;
  }

  if ((lfp = fopen(lfn, "a")) == (FILE *)NULL) {
	fprintf(efp, "sendbatch: cannot append(%s)\n", lfn);
	(void) fclose(efp);
	return;
  }

  (void) time(&now);
  mydate = localtime(&now);
  sprintf(buff, "%s %02d %02d:%02d  ",
	lgmonths[mydate->tm_mon], mydate->tm_mday,
			mydate->tm_hour, mydate->tm_min);
  va_start(args, fmt);
  if (err == 1) {
	fprintf(efp, buff);
	vfprintf(efp, fmt, args);
  } else {
	fprintf(lfp, buff);
	vfprintf(lfp, fmt, args);
  }
  if (debug > 0) {
	fprintf(stderr, buff);
	vfprintf(stderr, fmt, args);
  }
  (void) fclose(lfp);
  (void) fclose(efp);
}


/* Send a completed, news batch to a host via UUCP. */
int send_it(name)
char *name;
{
  char buff[256];
  struct stat stb;

  if (stat(name, &stb) < 0) {
	logmsg(1, "sendbatch: cannot stat(%s)\n", name);
	return(-1);
  }
  if (stb.st_size == 0L) return(0);     /* nothing to do! */

  sprintf(buff, "%s %s | %s %s!rnews\n", SYSCAT, batchname, UUX, remote);
  if ((system(buff) != 0)) {
	logmsg(1, "sendbatch: cannot exec(%s,%s)\n", UUX, remote);
  	return(-1);
  }
  return(0);
}


/* Create a batch of news articles. */
int cr_batch()
{
  strcpy(batchname, "/tmp/sb.XXXXXX");
  mktemp(batchname);

  if ((bfp = fopen(batchname, "w")) == (FILE *)NULL) {
	logmsg(1, "sendbatch: cannot create(%s)\n", batchname);
	return(-1);
  }
  cursize = 0L;
  return(0);
}


/*
 * Close up the current batch, and (optionally) run it through compress(1).
 * Then, ask UUX to send it to the UUCP host.
 */
int cl_batch()
{
  char buff[128];
  char iscomp[128], command[128];
  char temp[1024];
  int status, in, out;
  int countr, countw;

  (void) fflush(bfp);
  (void) fclose(bfp);
  if (opt_c == 1) {
	strcpy(buff, "/tmp/sbZ.XXXXXX");
	mktemp(buff);
        strcpy(iscomp, buff); 
	in = open(batchname, O_RDONLY);
        out = creat(iscomp, 0644);
	while ((countr = read(in, temp, 1024)) > 0) {
             countw = write(out, temp, countr);
	}
        (void) close(in);
        (void) close(out);

        sprintf(command,"%s -b%s %s 2>/dev/null\n", COMPRESS, complvl, iscomp);
        if ((system(command)) != 0) {
	    logmsg(1, "sendbatch: cannot exec(%s,%s)\n", COMPRESS, iscomp);
            return(-1); 
        }

        strcat(iscomp,".Z"); 
	in = open(iscomp, O_RDONLY);
        out = creat(buff, 0644);

	(void) write(out, COMPRHDR, strlen(COMPRHDR));
	while ((countr = read(in, temp, 1024)) > 0) {
             countw = write(out, temp, countr);
        }
        (void) close(out);
        (void) close(in);
 
	(void) unlink(batchname);
	strcpy(batchname, buff);
  }
  if ((status = send_it(batchname)) == 0) (void) unlink(batchname);
  if (opt_c == 1) (void) unlink(iscomp);
  return(status);
}


/* Add an article to the current batch. */
int add_art(name)
char *name;
{
  char buff[1024];
  struct stat stb;
  register FILE *fp;

  /* Fetch some file info. */
  if (stat(name, &stb) < 0) {
	logmsg(1, "sendbatch: cannot stat(%s)\n", name);
	return(0);
  }
  if ((fp = fopen(name, "r")) == (FILE *)NULL) {
	logmsg(1, "sendbatch: cannot open(%s)\n", name);
	return(0);
  }

  /*
   * Check to see if the article still fits in the current batch.
   * Note, that it is FORCED to fit if the batch is still empty.
   * This is needed to prevent the program from looping over an
   * article that is larger than 'maxsize' bytes...
   */
  if (cursize > 0L && (cursize + stb.st_size) > maxsize) {
	if (cl_batch() < 0) {
		(void) fclose(fp);
		return(-1);
	}
	if (cr_batch() < 0) {
		(void) fclose(fp);
		return(-1);
	}
  }

  /* Add size of this article to current size. */
  cursize += stb.st_size;

  /* Create an article-separation header. */
  fprintf(bfp, "%s%ld\n", RNEWSHDR, (long) stb.st_size);

  /* Copy the article into the batch. */
  while (fgets(buff, 1024, fp) != (char *)NULL) fprintf(bfp, "%s", buff);
  (void) fclose(fp);

  return(0);
}


/* Scan the batch list file for articles, and add them to the batch. */
int mk_batch(who)
char *who;
{
  char fname[128], temp[128];
  char buff[1024];
  FILE *fp;
  register char *sp;
  register int i;

  strcpy(remote, who);

  sprintf(fname, "%s/%s", CONFDIR, SYSFILE);
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	logmsg(1, "sendbatch: cannot open(%s)\n", fname);
	return(-1);
  }
  sp = (char *)NULL;
  do {
	if (fgets(buff, 1024, fp) == (char *)NULL) break;
	if (buff[0] == '0' || buff[0] == '\n' || buff[0] == '#') continue;
	if (!strncmp(buff, who, strlen(who))) sp = buff;
  } while (sp == (char *)NULL);
  (void) fclose(fp);

  if (sp == (char *)NULL) {
	logmsg(1, "sendbatch: unknown target %s\n", who);
	return(-1);
  }
  if ((sp = strrchr(buff, ':')) == (char *)NULL) {
	logmsg(1, "sendbatch: \"sys\" file format error\n");
	return(-1);
  }
  if (*++sp == '\n') sprintf(fname, "%s/%s", BATCHDIR, who);
    else strcpy(fname, sp);
  if ((sp = strchr(fname, '\n')) != (char *)NULL) *sp = '\0';
  sprintf(temp, "%s.work", fname);

  /* Does a batch file exist? */
  if (access(fname, 0) < 0) return(0);

  /* Does a work file exist? */
  if (access(temp, 0) < 0) {
	/* Try to link the batch list to a "work file". */
	i = 0;
	while (i < 10) {
		if (link(fname, temp) == 0) break;
		if (++i >= 10) {
			logmsg(1, "sendbatch: cannot link(%s,%s)\n",
							fname, temp);
			return(-1);
		}
		sleep(6);
	}
	(void) unlink(fname);
  }

  /* Open the batch list file. */
  if ((fp = fopen(temp, "r")) == (FILE *)NULL) {
	logmsg(1, "sendbatch: cannot open(%s)\n", temp);
	return(-1);
  }
  logmsg(0, "sendbatch: checking files for %s in %s\n", who, fname);

  /* Create the news batch. */
  if (cr_batch() < 0) {
	(void) fclose(fp);
	return(-1);
  }

  /* Now read article names from the list file. */
  while (fgets(buff, 128, fp) != (char *)NULL) {
	if (buff[0] == '0' || buff[0] == '\n') continue;

	if ((sp = strchr(buff, '\n')) != (char *)NULL) *sp = '\0';
	if (add_art(buff) < 0) {
		(void) fclose(fp);
		return(-1);
	}
  }
  (void) fclose(fp);

  /* Close up any pending batch file. */
  if (cursize > 0L) {
	if (cl_batch() < 0) return(-1);
  }
  (void) unlink(temp);
  return(0);
}


void usage()
{
  fprintf(stderr, "Usage: sendbatch [-c] [-l compress level] [-d] ");
  fprintf(stderr, "[-s maxsize] sysname\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  char name[128];
  register int i;
  extern int getopt(), optind;
#ifndef COHERENT
  extern int opterr;
#endif
  extern char *optarg;

  strncpy(complvl, COMPLVL, 2);
#ifndef COHERENT
  opterr = 0;
#endif
  while ((i = getopt(argc, argv, "cl:ds:")) != EOF) switch(i) {
	case 'c':
		opt_c = 1;
		break;
	case 'l':
		opt_c = 1;
		strncpy(complvl, optarg, 2);
		break;
	case 'd':
		debug = 1;
		break;
	case 's':
		maxsize = atol(optarg);
		break;
	default:
		usage();
  }

  /* At last one more argument needed. */
  if (optind >= argc) usage();

  (void) umask(033);                    /* set protection to rw-r--r-- */
  if (opt_c == 1) maxsize *= 2;         /* adjust 'compressed' batch size */

  /* Check if we are already (or still?) running. */
  sprintf(name, "%s/%s", CONFDIR, BATLOCK);
  if (access(name, 0) == 0) exit(0);

  /* No, we are not yet busy. */
  (void) creat(name, 0);

  i = 0;
  while (optind < argc) {
	if (mk_batch(argv[optind++]) < 0) i = -1;
  }

  /* Remove the lockfile. */
  (void) unlink(name);

  exit(i);
}
