/*
 * W-NEWS       A simple NEWS processing package.
 *              Inews is the program that posts a message to the local
 *              system, and optionally queues it for transmission to
 *              the news feed system.  It is called by the programs
 *              'unbatch' (for posting locally) and 'pnews' (for
 *              posting locally and to the network).
 *
 * Usage:       inews -h | {-t title -n newsgroups} [-e expires]
 *                    [-f sender] [-d distribution] [-F refs]
 *                    [-o organization] [-a approved] [-r reply-to]
 *                    [-x dontsend] [-c control] [artname]
 *
 *              inews [-d] -p artname
 *
 * Version:     @(#)inews.c	4.03	08/26/93
 *
 * Author:      Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *		Copyright 1988-1993 MicroWalt Corporation
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
#include "wnews.h"


typedef struct _hdr_ {                  /* the header field definition  */
  struct _hdr_  *next;
  char          *field;
  char          *text;
} HDR;
#define HDRSIZE (sizeof(HDR))
#define NILHDR  ((HDR *)NULL)

typedef struct _sys_ {                  /* the "sys" file definition    */
  struct _sys_  *next;
  char          *name;
  char          *nsys;
  char          *ngrp;
  char          *prog;
  char          *file;
} SYS;
#define SYSSIZE (sizeof(SYS))
#define NILSYS  ((SYS *)NULL)


static char *Release = VERSION;
static char *Version = "@(#) inews 4.02 (08/23/93)";


int debug = 0;                          /* debugging flag               */
int opt_h = 0;                          /* use article hdr for info     */
int opt_p = 0;                          /* post an incoming article     */
int old_uid;
char myname[128];                       /* various environment values   */
char myorg[128];
time_t mytime;                          /* current time and date        */
struct tm *mydate;
char remote[128];                       /* name of remote news feed     */
off_t bytes;                            /* length of completed article  */
char *dontsend = (char *)NULL;          /* list of "don't send" hosts   */
SYS *sys = NILSYS;                      /* the in-core "sys" file       */
struct {                                /* this is the in-core header   */
  char  *path,  *path_t;
  char  *from,  *from_t;
  char  *send,  *send_t;
  char  *repl,  *repl_t;
  char  *flup,  *flup_t;
  char  *ngrp,  *ngrp_t;
  char  *dist,  *dist_t;
  char  *ctrl,  *ctrl_t;
  char  *subj,  *subj_t;
  char  *keyw,  *keyw_t;
  char  *summ,  *summ_t;
  char  *mrk1,  *mrk1_t;
  char  *refs,  *refs_t;
  char  *iden,  *iden_t;
  char  *ide2,  *ide2_t;
  char  *orgn,  *orgn_t;
  char  *dept,  *dept_t;
  char  *date,  *date_t;
  char  *expr,  *expr_t;
  char  *appr,  *appr_t;
  char  *lins,  *lins_t;
  char  *xref,  *xref_t;
  char  *mrk2,  *mrk2_t;
  char  *null;
  HDR   *fields;
} hdr = {
  "Path:",              (char *)NULL,
  "From:",              (char *)NULL,
  "Sender:",            (char *)NULL,
  "Reply-To:",          (char *)NULL,
  "Followup-To:",       (char *)NULL,
  "Newsgroups:",        (char *)NULL,
  "Distribution:",      "world",
  "Control:",           (char *)NULL,
  "Subject:",           (char *)NULL,
  "Keywords:",          (char *)NULL,
  "Summary:",           (char *)NULL,
  "",                   (char *)NULL,
  "References:",        (char *)NULL,
  "Message-ID:",        (char *)NULL,
  "Message-Id:",        (char *)NULL,
  "Organization:",      (char *)NULL,
  "Department:",        (char *)NULL,
  "Date:",              (char *)NULL,
  "Expires:",           (char *)NULL,
  "Approved:",          (char *)NULL,
  "Lines:",             (char *)NULL,
  "Xref:",              (char *)NULL,
  "",                   (char *)NULL,
  (char *)NULL,
  NILHDR
};
static char *months[] = { 
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};
static char *days[] = {
	"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};


_PRO( char *xtime, (time_t *salt)					);
_PRO( char *gtof, (char *groupname)					);
_PRO( void acct_upd, (char *what, char *dist, off_t bytes, char *hosts)	);
_PRO( int sys_rd, (void)						);
_PRO( int ng_cmp, (char *ng1, char *ng2)				);
_PRO( int ngmatch, (char *name, char *ng, char *ds, char *pat)		);
_PRO( void hist_add, (char *msgid, char *name, unsigned int artno)	);
_PRO( int hist_chk, (char *msgid)					);
_PRO( unsigned int act_upd, (char *name)				);
_PRO( void ctrl_msg, (char *cmd, char *fname)				);
_PRO( int hdr_upd, (char *text, int level)				);
_PRO( void hdr_end, (int lines)						);
_PRO( int hdr_out, (char *temp, char *fname, int lines)			);
_PRO( int del_chk, (char *dir)						);
_PRO( void del_one, (char *temp, char *group, char *aname)		);
_PRO( void del_loc, (char *fname, char *aname)				);
_PRO( int del_cmd, (char *what, char *fname)				);
_PRO( int del_rmt, (char *fname, char *aname, SYS *ap)			);
_PRO( int do_post, (char *fname)					);
_PRO( void app_signature, (FILE *ofp)					);
_PRO( int inews, (FILE *ifp)						);
#ifdef COHERENT
_PRO( char *getenv, (char *text)					);
#endif


/* Write something to the log files. */
#ifdef __STDC__
void logmsg(int err, char *fmt, ...)
#else
void logmsg(err, fmt)
int err;
char *fmt;
#endif
{
  char buff[1024];
  FILE *lfp, *efp;
  char lfn[128], efn[128];
  struct tm *mydate;
  va_list args;
  time_t now;

  sprintf(lfn, "%s/%s", CONFDIR, LOGFILE);
  sprintf(efn, "%s/%s", CONFDIR, ERRLOG);

  if ((efp = fopen(efn, "a")) == (FILE *)NULL) {
	fprintf(stderr, "inews: cannot append(%s)\n", efn);
	return;
  }

  if ((lfp = fopen(lfn, "a")) == (FILE *)NULL) {
	fprintf(efp, "inews: cannot append(%s)\n", lfn);
	(void) fclose(efp);
	return;
  }

  (void) time(&now);
  mydate = localtime(&now);
  sprintf(buff, "%s %02d %02d:%02d  ",
	months[mydate->tm_mon], mydate->tm_mday,
			mydate->tm_hour, mydate->tm_min);
  va_start(args, fmt);
  if (err == 1) {
	fprintf(efp, buff);
	vfprintf(efp, fmt, args);
  } else {
	fprintf(lfp, buff);
	vfprintf(lfp, fmt, args);
  }
  if (debug > 0) {
	fprintf(stderr, buff);
	vfprintf(stderr, fmt, args);
  }
  (void) fclose(lfp);
  (void) fclose(efp);
}


/*
 * Make a decent DATE/TIME string.
 * Note, that there are TWO possible date formats:
 *
 *      Sat, 12 Oct 89 20:29:00\0
 * and
 *      Sat 12 Oct 20:29:00 1989\0
 *
 * Most Internet mailers use this first form, so we try
 * to this also. We use the function xtime() for the work...
 */
char *xtime(salt)
time_t *salt;
{
  static char buff[64];         /* date and time in MET format */
  struct tm *tm;

  tm = localtime(salt);
  sprintf(buff, "%s, %2d %s %2d %02d:%02d:%02d %s",
	days[tm->tm_wday], tm->tm_mday, months[tm->tm_mon], 
	    tm->tm_year, tm->tm_hour, tm->tm_min, tm->tm_sec,
		tzname[tm->tm_isdst]);

  return(buff);
}


/* Change 'groupname' into 'dirname'. */
char *gtof(groupname)
register char *groupname;
{
  static char buff[128];
  register char *bp;

  bp = buff;
  while (*groupname != '\0') {
	if (*groupname == '.') *bp++ = '/';
	  else *bp++ = *groupname;
	groupname++;
  }
  *bp = '\0';
  return(buff);
}


/* Perform some resource accouting. */
void acct_upd(what, dist, bytes, hosts)
char *what;
char *dist;
off_t bytes;
char *hosts;
{
#if HAVE_ACCT
  char fname[128];
  FILE *fp;

  sprintf(fname, "%s/%s", CONFDIR, ACCTFILE);
  if ((fp = fopen(fname, "a")) == (FILE *)NULL) {
	logmsg(1, "cannot append(%s)\n", fname);
	return;
  }
  fprintf(fp, "%02d/%02d/%02d %02d:%02d\t%s\t%s\t%ld\t%s\n",
	mydate->tm_year, mydate->tm_mon + 1, mydate->tm_mday, 
	mydate->tm_hour, mydate->tm_min,
	what, dist, (long) bytes, hosts);
  (void) fclose(fp);
#endif
}


/* Read the "sys" file into core. */
int sys_rd()
{
  char *mem = "out of memory\n";
  char buff[128], fname[128];
  register SYS *ap;
  register FILE *fp;
  register char *sp, *bp;
  char c;

  /* Open the "sys" file for reading. */
  sprintf(fname, "%s/%s", CONFDIR, SYSFILE);
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	fprintf(stderr, "cannot open(%s)\n", fname);
	return(-1);
  }

  /* Now, read the file, and store it in core. */
  while (fgets(buff, 128, fp) != (char *)NULL) {
	if (buff[0] == '0' || buff[0] == '\n' || buff[0] == '#') continue;

	/* Allocate a new frame for this system. */
	if (sys == NILSYS) {
		sys = (SYS *)malloc(SYSSIZE);
		ap = sys;
	} else {
		ap = sys;
		while (ap->next != NILSYS) ap = ap->next;
		ap->next = (SYS *)malloc(SYSSIZE);
		ap = ap->next;
	}
	if (ap == NILSYS) {
		fprintf(stderr, mem);
		(void) fclose(fp);
		return(-1);
	}
	ap->next = NILSYS;

	/* Decode the "sysname" field. */
	sp = buff;
	while (*sp && *sp != ':' && *sp != '/') sp++;
	c = *sp;
	*sp++ = '\0';
	if ((ap->name = (char *)malloc(strlen(buff) + 1)) == (char *)NULL) {
		fprintf(stderr, mem);
		(void) fclose(fp);
		return(-1);
	}
	strcpy(ap->name, buff);

	/* Check for and decode the "not_to_sys" field. */
	if (c == '/') {
		bp = sp;
		while (*sp && *sp != ':') sp++;
		*sp = '\0';
		ap->nsys = (char *)malloc(strlen(bp) + 1);
		if (ap->nsys == (char *)NULL) {
			fprintf(stderr, mem);
			(void) fclose(fp);
			return(-1);
		}
		strcpy(ap->nsys, bp);
	} else ap->nsys = (char *)NULL;

	/* Decode the "newsgroups" field. */
	bp = sp;
	while (*sp && *sp != ':') sp++;
	*sp++ = '\0';
	if ((ap->ngrp = (char *)malloc(strlen(bp) + 1)) == (char *)NULL) {
		fprintf(stderr, mem);
		(void) fclose(fp);
		return(-1);
	}
	strcpy(ap->ngrp, bp);

	/* Decode the "transport" field. */
	bp = sp;
	while (*sp && *sp != ':') sp++;
	*sp++ = '\0';
	if (*bp != '\0') {
		ap->prog = (char *)malloc(strlen(bp) + 1);
		if (ap->prog == (char *)NULL) {
			fprintf(stderr, mem);
			(void) fclose(fp);
			return(-1);
		}
		strcpy(ap->prog, bp);
	} else ap->prog = (char *)NULL;

	/* Decode the "filename" field. */
	bp = sp;
	while (*sp && *sp != '\n') sp++;
	*sp = '\0';
	if (*bp != '\0') {
		ap->file = (char *)malloc(strlen(bp) + 1);
		if (ap->file == (char *)NULL) {
			fprintf(stderr, mem);
			(void) fclose(fp);
			return(-1);
		}
		strcpy(ap->file, bp);
	} else ap->file = (char *)NULL;
  }
  (void) fclose(fp);
  return(0);
}


/* Compare two newsgroups for equality.
 * This function is taken from the "B News 2.11" software package,
 * (c) Rick Adams 1986.  I could not think of a better implemen-
 * tation of this function, so I used it- FvK.
 */
int ng_cmp(ng1, ng2)
register char *ng1, *ng2;
{
  while (*ng1 != ',' && *ng1 != '\0') {
	if (ng1[0] == 'a' && ng1[1] == 'l' && ng1[2] == 'l') {
		ng1 += 3;
		while (*ng2 != ',' && *ng2 != '.' && *ng2 != '\0')
					if (ng_cmp(ng1, ng2++)) return(1);
		return(ng_cmp(ng1, ng2));
	} else if (*ng1++ != *ng2++) return(0);
  }
  return(*ng2 == '.' || *ng2 == ',' || *ng2 == '\0');
}


/*
 * News group matching.
 * The main part of this function is taken from the "B News 2.11"
 * software package, (c) Rick Adams 1986.  I could not think of
 * a better implementation of this function, so I used it- FvK.
 */
int ngmatch(name, ng, ds, pat)
char *name;
char *ng;
char *ds;
char *pat;
{
  static char buff[128];
  register char *sp, *bp;
  register int i;

  /* Check for the "to.SYSNAME" newgroup.  It matches... */
  sprintf(buff, "to.%s", name);
  if (strstr(ng, buff) != (char *)NULL) return(1);

  /* Check for the "control" newgroup.  It also matches... */
  if (strstr(ng, "control") != (char *)NULL) return(1);

  /* Now, check the distribution. */
  if (ds != (char *)NULL) {
	if (strstr(pat, ds) == (char *)NULL) return(0);
  }

  /* Try to match the newgroup with the system's subscription list. */
  i = 0;
  for (sp = ng; *sp != '\0' && i == 0;) {
	for (bp = pat; *bp != '\0';) {
		if (*bp != '!') i = i || ng_cmp(bp, sp);
		  else i = i && !ng_cmp(bp+1, sp);

		while (*bp++ != ',' && *bp != '\0')
				;
	}
	while (*sp++ != ',' && *sp != '\0')
			;
  }
  return(i);
}


/* Add this article to the "history" file. */
void hist_add(msgid, name, artno)
char *msgid;
char *name;
unsigned int artno;
{
#if HAVE_HIST
  char fname[128];
  FILE *fp;

  sprintf(fname, "%s/%s", CONFDIR, HISTFILE);
  if ((fp = fopen(fname, "a")) == (FILE *)NULL) {
	logmsg(1, "cannot append(%s)\n", fname);
	return;
  }
  fprintf(fp, "%s\t%02d/%02d/%02d %02d:%02d\t%s/%u\n",
	msgid, mydate->tm_mon + 1, mydate->tm_mday, mydate->tm_year,
		mydate->tm_hour, mydate->tm_min, name, artno);
  (void) fclose(fp);
#endif
}


/* Check if we already have this article. */
int hist_chk(msgid)
char *msgid;
{
#if HAVE_HIST
  char buff[512];
  char fname[128];
  FILE *fp;
  char *sp;

  sprintf(fname, "%s/%s", CONFDIR, HISTFILE);
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	logmsg(1, "cannot open(%s)\n", fname);
	return(0);
  }

  while(fgets(buff, 512, fp) != (char *)NULL) {
	if (buff[0] == '\n') continue;
	if ((sp = strchr(buff, '\t')) != (char *)NULL) *sp = '\0';
	if (!strcmp(buff, msgid)) {
		logmsg(0, "%-8.8s  %s duplicate article rejected\n",
							remote, msgid);
		(void) fclose(fp);
		return(1);
	}
  }
  (void) fclose(fp);
#endif
  return(0);
}


/* Read and update the "active" file. */
unsigned int act_upd(name)
char *name;
{
  char buff[128];
  char fname[128], lckname[128];
  unsigned int ret;
  int try, foundit, fd;
  register FILE *afp;
  register char *bp, *sp;
#if HAVE_STDIO
  off_t offset;
#else
  char f2name[128], shadow[128], buff1[128];
  char value[16];
  register FILE *a2fp;
  register char *cp;
#endif

  /* Create file names and try to lock "active". */
  sprintf(fname, "%s/%s", CONFDIR, ACTIVE);
  sprintf(lckname, "%s.lock", fname);

  /* Try to lock "active".  We allow others one minute. */
  try = 0;
  fd = 0;
  while (try < 10) {
	if (creat(lckname, 0600) >= 0) break;
	if (++try >= 10) {
		logmsg(1, "cannot creat(%s, 0600)\n", lckname);
		return(0);
	}
	(void) sleep(6);
  }
  (void) close(fd);

  foundit = 0;
#if HAVE_STDIO
  /* Open the "active" file for read-and-update. */
  afp = fopen(fname, "r+");
  if (afp == (FILE *)NULL) {
	logmsg(1, "cannot open(%s)\n", fname);
	(void) unlink(lckname);
	return(0);
  }

  /* Now, find the desired newsgroup and its file position. */
  ret = 0;
  offset = (off_t)0;
  while (foundit == 0) {
	if (fgets(buff, 80, afp) == (char *)NULL) break;
	if (!strncmp(buff, name, strlen(name))) {
		/* right group, or deeper in hirarchy? */
		bp = buff + strlen(name);
		if ((*bp == ' ') || (*bp == '\t')) {
			offset += (off_t) (strlen(name) + 1);
			bp = buff + strlen(name) + 1;
			foundit++;
			sp = bp;
			while (*bp && *bp >= '0' && *bp <= '9') bp++;
			*bp = '\0';
			ret = (unsigned int) atol(sp);
			ret++;
			break;
		}
	}
	offset += (off_t) strlen(buff);
  }

  /* Finally, if we found the group, re-position for writing and update. */
  if (foundit) {
	  (void) fseek(afp, offset, SEEK_SET);
	  fprintf(afp, "%05u", ret);
  }
  (void) fclose(afp);
#else
  sprintf(f2name, "%s/%s2", CONFDIR, ACTIVE);

  /* Open the "active" file for reading. */
  afp = fopen(fname, "r");
  if (afp == (FILE *)NULL) {
	logmsg(1, "cannot open(%s)\n", fname);
	(void) unlink(lckname);
	return(ret);
  }

  /* Create the new "active" file. */
  a2fp = fopen(f2name, "w");
  if (a2fp == (FILE *)NULL) {
	logmsg(1, "cannot open(%s)\n", f2name);
	(void) unlink(lckname);
	(void) fclose(afp);
	return(ret);
  }

  /* Now, find the desired newsgroup and update it in the copy. */
  while (1) {
	if (fgets(buff, 80, afp) == (char *)NULL) break;

	/* Check if this is _really_ the desired newsgroup. */
	strncpy(buff1, buff, 80);
	for(sp = buff1; *sp ; sp++)
		if (*sp == ' ' || *sp == '\t') *sp-- = '\0';

	if (!strcmp(buff1, name)) {
		sp = shadow;
		bp = buff;
		while (*bp && *bp != ' ' && *bp != '\t') *sp++ = *bp++;
		*sp++ = ' ';
		bp++;

		cp = value;
		while (*bp && *bp >= '0' && *bp <= '9') *cp++ = *bp++;
		*cp = '\0';
		bp++;
		ret = (unsigned int) atol(value);
		ret++;

		sprintf(value, "%05.5u ", ret);
		cp = value;
		while (*cp != '\0') *sp++ = *cp++;
		while (*bp != '\0') *sp++ = *bp++;
		*sp = '\0';
		fprintf(a2fp, "%s", shadow);    
		foundit++;
	} else fprintf(a2fp, "%s", buff);
  }

  (void) fclose(a2fp);
  (void) fclose(afp);
  (void) unlink(fname);

  if (rename(f2name, fname) < 0) {
	logmsg(1, "cannot rename(%s,%s)\n", f2name, fname);
	ret = 0;
  }
  (void) unlink(f2name);
#endif

  (void) unlink(lckname);
  if (foundit == 0) return(0);
  return(ret);
}


/* Process a received control message. */
void ctrl_msg(cmd, fname)
char *cmd;
char *fname;
{
  char buff[128], arg[128], mailbuff[512];
  char *args[8], *sp;
  int st, pid, status;

  strcpy(arg, cmd);
#if HAVE_AUTO
  /* First of all, execute the control message. */
  if ((sp = strchr(arg, ' ')) != NULL) *sp++ = '\0';
  sprintf(buff, "%s/%s", LIBDIR, arg);
  st = 0;
  args[st++] = buff;
  do {
	args[st++] = sp;
	while (*sp && *sp != ' ' && *sp != '\t') sp++;
	if (*sp == ' ' || *sp == '\t') *sp++ = '\0';    
  } while (*sp != '\0');
  args[st] = (char *)NULL;

  if ((pid = fork()) == 0) {
	(void) close(0);
	(void) open(fname, O_RDONLY);
	st = execv(args[0], args);
	exit(st);
  } else if (pid > 0) {
	while (wait(&status) != pid)
		    ;
  }
  if (pid < 0 || WEXITSTATUS(status) != 0) {
	logmsg(1, "EXEC error in %s(%s) < %s\n", args[0], cmd, fname);
	sprintf(arg, "Error in %s(%s) < %s", args[0], cmd, fname);
  } else sprintf(arg, "Report of %s(%s) < %s", args[0], cmd, fname);
  strcpy(buff, arg);
#else
  sprintf(buff, "Incoming control message: %s < %s", cmd, fname);
#endif

#ifdef COHERENT
  sprintf(mailbuff, "%s %s <<!\nSubject: Control message: %s\n%s\n!",
  		   MAIL, USENET, cmd, buff);
#else
  sprintf(mailbuff, "%s -s\"Control message: %s: %s\" %s",
  		   MAIL, cmd, buff, USENET);
#endif
  		   
  st = 0;
  args[st++] = SHELL;
  args[st++] = "-c";
  args[st++] = mailbuff;
  args[st] = (char *)NULL;

  if ((pid = fork()) == 0) {
	(void) close(0);
	(void) open(fname, O_RDONLY);
	st = execv(args[0], args);
	exit(st);
  } else if (pid > 0) {
	while (wait(&status) != pid)
		    ;
  }
  if (pid < 0 || WEXITSTATUS(status) != 0)
	logmsg(1, "could not mail to %s subj '%s' fn %s\n",
						USENET, buff, fname);
}


/* Update our in-core header information. */
int hdr_upd(text, level)
char *text;
int level;
{
  static char **op = (char **)NULL;
  char buff[1024];
  struct passwd *pw;
  register char *sp;
  register HDR *hp;
  char *un, *ur;
  char **xp, **yp;

  /* Is this the "initialize header" call? */
  if (text == (char *)NULL) {
	hdr.fields = NILHDR;

	/* Is this generated locally or is it an incoming article? */
	if (opt_p == 0) {
		pw = getpwuid(old_uid);
		if (pw == (struct passwd *)NULL) {
			un = "unknown"; ur = "John Doe";
		} else {
			un = pw->pw_name; ur = pw->pw_gecos;
		}
		sprintf(buff, "From: %s@%s (%s)\n", un, myname, ur);
		(void) hdr_upd(buff, ++level);

		sprintf(buff, "Path: %s!%s\n", myname, un);
		(void) hdr_upd(buff, level);

		sprintf(buff, "Message-ID: <%d%02d%02d%d@%s>\n",
			mydate->tm_year, mydate->tm_mon + 1, mydate->tm_mday,
						getpid(), myname);
		(void) hdr_upd(buff, level);

		if (myorg[0] != '\0') sprintf(buff, "Organization: %s\n",
								myorg);
		(void) hdr_upd(buff, level);

		sprintf(buff, "Date: %s\n", xtime(&mytime));
		(void) hdr_upd(buff, level);
		strcpy(remote, "local");
	}
	if (opt_h == 1) return(0);      /* go into "scan header" state */
	  else return(1);               /* go into "read text" state */
  }

  /* Can this be a valid part of the header? */
  if (level == 0 && opt_h == 0) return(1);

  /* Yes.  Is this the end-of-header marker? */
  if (*text == '\n') return(1);

  /* No.  Remove the trailing newline. */
  if ((sp = strchr(text, '\n')) != (char *)NULL) *sp = '\0';

  /* First, split up the "Field: " and "value" parts. */
  if ((sp = strchr(text, ':')) == (char *)NULL) {
	if (op == (char **)NULL || (*text != ' ' && *text != '\t')) {
		logmsg(0, "bad header field: \"%s\"\n", text);
		return(0);
	}
	while (*text == ' ' || *text == '\t') text++;
	sp = (char *)malloc(strlen(--text) + strlen(*op) + 1);
	if (sp == (char *)NULL) {
		logmsg(1, "out of memory\n");
		return(-1);
	}
	strcpy(sp, *op);
	strcat(sp, text);
	free(*op);
	*op = sp;
	return(0);
  } else {
	*++sp = '\0';
	sp++;
	while (*sp == ' ' || *sp == '\t') sp++;
  }

  /* Then, add this header field to the in-core hdr. */
  xp = (char **) &hdr.path;
  yp = (char **)NULL;
  while (*xp != (char *)NULL) {
	if (!strncmp(text, *xp, strlen(text))) {
		yp = xp; yp++;
		break;
	}
	xp++;
	xp++;
  }
  if (yp == (char **)NULL) {
	if (hdr.fields != NILHDR) {
		hp = hdr.fields;
		while (hp->next != NILHDR) hp = hp->next;
		hp->next = (HDR *)malloc(HDRSIZE);  
		hp = hp->next;
	} else {
		hdr.fields = (HDR *)malloc(HDRSIZE);  
		hp = hdr.fields;
	}
	if (hp == NILHDR) {
		logmsg(1, "out of memory\n");
		return(-1);
	}
	hp->next = NILHDR;
	yp = &hp->text;
	hp->field = (char *)malloc(strlen(text) + 1);
	if (hp->field == NULL) {
		logmsg(1, "out of memory\n");
		return(-1);
	}
	strcpy(hp->field, text);
  }
  *yp = (char *)malloc(strlen(sp) + 1);  
  if (*yp == (char *)NULL) {
	logmsg(1, "out of memory\n");
	return(-1);
  }
  strcpy(*yp, sp);
  op = yp;
  return(0);
}


/* Patch the in-core header for various things. */
void hdr_end(lines)
int lines;
{
  char buff[1024];
  char addr[512];
  char temp[512];
  char *rest;
  register char *sp, *bp;

  /* 1. If feed uses Message-Id create Message-ID from it */
  if (hdr.iden_t == (char *)NULL && hdr.ide2_t != (char *)NULL)
						hdr.iden_t = hdr.ide2_t;

  /* 2. Include the "Lines:" field if not present. */
  if (hdr.lins_t == (char *)NULL) {
	sprintf(buff, "%d", lines);
	hdr.lins_t = (char *)malloc(strlen(buff) + 1);
	if (hdr.lins_t == (char *)NULL) {
		logmsg(1, "out of memory\n");
		return;
	} else strcpy(hdr.lins_t, buff);
  }

  /* 3. The rest of the header was generated locally. */
  if (opt_p == 0) return;

  /* 4. Fetch the "previous" host name. */
  sprintf(buff, "%s!%s", myname, hdr.path_t);
  sp = strchr(hdr.path_t, '!');
  if (sp != (char *)NULL) {
	*sp = '\0';
	strcpy(remote, hdr.path_t);
	sp = strchr(remote, '.');
	if (sp != (char *)NULL) *sp = '\0';
  } else strcpy(remote, "unknown");

  /* 5. Make a new "Path: myname!oldpath" field. */
  free(hdr.path_t);
  hdr.path_t = (char *)malloc(strlen(buff) + 1);
  if (hdr.path_t == (char *)NULL) {
	logmsg(1, "out of memory\n");
	return;
  } else strcpy(hdr.path_t, buff);

  /* 6. Change the "From: " line if needed. */
  if ((sp = strchr(hdr.from_t, '<')) == (char *)NULL) {
	bp = hdr.from_t;
	while (*bp && *bp != ' ' && *bp != '\t' && *bp != '(') bp++;
	if (*bp == ' ' || *bp == '\t' || *bp == '(') *bp++ = '\0';
	strcpy(addr, hdr.from_t);
	if ((sp = strchr(bp, '(')) != (char *)NULL) *sp++ = '\0';
	  else sp = bp;
	rest = sp;
	if ((bp = strchr(sp, ')')) != (char *)NULL) *bp = '\0';
  } else {
	*sp++ = '\0';
	if ((bp = strchr(sp, '>')) != (char *)NULL) *bp = '\0';
	strcpy(addr, sp);
	sp -= 2;
	rest = hdr.from_t;
	while (*sp == ' ' || *sp == '\t') *sp-- = '\0';
  }
  if ((sp = strchr(addr, ' ')) != (char *)NULL) *sp = '\0';
  if ((sp = strchr(addr, '@')) == (char *)NULL) {
	sprintf(temp, "%s!%s", myname, addr);
	strcpy(addr, temp);
  }
  if (rest == hdr.from_t) sprintf(temp, "%s <%s>", rest, addr);
    else sprintf(temp, "%s (%s)", addr, rest);
  free(hdr.from_t);
  hdr.from_t = (char *)malloc(strlen(temp) + 1);
  if (hdr.from_t == (char *)NULL) {
	logmsg(1, "out of memory\n");
	return;
  } else strcpy(hdr.from_t, temp);
}


/* Write out a new hdr. */
int hdr_out(temp, fname, lines)
char *temp;
char *fname;
int lines;
{
  char buff[1024];
  register HDR *hp;
  register char *bp;
  FILE *ifp, *ofp;
  register char **xp, **yp;

  /* Before writing it to disk, patch the in-core hdr. */
  hdr_end(lines);

  if ((ifp = fopen(temp, "r")) == (FILE *)NULL) {
	logmsg(1, "cannot open(%s)\n", temp);
	return(-1);
  }
  if ((ofp = fopen(fname, "w")) == (FILE *)NULL) {
	(void) fclose(ifp);
	logmsg(1, "cannot create(%s)\n", fname);
	return(-1);
  }

  /* Write the in-core header to the file. */
  xp = (char **) &hdr.path;
  while (**xp != '\0') {
	yp = xp; yp++;
	if (*yp != (char *)NULL) fprintf(ofp, "%s %s\n", *xp, *yp);
	xp = ++yp;
  }
  hp = hdr.fields;
  while (hp != NILHDR) {
	fprintf(ofp, "%s %s\n", hp->field, hp->text);
	hp = hp->next;
  }
  xp = (char **) &hdr.refs;
  while (**xp != '\0') {
	yp = xp; yp++;
	if (*yp != (char *)NULL) fprintf(ofp, "%s %s\n", *xp, *yp);
	xp = ++yp;
  }
  fprintf(ofp, "\n");

  /* Now, write the message body to the file. */
  do {
	if ((bp = fgets(buff, 1024, ifp)) == (char *)NULL) break;
	fprintf(ofp, "%s", buff);
  } while(1);

  (void) fclose(ifp);
  (void) fclose(ofp);
  return(0);
}


/* Check if we have a chance of a safe local deliverey. */
int del_chk(dir)
char *dir;
{
  struct stat buff;

  /* Directory present? */
  if (stat(dir, &buff) < 0) return(1);

  /* Access OK? */
  if ((buff.st_mode & 0700) != 0700) return(1);

  return(0);
}


/* Deliver the article into one newsgroup. */
void del_one(temp, group, aname)
char *temp;
char *group;
char *aname;
{
  static char fname[128];
  unsigned int art;

  /* Start out clean. */
  if (aname != (char *)NULL) strcpy(aname, "");

  /* Create a directory name for the article and check it. */
  sprintf(fname, "%s/%s", SPOOLDIR, gtof(group));
  if (del_chk(fname) != 0) {
	logmsg(1, "%s ng %s has no directory, moved to JUNK (%s)\n",
					hdr.iden_t, group, JUNK);
	group = JUNK;
	sprintf(fname, "%s/%s", SPOOLDIR, group);
	if (del_chk(fname) != 0) {
		logmsg(1, "%s ng JUNK (%s) has no directory, article is lost\n",
							hdr.iden_t, group);
		return;
	}
  }


  /* We now have a directory. Try to update the "active" file. */
  if ((art = act_upd(group)) == 0) {
	if (strcmp(group, JUNK)) {
		logmsg(0, "%s ng %s not localized, moved to JUNK (%s)\n",
						hdr.iden_t, group, JUNK);
		group = JUNK;
		sprintf(fname, "%s/%s", SPOOLDIR, group);
		if ((art = act_upd(group)) == 0) {
			logmsg(1,
			    "%s ng JUNK (%s) not localized, article is lost\n",
							hdr.iden_t, group);
			return;
		}
	} else {
		logmsg(1, "%s ng %s not localized, article is lost\n",
							hdr.iden_t, group);
		return;
	}
  }

  /* We updated "active" sucessfully.  Link the article. */
  sprintf(fname, "%s/%s/%u", SPOOLDIR, gtof(group), art);
  if (link(temp, fname) < 0) {
	logmsg(1, "cannot link(%s,%s)\n", temp, fname);
	return;
  } 

  /* Log the receipt of this article in the history file. */
  hist_add(hdr.iden_t, group, art);

  /* Create a file name that can be used by the batching scheme. */
  if (aname != (char *)NULL) strcpy(aname, fname);
}


/* Post an article on the local machine. */
void del_loc(fname, aname)
char *fname;
char *aname;
{
  register char *sp, *bp;

  /* Check for and execute the control message. */
  if (hdr.ctrl_t != (char *)NULL) {
	ctrl_msg(hdr.ctrl_t, fname);
	if (!strstr(hdr.ngrp_t, "control")) {
		sp = hdr.ngrp_t;
		bp = hdr.ctrl_t;
		hdr.ngrp_t = "control";
		hdr.ctrl_t = (char *)NULL;
		del_loc(fname, (char *)NULL);
		hdr.ngrp_t = sp;
		hdr.ctrl_t = bp;
		return;
	}
  }

  /* Loop over the Newsgroups: field, and post the message. */
  sp = hdr.ngrp_t;
  do {
	bp = sp;
	while (*sp && *sp != ',' && *sp != ' ') sp++;
	if (*sp == ',' || *sp == ' ') *sp++ = '\0';
	while (*sp == ' ') sp++;
	if (*bp == '\0') continue;

	del_one(fname, bp, aname);

  } while (*bp != '\0');

  /* Add a record to the resource accounting database. */
  acct_upd("rcvd", hdr.dist_t, bytes, remote);
}


/* Execute a delivery program with the article as standard input. */
int del_cmd(what, fname)
char *what;
char *fname;
{
  char *args[16];
  int i, pid, status;
  register char *sp;

  i = 0;
  sp = what;
  do {
	args[i++] = sp;
	while (*sp && *sp != ' ' && *sp != '\t') sp++;
	if (*sp == ' ' || *sp == '\t') *sp++ = '\0';    
  } while (*sp != '\0');
  args[i] = (char *)NULL;

  if ((pid = fork()) == 0) {
	(void) close(0);
	(void) open(fname, O_RDONLY);
	i = execv(args[0], args);
	exit(i);
  } else if (pid > 0) {
	while (wait(&status) != pid)
		    ;
  }
  if (pid < 0 || WEXITSTATUS(status) != 0) return(-1);
  return(0);
}


/* Send an article to a remote system. */
int del_rmt(fname, aname, ap)
char *fname;
char *aname;
SYS *ap;
{
  char buff[1024];
  FILE *fp;
  char c;
  int st;

  /* If no article available, fake it is "OK". */
  if ((aname == (char *)NULL) || (aname[0] == '\0')) return(0);

  c = (ap->prog == (char *)NULL) ? 'B' : *ap->prog;
  switch(c) {
	case 'A':       /* "A News" not supported */ 
		logmsg(1, "%s 'A' News format not supported.\n", hdr.iden_t);
		return(-1);
	case 'B':       /* "B News" format, single article */
		sprintf(buff, "%s %s!rnews", UUX, ap->name);
		st = del_cmd(buff, fname);
		break;
	case 'F':       /* "B News" format, batched */
		/* Add this article to the system's batch list. */
		if (ap->file == (char *)NULL) {
			sprintf(buff, "%s/%s", BATCHDIR, ap->name);
		} else strcpy(buff, ap->file);
		if ((fp = fopen(buff, "a")) != (FILE *)NULL) {
			fprintf(fp, "%s\n", aname);
			(void) fclose(fp);
			st = 0;
		} else st = -1;
		break;
	case 'X':       /* use a non-standard program */
		strcpy(buff, ap->file);
		st = del_cmd(buff, fname);
		break;
	default:
		/* Oops!  What else can we do but to cry out and ignore? */
		logmsg(1, "<%s> bad delivery code (%s) for %s\n",
					hdr.iden_t, ap->prog, ap->name);
		return(-1);
  }
  return(st);
}


/* Post a completed article. */
int do_post(fname)
char *fname;
{
  char allsys[1024];
  char artname[128];
  struct stat buf;
  register SYS *ap;
  register char *sp;

  /* No. Check the article header for completeness. */
  if (hdr.path_t == (char *)NULL ||
      hdr.from_t == (char *)NULL ||
      hdr.subj_t == (char *)NULL ||
      hdr.ngrp_t == (char *)NULL ||
      hdr.dist_t == (char *)NULL ||
      hdr.date_t == (char *)NULL ||
      hdr.iden_t == (char *)NULL) {
	logmsg(1, "incomplete article header\n");
	return(-1);
  }

  /*
   * Check if we saw this article before.
   * This only causes us to forget all about it, and exit, so
   * any "unbatch" program can continue running.
   */
  if (hist_chk(hdr.iden_t) == 1) return(0);

  /* Determine the length of the article. */
  if (stat(fname, &buf) < 0) {
	logmsg(1, "cannot stat(%s)\n", fname);
	return(-1);
  }
  bytes = buf.st_size;

  /* Loop over the in-core "sys" file to send the article to all hosts. */
  ap = NILSYS;
  strcpy(allsys, "");
  strcpy(artname, "");
  do {
	ap = (ap == NILSYS) ? sys : ap->next;
	if (ap == NILSYS) break;

	/* The local host must _always_ accept it. */
	if (!strcmp(ap->name, "ME") || !strcmp(ap->name, myname)) {
		if (!ngmatch(ap->name, hdr.ngrp_t, (char *)NULL, ap->ngrp)) {
			sp = hdr.ngrp_t;
			hdr.ngrp_t = JUNK;
		} else sp = (char *)NULL;
		del_loc(fname, artname);
		if (sp != (char *)NULL) hdr.ngrp_t = sp;
	} else {
		/* Skip this if posted locally. */
		if (!strcmp(hdr.dist_t, "local")) continue;

		/* Is this host already in the Path: field? */
		if (strstr(hdr.path_t, ap->name)) continue;

		/* Take care of the "-x sys" option. */
		if (dontsend != (char *)NULL)
			if (strstr(dontsend, ap->name)) continue;

		if (ngmatch(ap->name, hdr.ngrp_t, hdr.dist_t, ap->ngrp)) {
			if (del_rmt(fname, artname, ap) == 0) {
				strcat(allsys, ap->name);
				strcat(allsys, " ");
			} else logmsg(1, "cannot del_rmt(%s,%s,%s)\n",
						fname, artname, ap->name);
		}
	}
  } while (ap != NILSYS);

  if (strlen(allsys) > 0) {
	allsys[strlen(allsys) - 1] = '\0';
	logmsg(0, "%-8.8s  %s sent to %s\n", remote, hdr.iden_t, allsys);

	/* Add a record to the resource accounting database. */
	acct_upd("sent", hdr.dist_t, bytes, allsys);
  }
  return(0);
}


/* Add the ".signature" file to fp. */
void app_signature(ofp)
FILE *ofp;
{
  char fname[128];
  char buff[1024];
  struct passwd *pw;
  FILE *fp;
  char *sp;

  if ((sp = getenv("SIGNATURE")) == (char *)NULL) {
	pw = getpwuid(getuid());
	if (pw == (struct passwd *)NULL) {
		fprintf(stderr, "inews: cannot get home directory!\n");
		return;
	}
	sprintf(fname, "%s/%s", pw->pw_dir, SIGNATURE);
  } else strcpy(fname, sp);

  fprintf(ofp, "--\n");
  if ((fp = fopen(fname, "r")) != (FILE *)NULL) {
	while (fgets(buff, 1024, fp) != (char *)NULL) {
		fprintf(ofp, "%s", buff);
	}
	(void) fclose(fp);
  }
}


/* Process an incoming article. */
int inews(ifp)
register FILE *ifp;
{
  char buff[1024];
  char artname[128];
  char tempart[128];
  int state, ret;
  int lines;
  FILE *fp;

  /* Create a temporary article file. */
  strcpy(artname, "/tmp/inXXXXXX"); mktemp(artname);
  sprintf(tempart, "%s/inXXXXXX", SPOOLDIR); mktemp(tempart);

  if ((fp = fopen(artname, "w")) == (FILE *)NULL) {
	logmsg(1, "cannot create(%s)\n", artname);
	return(1);
  }

  /* Now, go and fetch the incoming article. */
  ret = -1;
  lines = 0;
  state = hdr_upd((char *)NULL, 0);
  while (state != -1) {
	if (fgets(buff, 1024, ifp) == (char *)NULL) state = 2;
	switch(state) {
		case -1:
			break;
		case 0:
			state = hdr_upd(buff, 0);
			break;
		case 1:
			fprintf(fp, "%s", buff);
			lines++;
			break;
		case 2:
			if (!opt_p) app_signature(fp);
			(void) fclose(fp);
			ret = hdr_out(artname, tempart, lines);
			state = -1;
			break;
	}
  }
  
  /* Record this in the log file. */
  if (opt_p == 1) {
	logmsg(0, "%-8.8s  received %s ng %s subj '%s' from %s\n",
		remote, hdr.iden_t, hdr.ngrp_t, hdr.subj_t,
							hdr.from_t);
  } else {
	logmsg(0, "%-8.8s  posted %s ng %s subj '%s' from %s\n",
		remote, hdr.iden_t, hdr.ngrp_t, hdr.subj_t,
							hdr.from_t);
  }

  /* We now have an article. Post it. */
  state = do_post(tempart);
  if (state != 0) ret = state;
    else (void) unlink(artname);

  (void) unlink(tempart);
  return(ret);
}


void usage()
{
  fprintf(stderr,
	"Usage: inews [-v] -h | {-t title -n newsgroups} [-e expires] ");
  fprintf(stderr,
	"[-f sender]\n             [-d distribution] [-F refs] [-o ");
  fprintf(stderr,
	"organization] [-a approved]\n             [-r reply-to] ");
  fprintf(stderr, "[-x dontsend] [-c control] [artname]\n");
  fprintf(stderr, "       inews [-v] -p artname\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  char buff[128];
  FILE *ifp, *fp;
  register char *sp;
  char *allflags;
  int st;
  extern int getopt(), optind;
  extern char *optarg;
#ifndef COHERENT
  extern int opterr;
#endif

  (void) umask(0133);
  (void) signal(SIGPIPE, SIG_IGN);

  old_uid = getuid();
  (void) time(&mytime);
  mydate = localtime(&mytime);

#ifndef COHERENT
  opterr = 0;
#endif
  allflags = "hpvF:a:c:d:e:f:n:o:r:t:x:";
  while ((st = getopt(argc, argv, allflags)) != EOF) switch(st) {
	case 'h':
		opt_h = 1;
		break;
	case 'p':
		opt_p = 1;
		opt_h = 1;      /* -p implies -h */
		break;
	case 'v':
		debug = 1;
		break;
	case 'F':
		hdr.refs_t = (char *)malloc(strlen(optarg) + 1);
		strcpy(hdr.refs_t, optarg);
		break;
	case 'a':
		hdr.appr_t = (char *)malloc(strlen(optarg) + 1);
		strcpy(hdr.appr_t, optarg);
		break;
	case 'c':
		hdr.ctrl_t = (char *)malloc(strlen(optarg) + 1);
		strcpy(hdr.ctrl_t, optarg);
		break;
	case 'd':
		hdr.dist_t = (char *)malloc(strlen(optarg) + 1);
		strcpy(hdr.dist_t, optarg);
		break;
	case 'e':
		hdr.expr_t = (char *)malloc(strlen(optarg) + 1);
		strcpy(hdr.expr_t, optarg);
		break;
	case 'f':
		hdr.send_t = (char *)malloc(strlen(optarg) + 1);
		strcpy(hdr.send_t, optarg);
		break;
	case 'n':
		hdr.ngrp_t = (char *)malloc(strlen(optarg) + 1);
		strcpy(hdr.ngrp_t, optarg);
		break;
	case 'o':
		hdr.orgn_t = (char *)malloc(strlen(optarg) + 1);
		strcpy(hdr.orgn_t, optarg);
		break;
	case 'r':
		hdr.repl_t = (char *)malloc(strlen(optarg) + 1);
		strcpy(hdr.repl_t, optarg);
		break;
	case 't':
		hdr.subj_t = (char *)malloc(strlen(optarg) + 1);
		strcpy(hdr.subj_t, optarg);
		break;
	case 'x':
		dontsend = (char *)malloc(strlen(optarg) + 1);
		strcpy(dontsend, optarg);
		break;
	default:
		usage();
  }

  if (opt_p == 1) {
	if (optind != (argc - 1)) usage();
	if (strcmp(argv[optind], "-")) {
		if ((ifp = fopen(argv[optind], "r")) == (FILE *)NULL) {
			logmsg(1, "cannot open file %s\n", argv[optind]);
			exit(-1);
		}
	} else ifp = stdin;
  } else {
	if (opt_h == 0) {
		if (hdr.subj_t == (char *)NULL ||
		    hdr.ngrp_t == (char *)NULL) usage();
	}
	if (optind != argc) {
		if ((ifp = fopen(argv[optind], "r")) == (FILE *)NULL) {
			logmsg(1, "cannot open(%s)\n", argv[optind]);
			exit(-1);
		}
	} else ifp = stdin;
  }

  /* Read the "sys" file into core. */
  if (sys_rd() != 0) {
	sprintf(buff, "cannot read sys file\n");
	logmsg(1, buff);
	exit(-1);
  }

  /* First of all, fetch our HOST.DOMAIN names. */
#if HAVE_HOST
  if (gethostname(myname, 128) < 0) {
	sprintf(buff, "cannot get HOSTNAME (%d)\n", errno);
	logmsg(1, buff);
	exit(-1);
  }
#endif	/* HAVE_HOSTNAME */
#ifdef HAVE_HOSTENV
  if ((sp = getenv(HAVE_HOSTENV)) == (char *)NULL) {
	sprintf(buff, "cannot get HOSTNAME from env\n");
	logmsg(1, buff);
	exit(-1);
  }
  strcpy(myname, sp);
#endif	/* HAVE_HOSTENV */
#ifdef HAVE_UUHOST
  if ((fp = fopen(HAVE_UUHOST, "r")) == (FILE *)NULL) {
	sprintf(buff, "cannot open(%s)\n", HAVE_UUHOST);
	logmsg(1, buff);
	exit(-1);
  }
  if ((sp = fgets(buff, 128, fp)) == (char *)NULL || buff[0] == '\n') {
	sprintf(buff, "cannot get HOSTNAME from %s\n", HAVE_UUHOST);
	logmsg(1, buff);
	exit(-1);
  } else {
	if ((sp = strchr(buff, '\n')) != NULL) *sp = '\0';
	strcpy(myname, buff);
  }
#ifndef HAVE_UUDOM
  /* Next, fetch our domain name. */
  if ((sp = fgets(buff, 128, fp)) == (char *)NULL || buff[0] == '\n') {
	sprintf(buff, "cannot get DOMAINNAME from %s\n", HAVE_UUHOST);
	logmsg(1, buff);
	exit(-1);
  } else {
	if ((sp = strchr(buff, '\n')) != NULL) *sp = '\0';
	strcat(myname, ".");
	strcat(myname, buff);
  }
#else
  (void) fclose(fp);
  if ((fp = fopen(HAVE_UUDOM, "r")) == (FILE *)0) {
	sprintf(buff, "cannot open(%s)\n", HAVE_UUDOM);
	logmsg(1, buff);
	exit(-1);
  }
  if ((sp = fgets(buff, 128, fp)) == NULL || buff[0] == '\n') {
	sprintf(buff, "cannot get DOMAINNAME from %s\n", HAVE_UUDOM); 
	logmsg(1, buff);
	exit(-1);
  } else {
	if ((sp = strchr(buff, '\n')) != NULL) *sp = '\0';
	strcat(myname, ".");
	strcat(myname, buff);
  }
#endif	/* HAVE_UUHOST */
  (void) fclose(fp);
#endif	/* HAVE_HOST */

  /* Now, get our organizational description. */
  sprintf(buff, "%s/%s", CONFDIR, ORGANIZAT);
  if ((fp = fopen(buff, "r")) != (FILE *)NULL) {
	if ((sp = fgets(buff, 128, fp)) != (char *)NULL && buff[0] != '\n') {
		if ((sp = strchr(buff, '\n')) != (char *)NULL) *sp = '\0';
		strcpy(myorg, buff);
	} else strcpy(myorg, "");
	(void) fclose(fp);
  } else strcpy(myorg, "");

  st = inews(ifp);
  (void) fclose(ifp);

  exit(st);
}
