/*
 * W-NEWS       A simple NEWS processing package for MINIX.
 *              This program implements the functions of several of the
 *              standard "control messages" that can be sent to a NEWS-
 *              running system.  Since we can save much effort by doing
 *              certain things only once, we have only one program with
 *              links to the names of the possible control messages.
 *
 * Usage:       newgroup newsgroup_name [ moderated ]
 *              rmgroup newsgroup_name
 *              version
 *              cancel <message_id>
 *              sendsys
 *              senduuname
 *              checkgroups
 *
 * Version:     @(#)control.c	4.02	02/23/93
 *
 * Author:      Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *		Copyright 1988-1993 MicroWalt Corporation
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
#include "wnews.h"


typedef struct _active_ {
  struct _active_ *next;
  char            *name;
} ACT;
#define ACTSIZE (sizeof(ACT))
#define NIL_ACT ((ACT *)NULL)


static char *Release = VERSION;
static char *Version = "@(#) control 4.02 (02/23/93)";


char *prog;                             /* what is our name?            */
int debug = 0;                          /* are we debugging?            */
ACT *active = NIL_ACT;                  /* in-core "active" file        */
struct {                                /* this is the in-core header   */
  char  *path,  *path_t;
  char  *from,  *from_t;
  char  *send,  *send_t;
  char  *repl,  *repl_t;
  char  *ngrp,  *ngrp_t;
  char  *dist,  *dist_t;
  char  *ctrl,  *ctrl_t;
  char  *subj,  *subj_t;
  char  *iden,  *iden_t;
  char  *appr,  *appr_t;
  char  *null;
} header = {
  "Path:",              (char *)NULL,
  "From:",              (char *)NULL,
  "Sender:",            (char *)NULL,
  "Reply-To:",          (char *)NULL,
  "Newsgroups:",        (char *)NULL,
  "Distribution:",      "local",
  "Control:",           (char *)NULL,
  "Subject:",           (char *)NULL,
  "Message-ID:",        (char *)NULL,
  "Approved:",          (char *)NULL,
  (char *)NULL
};
char *lgmonths[] = {                            /* for debugging        */
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};


_PRO( int act_lck, (char *name)						);
_PRO( int act_unl, (char *name)						);
_PRO( char *act_rd, (void)						);
_PRO( ACT *ng_find, (char *name)					);
_PRO( int ctrl_newgroup, (char *name, int mod)				);
_PRO( int ctrl_rmgroup, (char *name)					);
_PRO( int ctrl_cancel, (char *msgid)					);
_PRO( int ctrl_checkgroups, (void)					);
_PRO( int ctrl_version, (void)						);
_PRO( int ctrl_ssys, (void)						);
_PRO( int ctrl_suuname, (void)						);
_PRO( int hdr_upd, (char *text)						);


/* Write something to the log files. */
#ifdef __STDC__
void logmsg(int err, char *fmt, ...)
#else
void logmsg(err, fmt)
int err;
char *fmt;
#endif
{
  char buff[1024];
  FILE *lfp, *efp;
  char lfn[128], efn[128];
  struct tm *mydate;
  va_list args;
  time_t now;

  sprintf(lfn, "%s/%s", CONFDIR, LOGFILE);
  sprintf(efn, "%s/%s", CONFDIR, ERRLOG);

  if ((efp = fopen(efn, "a")) == (FILE *)NULL) {
	fprintf(stderr, "rnews: cannot append(%s)\n", efn);
	return;
  }

  if ((lfp = fopen(lfn, "a")) == (FILE *)NULL) {
	fprintf(efp, "rnews: cannot append(%s)\n", lfn);
	(void) fclose(efp);
	return;
  }

  (void) time(&now);
  mydate = localtime(&now);
  sprintf(buff, "%s %02d %02d:%02d  ",
	lgmonths[mydate->tm_mon], mydate->tm_mday,
			mydate->tm_hour, mydate->tm_min);
  va_start(args, fmt);
  if (err == 1) {
	fprintf(efp, buff);
	vfprintf(efp, fmt, args);
  } else {
	fprintf(lfp, buff);
	vfprintf(lfp, fmt, args);
  }
  if (debug > 0) {
	fprintf(stderr, buff);
	vfprintf(stderr, fmt, args);
  }
  (void) fclose(lfp);
  (void) fclose(efp);
}


/* Try to lock "active". We allow others one minute. */
int act_lck(name)
char *name;
{
  char buff[128];
  int fd, try;

  try = 0;
  sprintf(buff, "%s.lock", name);
  while (try < 10) {
	if ((fd = creat(buff, 0644)) >= 0) break;
	if (++try >= 10) {
		logmsg(1, "%s: cannot lock(%s)\n", prog, name);
		return(-1);
	}
	(void) sleep(6);
  }
  (void) close(fd);
  return(0);
}


/* Unlock the "active" file. */
int act_unl(name)
char *name;
{
  char buff[128];

  sprintf(buff, "%s.lock", name);
  if (unlink(buff) < 0) {
	logmsg(1, "%s: cannot unlock(%s)\n", prog, name);
	return(-1);
  }
  return(0);
}


/* Read the "active" file into core. */
char *act_rd()
{
  static char fname[128];
  char buff[128];
  register ACT *ap, *lp;
  register FILE *fp;
  register char *sp;

  /* Create file name and try to lock "active". */
  sprintf(fname, "%s/%s", CONFDIR, ACTIVE);
  if (act_lck(fname) != 0) return((char *)NULL);

  /* Open the file for reading. */
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	logmsg(1, "%s: cannot open(%s)\n", prog, fname);
	(void) act_unl(fname);
	return((char *)NULL);
  }

  /* Now, read the file, and store it in core. */
  lp = (ACT *)NULL;
  while (fgets(buff, 128, fp) != (char *)NULL) {
	if (buff[0] == '\n' || buff[0] == '0') continue;
	if ((sp = strchr(buff, ' ')) != (char *)NULL) *sp = '\0';
	if (active == NIL_ACT) {
		active = (ACT *)malloc(ACTSIZE);
		ap = active;
	} else {
		ap = lp;
		ap->next = (ACT *)malloc(ACTSIZE);
		ap = ap->next;
	}
	if (ap == NIL_ACT) {
		logmsg(1, "%s: out of memory\n", prog);
		(void) fclose(fp);
		(void) act_unl(fname);
		return((char *)NULL);
	}
	lp = ap;
	ap->next = NIL_ACT;
	ap->name = (char *)malloc(strlen(buff) + 1);
	if (ap->name == (char *)NULL) {
		logmsg(1, "%s: out of memory\n", prog);
		(void) fclose(fp);
		(void) act_unl(fname);
		return((char *)NULL);
	}
	strcpy(ap->name, buff);
  }
  (void) fclose(fp);
  return(fname);
}


/* Find a newsgroup in the in-core "active" file. */
ACT *ng_find(name)
char *name;
{
  register ACT *ap;

  ap = active;
  while (ap != NIL_ACT) {
	if (!strcmp(ap->name, name)) return(ap);
	ap = ap->next;
  }
  return(NIL_ACT);
}


/* Perform the "newgroup groupname [moderated]" command. */
int ctrl_newgroup(name, mod)
char *name;
int mod;
{
  char buff[512], *fname;
  char temp[128];
  register char *sp, *bp;
  register FILE *fp;

  /* Read the "active" file into core. */
  if ((fname = act_rd()) == (char *)NULL) return(-1);

  /* Does it already exist? */
  if (ng_find(name) != NIL_ACT) {
	logmsg(1, "%s: newsgroup %s already exists\n", prog, name);
	(void) act_unl(fname);
	return(-1);
  }

  /* No, update the "active" file. */
  if ((fp = fopen(fname, "a")) == (FILE *)NULL) {
	logmsg(1, "%s: cannot append(%s)\n", prog, fname);
	(void) act_unl(fname);
	return(-1);
  }
  fprintf(fp, "%s 00000 00000 %c\n", name, (mod == 1) ? 'm' : 'y');
  (void) fclose(fp);
  (void) act_unl(fname);

  /* Read the news group description from the input article. */
  sp = (char *)NULL;
  do {
	if (fgets(buff, 512, stdin) == (char *)NULL) break;
	if (buff[0] == '0' || buff[0] == '\n') continue;
	sp = buff;
  } while (sp == (char *)NULL);

  /* Update the "newsgroups" file. */
  sprintf(temp, "%s/%s", CONFDIR, NWGROUPS);
  if ((fp = fopen(temp, "a")) == (FILE *)NULL) {
	logmsg(1, "%s: cannot append(%s)\n", prog, temp);
	return(-1);
  }
  fprintf(fp, "%s\t\t%s", name,
	(sp == (char *)NULL) ? "<description unavailable>\n" : sp);
  (void) fclose(fp);

  /* Finally, create the directory(ies) for the new group. */
  strcpy(buff, name);
  sp = buff;
  while (*sp != '\0') {
	if (*sp == '.') *sp = '/';
	sp++;
  }
  sp = SPOOLDIR;
  if (chdir(sp) < 0) {
	logmsg(1, "%s: cannot chdir(%s)\n", prog, sp);
	return(-1);
  }

  /* Recursively create directories.  We use external commands. */
  sp = buff;
  while (sp != (char *)NULL) {
	if ((bp = strchr(sp, '/')) != (char *)NULL) *bp++ = '\0';
	if (chdir(sp) < 0) {
                sprintf(temp, "exec mkdir %s\n", sp);
		if (system(temp) != 0) {
			logmsg(1, "%s: cannot mkdir(%s)\n", prog, sp);
			return(-1);
		}
		sprintf(temp, "exec chmod 0755 %s\n", sp);
		if (system(temp) != 0) {
			logmsg(1, "%s: cannot chmod(%s, 0775)\n", prog, sp);
			return(-1);
		}
		if (chdir(sp) < 0) {
			logmsg(1, "%s: cannot chdir(%s)\n", prog, sp);
			return(-1);
		}
	}
	sp = bp;
  }
  return(0);
}


/* Perform the "rmgroup groupname" command. */
int ctrl_rmgroup(name)
char *name;
{
  char buff[512], *fname;
  char temp[128], more[128];
  register char *sp;
  register FILE *fp, *fp2;

  /* Read the active file into core. */
  if ((fname = act_rd()) == (char *)NULL) return(-1);

  /* Does it exist at all? */
  if (ng_find(name) == NIL_ACT) {
	logmsg(1, "%s: newsgroup %s does not exist\n", prog, name);
	(void) act_unl(fname);
	return(-1);
  }

  /* Yes, update the "active" file. */
  sprintf(more, "%s2", fname);
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	logmsg(1, "%s: cannot open(%s)\n", prog, fname);
	(void) act_unl(fname);
	return(-1);
  }
  if ((fp2 = fopen(more, "w")) == (FILE *)NULL) {
	logmsg(1, "%s: cannot create(%s)\n", prog, more);
	(void) fclose(fp);
	(void) act_unl(fname);
	(void) unlink(more);
	return(-1);
  }
  while (fgets(buff, 512, fp) != (char *)NULL) {
	if (buff[0] == '0' || buff[0] == '\n') continue;
	if ((sp = strchr(buff, ' ')) != (char *)NULL) *sp++ = '\0';
	if (strcmp(buff, name)) fprintf(fp2, "%s %s", buff, sp);
  }
  (void) fclose(fp);
  (void) fclose(fp2);
  if (unlink(fname) < 0) {
	logmsg(1, "%s: cannot unlink(%s)\n", prog, fname);
	(void) unlink(more);
	(void) act_unl(fname);
	return(-1);
  }
  if (rename(more, fname) < 0) {
	logmsg(1, "%s: cannot rename(%s,%s)\n", prog, more, fname);
	(void) unlink(more);
	(void) act_unl(fname);
	return(-1);
  }
  (void) unlink(more);
  (void) chmod(fname, 0644);
  (void) act_unl(fname);

  /* Update the "newsgroups" file. */
  sprintf(temp, "%s/%s", CONFDIR, NWGROUPS);
  sprintf(more, "%s2", temp);
  if ((fp = fopen(temp, "r")) == (FILE *)NULL) {
	logmsg(1, "%s: cannot open(%s)\n", prog, temp);
	return(-1);
  }
  if ((fp2 = fopen(more, "w")) == (FILE *)NULL) {
	logmsg(1, "%s: cannot create(%s)\n", prog, more);
	(void) fclose(fp);
	return(-1);
  }
  while (fgets(buff, 512, fp) != (char *)NULL) {
	if (buff[0] == '0' || buff[0] == '\n') continue;
	if ((sp = strchr(buff, '\t')) != (char *)NULL) *sp++ = '\0';
	if (strcmp(buff, name)) fprintf(fp2, "%s\t%s", buff, sp);
  }
  (void) fclose(fp);
  (void) fclose(fp2);
  if (unlink(temp) < 0) {
	logmsg(1, "%s: cannot unlink(%s)\n", prog, temp);
	(void) unlink(more);
	return(-1);
  }
  if (rename(more, temp) < 0) {
	logmsg(1, "%s: cannot rename(%s,%s)\n", prog, more, temp);
	(void) unlink(more);
	return(-1);
  }
  (void) unlink(more);
  (void) chmod(temp, 0644);

  /* Finally, remove the directory(ies) for this group. */
  sprintf(temp, "%s/%s", SPOOLDIR, name);
  sp = temp;
  while (*sp != '\0') {
	if (*sp == '.') *sp = '/';
	sp++;
  }
  sprintf(buff, "exec %s -rf %s", SYSRM, temp);
  (void) system(buff);
  return(0);
}


/* Perform the "cancel <msgid>" command. */
int ctrl_cancel(msgid)
char *msgid;
{
#if HAVE_HIST
  char buff[512], temp[128];
  char more[128], fname[128];
  register FILE *fp, *fp2;
  register char *sp, *bp;

  /* Step through the "history" file, deleting articles. */
  sprintf(fname, "%s/%s", CONFDIR, HISTFILE);
  sprintf(more, "%s2", fname);
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	logmsg(1, "%s: cannot open(%s)\n", prog, fname);
	return(-1);
  }
  if ((fp2 = fopen(more, "w")) == (FILE *)NULL) {
	logmsg(1, "%s: cannot create(%s)\n", prog, more);
	(void) fclose(fp);
	return(-1);
  }

  /* Now step through the file, looking for <msgid>. */
  while (fgets(buff, 512, fp) != (char *)NULL) {
	if (buff[0] == '\0' || buff[0] == '\n') continue;
	if ((sp = strchr(buff, '\t')) != (char *)NULL) *sp++ = '\0';
	if (!strcmp(buff, msgid)) {
		if ((bp = strchr(sp, '\n')) != (char *)NULL) *bp = '\0';
		if ((bp = strrchr(sp, '\t')) != (char *)NULL) {
			sp = ++bp;
			while (*sp != '\0') {
				if (*sp == '.') *sp = '/';
				sp++;
			}
			sprintf(temp, "%s/%s", SPOOLDIR, bp);
			if (unlink(temp) < 0) logmsg(1,
				"%s: cannot unlink(%s)\n", prog, temp);
		} else logmsg(1, "%s: %s format error!\n", prog, fname);
	} else fprintf(fp2, "%s\t%s", buff, sp);
  }
  (void) fclose(fp);
  (void) fclose(fp2);

  if (unlink(fname) < 0) {
	logmsg(1, "%s: cannot unlink(%s)\n", prog, fname);
	(void) unlink(more);
	return(-1);
  }
  if (rename(more, fname) < 0) {
	logmsg(1, "%s: cannot rename(%s,%s)\n", prog, more, fname);
	(void) unlink(more);
	return(-1);
  }
  (void) unlink(more);
  (void) chmod(fname, 0660);
#endif
  return(0);
}


/* Perform the "checkgroups" command. */
int ctrl_checkgroups()
{
  logmsg(1, "%s: this has not yet been implemented!\n", prog);
  return(1);
}


/* Perform the "version" command. */
int ctrl_version()
{
  logmsg(1, "%s: this has not yet been implemented!\n", prog);
  return(1);
}


/* Perform the "sendsys" command. */
int ctrl_ssys()
{
  logmsg(1, "%s: this has not yet been implemented!\n", prog);
  return(1);
}


/* Perform the "senduuname" command. */
int ctrl_suuname()
{
  logmsg(1, "%s: this has not yet been implemented!\n", prog);
  return(1);
}


/* Update our in-core header information. */
int hdr_upd(text)
char *text;
{
  register char *sp;
  char **xp, **yp;

  /* Is this the end-of-header marker? */
  if (*text == '\n') return(1);

  /* No. Remove the trailing newline. */
  if ((sp = strchr(text, '\n')) != (char *)NULL) *sp = '\0';

  /* First, split up the "Field: " and "value" parts. */
  if ((sp = strchr(text, ':')) == (char *)NULL) {
	logmsg(1, "%s: bad header field: \"%s\"\n", prog, text);
	return(-1);  
  } else {
	*++sp = '\0';
	sp++;
	while (*sp == ' ' || *sp == '\t') sp++;
  }

  /* Then, add this header field to the in-core header. */
  xp = (char **) &header.path;
  yp = (char **)NULL;
  while (*xp != (char *)NULL) {
	if (!strncmp(text, *xp, strlen(text))) {
		yp = xp; yp++;
		break;
	}
	xp++;
	xp++;
  }

  /* Ignore bad header fields. */
  if (yp == (char **)NULL) return(0);

  if ((*yp = (char *)malloc(strlen(sp) + 1)) == (char *)NULL) {
	logmsg(1, "%s: out of memory\n", prog);
	return(-1);
  }
  strcpy(*yp, sp);
  return(0);            /* stay in "scan header" state */
}


int main(argc, argv)
int argc;
char **argv;
{
  char buff[1024];
  int st;

  /* Determine the name by which we were called. */
  if ((prog = strrchr(*argv, '/')) != (char *)NULL) prog++;
    else prog = *argv;
  argv++;

  /* A quick check to allow for debugging during development. */
  if (!strcmp(*argv, "-d")) {
	debug = 1;
	argv++;
  }

  /* Read the header of the incoming article. */
  while (1) {
	if (fgets(buff, 1024, stdin) == (char *)NULL) break;
	if (hdr_upd(buff) != 0) break;
  }

  (void) umask(022);                    /* drwxr-xr-x */

  /* Carry out the correct control command. */
  if (!strcmp(prog, "newgroup")) {
	switch(argc) {
		case 2:
			st = ctrl_newgroup(*argv, 0);
			break;
		case 3:
			if (!strcmp(*(argv + 1), "moderated")) {
				st = ctrl_newgroup(*argv, 1);
				break;
			}
			/*FALLTHROUGH*/
		default:
			fprintf(stderr,
			"Usage: newgroup newsgroup_name [ moderated ]\n");
			st = -1;
			break;
	}
  } else if (!strcmp(prog, "rmgroup")) {
	if (argc != 2) fprintf(stderr, "Usage: rmgroup newsgroup_name\n");
	st = ctrl_rmgroup(*argv);
  } else if (!strcmp(prog, "cancel")) {
	if (argc != 2) fprintf(stderr, "Usage: cancel <message_id>\n");
	st = ctrl_cancel(*argv);
  } else if (!strcmp(prog, "sendsys")) {
	if (argc != 1) fprintf(stderr, "Usage: sendsys\n");
	st = ctrl_ssys();
  } else if (!strcmp(prog, "senduuname")) {
	if (argc != 1) fprintf(stderr, "Usage: senduuname\n");
	st = ctrl_suuname();
  } else if (!strcmp(prog, "version")) {
	if (argc != 1) fprintf(stderr, "Usage: version\n");
	st = ctrl_version();
  } else if (!strcmp(prog, "checkgroups")) {
	if (argc != 1) fprintf(stderr, "Usage: checkgroups\n");
	st = ctrl_checkgroups();
  } else {
	fprintf(stderr, "Usage: newgroup newsgroup_name [ moderated ]\n");
	fprintf(stderr, "       rmgroup newsgroup_name\n");
	fprintf(stderr, "       version\n");
	fprintf(stderr, "       cancel <message_id>\n");
	fprintf(stderr, "       sendsys\n");
	fprintf(stderr, "       senduuname\n");
	fprintf(stderr, "       checkgroups\n");
	st = -1;
  }

  exit(st);
}
