/*
 * W-NEWS       A simple NEWS processing package.
 *              This little program checks to see if the user has any
 *              news articles pending.  It can be used standalone, or
 *              it can be called from the /etc/profile startup file.
 *
 * Usage:       checknews [-nqvy] [ -N ng_list ]
 *
 * Version:     @(#)checknews.c	4.02	02/23/93
 *
 * Author:      Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *		Copyright 1988-1993 MicroWalt Corporation
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
#include "wnews.h"


typedef struct _active_ {
  struct _active_ *next;
  char            *name;
  unsigned int    seq;
} ACT;
#define ACTSIZE (sizeof(ACT))
#define NIL_ACT ((ACT *)NULL)


static char *Release = VERSION;
static char *Version = "@(#) checknews 4.02 (02/23/93)";


int opt_n = 0;                          /* "No News" is default         */
int opt_y = 1;                          /* "There is news" is default   */
int opt_q = 0;                          /* be quiet; just an exit value */
int opt_v = 0;                          /* tell which groups have news  */
ACT *active = NIL_ACT;                  /* in-core "active" file        */


_PRO( void act_rd, (void)						);
_PRO( unsigned int ng_find, (char *name)				);
_PRO( int chk_news, (char *ng_list)					);


/* Read the "active" file into core. */
void act_rd()
{
  char buff[128], fname[128];
  register ACT *ap, *lp;
  register FILE *fp;
  register char *sp, *bp;

  /* Open the "active" file for reading. */
  sprintf(fname, "%s/%s", CONFDIR, ACTIVE);
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	fprintf(stderr, "checknews: cannot open(%s)\n", fname);
	return;
  }

  /* Now, read the file, and store it in core. */
  lp = (ACT *)NULL;
  while (fgets(buff, 128, fp) != (char *)NULL) {
	if (buff[0] == '0' || buff[0] == '\n') continue;
	sp = strchr(buff, ' ');
	*sp++ = '\0';
	bp = strchr(sp, ' ');
	*bp++= '\0';

	if (active == NIL_ACT) {
		active = (ACT *)malloc(ACTSIZE);
		ap = active;
	} else {
		ap = lp;
		ap->next = (ACT *)malloc(ACTSIZE);
		ap = ap->next;
	}
	if (ap == NIL_ACT) {
		fprintf(stderr, "checknews: out of memory\n");
		(void) fclose(fp);
		return;
	}
	lp = ap;
	ap->next = NIL_ACT;
	ap->name = (char *)malloc(strlen(buff) + 1);
	if (ap->name == NULL) {
		fprintf(stderr, "checknews: out of memory\n");
		(void) fclose(fp);
		return;
	}
	strcpy(ap->name, buff);
	ap->seq = (unsigned int) atol(sp);
  }
  (void) fclose(fp);
}


/* Find a newsgroup in the in-core "active" file. */
unsigned int ng_find(name)
char *name;
{
  register ACT *ap, *bp;

  ap = active;
  bp = active;
  while (ap != NIL_ACT) {
	if (!strcmp(ap->name, name)) {
		/*
		 * This is a dirty hack, we just remove the found
		 * newsgroup from the linked list. This should make
		 * it a lot faster.
		 */
		if (ap == active) {
			active = ap->next;
		} else {
			bp->next = ap->next;
		}
		return(ap->seq);
	}
	bp = ap;
	ap = ap->next;
  }
  return((unsigned int) 0);
}


/* Check if there is any news pending. */
int chk_news(ng_list)
char *ng_list;
{
  char buff[128], fname[128];
  struct passwd *pw;
  register FILE *fp;
  register char *sp, *bp;
  unsigned int seq, act;
  int found;

  act_rd();

  /* Open the ACTIVE file for reading. */
  pw = getpwuid(getuid());
  if (pw == (struct passwd *)NULL) {
	fprintf(stderr, "checknews: cannot find myself\n");
	return(-1);
  }
  sprintf(fname, "%s/%s", pw->pw_dir, NEWSRC);
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	if (opt_v == 1) fprintf(stderr, "checknews: cannot open(%s)\n", fname);
	return(0);
  }

  /* Now, read the file, and check the numbers. */
  found = 0;
  while (fgets(buff, 128, fp) != (char *)NULL) {
	if (buff[0] == '0' || buff[0] == '\n') continue;
	sp = buff;
	while (*sp && *sp != ':') sp++;
	*sp++ = '\0';
	while (*sp && *sp != '-') sp++;
	bp = ++sp;
	while (*sp && *sp >= '0' && *sp <= '9') sp++;
	*sp = '\0';
	seq = (unsigned int) atol(bp);

	if (ng_list != (char *)NULL) {
		if (strcmp(ng_list, buff)) continue;
	}
	if ((act = ng_find(buff)) > seq) {
		if (opt_v == 1) {
			if (!found++)
				printf("Newsgroup:                             new articles:\n");
			if ((seq+1)==act) {
				printf("%-20s                        %d\n", 
							buff, act);
			} else {
				printf("%-20s                   %5u - %d\n",
							buff, seq+1, act);
			}
		} else {
			(void) fclose(fp);
			return(1);
		}
	}
  }
  (void) fclose(fp);
  return((found == 0) ? 0 : 1);
}


void usage()
{
  fprintf(stderr, "Usage: checknews [-nqvy] [ -N ng_list ]\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  register int c;
  int st;
  char *ng_list;
  extern int getopt(), optind;
#ifndef COHERENT
  extern int opterr;
#endif
  extern char *optarg;

  ng_list = (char *)NULL;               /* all groups by default        */

#ifndef COHERENT
  opterr = 0;
#endif
  while ((c = getopt(argc, argv, "nqvyN:")) != EOF) switch(c) {
	case 'n':
		opt_n = 1;
		break;
	case 'q':
		opt_q = 1;
		break;
	case 'v':
		opt_v = 1;
		break;
	case 'y':
		opt_y = 1;
		break;
	case 'N':
		ng_list = (char *)malloc(strlen(optarg) + 1);
		if (ng_list == (char *)NULL) {
			fprintf(stderr, "checknews: out of memory\n");
			exit(-1);
		}
		strcpy(ng_list, optarg);
		break;
	default:
		usage();
  }

  /* No more arguments allowed. */
  if (optind != argc) usage();

  /* Check the news system. */
  st = 0;
  switch(chk_news(ng_list)) {
	case 0:         /* no news */
		if (opt_q == 0) {
			if (opt_n == 1) printf("No news.\n");
		} else st = 0;
		break;
	case 1:         /* there is news */
		if (opt_q == 0) {
			if (opt_y == 1 && opt_v == 0)
					printf("There is news.\n");
		} else st = 1;
		break;
	case -1:        /* error */
		st = -1;
		break;
  }

  exit(st);
}
