/******************************************************************************
 * $Id: display.c,v 1.2 92/08/05 01:37:20 romel Exp Locker: romel $
 * Name:	Display
 *		DispFstats
 *		DispXcmd
 *		GetFilename
 *		GetUUfsize
 * Proj:	uustat
 * Desc:	Display: Displays information about a uucp job.  If the file is zero-
 *			length, it is a dummy file used to cause a poll (from uutouch),
 *			and a message is printed saying so.
 *		DispFstats: Displays the stats of a uucp data file or remote
 *			execution file
 *		DispXcmd: Displays a remote execution command
 *		GetFilename: Retrieves the name of the data file for a uucp job
 *		GetUUfsize: Retrieves the filesize of a uucp data file
 * Synop:	int Display (char *system, uuflags *uflags, char *filename);
 *		int DispFstats (char *system, char *line);
 *		int DispXcmd (char *file);
 *		char * GetFilename (char *line);
 *		long GetUUfsize (char *file);
 * Entry:	Display:
 *			system: string describing the remote system name
 *			uflags: struct containing option flags
 *			file: pointer to a char array containing the filename of the
 *				control file
 *		DispFstats:
 *			system: string describing the remote system name
 *			line: a description line in the control file
 *		DispXcmd:
 *			file: pointer to a char array containing the filename
 *		GetFilename:
 *			line: a description line in the control file
 *		GetUUfsize:
 *			file: pointer to a char array containing the filename
 * Exit:	Display: (int):
 *			0 -	successful
 *			-1 -	could not open control file
 *			-2 -	could not stat control file
 *			-3 -	could not obtain owner name of control file
 *			-4 -	error from DispFstats (maybe trouble reading data or execute
 *				file)
 *		DispFstats: (int) 0 if successful, -1 if not.
 *		DispXcmd: (int) 0 if successful, -1 if not.
 *		GetFilename: (char*) a pointer to the name of the data file or NULL.
 *			If valid, this pointer points inside the character array pointed
 *			to by line.
 *		GetUUfsize: (long) the size of the file or -99999 if the file could
 *			not be accessed by uustat.
 * Notes:	None
 * $Log:	display.c,v $
 * Revision 1.2  92/08/05  01:37:20  romel
 * minor changes
 * 
 * Revision 1.1  92/08/04  16:54:04  romel
 * Initial revision
 * 
 *****************************************************************************/
#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "uustat.h"


/* functions defined in this file: */
int Display ();
int DispFstats ();
int DispXcmd ();
char * GetFilename ();
long GetUUfsize ();

/*
*	Function: Display
*		Displays a file's statistics and other information, depending on
*		the given flags:
*		*	if fdispl is false, nothing will be displayed.  This function
*			will just return with 0.
*		*	if uflags->fuser is true and real userid of user issuing this
*			command matches the userid of user who created the given
*			file (given by filename), the stats will be displayed.
*		*	if uflags->fall is true, every filename passed will have its
*			stats displayed.  If both fuser and fall are true, fall will
*			have precedence.
*/
int Display (system, uflags, filename)
char * system;
char * filename;
uuflags * uflags;
	{
	static char line[LLEN+1], sender[LLEN+1];
	FILE * fp;
	struct stat fs;
	char *tmp;
	int first = 1;
	struct tm * dts;

	/* if no display is desired, just return with no error */
	if (!uflags->fdispl)
		{
		return 0;
		}
		
	/* 
	*	gather information about the file 
	*/
	/* open file for reading */
	fp = fopen (filename, "r");
	if (fp == NULL)
		{
		return -1;
		}

	/* get some info from os */
	if (stat (filename, &fs) == -1)
		{ /* error during fstat */
		fclose (fp);
		return -2;
		} /* if fstat */

	/* 
	*	obtain userid of user who created job.  GetSend will retrieve
	*	this info and then rewind the file pointer.  Allocate space for
	*	the string and save it.  If fs.st_size = 0 (file size is zero), 
	*	then file is a dummy control file used to cause a poll to that site.
	*/
	if (fs.st_size)
		{
		tmp = GetSender (fp);
		if (tmp == NULL)
			{ /* GetSender could not retrieve owner name - maybe corrupt or
				invalid file */
			fclose (fp);
			return -3;
			}
			
		strcpy (sender, tmp);	
		} /* if fs.st_size */
	else
		{
		*sender = '\0';	
		}
		
	
	/* 
	*	=== determine whether the file should be displayed ===
	*	if uflags->fuser is true and real userid of user issuing this
	*	command matches the userid of user who created the given
	*	file (given by filename), the stats will be displayed.
	*
	*	if uflags->fall is true, every filename passed will have its
	*	stats displayed.  If both fuser and fall are true, fall will
	*	have precedence.
	*/
	if (uflags->fall || (uflags->fuser && !strcmp (sender, uflags->userid)))
		{
		/* 
		*	display the file's information 
		*/
		/* convert the last time the file was modified into a more useful
		form */
		dts = localtime (&fs.st_mtime);
			
		/* print the jobid of the file */
		printf ("%-12s  ", GetJobId (filename));
				
		/* if file size (fs.st_size=0) file is zero length - dummy file.
		In this situation, display only if fall is true */
		if (fs.st_size == 0)
			{
			/* print the date and time stamp of the control file */
			printf ("%02d/%02d-%02d:%02d ", dts->tm_mon, dts->tm_mday, dts->tm_hour,
			dts->tm_min);

			/* say it's a dummy file */
			printf ("*** Empty control file ***\n");
			}
		else
			{	
			/* read in each line and display the information */
			while ((fgets (line, LLEN, fp)) != NULL)
				{
				if (first)
					first = 0;
				else
					printf ("              ");
					
				/* print the date and time stamp of the control file */
				printf ("%02d/%02d-%02d:%02d ", dts->tm_mon, dts->tm_mday, dts->tm_hour,
						dts->tm_min);
				
				/* do not put more than one occurrance of GetField() in one
				statement. GetField keeps a static variable to hold a string
				and it stays intact until the next call to GetField.  Two or
				more GetFields (say, in a printf() call) will result in garbage
				output.
				*/		
				printf ("%s %s %s ",GetField(line,1), system, sender); 
				if (DispFstats (system, line) == -1)
					{
					fclose (fp);
					return -4;
					}
				} /* while */
			} /* else - if (tmp == NULL) */	

		/* now increment number of jobs displayed */
		uflags->count++;
		} /* if uflags->... */

	fclose (fp);
	return 0;
	} /* Display */


/* 
*	DispFstats:  display uustats depending on type of file.
*	if file is uucp data file (starts with 'D.'), just display the name
*	if file is uucp execute file (starts with 'X.'), open the file and
*		display the command to execute (plus arguments)
*	if file is regular file (none of above), display the name (including
*		path)
*
*	In the first and third cases, the size of the file is also displayed 
*/
int DispFstats (system, line)
char * line;
char * system;
	{
	static char buff[LLEN+1];
	char * file;

	/* 
		Check the type of file and take a course of action.
		The file type is given by the third field of the line
		passed from the caller.  Once the file type is determined,
		the filename can be obtained by checking the sixth field.
		If the filename in the sixth field is "D.0" or "dummy" then
		the valid filename is in the second field
	*/
	file = GetField(line,3);		/* get the file type */
	switch (*file)
		{
		case 'D':
			/* data file */
			file = GetFilename (line);
			makeuupath (buff, system, file);
			printf ("%ld %s\n", GetUUfsize(buff), file);
			break;
			
		case 'X':
			/* execute file */
			file = GetFilename (line);
			makeuupath (buff, system, file);
			if (DispXcmd (buff) == -1)
				return -1;
			break;
			
		default:
			/* regular file */
			file = GetField (line,6);
			if ((!strncmp (file, "D.0", 3)) || (!strncmp (file, "dummy", 5)))
				{
				file = GetField (line, 2);
				printf ("%ld %s\n", GetUUfsize(file), file);
				}
			else
				{
				makeuupath (buff, system, file);
				printf ("%ld %s\n", GetUUfsize(buff), file);
				}
			break;
		} /* switch */
	return 0;
	} /* DispFstats */




/*
	Displays the command contained in a uucp execute command file.
	The command itself is on a line by itself that starts with a 'C'.
	If successful, 0 is return.  -1 is return on an error.
*/
int DispXcmd (file)
char * file;
	{
	static char buff[LLEN+1];
	FILE * fp;
	int ret;

	/* open file */
	fp = fopen (file, "r");
	if (fp == NULL)
		return -1;
			
	/* read a line until one is found beginning with a 'C' or EOF */
	ret = 0;
	while (!feof (fp))
		{
		if (fgets (buff, LLEN, fp) == NULL)
			{
			ret = -1;
			break;
			}
			
		if (*buff == 'C')
			{
			printf ("%s", buff+2);	/* don't need a newline, fgets reads it */
			break;
			}
		} /* while */
	
	fclose (fp);
	return ret;
	} /* DispXcmd */



char * GetFilename (line)
char * line;
	{
	char * file;
	
	/* get field 6 and see if it exists */
	file = GetField (line, 6);

	/* if field 6 is "D.0" or "dummy", use field two */
	if (!strncmp (file, "D.0", 3) || !strncmp (file, "dummy", 5))
		file = GetField (line, 2);

	return file;
	} /* GetFilename */



/*
	Get the size of the given file and return it
	If an error occurs (such as when the file does not exist -- a situation
	where the user created a job and then later deleted the required file),
	a -99999 is returned instead.
*/
long GetUUfsize (file)
char * file;
	{
	struct stat fs;
	

	if (stat (file, &fs) == -1)
		{ /* error during fstat */
		return -99999L;
		} /* if fstat */
		
	return fs.st_size;	
	} /* GetUUfsize */


/* #define TEST */
#ifdef TEST

main (argc, argv)
int argc;
char *argv[];
	{
	int ret;
	uuflags uflags;
	
	if (argc < 2)
		{
		printf ("usage: display file\n");
		exit (1);
		}
	printf ("displaying: %s\n", *(argv+1));	
	uflags.fdispl = 1;
	ret = Display ("test", &uflags, *(argv+1));
	printf ("returned: %d\n",ret);
	exit (ret);
	}


#endif
