#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <fcntl.h>
#include <memory.h>
#include <errno.h>
#include "socket.h"
#include "in.h"
#include "un.h"
#include "socketvar.h"

extern char *malloc(), *strdup();
extern void free();
struct socket *_socktab[NOFILE];
fd_set _socketmap;		/* File descriptor map */
static int seq = 0;		/* Sequence number for file creation */

/*
 * Sockets are emulated by named pipes. A socket is characterized
 * by a file in the "/usr/spool/socket" directory.
 * The filename is randomly generated to always be unique.
 * The actual connection consist of two other named pipe files
 * in the "/tmp" directory. Since there is only one channel
 * for the pipes, we need two pipes to simulate one socket.
 */
int
socket(domain, type, protocol)
int domain, type, protocol;
{
	register struct socket *so;
	int sofd, old_umask;
	char soname[30];

	if ((domain != AF_UNIX && domain != AF_INET)
	    || type != SOCK_STREAM || protocol != 0) {
		 errno = EPROTONOSUPPORT;
		 return NULL;
	}
	if ((so = (struct socket *)malloc(sizeof(struct socket))) == NULL)
		return NULL;
	so->so_options = 0;
	so->so_state = 0;
	so->so_type = type;
	so->so_rfd = so->so_wfd = so->so_pfd = -1;
	so->so_domain = domain;
	so->so_protocol = protocol;
	so->so_rname = so->so_wname = NULL;
	so->so_conn = NULL;

	/*
	 * create the main socket pipe
	 */
	sprintf(soname, "/tmp/SO.%d.%d", getpid(), ++seq);
	old_umask = umask(0);
	if (mknod(soname, S_IFIFO|0777, 0) == -1) {
		free((char *)so);
		umask(old_umask);
		return -1;
	}
	umask(old_umask);
	if ((sofd = open(soname, O_RDWR)) == -1) {
		unlink(soname); free((char *)so); 
		return -1;
	}
	fcntl(sofd, F_SETFD, O_RDONLY);
	so->so_name = strdup(soname);
	so->so_fd = sofd;
	FD_SET(sofd, &_socketmap);
	_socktab[sofd] = so;

	/*
	 * Indicate socket is active
	 */
	_setsolock(sofd, 0);
	return sofd;
}

/*
 * To be able to have a few status bits for the pipe files
 * we use file locks, at a high address.
 * "_setsolock" can be used to set a flag, and "_checksolock" to
 * examine the state of the flag.
 */
int
_setsolock(fd, type)
int fd, type;
{
#if CAN_LOCK	/* Coherent 3.2 can't lock files */
	struct flock lockblk;

	lockblk.l_type = F_WRLCK;
	lockblk.l_whence = 0;
	lockblk.l_start = type + 10000000L;
	lockblk.l_len = 1;
	return fcntl(fd, F_SETLK, &lockblk);
#endif
	return 0; /* hm, always ok */
}

int
_clearsolock(fd, type)
int fd, type;
{
#if CAN_LOCK	/* Coherent 3.2 can't lock files */
	struct flock lockblk;

	lockblk.l_type = F_UNLCK;
	lockblk.l_whence = 0;
	lockblk.l_start = type + 10000000L;
	lockblk.l_len = 1;
	return fcntl(fd, F_SETLK, &lockblk);
#endif
	return 0; /* hm, always ok */
}

/*
 * Check for a lock.
 * Returns:
 *	-1 if error
 *	0 if no lock
 *	1 if lock
 */
int
_checksolock(fd, type)
int fd;
int type;
{
#if CAN_LOCK	/* Coherent 3.2 can't lock files */
	struct flock lockblk;

	lockblk.l_type = F_WRLCK;
	lockblk.l_whence = 0;
	lockblk.l_start = type + 10000000L;
	lockblk.l_len = 1;
	if (fcntl(fd, F_GETLK, &lockblk) == -1)
		return -1;
	return lockblk.l_type != F_UNLCK;
#endif
	return 0; /* hm, always ok */
}

char *
_socketname(addr)
struct sockaddr *addr;
{
	static char path[108];

	switch (addr->sa_family) {
	case AF_UNIX:
		strcpy(path, ((struct sockaddr_un *)addr)->sun_path);
		break;

	case AF_INET:
#ifndef NO_SPOOL
		sprintf(path, "/usr/spool/socket/PORT.%d",
#else
		sprintf(path, "/tmp/PORT.%d",
#endif
			(unsigned)((struct sockaddr_in *)addr)->sin_port);
		break;
	default:
		return NULL;
	}
	return path;
}
