/* Client for mtalk - multiuser talk program.  This program is invoked
 * by a user and sends and receives messages from the mtalk daemon.
 *
 * Copyright (c) 1991, Matthew Kimmel.  Permission granted for unlimted
 * non-commercial use and distribution.
 */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>
#include <sgtty.h>
#ifndef _I386
#include <sys/fcntl.h>
#else
#include <fcntl.h>
#endif
#include <sys/msg.h>
#include "mtalk.h"

struct genmsg msgin;  /* Buffer for unprocessed messages */
int ouruid; /* uid of this user */
int ourmqid; /* our message queue id */
int dmqid; /* mtalk daemon's message queue id */
struct sgttyb t0, t1; /* tty structs for stdin and stdout */
int oldfcntl; /* preserved old fcntl value for stdin */

main()
{
  char buf[MAX_MESSAGE];
  int pos = 0; /* position in message buffer */
  char ch;
  int i;

  puts("Welcome to mtalk!  Attempting connection...");
  init();
  login();
  buf[0] = '\0';

  /* The program's main loop.  This could be accomplished more easily
     by forking another process, one to process user input and one
     to process messages from the server, but that would double the
     memory that this program uses. */
  for(;;) {
    /* First check for input from the terminal */
    if(read(0,&ch,1) > 0)
      switch(ch) {
        case 127 :
        case 8 : /* Erase */
                 if(pos > 0) {
                   pos--;
                   buf[pos] = '\0';
                   write(1,"\b \b",3);
                   }
                 break;
        case 3 : /* CTRL-C */
                 doexit();
        case 21 : /* Kill - CTRL-U */
                  pos = 0;
                  buf[0] = '\0';
                  write(1,"^U\r\n",4);
                  break;
        case 18 : /* CTRL-R - refresh line */
                  if(pos > 0) {
                    write(1,"^R\r\n",4);
                    write(1,buf,strlen(buf));
                    }
                  break;
        case 13 : write(1,"\r\n",2);
                  if(pos > 0) {
                    sendline(buf);
                    pos = 0;
                    buf[0] = '\0';
                    }
                  break;
        default : if(pos <= (MAX_MESSAGE - 2)) {
                    write(1,&ch,1);
                    buf[pos] = ch;
                    pos++;
                    buf[pos] = '\0';
                    }
                  break;
        }

    /* Now check for incoming messages */
    if((i = msgrcv(ourmqid,&msgin,1024,0L,IPC_NOWAIT)) == -1)
      if(errno == EDOM) {
        perror("mtalk: msgrcv");
        cleanup();
        exit(1);
        }
    if(i >= 0) {
      if(msgin.msgtype == M_DMNTEXT) {
        write(1,msgin.contents,strlen(msgin.contents));
        write(1,"\r\n",2);
        continue;
        }
      if(msgin.msgtype == M_DMNFULL) {
        puts("Sorry, mtalk is full now.  Try again later.\r");
        cleanup();
        exit(0);
        }
      }
    }
}

/* Initialize various things--message queues, signals, cbreak, etc. */
init()
{
  key_t ftok();
  int doexit();

  /* First try to connect with the daemon's message queue...using the secret formula! */
  if((dmqid = msgget(ftok(DAEMON_PATH,1),0)) == -1) {
    perror("mtalk: msgget");
    exit(1);
    }

  /* Now get our uid, and attempt to create our own message queue. */
  ouruid = getuid();
  if((ourmqid = msgget(ftok(CLIENT_PATH,ouruid),(0622 | IPC_CREAT))) == -1) {
    perror("mtalk: msgget");
    exit(1);
    }

  /* Set up to handle signals - we DON'T want to exit without notifying
     the server, if at all possible. */
  signal(SIGHUP,doexit);
  signal(SIGQUIT,doexit);
  signal(SIGTERM,doexit);
#ifndef _I386
  signal(SIGREST,doexit);
#endif

  /* Now set up stdin */
  oldfcntl = fcntl(0,F_GETFL,0);
  fcntl(0,F_SETFL,(oldfcntl | O_NDELAY));
  gtty(0,&t0);
  t0.sg_flags |= RAW;
  t0.sg_flags &= ~ECHO;
  stty(0,&t0);

  /* Set up stdout */
  gtty(1,&t1);
  t1.sg_flags |= RAW;
  stty(1,&t1);
}

/* Send a login attempt to the daemon */
login()
{
  struct {
    long mtype;
    struct usrin info;
    } mbuf;

  mbuf.mtype = M_USRIN;
  mbuf.info.uid = ouruid;
  msgsnd(dmqid,&mbuf,sizeof(struct usrin),0);
}

/* This function takes a line of text that the user has entered, and
   determines whether it is a public message or a command.  If it is
   a public message, it sends it; otherwise, it tries to process the
   command. */
sendline(txt)
char *txt;
{
  struct {
    long mtype;
    struct usrmsg info;
    } mbuf;

  if(txt[0] != '/') { /* Not a command */
    mbuf.mtype = M_USRMSG;
    mbuf.info.uid = ouruid;
    strcpy(mbuf.info.text,txt);
    msgsnd(dmqid,&mbuf,sizeof(struct usrmsg),0);
    return;
    }

  /* Determine which command it is by letter following slash */
  switch(tolower(txt[1])) {
    case 'w' : /* Whisper */
               dowhisper(txt);
               return;
    case 'u' : /* Who (Users) */
               dowho();
               return;
    case 'e' : /* Exit */
               doexit();
    case '?' :
    case 'h' : /* Help */
               puts("Commands available\r");
               puts("  /users - show current mtalk users\r");
               puts("  /whisper name message - send message privately to user name\r");
               puts("  /exit - exit the program\r");
               puts("  /help - see this help screen\r");
               return;
    default : puts("Unknown command.\r");
              return;
    }
}

/* Process the whisper command */
dowhisper(txt)
char *txt;
{
  struct {
    long mtype;
    struct usrcmd info;
    } mbuf;
  char dummy[20];
  char *p;

  mbuf.mtype = M_USRCMD;
  mbuf.info.uid = ouruid;
  mbuf.info.cmd = U_WHISPER;

  /* Make sure there is a first space and hence a name in the line */
  if((p = (char *)index(txt,' ')) == NULL) {
    puts("usage: /whisper name text\r");
    return;
    }

  /* Attempt to extract the name.  The line should be of the form
     /whisper name lots of text */
  sscanf(txt,"%s %s",dummy,mbuf.info.user);

  /* Now obtain a pointer to the text of the message */
  /* We want to find the second space in the string. */
  p++;
  if((p = (char *)index(p,' ')) == NULL) {
    puts("usage: /whisper name text\r");
    return;
    }
  p++;

  /* p now points to the message text.  Copy it into our structure and send it. */
  strcpy(mbuf.info.text,p);
  msgsnd(dmqid,&mbuf,sizeof(struct usrcmd),0);
}

/* Process a who command--send a request for a list of users. */
dowho()
{
  struct {
    long mtype;
    struct usrcmd info;
    } mbuf;

  mbuf.mtype = M_USRCMD;
  mbuf.info.uid = ouruid;
  mbuf.info.cmd = U_WHO;
  msgsnd(dmqid,&mbuf,sizeof(struct usrcmd),0);
}

/* Process an exit command--send an exit message, and exit the program.
   Also called by some signals. */
doexit()
{
  struct {
    long mtype;
    struct usrcmd info;
    } mbuf;

  mbuf.mtype = M_USRCMD;
  mbuf.info.uid = ouruid;
  mbuf.info.cmd = U_EXIT;
  msgsnd(dmqid,&mbuf,sizeof(struct usrcmd),0);

  write(1,"\r\n",2);
  cleanup();

  exit(0);
}

/* un-initialize various things--called before exiting. */
cleanup()
{
  /* Kill our message queue */
  msgctl(ourmqid,IPC_RMID,(struct msqid_ds *)0);

  /* Set terminal back to normal */
  fcntl(0,F_SETFL,oldfcntl);
  gtty(0,&t0);
  t0.sg_flags &= ~RAW;
  t0.sg_flags |= ECHO;
  stty(0,&t0);
  gtty(1,&t1);
  t1.sg_flags &= ~RAW;
  stty(1,&t1);
}
 
