/*
 * GNU m4 -- A simple macro processor
 * Copyright (C) 1989, 1990, 1991 Free Software Foundation, Inc. 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "m4.h"

/* 
 * Output functions.  Most of the complexity is for handling cpp like
 * sync lines.
 *
 * This code is fairly entangled with the code in input.c, and maybe it
 * belongs there?
 */

/* number of output lines for current input line */
int output_lines;

/* number of input line we are generating output for */
int output_current_line;


/* current output stream */
static FILE *output;

/* table of diversion files */
static FILE **divtab;


/* 
 * Output initialisation.  It handles allocation of memory for
 * diversions.  This is done dynamically, to allow customisation of the
 * number of available divertions.
 */
void 
output_init()
{
    int i;

    output = stdout;

    divtab = (FILE **)xmalloc(ndivertion * sizeof(FILE *));
    for (i = 0; i < ndivertion; i++)
	divtab[i] = nil;
    divtab[0] = stdout;
}


/* 
 * Output a sync line for line number LINE, with an optional file name
 * FILE specified.
 */

void 
sync_line(line, file)
    int line;
    char *file;
{
    if (output == nil)
	return;

    fprintf(output, "#line %d", line);
    if (file != nil)
	fprintf(output, " \"%s\"", file);
    putc('\n', output);
}

/* 
 * Output TEXT to either an obstack or a file.  If OBS is nil, and there
 * is no output file, the text is discarded.
 *
 * If we are generating sync lines, the output have to be examined,
 * because we need to know how much output each input line generates.
 * In general, sync lines are output whenever a single input lines
 * generates several output lines, or when several input lines does not
 * generate any output.
 */
void 
shipout_text(obs, text)
    struct obstack *obs;
    char *text;
{
    static boolean start_of_output_line = true;

    if (obs != nil) {			/* output to obstack OBS */
	obstack_grow(obs, text, strlen(text));
	return;
    }
    if (output == nil)			/* discard TEXT */
	return;

    if (!sync_output)
	fputs(text, output);
    else {
	for (; *text != '\0'; text++) {
	    if (start_of_output_line) {
		start_of_output_line = false;

#ifdef DEBUG_OUTPUT
		printf("DEBUG: cur %d, cur out %d, out %d\n",
		       current_line, output_current_line, output_lines);
#endif

		if (current_line - output_current_line > 1 || output_lines > 1)
		    sync_line(current_line, nil);
		
		output_current_line = current_line;
	    }
	    putc(*text, output);
	    if (*text == '\n') {
		output_lines++;
		start_of_output_line = true;
	    }
	}
    }
}

/* 
 * Functions for use by diversions.
 */
#ifndef HAVE_TMPFILE
/* 
 * Implement tmpfile(3) for non-USG systems.
 */
int mkstemp();
extern int unlink();

FILE *
tmpfile()
{
    char buf[32];
    int fd;

    strcpy(buf, "/tmp/m4XXXXXX");
    fd = mkstemp(buf);
    if (fd < 0)
	return nil;

    unlink(buf);
    return fdopen(fd, "w+");
}

#ifndef HAVE_MKSTEMP
/* 
 * This implementation of mkstemp(3) does not avoid any races, but its
 * there.
 */
#include <sys/types.h>
#include <fcntl.h>

int 
mkstemp(tmpl)
    char *tmpl;
{
    mktemp(tmpl);
    return open(tmpl, O_RDWR|O_TRUNC|O_CREAT, 0600);
}
#endif /* not HAVE_MKSTEMP */
#endif /* not HAVE_TMPFILE */

/* 
 * Make a file for diversion DIVNUM, and install it in the diversion
 * table "divtab".  The file is opened read-write, so we can unlink it
 * immediately.
 */
void 
make_divertion(divnum)
    int divnum;
{
    if (output != nil)
	fflush(output);

    if (divnum < 0 || divnum >= ndivertion) {
	output = nil;
	return;
    }

    if (divtab[divnum] == nil) {
	divtab[divnum] = tmpfile();
	if (divtab[divnum] == nil)
	    fatal("can't create file for diversion: %s", syserr());
    }
    output = divtab[divnum];

    if (sync_output)
	sync_line(current_line, current_file);
}

/* 
 *
 * Insert a FILE into the current output file, in tha same manner
 * divertions are handled.  This allows files to be included, without
 * having them rescanned by m4.
 */

void
insert_file(file)
    FILE *file;
{
    int ch;

    while ((ch = getc(file)) != EOF)
	putc(ch, output);
}

/* 
 * Insert diversion number DIVNUM into the current output file.  The
 * diversion is NOT placed on the expansion obstack, because it must not
 * be rescanned.  When the file is closed, it is deleted by the system.
 */
void 
insert_divertion(divnum)
    int divnum;
{
    FILE *div;

    if (divnum < 0 || divnum >= ndivertion)
	return;

    div  = divtab[divnum];
    if (div == nil || div == output)
	return;

    if (output != nil) {
	rewind(div);
	insert_file(div);
	
	if (sync_output)		/* BUG HERE -- undivert in the middle of line */
	    sync_line(current_line, current_file);
    }
    fclose(div);
    divtab[divnum] = nil;
}

/* 
 * Get back all diversions.  This is done just before exiting from
 * main(), and from m4_undivert(), if called without arguments.
 */
void 
undivert_all()
{
    int divnum;

    for (divnum = 1; divnum < ndivertion; divnum++)
	insert_divertion(divnum);
}
