/*
 * GNU m4 -- A simple macro processor
 * Copyright (C) 1989, 1990, 1991 Free Software Foundation, Inc. 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "m4.h"
#include "version.h"
#include "getopt.h"
#include <sys/types.h>
#ifndef COHERENT
#include <sys/signal.h>
#else
#include <signal.h>
#endif

void cmd_error();


/* Operate interactively (-e) */
int interactive = 0;

/* Enable sync output for /lib/cpp (-s) */
int sync_output = 0;

/* Debug (-d[flags]) */
int debug_level = 0;

/* Hash table size (should be a prime) (-Hsize) */
int hash_table_size = HASHMAX;

/* Number of diversions allocated */
int ndivertion = NDIVERTIONS;

/* Disable GNU extensions (-G) */
int no_gnu_extensions = 0;

/* Max length of arguments in trace output (-lsize) */
int max_debug_argument_length = 0;

/* Suppress warnings about missing arguments. */
int suppress_warnings = 0;

/* Program name */
char *progname = nil;

static struct option m4_options[] = {
    { "interactive",		GETOPT_NO_ARG,	nil, 'e' },
    { "version",		GETOPT_NO_ARG,	nil, 'V' },
    { "synclines",		GETOPT_NO_ARG,	nil, 's' },
    { "debug",			GETOPT_OPT_ARG,	nil, 'd' },
    { "arglength",		GETOPT_REQ_ARG,	nil, 'l' },
    { "erroroutput",		GETOPT_REQ_ARG,	nil, 'o' },
    { "hashsize",		GETOPT_REQ_ARG,	nil, 'H' },
    { "divertions",		GETOPT_REQ_ARG,	nil, 'N' },
    { "include",		GETOPT_REQ_ARG,	nil, 'I' },
    { "no_gnu_extensions", 	GETOPT_NO_ARG,	nil, 'G' },
    { "no_warnings",	 	GETOPT_NO_ARG,	nil, 'Q' },
    { "quiet",		 	GETOPT_NO_ARG,	nil, 'Q' },

    /* These are somewhat troublesome */
    { "define",			GETOPT_REQ_ARG,	nil, 'D' },
    { "undefine",		GETOPT_REQ_ARG,	nil, 'U' },
    { "trace",			GETOPT_REQ_ARG,	nil, 't' },
    { nil },
};

#define OPTSTRING "esGQVD:U:t:B:H:o:l:d::S:T:I:N:"

struct macro_definition {
    struct macro_definition *next;
    int code;				/* D, U or t */
    char *macro;
};
typedef struct macro_definition macro_definition;


/* 
 * usage --- Print usage message on stderr.
 */
void 
usage()
{
    fprintf(stderr, "Usage: m4 [options] file ....\n");
    exit(1);
}


/* 
 * Split a string of arguments in ENV, taken from the environment.
 * Return generated argv, and the number through the pointer ARGCP.
 */
char **
split_env_args(env, argcp)
    char *env;
    int *argcp;
{
    char **argv;
    int argc;

    if (env == nil) {
	*argcp = 0;
	return nil;
    }

    /* this should be enough */
    argv = (char **)xmalloc((strlen(env)/2 + 3) * sizeof(char *));

    argc = 1;
    for (;;) {
	while (*env && (*env == ' ' || *env == '\t' || *env == '\n'))
	    env++;

	if (*env == '\0')
	    break;

	argv[argc++] = env;

	while (*env && !(*env == ' ' || *env == '\t' || *env == '\n'))
	    env++;

	if (*env == '\0')
	    break;

	*env++ = '\0';
    }
    argv[argc] = nil;

    *argcp = argc;
    return argv;
}

/* 
 * Parse the arguments ARGV.  Arguments that cannot be handled until
 * later (i.e., -D, -U and -t) are accumulated on a list, which is
 * returned through DEFINES.  The function returns the rest of the
 * arguments.
 */
char **
parse_args(cmdargc, cmdargv, defines)
    int cmdargc;
    char **cmdargv;
    macro_definition **defines;
{
    int option;				/* option character */
    int dummy;				/* dummy option_index, unused */
    boolean print_version;		/* print version before return */
    char **argv;			/* argv from environment M4OPTS */
    int argc;				/* argc fron environment M4OPTS */
    macro_definition *head;		/* head of deferred argument list */
    macro_definition *tail;
    macro_definition *new;

    head = tail = nil;
    print_version = false;

    argv = split_env_args(getenv("M4OPTS"), &argc);
    if (argv == nil) {
	argv = cmdargv;
	argc = cmdargc;
	cmdargv = nil;
    }

    while (true) {
	option = getopt_long(argc, argv, OPTSTRING, m4_options, &dummy);

	if (option == EOF) {
	    if (cmdargv != nil) {
		if (optind != argc)
		    cmd_error("warning: excess file arguments in M4OPTS ignored");

		argv = cmdargv;
		argc = cmdargc;
		cmdargv = nil;
		optind = 0;
		continue;
	    } else
		break;
	}

	switch (option) {
	case 'e':
	    interactive = 1;
	    break;

	case 's':
	    sync_output = 1;
	    break;

	case 'G':
	    no_gnu_extensions = 1;
	    break;

	case 'Q':
	    suppress_warnings = 1;
	    break;

	case 'V':
	    print_version = true;
	    break;

	case 'd':
	    debug_level = debug_decode(optarg);
	    if (debug_level < 0) {
		cmd_error("bad debug flags: `%s'", optarg);
		debug_level = 0;
	    }
	    break;

	case 'l':
	    max_debug_argument_length = atoi(optarg);
	    if (max_debug_argument_length <= 0)
		max_debug_argument_length = 0;
	    break;

	case 'o':
	    if (!debug_set_output(optarg))
		cmd_error("cannot set error file %s: %s", optarg, syserr());
	    break;

	case 'H':
	    hash_table_size = atoi(optarg);
	    if (hash_table_size <= 0)
		hash_table_size = HASHMAX;
	    break;

	case 'N':
	    ndivertion = atoi(optarg);
	    if (ndivertion <= 0)
		ndivertion = NDIVERTIONS;
	    break;

	case 'I':
	    add_include_directory(optarg);
	    break;

	case 'B':			/* compatibility junk */
	case 'S':
	case 'T':
	    break;

	case 'D':
	case 'U':
	case 't':
	    new = (struct macro_definition *)xmalloc(sizeof(struct macro_definition));
	    new->code = option;
	    new->macro = optarg;
	    new->next = nil;

	    if (head == nil)
		head = new;
	    else
		tail->next = new;
	    tail = new;

	    break;

	default:
	    usage();
	}
    }

    if (print_version)
	fprintf(stderr, "\
GNU m4 %s, Copyright (C) 1989, 1990, 1991 Free Software Foundation, Inc.\n\
There is ABSOLUTELY NO WARRANTY for GNU m4.  See the file\n\
COPYING in the source distribution for more details.\n",
		version);

    *defines = head;
    return argv + optind;
}


int 
main(argc, argv)
    int argc;
    char **argv;
{
    macro_definition *defines;
    FILE *fp;

    progname = rindex(argv[0], '/');
    if (progname == nil)
	progname = argv[0];
    else
	progname++;

    include_init();
    debug_init();

    /* 
     * First, we decode the arguments, to size up tables and stuff.
     */
    argv = parse_args(argc, argv, &defines);

    /* 
     * Do the basic initialisations.
     */
    input_init();
    output_init();
    symtab_init();
    builtin_init();
    include_env_init();
    
    /* 
     * Handle deferred command line macro definitions.  Must come after
     * initialisation of the symbol table.
     */
    while (defines != nil) {
	macro_definition *next;
	char *macro_value;
	symbol *sym;

	switch (defines->code) {
	case 'D':
	    macro_value = index(defines->macro, '=');
	    if (macro_value == nil)
		macro_value = "";
	    else
		*macro_value++ = '\0';
	    define_user_macro(defines->macro, macro_value, SYMBOL_INSERT);
	    break;

	case 'U':
	    lookup_symbol(defines->macro, SYMBOL_DELETE);
	    break;

	case 't':
	    sym = lookup_symbol(defines->macro, SYMBOL_INSERT);
	    SYMBOL_TRACED(sym) = true;
	    break;

	default:
	    internal_error("bad code in deferred arguments.");
	}

	next = defines->next;
	xfree(defines);
	defines = next;
    }

    /* 
     * Interactive mode means unbuffered output, and interrupts ignored.
     */
    if (interactive) {
	signal(SIGINT, SIG_IGN);
	setbuf(stdout, (char *)NULL);
    }

    /* 
     * Handle the various input files.  Each file is pushed on the
     * input, and the input read.  Wrapup text is handled separately
     * later.
     */
    if (*argv == nil) {
	push_file(stdin, "stdin");
	expand_input();    
    } else {
	for ( ; *argv != nil; argv++) {
	    if (strcmp(*argv, "-") == 0) {
		push_file(stdin, "stdin");
	    } else {
		fp = path_search(*argv);
		if (fp == nil) {
		    cmd_error("can't open %s: %s", *argv, syserr());
		    continue;
		} else
		    push_file(fp, *argv);
	    }
	    expand_input();
	}
    }
#undef NEXTARG
    
    /* Now handle wrapup text. */
    while (pop_wrapup())
	expand_input();

    undivert_all();

    return 0;
}


/* 
 * The rest of this file contains error handling functions, and memory
 * allocation.
 */

/* Non m4 specific error -- just complain */
/* VARARGS */
void 
cmd_error(va_alist)
    va_dcl
{
    va_list args;
    char *fmt;

    fprintf(stderr, "%s: ", progname);

    va_start(args);
    fmt = va_arg(args, char*);
    vfprintf(stderr, fmt, args)
    va_end(args);

    putc('\n', stderr);
}

/* Basic varargs function for all error output */
void 
vmesg(level, args)
    char *level;
    va_list args;
{
    char *fmt;

    fflush(stdout);
    fmt = va_arg(args, char*);
    fprintf(stderr, "%s: %s: %d: ", progname, current_file, current_line);
    if (level != nil)
	fprintf(stderr, "%s: ", level);
    vfprintf(stderr, fmt, args);
    putc('\n', stderr);
}

/* Internal errors -- print and dump core */
/* VARARGS */
volatile void 
internal_error(va_alist)
    va_dcl
{
    va_list args;
    va_start(args);
    vmesg("internal error", args);
    va_end(args);

    abort();
}

/* Fatal error -- print and exit */
/* VARARGS */
volatile void 
fatal(va_alist)
    va_dcl
{
    va_list args;
    va_start(args);
    vmesg("fatal error", args);
    va_end(args);

    exit(1);
}

/* "Normal" error -- just complain */
/* VARARGS */
void 
error(va_alist)
    va_dcl
{
    va_list args;
    va_start(args);
    vmesg((char *)nil, args);
    va_end(args);
}

/* Warning --- for potential trouble */
/* VARARGS */
void 
warning(va_alist)
    va_dcl
{
    va_list args;
    va_start(args);
    vmesg("warning", args);
    va_end(args);
}


/* 
 * Memory allocation functions
 */

/* Out of memory error -- die */
void 
no_memory()
{
    fatal("Out of memory");
}

/* Free previously allocated memory */
void 
xfree(p)
    char *p;
{
    if (p != nil)
	free(p);
}

/* Semi-safe malloc -- never returns nil */
char *
xmalloc(size)
    unsigned int size;
{
    register char *cp = malloc(size);
    if (cp == nil)
	no_memory();
    return cp;
}

#if 0
/* Ditto realloc */
char *
xrealloc(p, size)
    char *p;
    unsigned int size;
{
    register char *cp = realloc(p, size);
    if (cp == nil)
	no_memory();
    return cp;
}
#endif

/* and strdup */
char *
xstrdup(s)
    char *s;
{
    return strcpy(xmalloc((unsigned int)strlen(s)+1), s);
}
