/*
 * This program is supposed to run as suid to daemon to get access to
 * the spool files.
 * It has several functions, depending on name of invocation:
 *
 * lpjob:  reports spooled jobs 
 * lpcmd:  reports commandlines
 *
 * These commands reports the contents of the lpr/hpr control file(s)
 * entered as commandline arguments, assuming they have valid format.
 * If any error is encountered opening the file(s),
 * they return ERRDATA and/or ERRNOFILE.
 *
 * lpcan:  cancel job(s).
 *
 * This command finds the callers UID, and removes all files associated
 * with the lpr/hpr control file(s) entered as commandline arguments,
 * after verifying owner ID if caller is NOT root.
 * If any error is encountered opening the file(s), or they don't contain
 * valid data, the corresponding file is left untouched, and the
 * program returns ERRDATA and/or ERROWNER and/or ERRNOFILE.
 *
 * lpid:   reports pid found in lock file.
 * rmlpid: report pid and remove lock file.
 *
 * These commands reports the contents of the lpr/hpr lock file
 * entered as commandline argument, or `dpid' as a default
 * If compiled with UNLINKNAME defined to a filename, and invoked
 * with this name, the file is removed regardless of caller UID.
 * (but only the name LOCKNAME, to prevent misuse)
 *
 * On success, the process id found in the file is printed on stdout,
 * and a value of 0 is returned. if file cannot be opened, - or read -
 * ERRNOFILE or ERRNOREAD is returned.
*/

#include <stdio.h>
#include <string.h>
#include <pwd.h>

#define ERRNOFILE 1
#define ERRNOREAD 2
#define ERRPWD	  3
#define ERROWNER  4
#define ERRDATA   8
#define ERRNAME  16

#define UNLINKNAME  "rmlpid"  /* define to NULL to disable */
#define LOCKNAME    "dpid"

#define JOBS 0		/* implemented functions */
#define CMDS 1
#define CANCEL 2
#define REPORT 3
#define UNLINK 4

static int printjob();
static int killjob();
static int lockid();

static char caller[256];
static char owner[20];
static char root[] = "root";
static char *dflt = LOCKNAME;

main(argc, argv)
int argc;
char **argv;
{
   static char *cmdnames[] = 
          {"lpjob", "lpcmd", "lpcan", "lpid", UNLINKNAME, NULL};
   int errors = 0, function = 0, status = 0, myuid;
   char **command, *myname;

   if ((myname = strrchr(*argv, '/')) == NULL)
      myname = *argv;
   else
      myname++;
   for (command = cmdnames; *command; ++function, ++command)
      if (strcmp(*command,myname) == 0)
         break;

   switch (function)  {
   case JOBS:
   case CMDS:
            for(++argv; --argc; argv++)
               if (printjob(*argv, function) && ++errors && (status|=ERRDATA))
                   printf("** Error reading file:\t%s\n",*argv);
            break;
   case CANCEL:
            if (!(myuid = getuid()))
               strcpy(caller,root);
            else if ( getpw(myuid,caller) == 0)
               *strchr(caller,':') = '\0';
            else {
               printf("Error reading passwd file");
               return ERRPWD;
            }

            for(++argv; --argc; argv++)
               switch (killjob(*argv))   {
                  case 0: break;
               case 1: ++errors;
                  printf("** Error opening file:\t%s\n", *argv);
                  status |= ERRDATA; break;
               case 2: ++errors;
                  printf("** Not the owner (%s); file:\t%s\n", owner, *argv);
                  *owner = '\0';
                  status |= ERROWNER; break;
               }
            break;
   case REPORT:
   case UNLINK:
            if (--argc) ++argv;
            else *argv = dflt;
            return(lockid(*argv, function));
   default: printf("Unimplemented name: %s\n", *argv);
            return ERRNAME;
   }
   if (errors)
      printf("\nthere was %d errors detected\n",errors);
   return status;
}

static int printjob(file, action)
char *file;
int action;
{
   char buf[512];
   char cmdline[512];
   FILE *job = fopen(file, "r");
   char *jobno = strrchr(file,'/');
   int no_banner, files = 0;

   printf("%6d  ",(jobno) ? atoi(jobno+3) : atoi(file+2));
   if (job == (FILE *) NULL || !fgets(cmdline,512,job) || *cmdline != 'D')
      return ERRNOFILE;
   fgets(buf,512,job);
   if (no_banner = (*buf == 'B')) 
      fgets(buf,512,job);
   fgets(buf,512,job);          /* strip 2 lines */
   fgets(buf,512,job);
   *strchr(buf,'\n') = '\0';
   printf("%s  ", buf+5);       /* print date without weekday */
   fgets(buf,512,job);
   *strchr(buf,'\n') = '\0';
   printf("%s\t", buf+1);       /* print user name */
   if (action == CMDS)  {
      fclose(job);
      printf(cmdline+1);        /* print only commandline */
      return 0;
   }
   while (fgets(buf,512,job))   /* it seems, the only reliable filename */
      if (*buf == 'A')     {    /* is the absolute path */
	 if (++files > 1)
            printf("\t\t\t\t");
         *strchr(buf,'\n') = '\0';
         printf("%s\n", buf+1);
      }
   fclose(job);
   return 0;
}

static int killjob(file)
char *file;
{
   char buf[512];
   char cmdline[512];
   FILE *job = fopen(file, "r");
   int no_banner;

   if (job == (FILE *) NULL || !fgets(cmdline,512,job) || *cmdline != 'D')  {
      fclose(job);
      return ERRNOFILE;
   }
   fgets(buf,512,job);
   if (no_banner = (*buf == 'B')) 
      fgets(buf,512,job);
   fgets(buf,512,job);          /* strip 3 lines */
   fgets(buf,512,job);
   fgets(buf,256,job);
   *strchr(buf,'\n') = '\0';
   strcpy(owner,buf+1);
   if (strcmp(caller,root) && strcmp(caller, owner) != 0)   {
      fclose(job);
      return ERROWNER;                 /* invalid user name */
   }
   while (fgets(buf,512,job))
      if (*buf == 'U')     {
         *strchr(buf,'\n') = '\0';
         unlink(buf+1);
      }
   fclose(job);
   unlink(file);
   return 0;
}

static int lockid(name, action)
char *name;
int action;
{
   int  lock;
   int  nread, lckpid;
   
   if (*name == '\0')
      name = dflt;
   lock = open(name, 0);
   if (lock == -1)			/* no file */
      return ERRNOFILE;
   if ((nread = read(lock, &lckpid, sizeof(int))) ==  2)
      printf("%d\n",lckpid);
   close(lock);
   if (action == UNLINK && (strstr(name,dflt) != NULL))
      unlink(name);
   return ((nread == 2 && (lckpid >1)) ? 0 : ERRNOREAD);  /* check validity */
}

