/* ta=4     tabs are set to 4.  vi: "set ts=4" */
/****************************************************************************
*                   l o g i n . c                                          	*
*																			*
*	login replacement for AT&T SysV/386 3.2 and Interactive Unix.			*
*																			*
* tony field	tony@ajfcal.cuc.ab.ca										*
****************************************************************************/
/*
 * $Id: login.c,v 2.1 1991/11/14 01:48:07 ajf Exp ajf $
 * $Log: login.c,v $
 * Revision 2.1  1991/11/14  01:48:07  ajf
 *    1. fix problem with 0 or blank expiry in /etc/shadow
 *
 * Revision 2.0  1991/11/10  23:01:36  ajf
 *    1. added slip, /etc/nologin, general cleanup and bugfix.
 *    2. added /etc/netlogin support (for use with rlogind).
 *    3. bug fix plus "!" from Uwe Doering (gemini@geminix.in-berlin.de)
 *
 * Revision 1.2  1991/04/04  05:06:17  ajf
 * added to rcs
 *
*/
/*	If compiled for Interactive unix with TCP/IP, /etc/netlogin
	should be a link to /bin/login.

	Wollongong (on AT&T) rlogind uses an incompatible format for the utmp
	entries: do NOT use this login for /usr/etc/netlogin!  Wollongong
	puts "ttypxx.hostname" into utmp.

Calling Parameters:
	called by getty:    login <username>
	called by telnet:   login -p		 	(-p = preserve environment)
	called by rlogind:  login -r <systemid>

	If called by rlogind, read from the socket to get the
	remote user, local user and terminal type/speed as null
	terminated strings:

			ajf ajf AT386-M/9600

Makefile manifests:
	-DTCPIP			if compilation for use with tcp/ip
	-DWOLLONGONG	if for AT&T SysV/386 + Wollongong TCP/IP
	-DUSRACCESS		if /etc/usrtime file is to be used
	-DNOISY			if login user id errors are reported to user
	-DNEEDUTIME		if /usr/include/utime.h is missing
    -DSLIP="xx"     name of slip login binary (e.g. "sllogin")

For "large" systems with many users, /etc/usrtime should be a dbm
file to speed user lookups.
*/

#include <stdio.h>
#include <sys/types.h>
#ifdef NEEDUTIME
struct utimbuf			/*	missing from /usr/include	*/
{	time_t	actime;		/*	access time					*/
	time_t	modtime;	/*	modification time			*/
} ;
#else
#include <utime.h>
#include <unistd.h>
#endif
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <ctype.h>

#include <termio.h>

#ifndef COHERENT
#include <sys/ioctl.h>
#endif

#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <utmp.h>
#include <pwd.h>

#ifdef __GNUC__
#define alloca __builtin_alloca
#else /* not __GNUC__ */
char *alloca ();
#endif

#define	TIMEOUT			120					/* 	seconds before login aborted*/
#define MAXBAD			5					/*	max allowed login attempts 	*/
#define BADLOGINTIME	200					/*	wait time before allowing new logins */
#define	MAXUSERNAME		25					/*	size of user name			*/

#define DEFAULTSHELL		"/bin/sh"
#define USERPATH		"/bin:/usr/bin"
#define ROOTPATH		"/bin:/etc:/usr/bin"
#define MAIL			"/usr/spool/mail"
#define DEFAULTTZ		"EST5EDT"

#define BINPASSWD		"/bin/passwd"
#define DIALUPTTY		"/etc/dialups"
#define DIALUPPWD		"/etc/d_passwd"
#define SHADOW			"/etc/shadow"
#define DEFAULTFILE		"/etc/default/login"
#define TTYTYPE			"/etc/ttytype"
#define	WTMP			"/usr/adm/wtmp"
#define	USRTIME			"/etc/usrtime"
#define	NOLOGIN			"/etc/nologin"
#define	TRUSTED			"/etc/trustme"
#define LOGINLOG		"/usr/adm/loginlog"
#define LASTLOGIN		".lastlogin"

#ifndef SLIP
#define	SLIP			"/usr/ucb/sllogin"
#endif

#define	ROOT			0

/*	user is forbidden to modify these with login command line options */
char *forbidden[] = {"HOME=", "PATH=",   "SHELL=",  "MAIL=",  "IFS=",
					 "HZ=",   "CDPATH=", "LOGNAME=", NULL };

#define MAXPATHLEN		300
#define LONGLINE		500
#define SHORTLINE		100

#define UNLOCK		0
#define LOCK		1

#ifdef USRACCESS
struct usrtime			/* fields within /etc/usrtime	*/
{	char	*enable;
	char	*ttyline;
	char	*days;
	char	*timerange;
} ;
#endif

long	shadow_lstchg, shadow_min, shadow_max;

#ifdef __STDC__
/* login.c */
int main(int argc, char **argv);
int dialup_pwd(char *ttyfname, char *pwdfname, char *tty_name, struct passwd *pwd);
void setenv(char *this, char *val);
void check_putenv(char *val);
char *get_shadow(char *fname, char *user);
char *search_ttytype(char *tty);
int get_defaults(char *fname, char *tz, char *hz, char **console, long *ulim, int *passreq, int *altshell, char **path, char **supath, long *timeout, int *cmask, int *idleweeks);
void trim(char *s);
int atoo(char *s);
void user_timeout(void);
int tscan(char s[], char t[]);
int qtscan(char s[], char t[]);
void get_remote(char *buf, int cnt);
int get_remote_user(char *host);
void set_tcp_ioctl();
void slow_exit (int rc);
void etc_nologin(char *c);
#ifdef USRACCESS
int get_usrtime(char *fname, char *user, char *shell, char *tty);
int check_user(char *note, long now, char *tty, struct usrtime *usr);
int check_tty(char *note, char *tty, char *list);
int check_day(char *note, long now, char *list);
int check_time(char *note, long now, char *list, int timerange);
#endif	/* USERACCESS */
#else
int main();
int dialup_pwd();
void setenv();
void check_putenv();
char *get_shadow();
char *search_ttytype();
int get_defaults();
void trim();
int atoo();
void user_timeout();
int tscan();
int qtscan();
void get_remote();
int get_remote_user();
void set_tcp_ioctl();
void slow_exit ();
void etc_nologin();
#ifdef USRACCESS
int get_usrtime();
int check_user();
int check_tty();
int check_day();
int check_time();
#endif	/* USRACCESS */
#endif	/* __STDC__ */


extern char *getpass(), *crypt(), *malloc();
extern int errno;
extern char *sys_errlist[];
long	timeout;				/*	TIMEOUT from /etc/default/login		*/
long	atol();
int		atoi();

char	rname[100], lname[100], tname[100];
struct passwd *pwd;
char	*tty_name;
char	*hostname;
int		last_match;				/*	qtscan last character matched		*/

main (argc, argv)
int argc;
char **argv;
{
	char *login_name;							/* user login id  		*/
	char user_name[SHORTLINE+1];
	char *tty_n, *ttyname();					/* tty name for user	*/
	int	 i;
	struct passwd *getnam();
	char	shell[MAXPATHLEN + 1];
	char	term[SHORTLINE + 1];
	char	buffer[MAXPATHLEN+1], *p;
	char	*shadow, *my_pass;
	char	*ctime();
	long	ulimit_blocks;					/*	ULIMIT from /etc/default/login */
	char	tz[50];							/*	TZ			"			*/
	char	hz[50];							/*	HZ			"			*/
	char	*console;						/*	CONSOLE		"			*/
	int		passreq;						/*	PASSREQ		"			*/
	int		altshell;						/*	ALTSHELL	"			*/
	char	*path;							/*	PATH		"			*/
	char	*supath;						/*	SUPATH		"			*/
	int		cmask;							/*	UMASK		" 			*/
	int		idleweeks;						/*	IDLEWEEKS	"			*/
	int		passwd_needed = 0;
	int		null_user_count;
	int		lcreate;
	char	bad_user[MAXBAD][MAXUSERNAME+1];
	extern char	**environ, *optarg;
	extern int  optind, getopt();
	int		c;
	char	*domain, *environ_init[1];
	int		preserve = 0;


	/* 	To be really safe, two binaries for login should exist.
		For /bin/login, the user must be at the lowest shell for
		any login attempt.  For /etc/netlogin, the uid must be ROOT
		and the ppid must be != 1.  Unfortunately, we cannot check
		argv[0] since getty, rlogind and telnetd force "login" to
		be the argv[0] name.  The following test could be #ifdef'ed
		out for TCPIP.  For /bin/login, the test should be
		for getppid() != 1 only.
	*/
	if (getuid()  &&  getppid () != 1)		/* bypassed by rlogind */
	{	fprintf (stdout, "%s must be exec'ed from lowest level shell.\n", argv[0]);
		fflush (stdout);
		exit (1);
	}

#ifdef TCPIP
	gethostname(buffer, sizeof(buffer));
	domain = strchr (buffer, '.');
#endif
	hostname = NULL;

	tty_name = ttyname (0);
	if (tty_name == NULL  ||  *tty_name == '\0')
		tty_name = "/dev/tty??";
	if ((tty_n = strrchr (tty_name, '/')))
		tty_n++;
	else
		tty_n = tty_name;

	while ((c = getopt(argc, argv, "pr:")) != -1)
	{
		switch (c)
		{
		case 'p':	/*	preserve environment  */
					preserve = 1;
					break;

#ifdef TCPIP
		case 'r':	/*	rlogind initiated this call, remote host name */
					if (getuid())
					{	fprintf (stdout, "%s: -r for superuser only\n", argv[0]);
						slow_exit (1);
					}
					if (domain && (p = strchr (optarg, '.')) &&
					    strcasecmp(p, domain) == 0)
						*p = 0;
					hostname = optarg;
					if (get_remote_user (hostname))
						slow_exit (1);
					break;
#endif

		default:	break;
		}
	}

	if (hostname)
		login_name = lname;
	else if (optind < argc)
		login_name = argv[optind++];
	else
		login_name = NULL;
	if (login_name)
		etc_nologin (login_name);

	/*	get defaults from /etc/default/login	*/
	
	console = path = supath = NULL;
	*tz = *hz = '\0';
	idleweeks = timeout = cmask = -1;
	ulimit_blocks = 0;
	passreq = 1;
	altshell = 1;
	get_defaults (DEFAULTFILE, tz, hz, &console, &ulimit_blocks, &passreq, 
				  &altshell, &path, &supath, &timeout, &cmask, &idleweeks);

	if (path == NULL)
		path = USERPATH;
	if (supath == NULL)
		supath = ROOTPATH;
	if (*tz == '\0')
		strcpy (tz, DEFAULTTZ);
	if (timeout == -1)
		timeout = TIMEOUT;

#ifdef __STDC__
	signal (SIGALRM, (void (*)(int)) user_timeout);
#else
	signal (SIGALRM, (void (*)()) user_timeout);
#endif
	alarm  (timeout);
	signal (SIGQUIT, SIG_IGN);
	signal (SIGINT, SIG_IGN);

	/*	accept a user's login user name and password	*/

	shadow_lstchg = -1;
	for (null_user_count = i = 0;  i < MAXBAD;  i++)
	{	char	*salt, *pass, *pass_crypt;

		if (hostname)				/*	network rlogin already ok */
		{	strcpy (user_name, login_name);
			break;
		}

		if (login_name)
		{	strcpy (user_name, login_name);
			login_name = NULL;
		}
		else
		{	fprintf (stdout, "login: ");
			fgets (user_name, SHORTLINE, stdin);
			user_name[SHORTLINE] = '\0';
			trim (user_name);
			user_name[MAXUSERNAME] = '\0';
			strcpy (bad_user[i], user_name);
			if (*user_name == '\0')
			{	null_user_count++;
				continue;
			}
		}

		pwd = getpwnam (user_name);
		if (pwd)
		{	if (pwd->pw_uid)
				etc_nologin (user_name);
			if (*(pwd->pw_shell) == '*')
			{	/*	see 'real' man page for login for * in shell field.
					must change the root directory, then rerun login
					with the new root looking at new password file...
					HOWEVER, i cannot see any use of this feature!
				*/
				if (chroot (pwd->pw_dir))
				{	perror ("Could not change to your root directory: ");
					slow_exit (1);
				}
				fprintf (stdout, "Subsystem root = %s\n", pwd->pw_dir);
				execlp ("login", "login", 0);
				perror ("No login on subsystem: ");
				slow_exit (1);
			}

			/*	valid user.  shadow file may or may not exist */
			if ((strcmp (pwd->pw_passwd, "x") == 0)  
						&&  (shadow = get_shadow (SHADOW, user_name)))
			{	salt = shadow;
				my_pass = shadow;
			}
			else 
				my_pass = salt = pwd->pw_passwd;
		}
		else /* invalid user */
		{	my_pass = salt = "xx";
			etc_nologin ("invalid user name");
		}

		if (*my_pass == '\0')
		{	/*	bypass prompting if zero length */
			if (passreq)
				passwd_needed = 1;		/*	ask user later */
#ifdef USRACCESS
			if (get_usrtime (USRTIME, user_name, pwd->pw_shell, tty_name))
				slow_exit (1);
#endif
			break;
		}

		pass = getpass ("password: ");
		if (pwd)
		{	/*	get regular + dialup password + restrict time access */
			pass_crypt = crypt (pass, salt);
			if ( !strcmp (pass_crypt, my_pass)  
					&&  !dialup_pwd (DIALUPTTY, DIALUPPWD, tty_name, pwd)
#ifdef USRACCESS
					&&  !get_usrtime (USRTIME, user_name, pwd->pw_shell, tty_name)
#endif
				)
				break;
		}
#ifdef NOISY
		if (i > 1)
			sleep (i);
		fprintf (stdout, "Login incorrect: try again.\n");
		fflush (stdout);
#endif
	}
	alarm (0);						/*	turn of pending alarm	*/

	if (i == MAXBAD)
	{	/*	write the bad attempt to /usr/adm/loginlog. */
		FILE *fp;
		time_t	now;

		if (null_user_count < 3)
		{	if ((fp = fopen (LOGINLOG, "a")))
			{	now = time (0L);
				for (i = 0;  i < MAXBAD;  i++)
				{	if (*bad_user[i])
					fprintf (fp, "%s:%s: %s", bad_user[i], tty_name, ctime (&now));
				}
				fclose (fp);
			}
			sleep (BADLOGINTIME);		/* discourage further abuse */
		}
		slow_exit (1);
	}

	/*	if CONSOLE= is defined in /etc/default/login, then root may
		root may log in only on the CONSOLE=device
	*/
	if (pwd->pw_uid == ROOT  &&  console  &&  *console)
	{	strcpy (buffer, tty_name);
		strcat (buffer, ":");
		if (qtscan (console, buffer) < 0)
		{	fprintf (stdout, "Superuser must be on %s\n", console);
			slow_exit (1);
		}
	}

	if (*(pwd->pw_shell) == '\0')
		strcpy (shell, DEFAULTSHELL);
	else
		strcpy (shell, pwd->pw_shell);

	if (chdir (pwd->pw_dir) < 0)
	{	perror ("Could not locate login directory");
		slow_exit (1);
	}

	{	/*	update utmp and wtmp (if wtmp exists)
			For an original tty login, utmp will contain:
				LOGIN   /dev/vt01
			For exec login:
				ajf		vt01
			Must scan for either getutline() form.
		*/
		struct utmp *utmp;
		struct utmp line, id;
#ifndef COHERENT
		struct utmp *getutline();

		/* first check for /dev/tty?? format */

		memset ((char *)&line, '\0', sizeof(line));
		strncpy (line.ut_line, tty_name, sizeof(line.ut_line));
		if ((utmp = getutline (&line)) == NULL)
		{
			/*	for tty?? format */
			setutent();
			memset ((char *)&line, '\0', sizeof(line));
			strncpy (line.ut_line, tty_n, sizeof(line.ut_line));
			if ((utmp = getutline (&line)) == NULL)
			{	fprintf (stdout, "No utmp entry.\n");
				slow_exit (1);
			}
		}
#endif
		memset ((char *)&id, '\0', sizeof(id));
#ifndef COHERENT
		strncpy (id.ut_id, utmp->ut_id, sizeof(id.ut_id));
#endif
		(void) time (&id.ut_time);
		strncpy (id.ut_name, user_name, sizeof(id.ut_name));
		strncpy (id.ut_line, tty_n, sizeof(id.ut_line));
#ifndef COHERENT
		id.ut_pid = getpid();
		id.ut_type = USER_PROCESS;
#endif
		pututline (&id);

		if (access (WTMP, W_OK) == 0)
		{	FILE	*wtmp;
			if ((wtmp = fopen (WTMP, "a")))
			{	fwrite (&id, sizeof(id), 1, wtmp);
				fclose (wtmp);
			}
		}
	}

	if (pwd->pw_uid != ROOT  &&  ulimit_blocks)
	{	if (ulimit (2, ulimit_blocks) == -1)
			fprintf (stdout, "Could not set ULIMIT=%ld\n", ulimit_blocks);
	}

	/*	set user context: tty owner/group, uid/gid and environment */
	
	if (strcmp (tty_name, "/dev/tty??") != 0)
	{	chown (tty_name, pwd->pw_uid, pwd->pw_gid);
		chmod (tty_name, 0622);		/* some people believe 0620 */
	}

	if (setgid (pwd->pw_gid))
	{	fprintf(stdout, "Could not set gid=%d: %s\n", pwd->pw_gid, sys_errlist[errno]);
		slow_exit (1);
	}
#ifdef POSIX_GROUPS
	initgroups (user_name, pwd->pw_gid);	/* posix supplimentary groups */
#endif
	if (setuid (pwd->pw_uid))
	{	fprintf(stdout, "Could not set uid=%d: %s\n", pwd->pw_uid, sys_errlist[errno]);
		slow_exit (1);
	}

	if (passwd_needed)
	{	fprintf (stdout, "A password is needed to log into this system. Choose one.\n");
		if (system (BINPASSWD))
			slow_exit (1);
	}
	else if (shadow_lstchg >= 0 && shadow_max)
	{	if ((time(NULL) / (24L*60*60) - shadow_lstchg) > shadow_max)
		{	fprintf (stdout, "Your password has expired. Choose a new one.\n");
			if (system (BINPASSWD))
				slow_exit (1);
		}
	}

	/*	lastlogin magic seems to use a 2 second difference in
		the modified and access times of the .lastlogin file.
		This is a guess....but it seems to work.
	*/
	sprintf (buffer, "%s/%s", pwd->pw_dir, LASTLOGIN);
	if (access (buffer, 0))
	{	FILE	*fp;
		fprintf (stdout, "Warning: creating .lastlogin\n");
		lcreate = 1;
		if (fp = fopen (buffer, "w"))
		{	fclose (fp);
			chmod (buffer, 0400);
		}
	}
	else
		lcreate = 0;
	if (access (buffer, 0) == 0)
	{	struct stat sb;
		struct utimbuf timset;
		time_t	now;

		if (stat (buffer, &sb) == 0)
		{	if (lcreate == 0)
			{	if (sb.st_mtime - sb.st_atime < 2)
					fprintf (stdout, "Warning: .lastlogin has been modified since your previous login.\n");
				fprintf (stdout, "last login: %s", ctime (&sb.st_mtime));
			}
			now = time (0L);
			if (idleweeks > 0  &&  pwd->pw_uid != ROOT
			    &&  ((now - sb.st_atime) / (24L * 60 * 60 * 7 * (long) idleweeks)) >= idleweeks)
			{	fprintf (stdout, "Login disabled due to lack of usage.\n");
				slow_exit (1);
			}			
			timset.actime  = now - 2; 
			timset.modtime = now;
			utime (buffer, &timset);
		}
	}
	else
		fprintf (stdout, "Warning: could not find .lastlogin\n");

	if (cmask != -1)
		umask (cmask);

	if (!preserve)
	{	environ_init[0] = NULL;
		environ = environ_init;
		if (hostname == NULL)
		{	strncpy(term, search_ttytype(tty_n), sizeof(term));
			term[sizeof(term) - 1] = 0;
		}
		else
		{	char *c, *strchr();
			c = strchr (tname, '/');
			*c = '\0';
			strcpy (term, tname);
		}
		setenv ("TERM",    term);
	}
	sprintf (buffer, "%s/%s", MAIL, user_name);
	setenv ("MAIL",    buffer);
	setenv ("HOME",    pwd->pw_dir);
	setenv ("USER", user_name);
	setenv ("LOGNAME", user_name);
	if (altshell)
		setenv ("SHELL",   shell);
	if (pwd->pw_uid == ROOT)
		setenv ("PATH", supath);
	else
		setenv ("PATH", path);
	if (*tz)
		setenv ("TZ", tz);

	for (i = optind;  i < argc;  i++)		/*	environment from command line */
		check_putenv (argv[i]);
	
	signal (SIGALRM, SIG_DFL);
	signal (SIGQUIT, SIG_DFL);
	signal (SIGINT,  SIG_DFL);
#ifdef SIGTSTP
	/*	Interactive has job control signals, AT&T does not */
	signal (SIGTSTP, SIG_IGN);
#endif

	/*	exec the desired shell */
	buffer[0] = '-';
	strcpy (buffer + 1, (p = strrchr (shell, '/')) ? p + 1 : shell);
	execlp (shell, buffer, 0);
	fprintf (stdout, "%s: no shell: ", argv[0]);
	perror (shell);
	slow_exit (1);
}

/****************************************************************************
*	read /etc/dialups & /etc/d_passwd, if they exist						*
*	Return 0 if user has access, 1 if no access								*
****************************************************************************/

int dialup_pwd (ttyfname, pwdfname, tty_name, pwd)
char	*ttyfname;		/*	/etc/dialups									*/
char	*pwdfname;		/*	/etc/d_passwd									*/
char	*tty_name;		/*	user's tty line									*/
struct passwd *pwd;		/*	user;s /etc/passwd file entry					*/
{	FILE	*fp;
	char	dtty[SHORTLINE+1];
	char	*dpass, *dcrypt, *salt, *shell, *dialpass, *crypt();

	if (pwd == NULL)
		return (0);

	if (*(pwd->pw_shell))
		shell = pwd->pw_shell;
	else
		shell = DEFAULTSHELL;
	if (access (ttyfname, 0))
		return (0);
	if (fp = fopen (ttyfname, "r"))
	{	while (fgets (dtty, SHORTLINE, fp))
		{	dtty[SHORTLINE] = 0;
			trim (dtty);
			if (strcmp (dtty, tty_name) == 0)
			{	fclose (fp);
				if (dialpass = get_shadow (DIALUPPWD, shell))
				{	if (*dialpass == '\0')		/*	no password needed */
						return (0);
					salt = dialpass;
				}
				else 
					return (1);
				dpass = getpass ("dialup password: ");
				dcrypt = crypt (dpass, salt);
				if (strcmp (dcrypt, dialpass))
					return (1);
				else
					return (0);
			}
		}
		fclose (fp);
	}
	return (0);
}

#ifdef USRACCESS
/****************************************************************************
*	process /etc/usrtime, if it exists										*
*	Return 0 if user has access, 1 if no access								*
****************************************************************************/

int get_usrtime (fname, user, shell, tty)
char	*fname;			/*	/etc/usrtime									*/
char	*user;			/*	user login name									*/
char	*shell;			/*	user's shell (from /etc/passwd)					*/
char	*tty;			/*	user's login tty line							*/
{
	struct usrtime	all, uucp, slip, interact, genuser, thisuser, *which;
	char	line[LONGLINE+1];
	FILE	*fp;
	char	*var[5], *colon;
	char	*uname, save_letter;
	int		i;
	long	now;
	char	*uucico;

	if (strcmp (user, "root") == 0)
		return (0);

	if ((fp = fopen (fname, "r")) == NULL)
		return (0);

	uucico = "uucico";
	all.enable = uucp.enable = slip.enable = interact.enable
			   = thisuser.enable = genuser.enable = NULL;
	
	uname = alloca (strlen (user) + 3);
	strcpy (uname, user);
	strcat (uname, ",");

	while (fgets (line, LONGLINE, fp))
	{	line[LONGLINE] = '\0';
		if (*line == '#')
			continue;
		trim (line);
		for (colon = line, i = 0;  i < 5;  i++)
		{	var[i] = colon;
			colon = strchr (colon, ':');
			if (colon == NULL)
				break;
			*colon++ = '\0';
		}
		if (colon == NULL)					/*	need min of 5 colons */
			continue;
		which = NULL;

		/*	select first instance of lines */

		if (strcmp (var[0], "ALL") == 0)
		{	if (all.enable)
				continue;
			which = &all;
		}
		else if (strcmp (var[0], "UUCP") == 0)
		{	if (uucp.enable)
				continue;
			which = &uucp;
		}
		else if (strcmp (var[0], "SLIP") == 0)
		{	if (slip.enable)
				continue;
			which = &slip;
		}
		else if (strcmp (var[0], "INTERACTIVE") == 0)
		{	if (interact.enable)
				continue;
			which = &interact;
		}
		else if (var[0][0] == '\0')
		{	/*	general default user setup */
			if (genuser.enable)
				continue;
			which = &genuser;
		}

		if (which)
		{	/*	ALL, UUCP, SLIP, INTERACTIVE or default user lines */
			which->enable 	 = alloca (strlen (var[1]) + 2);
			which->ttyline   = alloca (strlen (var[2]) + 2);
			which->days    	 = alloca (strlen (var[3]) + 2);
			which->timerange = alloca (strlen (var[4]) + 2);
			strcpy (which->enable, 	  var[1]);
			strcpy (which->ttyline,   var[2]);
			strcpy (which->days,      var[3]);
			strcpy (which->timerange, var[4]);
		}
		else
		{	/*	standard user line, could be list of users */
			i = strlen (var[0]);
			save_letter = var[1][0];
			strcat (var[0], ",");
			if (tscan (var[0], uname) >= 0)
			{	var[0][i] = '\0';				/*	undo comma */
				var[1][0] = save_letter;
				thisuser.enable    = var[1];
				thisuser.ttyline   = var[2];
				thisuser.days      = var[3];
				thisuser.timerange = var[4];
				break;							/* got user exactly */
			}
		}
	}
 	fclose (fp);
	now = time (NULL);

	if (   (all.enable      
					&& check_user ("All", now, tty, &all))
		|| (uucp.enable     
					&& tscan (shell, uucico) >= 0 
					&& check_user ("Uucp", now, tty, &uucp))
		|| (slip.enable     
					&& tscan (shell, SLIP)   >= 0 
					&& check_user ("Slip", now, tty, &slip))
		|| (interact.enable 
					&& tscan (shell, uucico) < 0 
					&& tscan (shell, SLIP) < 0 
					&& check_user ("Interactive", now, tty, &interact))
		|| (genuser.enable  
					&& thisuser.enable == NULL 
					&& check_user ("General", now, tty, &genuser))
		|| (thisuser.enable 
					&& check_user ("Your", now, tty, &thisuser)))
	{	return (1);		/* user rejected */
	}
	return (0);			/* user accepted */
}

/****************************************************************************
*	usrtime: check fields in sequence to see if user has access permission	*
****************************************************************************/

int check_user (note, now, tty, usr)
char	*note;			/*	comment to user									*/
char	*tty;			/*	user's tty line									*/
long	now;			/*	current time of day (now=time(NULL))			*/
struct usrtime *usr;	/*	various values from /etc/usrtime				*/
{
	if (strcmp (usr->enable, "NOLOGIN") == 0)
	{	fprintf (stdout, "%s logins are disabled.\n", note);
		return (1);
	}
	return (   check_time (note, now, usr->enable, 0)		/* yymmdd */
			|| check_tty  (note, tty, usr->ttyline)  
			|| check_day  (note, now, usr->days)  
			|| check_time (note, now, usr->timerange, 1));	/* hhmm */
}

/****************************************************************************
*	usrtime: ensure usr is on acceptable tty line							*
****************************************************************************/

int check_tty (note, tty, list)
char	*note;			/*	comment to user									*/
char	*tty;			/*	user's tty line									*/
char	*list;			/*	list of valid tty lines							*/
{
	char	*tt, *ls, *short_tt;
	int		inverse = 0;

	if (*list == '\0')
		return (0);
	if (*list == '!')
	{	inverse = 1;
		list++;
	}
	tt = alloca (strlen (tty) + 2);
	strcpy (tt, tty);
	strcat (tt, ",");
	if (short_tt = strrchr (tt, '/'))
		tt = short_tt + 1;
	ls = alloca (strlen (list) + 2);
	strcpy (ls, list);
	strcat (ls, ",");
	if (qtscan (ls, tt) >= 0  &&  last_match == ',')
	{	if (!inverse)
			return (0);
	}
	else if (inverse)
		return (0);
	fprintf (stdout, "%s logins are %s on %s.\n", 
				note, inverse ? "not enabled" : "enabled", list);
	return (1);
}

/****************************************************************************
*	usrtime: ensure usr is on acceptable week day							*
****************************************************************************/

int check_day (note, now, list)
char	*note;			/*	comment to user									*/
long	now;			/*	current time = time(NULL)						*/
char	*list;			/*	list of days (Mon, Tue...)						*/
{
	char	weekday[50];
	int		inverse = 0;
	
	if (*list == '\0')
		return (0);
	if (*list == '!')
	{	inverse = 1;
		list++;
	}
	cftime (weekday, "%a", &now);
	if (tscan (list, weekday) >= 0)
	{
		if (!inverse)
			return (0);
	}
	else if (inverse)
		return (0);
	fprintf (stdout, "%s logins are %s on %s.\n", 
				note, inverse ? "not enabled" : "enabled", list);
	return (1);
}

/****************************************************************************
*	usrtime: ensure usr is on acceptable time or date range					*
****************************************************************************/

int check_time (note, now, list, timerange)
char	*note;			/*	comment to user									*/
long	now;			/*	current time = time(NULL)						*/
char	*list;			/*	hhmm-hhmm or yymmdd-yymmdd list					*/
int		timerange;		/*	0 = check yyyymmdd,  1 = check hhmm				*/
{
	char	clock[50];
	char	*begin, *end;
	char	*ls;
	long	present, first, last;
	int		inverse = 0;
	
	if (*list == '!')
	{	inverse = 1;
		list++;
	}
	if (*list < '0'  ||  *list > '9'  ||  strchr (list, '-') == NULL)
		return (0);
	ls = end = alloca (strlen (list) + 2);
	strcpy (end, list);
	if (timerange)
		cftime (clock, "%H%M", &now);
	else
		cftime (clock, "%Y%m%d", &now);
	present = atol (clock);
	while (*end)
	{	begin = end;
		if ((end = strchr (begin, '-')) == NULL)
			break;
		*end++ = '\0';
		first = atol (begin);
		last = atol (end);
		if (first <= present  &&  present <= last)
		{	if (!inverse)
				return (0);
			else
				break;
		}
		if ((end = strchr (end, ',')) == NULL)
			break;
		end++;
	}
	if ((inverse  &&  end)  ||  (!inverse  &&  end == NULL))
	{	if (inverse)
			end = "are not";
		else
			end = "are";
		if (timerange)
			fprintf (stdout, "%s logins %s enabled between %s hours.\n", note, end, list);
		else
			fprintf (stdout, "%s logins %s enabled for dates %s.\n", note, end, list);
		return (1);
	}
	return (0);
}

#endif

/****************************************************************************
*	set the users environment with values.									*
****************************************************************************/

void setenv (this, val)
char *this;
char *val;
{
	char	s[MAXPATHLEN+1];
	char	*ss;

	sprintf (s, "%s=%s", this, val);
	ss = malloc (strlen(s) + 1);
	strcpy (ss, s);
	putenv (ss);
}

/****************************************************************************
*	user-provided environment variables must be checked						*
****************************************************************************/

void check_putenv (val)
char *val;
{	static count = 0;
	char	*env;
	char	**forbid;
	
	/*	check forbidden environment variables */
	for (forbid = forbidden;  *forbid;  forbid++)
	{	if (strncmp (*forbid, val, strlen (*forbid)) == 0)
			return;
	}

	/*	 variables without an '=' sign assume the form of "Lnn=val" */
	if (strchr (val, '=') == NULL)
	{	env = malloc (strlen (val)+ 6);
		sprintf (env, "L%d=%s", count++, val);
		val = env;
	}
	putenv (val);
}

/****************************************************************************
*	get the encrypted password from /etc/shadow or /etc/d_passwd			*
*	allow "null" user for /etc/d_passwd general shell password.				*
*	Return encrypted password if found, \0 if user/passwd not found, or 	*
* 	NULL if shadow file missing.											*
****************************************************************************/

/*	could have used getspnam(user) except for /etc/d_passwd */

char *get_shadow (fname, user)
char *fname;		/*	/etc/shadow or /etc/d_passwd						*/
char *user;			/*	user's login name									*/
{	FILE	 *shadow;
	char	line[SHORTLINE+1];
	char	*pwd, *colon;
	static char pass[50];

	*pass = '\0';
	shadow_lstchg = shadow_min = shadow_max = 0;
	if (shadow = fopen (fname, "r"))
	{	while (fgets (line, SHORTLINE, shadow))
		{	line[SHORTLINE] = '\0';
			if (pwd = strchr (line, ':'))
			{	*pwd++ = '\0';
				if (colon = strchr (pwd, ':'))
				{	*colon++ = '\0';
					if (*line)
					{	if (strcmp (user, line) == 0)
						{	strcpy (pass, pwd);
							if (colon)
							{	shadow_lstchg = atol (colon);
								colon = strchr (colon, ':') + 1;
								shadow_min = atol (colon);
								colon = strchr (colon, ':') + 1;
								shadow_max = atol (colon);
							}
							break;
						}
					}
				}
			}
		}
		fclose (shadow);
		return (pass);		/* \0 or valid user passwd */
	}
	return (NULL);			/* no file */
}

/****************************************************************************
*	search for tty type in /etc/ttytype										*
****************************************************************************/

char *search_ttytype (tty)
char *tty;			/*	user's tty line										*/
{	FILE	*fp;
	char	line[SHORTLINE+1];
	static char *tname;
	char	*dev;
	
	if (fp = fopen (TTYTYPE, "r"))
	{	while (fgets (line, SHORTLINE, fp))
		{	line[SHORTLINE] = '\0';
			trim (line);
			dev = line;
			while (*dev)				/*	skip tty name 	*/
			{	if (*dev <= ' ')
					break;
				dev++;
			}
			while (*dev)				/*	locate device	*/
			{	if (*dev > ' ')
					break;
				*dev++ = '\0';
			}
			if (strcmp (dev, tty) == 0)
			{	fclose (fp);
				tname = malloc (strlen(line)+1);
				strcpy (tname, line);
				return (tname);
			}
		}
	}
	fclose (fp);
	return ("UNKNOWN");
}


/****************************************************************************
*	read /etc/default/login file											*
****************************************************************************/

int get_defaults (fname, tz, hz, console, ulim, passreq, altshell, 
				  path, supath, timeout, cmask, idleweeks)
char *fname;		/*	/etc/default/login									*/
char *tz;			/*	TIMEZONE=xxx	should be same as in /etc/TIMEZONE	*/
char *hz;			/*	HZ=nn												*/
char **console;		/*	CONSOLE=xxx     root must be on this device if given*/
long *ulim;			/*	ULIMIT=xxx		default ulimit for users			*/
int	*passreq;		/*	PASSREQ=YES/NO	is password required				*/
int *altshell;		/*	ALTSHELL=YES/NO	YES==set shell name in environment	*/
char **path;		/*	PATH=xxx		default user path					*/
char **supath;		/*	SUPATH=xxx		defaut root path					*/
long *timeout;		/*	TIMEOUT=nn		login abort timeout in seconds		*/
int  *cmask;		/*	UMASK=ooo		default umask for users				*/
int	 *idleweeks;	/*	IDLEWEEKS=nn	idle weeks							*/
{
	FILE	*fp;
	char	line[LONGLINE+1];
	char	*equal;
	
	if ((fp = fopen (fname, "r")) == NULL)
		return (2);
	while ((fgets (line, LONGLINE, fp)))
	{	if (*line == '#')				/*	must be first character */
			continue;
		line[LONGLINE] = '\0';
		trim (line);
		if (equal = strchr (line, '='))
		{	*equal++ = '\0';
			if (strcmp (line, "TIMEZONE") == 0)
				strcpy (tz, equal);
			else if (strcmp (line, "HZ") == 0)
				strcpy (hz, equal);
			else if (strcmp (line, "CONSOLE") == 0)
			{	if (*console == NULL)
				{	*console = malloc (strlen (equal) + 2);
					strcpy (*console, equal);
					if ((*console)[strlen(*console) - 1] != ':')
						strcat (*console, ":");
				}
			}
			else if (strcmp (line, "ULIMIT") == 0)
				*ulim = atol (equal);
			else if (strcmp (line, "PATH") == 0)
			{	if (*path == NULL)
				{	*path = malloc (strlen (equal) + 1);
					strcpy (*path, equal);
				}
			}
			else if (strcmp (line, "SUPATH") == 0)
			{	if (*supath == NULL)
				{	*supath = malloc (strlen (equal) + 1);
					strcpy (*supath, equal);
				}
			}
			else if (strcmp (line, "TIMEOUT") == 0)
				*timeout = atol (equal);
			else if (strcmp (line, "UMASK") == 0)
				*cmask = atoo (equal);
			else if (strcmp (line, "PASSREQ") == 0)
			{	if (toupper (*equal) == 'Y')
					*passreq = 1;
				else
					*passreq = 0;
			}
			else if (strcmp (line, "ALTSHELL") == 0)
			{	if (toupper (*equal) == 'Y')
					*altshell = 1;
				else
					*altshell = 0;
			}
			else if (strcmp (line, "IDLEWEEKS") == 0)
				*idleweeks = atoi (equal);
		}
	}
	fclose (fp);
	return (0);
}

/****************************************************************************
*	check for and display /etc/nologin file									*
****************************************************************************/

void etc_nologin (trustme)
char	*trustme;
{
	int fd, nchars;
	FILE *fp;
	char why[8192];
	static int msg_done = 0;

	if (strcmp (trustme, "root") == 0  ||  msg_done)
		return;
	if ((fd = open (NOLOGIN, O_RDONLY, 0)) >= 0) 
	{	while ((nchars = read(fd, why, sizeof(why))) > 0)
			write (fileno(stdout), why, nchars);
		close (fd);
		if (fp = fopen (TRUSTED, "r"))
		{	fgets (why, 100, fp);
			fclose (fp);
			trim (why);
			
			/*	trusted user can bypass nologin to allow the sysadmin
				to login on the network (he cannot be root - usually)
			*/
			if (strcmp (why, trustme))
				slow_exit (0);
		}
		else slow_exit (0);
	}
	msg_done = 1;
}

/****************************************************************************
*	sleep before exit so network does not close before msg is out.			*
****************************************************************************/

void slow_exit (rc)
int	rc;
{
#ifdef TCPIP
	fflush (stdout);
	sleep (1);		/* maybe this should be longer? */
#endif
	exit (rc);
}

/****************************************************************************
*	trim line at end														*
****************************************************************************/

void trim (s)
char *s;
{	char	*t;

	t = s + strlen (s);
	do
	{	if (*t > ' ')
			break;
		*t = '\0';
	} while (t-- != s);
}

/****************************************************************************
*	ascii to octal															*
****************************************************************************/

int atoo (s)
char *s;
{
	unsigned int v;
	
	v = 0;
	while (*s)
		v = v * 8 + (*s++ - '0');
	return (v);
}

/****************************************************************************
*	timeout error message if user takes too long to log in.					*
****************************************************************************/

void user_timeout()
{
	fprintf (stdout, "login timed out after %d seconds\n", timeout);
	slow_exit(1);
}


/****************************************************************************
*	search for string t in string s.  return -1 if it does not exist		*
*	otherwise return an integer, the first byte of s that matches t.		*
****************************************************************************/

int tscan (s, t)
char 	s[], t[];
{
	int	i, j, k;
	for (i = 0;  s[i] != '\0';  i++)
	{	for (j = i, k=0;  t[k] != '\0'  &&  s[j] == t[k];  j++, k++)
			;
		if (t[k] == '\0')
			return (i);
	}
	return (-1);
}

/****************************************************************************
*	search for string t in string s.  return -1 if it does not exist		*
*	otherwise return an integer, the first byte of s that matches t.		*
*	Any '?' in s  is a wild card match for t								*
****************************************************************************/

int qtscan (s, t)
char 	s[], t[];
{
	int	i, j, k;
	for (i = 0;  s[i] != '\0';  i++)
	{	for (j = i, k=0;  t[k] != '\0'  &&  ((s[j] == t[k]) || (s[j] == '?')) ;  j++, k++)
			last_match = s[j];
		if (t[k] == '\0')
			return (i);
	}
	last_match = 0;
	return (-1);
}

#ifdef TCPIP
/****************************************************************************
*	read remotename, localname tty info from rlogind						*
****************************************************************************/

void get_remote (buf, cnt)		/* read null terminated string */
char *buf;
int cnt;
{
	char c;

	do 
	{	if (read (0, &c, 1) != 1)
			slow_exit(1);
		if (--cnt < 0) 
			slow_exit(1);		/* should never happen! */
		*buf++ = c;
	} while (c != 0);
}

int get_remote_user (host)		/* read from rlogind */
char	*host;
{	int	ruserok();

	if (host == NULL  ||  *host == '\0')
		return (-1);
	get_remote (rname, sizeof (rname));		/*	remote user name		*/
	get_remote (lname, sizeof (lname));		/*	desired local user name */
	get_remote (tname, sizeof (tname));		/*	tty / baud rate			*/
	set_tcp_ioctl ();						/*	clean up socket.		*/
	pwd = getpwnam (lname);

	/* check hosts.equiv and all those good things */
	if (pwd == NULL  ||  ruserok (host, (pwd->pw_uid == 0), rname, lname))
	{	fprintf (stdout, "Who is %s@%s?\n", rname, host);
		return (-1);
	}
#ifdef USRACCESS
	if (get_usrtime (USRTIME, lname, pwd->pw_shell, tty_name))
		return (-1);
#endif	/* USRACCESS */
	return (0);
}


/****************************************************************************
*	set up the tty line ioctl for network connection						*
*																			*
*	The default ioctl() conditions are for Interactive Unix.  If WOLLONGONG	*
*	is defined, then the ioctls for AT&T + Wollongong rlogind are set.		*
****************************************************************************/

char	*rates[] =  { "0", "50", "75", "110", "134", "150", "200", "300",
				      "600", "1200", "1800", "2400", "4800", "9600", 
				      "19200", "38400" 
				    };
#define	NRATE	(sizeof (rates) / sizeof (rates[0]))

void set_tcp_ioctl ()
{	struct termio tio;
	char	*baudrate;
	int		i;
	int		t;

	ioctl(0, TCGETA, &tio);
	if (baudrate = strchr (tname, '/'))
	{	baudrate++;
		for (i = 0;  i < NRATE;  i++)
		{	if (strcmp (baudrate, rates[i]) == 0)
			{
#ifdef WOLLONGONG
				tio.c_cflag = (B0 + i) |  CS8;
#else
				tio.c_cflag = (B0 + i) | HUPCL | CREAD | CS8;
#endif
				break;
			}
		}
	}
#ifdef WOLLONGONG
	tio.c_iflag = ICRNL | ISTRIP | IGNPAR;
	tio.c_oflag = TAB3 | ONLCR | OPOST;
	tio.c_lflag = ECHO | ECHOE | ECHOK | ISIG | ICANON;
#else
	tio.c_iflag = IXON | IXANY | ICRNL | ISTRIP | IGNPAR | BRKINT;
	tio.c_oflag = ONLCR | OPOST;
	tio.c_lflag = ECHO | ECHOE | ECHOK | ISIG | ICANON;
#endif
	ioctl (0, TCSETA, &tio);

	for (t = getdtablesize(); t > 2; t--)
		close(t);

}
#endif	/* TCPIP */
