/*
**
** dired - a directory editor modelled after GNU Emacs Dired mode.
**
** Written in C++ using the termcap(3) library
**
** dired.C 1.35  Delta'd: 14:09:27 3/9/92  Mike Lijewski, CNSF
**
** Copyright (c) 1991 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: (1) source distributions retain this entire copyright
** notice and comment, and (2) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University'' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/

#ifndef _IBMR2
#include <libc.h>
#endif
#include <new.h>
#include <osfcn.h>
#include <stdio.h>
#include <string.h>

#include "classes.h"
#include "dired.h"
#include "display.h"

//
// the definition of our stack of directories
//
DirStack *dir_stack;

//
// The definition of the variable controlling how we sort listings.
// The default sort order is ALPHABETICALLY.
//
sort_order how_to_sort = ALPHABETICALLY;

//
// commands to get long directory listings:
//
//    ls_cmd[0] gives alphabetical listing
//    ls_cmd[1] gives listing sorted by modification time
//    ls_cmd[2] gives listing sorted by access time
//    ls_cmd[3] gives listing sorted by inode-change time
//
const char *const ls_cmd[4] =
#ifdef NEED_LG
  { "ls -agl ", "ls -aglt ", "ls -agltu ", "ls -acglt " };
#else
  { "ls -al ", "ls -alt ", "ls -altu ", "ls -aclt " };
#endif

//
// the definition of our modeline prefix
//
const char *const modeline_prefix = "----- Dired: ";

/*
** exception handler for new() - called once in main()
*/
static void free_store_exception()
{
    error("File %s, line %d: memory exhausted", __FILE__, __LINE__);
}

/*
** dired - Edit the given directory. We must have read and execute
**         permission to edit a directory; calling routines must
**         guarantee this.
*/
void dired(char *dirname)
{
    if (chdir(dirname) < 0)
        error("File %s, line %d: couldn't chdir() to `%s'",
              __FILE__, __LINE__, dirname);

    char *pathname = get_current_directory();
    if (pathname == 0)
        error("File %s, line %d: get_current_ directory() in `%s' failed.",
              __FILE__, __LINE__, dirname);

    char *full_directory_name = new char[strlen(pathname)+1];
    (void)strcpy(full_directory_name, pathname);

    DirList *dir_list = get_directory_listing(full_directory_name);
    if (dir_list == 0)
        error("File %s, line %d: couldn't read directory `%s'",
              __FILE__, __LINE__, full_directory_name);

    /*
     * We track the CWD and PWD variables, if they're defined, so that
     * applications such as emacs which use them will work properly.
     */
    if (getenv("CWD")) {
        static char *str;
        if (str) delete str;
        str = new char[strlen(full_directory_name)+5];
        (void)strcpy(str, "CWD=");
        (void)strcat(str, full_directory_name);
        if (putenv(str) < 0)
            error("File %s, line %d: putenv(%s) failed.",
                  __FILE__, __LINE__, full_directory_name);
    }

    if (getenv("PWD")) {
        static char *str;
        if (str) delete str;
        str = new char[strlen(full_directory_name)+5];
        (void)strcpy(str, "PWD=");
        (void)strcat(str, full_directory_name);
        if (putenv(str) < 0)
            error("File %s, line %d: putenv(%s) failed.",
                  __FILE__, __LINE__, full_directory_name);
    }

    dir_stack->push(dir_list);

    initial_listing(dir_list);

    update_modeline(modeline_prefix, full_directory_name);

    dir_list->saveYXPos(0, goal_column(dir_list));

    if (dir_list->currLine()->length() > columns())    
        leftshift_current_line(dir_list);
    else
        move_cursor(dir_list->savedYPos(), dir_list->savedXPos());

    synch_display();

    read_commands(dir_list);
}

///////////////////////////////////////////////////////////////////////////////
//
// ******************** main *************************************************
//
///////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
    char *dirname;

    //
    // process options - the only options we accept are -t -u or -c.
    //
    if (argc < 2)
        //
        // edit current directory
        //
        dirname = ".";
    else {
        while(**++argv == '-') {
            if (strcmp(*argv, "-t") == 0) {
                set_sort_order(MODIFICATION_TIME);
                continue;
            }
            if (strcmp(*argv, "-u") == 0) {
                set_sort_order(ACCESS_TIME);
                continue;
            }
            if (strcmp(*argv, "-c") == 0) {
                set_sort_order(INODE_CHANGE_TIME);
                continue;
            }
        }
        dirname = *argv ? *argv : ".";
    }

    if (!isatty(0) || !isatty(1)) {
        (void)fprintf(stderr, "stdin & stdout must be terminals\n");
        exit(EXIT_FAILURE);
    }

    set_new_handler(free_store_exception);

    init_display();

    set_signals();

    char *cwd = get_current_directory();
    if (cwd == 0)
        error("File %s, line %d: couldn't grok current directory.",
              __FILE__, __LINE__);

    if (strcmp(cwd, "/") && cwd[strlen(cwd)-1] == '/')
        cwd[strlen(cwd)-1] = '\0';

    if (!is_directory(dirname))
        error("File %s, line %d: `%s' isn't a directory",
              __FILE__, __LINE__, dirname);

    if (!read_and_exec_perm(dirname))
        error("File %s, line %d: need read & exec permission to edit `%s'",
              __FILE__, __LINE__, dirname);

    dir_stack = new DirStack();

    dired(dirname);

    return 0;
}
