/*
** command2.C - the main command loop and some of the commands themselves.
**
**               Command1.C and command2.C
**               are concatenated during the make into
**               commands.C, which consists of the main
**               command loop and all commands called from
**               within that loop.
**
** command2.C 1.9   Delta'd: 20:42:12 3/9/92   Mike Lijewski, CNSF
**
** Copyright (c) 1991 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: (1) source distributions retain this entire copyright
** notice and comment, and (2) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University'' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/

/*
** delete_file - delete the current file which is a regular file.
**               Update the window appropriately.
*/

static void delete_file(DirList *dl, const char *file)
{
    if (unlink(file) != -1)
        remove_listing_line(dl, dl->savedYPos());
    else {
        message("`deletion' failed");
        move_cursor(dl->savedYPos(), dl->savedXPos());
    }
}

/*
** delete_directory - delete the current file which is a directory.
**                    Update the window appropriately.
*/

static void delete_directory(DirList *dl, const char *dirname)
{
    if (rmdir(dirname) != -1)
        remove_listing_line(dl, dl->savedYPos());
    else {
        message("`deletion' failed");
        move_cursor(dl->savedYPos(), dl->savedXPos());
    }
}

/*
** delete_current_file - attempt to delete current file.  `key' is
**                       the actual key which resulted in calling
**                       this routine.  One of `d' or `D'.
*/

static void delete_current_file(DirList *dl, int key)
{
    char *file = get_file_name(dl);
    
    if (strcmp(file, ".")  == 0 || strcmp(file, "..") == 0) {
        message("`deletion' of `.' and `..' is not allowed");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        return;
    }

    char *msg = new char[strlen(file)+18];
    (void)strcpy(msg, "delete `");
    (void)strcat(msg, file);
    (void)strcat(msg, "'? (y|n) ");
    
    if (is_directory(file) && (key == KEY_D || yes_or_no(msg)))
        delete_directory(dl, file);
    else if (key == KEY_D || yes_or_no(msg))
        delete_file(dl, file);
    else
        //
        // don't do the delete
        //
        move_cursor(dl->savedYPos(), dl->savedXPos());
    
    synch_display();

    delete file;
    delete msg;
}

/*
** rename_file - rename src to dest and update window appropriately.
*/

static void rename_file(DirList *dl, const char *src, const char *dest)
{
    if (rename(src, dest) != -1) {
        message("`rename' was successful");
        //
        // Is `dest' in our current directory?
        //
        if (strchr(dest, '/') == NULL) {
            //
            // It must be in our current directory.
            //
            // The important point to note about the following code
            // is that if `dest' is a directory, we need to add a 'd'
            // to the ls command as in
            //
            //   ls -ld source
            //
            int is_dir = is_directory(dest);
            size_t size = strlen(ls_cmd[the_sort_order()]) + strlen(dest) + 3;

            if (is_dir) size++;

            char *command = new char[size];
            (void)strcpy(command, ls_cmd[the_sort_order()]);
            if (is_dir) {
                size_t pos = strlen(command);
                command[pos-1] = 'd';
                command[pos]   = ' ';
                command[pos+1] = 0;
            }
            (void)strcat(command, "'");
            (void)strcat(command, dest);
            (void)strcat(command, "'");
            
            FILE *fp = popen(command, "r");
            if (fp == 0)
                error("File %s, line %d: popen() error", __FILE__, __LINE__);

            delete command;

            char *new_line = fgetline(fp);
            if (fp == 0)
                error("File %s, line %d: couldn't read from popen() pipe",
                      __FILE__, __LINE__);

            (void)pclose(fp);

            dl->currLine()->update(new_line);
            dl->saveYXPos(dl->savedYPos(), goal_column(dl));

            if (dl->currLine()->length() > columns())
                leftshift_current_line(dl);
            else {
                move_cursor(dl->savedYPos(), 0);
                clear_to_end_of_line();
                display_string(new_line);
                move_cursor(dl->savedYPos(), dl->savedXPos());
            }
        }
        else
            //
            // Assume it isn't in our current directory.
            //
            remove_listing_line(dl, dl->savedYPos());
    }
    else {
        message("`rename' failed");
        move_cursor(dl->savedYPos(), dl->savedXPos());
    }
}

/*
** rename_current_file - attempt to rename the current file
*/

static void rename_current_file(DirList *dl)
{
    char *from_file = get_file_name(dl);
    
    if (strcmp(from_file, ".")  == 0 ||
        strcmp(from_file, "..") == 0) {
        message("`renaming' of `.' and `..' is not allowed");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        return;
    }
    
    char *to_file = prompt("Rename to: ");
    
    if (to_file == 0) {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete to_file;
        delete from_file;
        return;
    }
    else if (*to_file == 0) {
        message("not a valid file name");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete to_file;
        delete from_file;
        return;
    }
    
    rename_file(dl, from_file, to_file);
    
    synch_display();
    
    delete to_file;
    delete from_file;
}

/*
** compress_file - compress the current file and update the window.
*/

static void compress_file(DirList *dl, const char *file)
{
    message("Compressing ... ");

    char *command = new char[strlen(file)+32];
    (void)strcpy(command, "compress -f ");
    (void)strcat(command, "'");
    (void)strcat(command, file);
    (void)strcat(command, "' >/dev/null 2>&1");
    
    if (!system(command)) {
        
        (void)strcpy(command, ls_cmd[the_sort_order()]);
        (void)strcat(command, "'");
        (void)strcat(command, file);
        (void)strcat(command, ".Z");
        (void)strcat(command, "'");
        
        FILE *fp = popen(command, "r");
        if (fp == 0)
            error("File %s, line %d: error on popen()", __FILE__, __LINE__);

        char *new_line = fgetline(fp);
        if (fp == 0)
            error("File %s, line %d: couldn't read from popen pipe",
                  __FILE__, __LINE__);

        (void)pclose(fp);
        
        dl->currLine()->update(new_line);

        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else {
            move_cursor(dl->savedYPos(), 0);
            clear_to_end_of_line();
            display_string(new_line);
        }

        message("Compressing ... done");
    }
    else
        message("`compress' failed");
    
    delete command;
}

/*
** compress_current_file - attempt to compress current file
*/

static void compress_current_file(DirList *dl)
{
    char *file = get_file_name(dl);

    //
    // Disallow compressing of symbollically linked files.
    //
    if (strstr(&(dl->currLine()->line())[dl->savedXPos()], " -> ")) {
        message("compress'ing symbollically linked files not allowed");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        return;
    }
    
    if (file[strlen(file)-1] == 'Z' && file[strlen(file)-2] == '.') {
        message("file appears to already be compressed");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        return;
    }
    
    if (is_regular_file(file))
        compress_file(dl, file);
    else
        message("can only `compress' regular files");
    
    move_cursor(dl->savedYPos(), dl->savedXPos());

    synch_display();
    
    delete file;
}

/*
** chgrp_file - change the group of the current file.
**              Update the window appropriately.
*/

static void chgrp_file(DirList *dl, const char *file, const char *group)
{
    char *command = new char[strlen(file)+strlen(group)+25];
    (void)strcpy(command, "chgrp ");
    (void)strcat(command, group);
    (void)strcat(command, " '");
    (void)strcat(command, file);
    (void)strcat(command, "' >/dev/null 2>&1");
    
    if (!system(command)) {

        (void)strcpy(command, ls_cmd[the_sort_order()]);
        if (is_directory(file)) {
            size_t pos = strlen(command);
            command[pos-1] = 'd';
            command[pos]   = ' ';
            command[pos+1] = 0;
        }
        (void)strcat(command, "'");
        (void)strcat(command, file);
        (void)strcat(command, "'");
        
        FILE *fp = popen(command, "r");
        if (fp == 0)
            error("File %s, line %d: error on popen()", __FILE__, __LINE__);
        
        char *new_line = fgetline(fp);
        if (fp == 0)
            error("File %s, line %d: couldn't read from popen() pipe",
                  __FILE__, __LINE__);
        
        (void)pclose(fp);
        
        dl->currLine()->update(new_line);

        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else {
            move_cursor(dl->savedYPos(), 0);
            clear_to_end_of_line();
            display_string(new_line);
        }
    }
    else
        message("`chgrp' failed");

    delete command;
}

/*
** chgrp_current_file - attempt to chgrp current file
*/

static void chgrp_current_file(DirList *dl)
{
    char *file = get_file_name(dl);
    
    char *group = prompt("Change to Group: ");
    
    if (group == 0) {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete group;
        delete file;
        return;
    }
    else if (*group == 0) {
        message("not a valid group");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete group;
        delete file;
        return;
    }
    
    chgrp_file(dl, file, group);
    
    delete group;
    delete file;
    
    move_cursor(dl->savedYPos(), dl->savedXPos());

    synch_display();
}

/*
** chmod_file - change the mode of the current file.
**              Update the window appropriately.
*/

static void chmod_file(DirList *dl, const char *file, const char *mode)
{
    char *command = new char[strlen(file)+strlen(mode)+25];
    (void)strcpy(command, "chmod ");
    (void)strcat(command, mode);
    (void)strcat(command, " '");
    (void)strcat(command, file);
    (void)strcat(command, "' >/dev/null 2>&1");

    if (!system(command)) {

        (void)strcpy(command, ls_cmd[the_sort_order()]);
        if (is_directory(file)) {
            size_t pos = strlen(command);
            command[pos-1] = 'd';
            command[pos]   = ' ';
            command[pos+1] = 0;
        }
        (void)strcat(command, "'");
        (void)strcat(command, file);
        (void)strcat(command, "'");
        
        FILE *fp = popen(command, "r");
        if (fp == 0)
            error("File %s, line %d: error on popen()", __FILE__, __LINE__);
        
        char *new_line = fgetline(fp);
        if (fp == 0)
            error("File %s, line %d: couldn't read from popen pipe",
                  __FILE__, __LINE__);
        
        (void)pclose(fp);
        
        dl->currLine()->update(new_line);

        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else {
            move_cursor(dl->savedYPos(), 0);
            clear_to_end_of_line();
            display_string(new_line);
        }

    }
    else
        message("`chmod' failed");
    
    delete command;
}

/*
** chmod_current_file - attempt to chmod the current file
*/

static void chmod_current_file(DirList *dl)
{
    char *file = get_file_name(dl);
    
    char *mode = prompt("Change to Mode: ");
    
    if (mode == 0) {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete mode;
        delete file;
        return;
    }
    else if (*mode == 0) {
        message("not a valid file mode");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete mode;
        delete file;
        return;
    }
    
    chmod_file(dl, file, mode);
    
    delete mode;
    delete file;
    
    move_cursor(dl->savedYPos(), dl->savedXPos());
    synch_display();
}

/*
** print_current_file - attempt to print current file
*/

static void print_current_file(DirList *dl)
{
    char *file = get_file_name(dl);
    
    char *printer = getenv("DIREDPRT");
    if (printer == 0) printer = "lpr";
    
    char *cmd = new char[strlen(printer)+strlen(file)+4];
    (void)strcpy(cmd, printer);
    (void)strcat(cmd, " '");
    (void)strcat(cmd, file);
    (void)strcat(cmd, "'");
    
    if (is_regular_file(file))
        (void)system(cmd);
    else
        message("can only print regular files");
    
    move_cursor(dl->savedYPos(), dl->savedXPos());
    synch_display();
    
    delete file;
    delete printer;
    delete cmd;
}

/*
** uncompress_file - uncompress the file at line y in the DirList.
**                   Update the window appropriately.
*/

static void uncompress_file(DirList *dl, const char *file)
{
    message("Uncompressing ... ");
    
    char *command = new char[strlen(file)+33];
    (void)strcpy(command, "uncompress -f ");
    (void)strcat(command, "'");
    (void)strcat(command, file);
    (void)strcat(command, "' >/dev/null 2>&1");

    if (!system(command)) {
        
        char *dot = strrchr(file, '.');
        *dot = 0;  // remove ".Z' suffix
        (void)strcpy(command, ls_cmd[the_sort_order()]);
        (void)strcat(command, "'");
        (void)strcat(command, file);
        (void)strcat(command, "'");
        
        FILE *fp = popen(command, "r");
        if (fp == 0)
            error("File %s, line %d: error on popen()", __FILE__, __LINE__);
        
        char *new_line = fgetline(fp);
        if (fp == 0)
            error("File %s, line %d: couldn't read from popen pipe",
                  __FILE__, __LINE__);
        
        (void)pclose(fp);

        dl->currLine()->update(new_line);

        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else {
            move_cursor(dl->savedYPos(), 0);
            clear_to_end_of_line();
            display_string(new_line);
        }

        message("Uncompressing ... done");
    }
    else
        message("`uncompress' failed");

    delete command;
}

/*
** uncompress_current_file - attempt to uncompress current file
*/

static void uncompress_current_file(DirList *dl)
{
    char *file = get_file_name(dl);
    
    if (file[strlen(file)-1] != 'Z' && file[strlen(file)-1] != '.') {
        message("can only `uncompress' compressed files");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        return;
    }
    
    uncompress_file(dl, file);
    
    move_cursor(dl->savedYPos(), dl->savedXPos());
    synch_display();
    
    delete file;
}

/*
** search - search through the line list, starting with the given DirLine,
**          for a line whose filename contains a match for `str'.  The
**          matching is strictly a string match using `strstr', not a
**          regex match.  Returns the first matching line found, or 0
**          if none found.  `pos' is a pointer to an int containing
**          the position in the window of `ln'.  On exit, it contains the
**          position of the matching line, relative to the starting position.
**          If `forward' is nonzero (the default), we search forward,
**          otherwise we search backwards.  If str is NULL, we redo the
**          previous search.
*/

static DirLine *search(DirLine *ln, const char *str, int *pos, char forward=1)
{
    static char *saved_str = 0;
    const char *name;

    if (*str) {
        //
        // save copy of `str' so we can redo searches
        //
        delete saved_str;
        saved_str = new char[sizeof(str)+1];
        (void)strcpy(saved_str, str);
    }
    else if (saved_str)
        //
        // we have a previous search string
        //
        str = saved_str;
    else
        return 0;

    while (ln) {
#ifndef NO_SYMLINKS
        if ((name = strstr(ln->line(), " -> ")) != 0)
            //
            // we have a symbolic link
            //
            --name;
        else
#endif
            name = ln->line() + ln->length();

        while (!isspace(*name)) --name;

        if (strstr(name, str)) return ln;

        if (forward) {
            ln = ln->next();
            (*pos)++;
        }
        else {
            ln = ln->prev();
            (*pos)--;
        }
    }

    return 0;
}

/*
** search_forward - search forward for file or directory matching string.
**                  The search always starts with the file following
**                  the current file.  If a match occurs, the cursor
**                  is placed on the new current file, with that line
**                  placed at the top of the window.
*/

static void search_forward(DirList *dl)
{
    char *str = prompt("Search forward for: ");
    if (str == 0) {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete str;
        return;
    }                  
    
    DirLine *ln = dl->currLine();
    if (ln == dl->tail()) {
        message("no match");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        return;
    }
    
    int pos = dl->savedYPos()+1;
    DirLine *found = search(ln->next(), str, &pos);
    if (found) {
        if (dl->currLine()->length() > columns())
            rightshift_current_line(dl);
        dl->setCurrLine(found);
    }
    
    if (found && pos < rows()-2) {
        //
        // We found a match in our current window.
        // Only need to update the cursor position.
        //
        dl->saveYXPos(pos, goal_column(dl));
        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else
            move_cursor(pos, dl->savedXPos());
    }
    else if (found) {
        //
        // We found a match, but it is out of our
        // current view.  Place the matched line at the top
        // of the window.
        //
        clear_listing();
        ln = found;
        dl->setFirst(ln);
        for (int i = 0; i < rows()-2 && ln; i++, ln = ln->next())
            display_string(ln->line(), ln->length());

        ln ? dl->setLast(ln->prev()) : dl->setLast(dl->tail());
        
        dl->saveYXPos(0, goal_column(dl));

        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else
            move_cursor(0, dl->savedXPos());
    }
    else {
        message("no match");
        move_cursor(dl->savedYPos(), dl->savedXPos());
    }
    
    synch_display();
    
    delete str;
}

/*
** search_backward - search backward for file or directory matching string.
**                   The search always starts with the file following
**                   the current file.  If a match occurs, the cursor
**                   is placed on the new current file, with that line
**                   placed at the top of the window.
*/

static void search_backward(DirList *dl)
{
    char *str = prompt("Search backward for: ");
    if (str == 0) {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        
        delete str;
        return;
    }
    
    DirLine *ln = dl->currLine();
    if (ln == dl->head()) return;
    
    int pos = dl->savedYPos()-1;
    DirLine *found = search(ln->prev(), str, &pos, 0);
    if (found) {
        if (dl->currLine()->length() > columns())
            rightshift_current_line(dl);
        dl->setCurrLine(found);
    }
    
    if (found && pos >= 0) {
        //
        // We found a match in our current window.
        // Only need to update the cursor position.
        //
        dl->saveYXPos(pos, goal_column(dl));
        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else
            move_cursor(pos, dl->savedXPos());
    }
    else if (found) {
        //
        // We found a match, but it is out of our
        // current view.  Place the matched line at the top
        // of the window.
        //
        clear_listing();
        dl->setFirst(ln = found);
        for (int i = 0; i < rows()-2 && ln->next() != 0; i++, ln = ln->next())
            display_string(ln->line(), ln->length());
        
        if (i != rows()-2) {
            //
            // We hit last line before outputing all that we could.
            // Must output lastLine() == tail().
            //
            display_string(ln->line(), ln->length());
            dl->setLast(ln);
        }
        else
            dl->setLast(ln->prev());
        
        dl->saveYXPos(0, goal_column(dl));
        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else
            move_cursor(0, dl->savedXPos());
    }
    else {
        message("no match");
        move_cursor(dl->savedYPos(), dl->savedXPos());
    }
    
    synch_display();
    
    delete str;
}

/*
** help - give some help
*/

static void help()
{
    clear_modeline();
    int center = columns()/2 - 3;
    enter_standout_mode();
    for (int i = 0; i < center; i++) putchar('-');
    (void)printf(" HELP ");
    for (i = center+6; i < columns(); i++) putchar('-');
    end_standout_mode();
    
    int position = 0;
    do {
        int quit_now = 0, i = position;
        
        clear_listing();
        for (i = position; i < rows()-2+position && i < HELP_FILE_DIM; i++)
            display_string(help_file[i]);
        position += rows()-2;
        
        clear_message_line();
        if (position < HELP_FILE_DIM-1)
            (void)printf(HELP_MSG[0]);
        else
            (void)printf(HELP_MSG[1]);
        
        synch_display();
        
        switch (getchar()) {
          default:
            quit_now = 1;
            // fall through
          case KEY_SPC:
            break;
        }
        if (quit_now) break;
    }
    while (position < HELP_FILE_DIM-1);
    
    redisplay();
}

/*
** reread_current_directory - reread the current directory and
**                            put it up fresh on the screen.  We
**                            attempt to put the cursor back onto
**                            the previous current file, if that file
**                            still exists.
*/

static void reread_current_directory(DirList **dlp)
{
    char *dirname = new char[strlen((*dlp)->name())+1];
    (void)strcpy(dirname, (*dlp)->name());
    char *old_current_file = get_file_name(*dlp);

    int old_pos;  // position in old DirList of old_current_file
    DirLine *ln = (*dlp)->head();
    for (old_pos = 0; ln != (*dlp)->currLine(); ln = ln->next(), old_pos++) ;

    delete dir_stack->pop();
    
    DirList *dl = get_directory_listing(dirname);
    if (dl == 0)
        error("File %s, line %d: couldn't read directory `%s'",
              __FILE__, __LINE__, dirname);
    
    *dlp = dl;  // update 'dir_list' in read_commands()

    dir_stack->push(dl);
    
    clear_listing();

    ln = dl->head();
    int pos = 0;
    const char *name;
    int namelen = 0;
    while (ln) {

#ifndef NO_SYMLINKS
        if ((name = strstr(ln->line(), " -> ")) != 0)
            //
            // we have a symbolic link
            //
            --name;
        else
#endif
            name = ln->line() + ln->length();

        for (namelen = 0; isspace(*name) == 0; name--, namelen++) ;

        if (strncmp(name+1, old_current_file, namelen) == 0) break;

        ln = ln->next();
        pos++;
        namelen = 0;
    }

    if (ln == 0) {
        //
        // Update `ln' and `pos' to refer to a suitable file on
        // which to place cursor  .
        //
        ln = dl->head();
        for (pos = 0; pos < old_pos && ln; ln = ln->next(), pos++) ;
    }

    dl->setCurrLine(ln);

    if (pos < rows()-2) {
        //
        // Display starting at the head.
        //
        dl->setFirst(ln = dl->head());
        dl->saveYXPos(pos, goal_column(dl));
    }
    else {
        //
        // We center dl->currLine() at (rows()-1)/2.
        //
        ln = dl->currLine();
        for (int i = 0; i < (rows()-1)/2; i++, ln = ln->prev()) ;
        dl->setFirst(ln);
        dl->saveYXPos((rows()-1)/2, goal_column(dl));
    }

    for (int i = 0; i < rows()-2 && ln; ln = ln->next(), i++)
        display_string(ln->line(), ln->length());
    ln ? dl->setLast(ln->prev()) : dl->setLast(dl->tail());
    
    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(dl->savedYPos(), dl->savedXPos());
    
    synch_display();
}

/*
** expand_percent - expand any `%' signs in cmd to file.
**                  Return the result in new space.
*/

static char *expand_percent(const char *cmd, const char *file)
{
    char *expanded_cmd  = new char[strlen(cmd)+1];
    (void)strcpy(expanded_cmd, cmd);
    char *pos = strchr(expanded_cmd, '%');

    while (pos) {
        char *full_cmd = new char[strlen(expanded_cmd)+strlen(file)];
        (void)strncpy(full_cmd, expanded_cmd, pos-expanded_cmd);
        full_cmd[pos-expanded_cmd] = '\0';
        (void)strcat(full_cmd, file);
        (void)strcat(full_cmd, pos+1);
        delete expanded_cmd;
        expanded_cmd = full_cmd;
        pos = strchr(expanded_cmd, '%');
    }

    return expanded_cmd;
}


/*
** shell_command - execute a shell command.  `%' is expanded to
**                 the current filename.
**                 If *cmd == '\0', start up a shell.
**                 If *cmd == '!', reexecute most recent shell command.
*/

static void shell_command(DirList *dl)
{
    static char *original_cmd = 0;
    static char *saved_cmd    = 0;
    static char *saved_shell  = 0;
    char *file = get_file_name(dl);
    char *cmd  = prompt("!");

    if (cmd == 0) {
        delete file;
        return;
    }
    else if (*cmd == '\0') {
        //
        // start up a shell
        //
        if (!saved_shell) saved_shell = getenv("SHELL");
        if (saved_cmd) delete saved_cmd, delete original_cmd;
        if (!saved_shell) {
            char *shell = "sh";
            saved_shell = new char[strlen(shell)+1];
            (void)strcpy(saved_shell, shell);
        }
        saved_cmd    = new char[strlen(saved_shell)+1];
        original_cmd = new char[strlen(saved_shell)+1];
        (void)strcpy(saved_cmd,    saved_shell);
        (void)strcpy(original_cmd, saved_shell);
        message("Starting interactive shell ...");
        cursor_wrap();
        synch_display();
        exec_command(saved_shell, 0);
        delete cmd;
    }
    else if (*cmd == '[') {
        //
        // Re-expand previous original command, if it contains a `%'.
        // Otherwise, re-execute previous saved command since the original
        // and saved will be the same.
        //
        if (original_cmd) {
            //
            // expand the `%'
            //
            delete saved_cmd;
            saved_cmd = expand_percent(original_cmd, file);
            message(saved_cmd);
            cursor_wrap();
            synch_display();
            exec_command(saved_cmd);
            delete cmd;
        }
        else {
            message("No Previous Shell Command");
            move_cursor(dl->savedYPos(), dl->savedXPos());
            synch_display();
            delete cmd;
            delete file;
            return;
        }
    }
    else if (*cmd == '!') {
        //
        //  re-execute previously saved command
        //
        if (saved_cmd) {
            message(saved_cmd);
            cursor_wrap();
            synch_display();
            exec_command(saved_cmd);
            delete cmd;
        }
        else {
            message("No Previous Shell Command");
            move_cursor(dl->savedYPos(), dl->savedXPos());
            synch_display();
            delete cmd;
            delete file;
            return;
        }
    }
    else {
        //
        // expand and then execute command
        //
        if (saved_cmd) {
            delete saved_cmd;
            delete original_cmd;
        }
        original_cmd = cmd;
        saved_cmd    = expand_percent(original_cmd, file);
        message(saved_cmd);
        cursor_wrap();
        synch_display();
        exec_command(saved_cmd);
    }

    delete file;
    redisplay();
}

/*
** change_sort_order - query the user for a new sorting order - expects to
**                     read a single character - and change the sort order
**                     to that value, if it is a valid value.
*/
static void change_sort_order(DirList *dl)
{
    message("Set sort order to? [a, c, t, u]: ");

    switch(get_key(dl)) {
      case 'a':
        set_sort_order(ALPHABETICALLY);
        message("Sorting `alphabetically' now.");
        break;
      case 'c':
        set_sort_order(INODE_CHANGE_TIME);
        message("Sorting by `inode-change time' now.");
        break;
      case 't':
        set_sort_order(MODIFICATION_TIME);
        message("Sorting by `modification time' now.");
        break;
      case 'u':
        set_sort_order(ACCESS_TIME);
        message("Sorting by `access time' now.");
        break;
      default:
        message("Sort order not changed.");
        break;
    }

    move_cursor(dl->savedYPos(), dl->savedXPos());
    synch_display();
}

/*
** read_commands - the command loop
*/
void read_commands(DirList *dir_list)
{
    int key;
    for (;;) {
        switch (key = get_key(dir_list)) {
          case KEY_j    :
          case KEY_n    :
          case KEY_CTL_N:
          case KEY_SPC  :
          case KEY_CR   :

              if (dir_list->atEndOfList()) break;

              if (dir_list->currLine()->length() > columns()) 
                  rightshift_current_line(dir_list);
    
              if (dir_list->savedYPos() < rows()-3) {
                  //
                  // There are still more lines below us in the window
                  // so we just mOve the cursor down one line.
                  //
                  dir_list->setCurrLine(dir_list->currLine()->next());
                  int x = goal_column(dir_list);
                  if (x == dir_list->savedXPos())
                      cursor_down();
                  else
                      move_cursor(dir_list->savedYPos()+1, x);
                  dir_list->saveYXPos(dir_list->savedYPos()+1, x);
              }
              else
                  //
                  // We are on the last line on the screen and there
                  // are more lines to display.  Scroll up one line
                  // and leave the cursor on the next logical line.
                  //
                  scroll_up_one_line(dir_list);

              if (dir_list->currLine()->length() > columns())
                  leftshift_current_line(dir_list);

              synch_display();
              break;

          case KEY_k    :
          case KEY_p    :
          case KEY_CTL_P:
          case KEY_CTL_Y:

              if (dir_list->atBegOfList()) break;

              if (dir_list->currLine()->length() > columns())
                  rightshift_current_line(dir_list);

              if (dir_list->savedYPos() != 0) {
                  //
                  // We aren't at the top of the window so can mOve up.
                  //
                  dir_list->setCurrLine(dir_list->currLine()->prev());
                  int x = goal_column(dir_list);
                  if (x == dir_list->savedXPos() && UP)
                      cursor_up();
                  else
                      move_cursor(dir_list->savedYPos()-1, x);
                  dir_list->saveYXPos(dir_list->savedYPos()-1, x);
              }
              else
                  //
                  // We are on the first line of the window and there are
                  // lines preceding us in the directory listing.
                  //
                  scroll_down_one_line(dir_list);

              if (dir_list->currLine()->length() > columns())
                  leftshift_current_line(dir_list);

              synch_display();
              break;

          case KEY_CTL_F:
          case KEY_CTL_V:

              if (dir_list->lastLine() == dir_list->tail()) break;
              scroll_up_full_window(dir_list);

              break;
          case KEY_b    :
          case KEY_CTL_B:

              if (dir_list->firstLine() == dir_list->head()) break;
              scroll_down_full_window(dir_list);

              break;
          case KEY_CTL_D:

              if (dir_list->lastLine() == dir_list->tail()) break;
              scroll_up_half_window(dir_list, dir_list->savedYPos());

              break;
          case KEY_CTL_U:

              if (dir_list->firstLine() == dir_list->head()) break;
              scroll_down_half_window(dir_list, dir_list->savedYPos());

              break;
          case KEY_TOP:

              if (dir_list->atBegOfList()) break;
              goto_first(dir_list);

              break;
          case KEY_BOT:

              if (dir_list->atEndOfList()) break;
              goto_last(dir_list);

              break;
          case KEY_e:
          case KEY_f:

              edit_current_file(dir_list); break;

          case KEY_m:
          case KEY_v:

              page_current_file(dir_list); break;

          case KEY_c:

              copy_current_file(dir_list); break;

          case KEY_d:
          case KEY_D:

              delete_current_file(dir_list, key); break;

          case KEY_r:

              rename_current_file(dir_list); break;

          case KEY_C:

              compress_current_file(dir_list); break;

          case KEY_E:

              edit_prompted_for_directory(dir_list); break;

          case KEY_G:

              chgrp_current_file(dir_list); break;

          case KEY_QM:
          case KEY_H :

              help(); break;

          case KEY_L:

              link_current_file(dir_list); break;

          case KEY_M:

              chmod_current_file(dir_list); break;

          case KEY_O:

              change_sort_order(dir_list); break;

          case KEY_P:

              print_current_file(dir_list); break;

          case KEY_g:
          case KEY_R:

              reread_current_directory(&dir_list); break;

#ifndef NO_SYMLINKS
          case KEY_S:

              symlink_current_file(dir_list); break;
#endif

          case KEY_U:

              uncompress_current_file(dir_list); break;

          case KEY_SLASH:

              search_forward(dir_list); break;

          case KEY_BKSLASH:

              search_backward(dir_list); break;

          case KEY_BANG:

              shell_command(dir_list); break;

          case KEY_V:

              message(version);
              move_cursor(dir_list->savedYPos(), dir_list->savedXPos());
              synch_display();
              break;

          case KEY_CTL_L:

              redisplay(); break;

          case KEY_q:

              if (dir_stack->nelems() > 1) {
                  delete dir_stack->pop();
                  dir_list = dir_stack->top();
                  //
                  // update our current directory
                  //
                  if (chdir(dir_list->name()) < 0)
                      error("File %s, line %d: couldn't chdir() to `%s'",
                            __FILE__, __LINE__, dir_list->name());
                  /*
                   * We track the CWD and PWD variables if they're defined,
                   * so that applications, such as emacs, which use them
                   * will work properly.
                   */
                  if (getenv("CWD")) {
                      char *str = new char[strlen(dir_list->name())+6];
                      (void)strcpy(str, "CWD=");
                      (void)strcat(str, dir_list->name());
                      if (putenv(str) < 0)
                          error("File %s, line %d: putenv(%s) failed.",
                                __FILE__, __LINE__, dir_list->name());
                  }
                  
                  if (getenv("PWD")) {
                      char *str = new char[strlen(dir_list->name())+6];
                      (void)strcpy(str, "PWD=");
                      (void)strcat(str, dir_list->name());
                      if (putenv(str) < 0)
                          error("File %s, line %d: putenv(%s) failed.",
                                __FILE__, __LINE__, dir_list->name());
                  }

                  redisplay();
              }
              else {
                  term_display();
                  exit(0);
              }

              break;

          case KEY_ESC:

              /*
              ** some Emacs ESC key bindings
              */

              switch(getchar()) {
                case KEY_v:
                  if (dir_list->firstLine() == dir_list->head()) break;
                  scroll_down_full_window(dir_list);
                  break;
                case KEY_TOP:
                  if (dir_list->atBegOfList()) break;
                  goto_first(dir_list);
                  break;
                case KEY_BOT:
                  if (dir_list->atEndOfList()) break;
                  goto_last(dir_list);
                  break;
              }

          break;

          case KEY_Q:

              term_display(); exit(0); break;
        }
    }
}
