/**********************************************************************/
/*   Functions for manipulating the accumulator.		      */
/**********************************************************************/
# include	"list.h"

/**********************************************************************/
/*   Size to round up to when expanding accumulator memory objects.   */
/**********************************************************************/
# define	BOUNDARY_SIZE	0x3ff

/**********************************************************************/
/*   Following  data  structures  used  to hold the current value of  */
/*   the  accumulator.  We keep old values of numeric/memory objects  */
/*   lying  around.  This  is  done  mostly  so that if we put large  */
/*   things  in  the accumulator, we keep the memory because chances  */
/*   are   we'll   need   a  big  memory  block  again,  so  we  can  */
/*   dynamically expand the accumulator size, but never contract it.  */
/*   								      */
/*   The following accumulator types are defined:		      */
/*   								      */
/*   F_LIT      accumulator points to a literal string which can      */
/*   		never be freed. 				      */
/*   								      */
/*   F_STR      accumulator points to a string which we need to take  */
/*   		a private copy of.      			      */
/*   								      */
/*   F_RSTR     accumulator points to a reference string.             */
/*   								      */
/*   F_LIST     accumulator points to a list. (We copy list, but      */
/*   		increment all ref-string pointers).     	      */
/*   								      */
/*   F_INT      accumulator contains a 32-bit number.   	      */
/*   								      */
/*   F_FLOAT    Accumulator contains a floating point number.         */
/**********************************************************************/
typedef struct acc_t {
	OPCODE  	type;	/* Current type of accumulator.		*/
	unsigned int 	length; /* Length of composite object.		*/
	
				/* Union for numeric objects.		*/
	char		*memptr;
	union n {
		long	_int_val;	/* F_INT			*/
		double	_float_val;	/* F_FLOAT			*/
		} n;

	char	*str_val;
	ref_t	*ref_val;
	char	*lit_ptr;
	} acc_t;
# define	int_val		n._int_val
# define	float_val	n._float_val

static acc_t	accum = {F_INT};

void	clear_acc PROTO((void));
void	expand_acc PROTO((int));
/**********************************************************************/
/*   Function  to  assign  a  string  to  the accumulator. If len is  */
/*   less  than  zero,  then  we  need  to perform a strlen() on the  */
/*   string to see how long it is.				      */
/**********************************************************************/
void
acc_assign_str(str, len)
char	*str;
int	len;
{

	/***********************************************/
	/*   Free   any  memory  in  the  accumulator  */
	/*   before assigning the new value.	       */
	/***********************************************/
	clear_acc();

	if (len < 0)
		len = strlen(str);
	expand_acc(len+1);
	accum.str_val = accum.memptr;
	memcpy(accum.str_val, str, len);
	accum.str_val[len] = NULL;
	
	/***********************************************/
	/*   Mark accumulator as containing a string.  */
	/***********************************************/
	accum.type = F_STR;
}
/**********************************************************************/
/*   Assign  a  literal  string  (one  that  cannot  be  modified or  */
/*   freed) to the accumulator.					      */
/**********************************************************************/
void
acc_assign_lit(str)
char	*str;
{
	clear_acc();
	accum.lit_ptr = str;
	accum.type = F_LIT;
}
/**********************************************************************/
/*   Function to assign an argv element to the accumulator.	      */
/**********************************************************************/
void
acc_assign_argv(lvp)
LISTV	*lvp;
{
	switch (lvp->l_flags) {
	  case F_INT:
	  	acc_assign_int(lvp->l_int);
		break;
	  case F_FLOAT:
	  	acc_assign_int(lvp->l_float);
		break;
	  case F_RLIST:
	  case F_RSTR:
	  	acc_assign_ref(lvp->l_ref);
		break;
	  case F_STR:
	  case F_LIT:
	  	acc_assign_str(lvp->l_str, -1);
		break;
	  case F_NULL:
	  	acc_assign_null();
		break;
	  default:
	  	panic("acc_assign_argv: what type?");
	  }
}
/**********************************************************************/
/*   Function   to   make   sure   we  have  enough  memory  in  the  */
/*   accumulator  memory  block  for  the  object  we  are  about to  */
/*   assign to it.						      */
/**********************************************************************/
void
expand_acc(len)
int	len;
{
	/***********************************************/
	/*   If  current  string  we're  holding onto  */
	/*   is  too  small  for the new string, then  */
	/*   expand it.				       */
	/***********************************************/
	if (len > accum.length) {
		if (accum.length)
			chk_free(accum.memptr);
		accum.length = (len | BOUNDARY_SIZE) + 1;
		accum.memptr = (char *) chk_alloc(accum.length);
		}
}
/**********************************************************************/
/*   Clear memory in accumulator before assigning new value.	      */
/**********************************************************************/
void
clear_acc()
{

	switch (accum.type) {
	  case F_LIT:
	  case F_INT:
	  case F_FLOAT:
	  case F_STR:
	  case F_NULL:
		/***********************************************/
		/*   No memory to free for these.	       */
		/***********************************************/
	  	break;
		
	  case F_LIST:
	  case F_RLIST:
	  case F_RSTR:
	  	r_dec(accum.ref_val);
		break;

	  default:
	  	panic("Bad accumulator type.");
	  }
}
/**********************************************************************/
/*   Function to assign a reference pointer to the accumulator.	      */
/**********************************************************************/
void
acc_assign_ref(rp)
ref_t	*rp;
{
	clear_acc();
	accum.ref_val = r_inc(rp);
	accum.type = rp->r_type == F_LIST ? F_RLIST : F_RSTR;
}
/**********************************************************************/
/*   Function to assign a list to the accumulator.		      */
/**********************************************************************/
void
acc_assign_list(lp, len)
LIST	*lp;
int	len;
{
	clear_acc();
	accum.ref_val = r_init(F_LIST, (char *) copy_list(lp, 0), len);
	accum.type = F_RLIST;
}
/**********************************************************************/
/*   Function  to  assign  a  newly created list to the accumulator.  */
/*   We  are  given  the  list so we don't need to allocate any more  */
/*   memory to it.						      */
/**********************************************************************/
void
acc_donate_list(lp, len)
LIST	*lp;
int	len;
{
	clear_acc();
	accum.ref_val = r_init(F_RLIST, (char *) lp, len);
	accum.type = F_RLIST;
}
/**********************************************************************/
/*   Function to assign the NULL value to the accumulator.	      */
/**********************************************************************/
void
acc_assign_null()
{
	clear_acc();
	accum.type = F_NULL;
}
/**********************************************************************/
/*   Following   function   used   to   assign  an  integer  to  the  */
/*   accumulator.						      */
/**********************************************************************/
void
acc_assign_int(val)
long	val;
{

	/***********************************************/
	/*   Lose  any  references  to strings in the  */
	/*   accumulator before changing its type.     */
	/***********************************************/
	clear_acc();

	accum.type = F_INT;
	accum.int_val = val;
}
/**********************************************************************/
/*   Following function used to assign a float to the accumulator.    */
/**********************************************************************/
void
acc_assign_float(val)
double	val;
{

	/***********************************************/
	/*   Lose  any  references  to strings in the  */
	/*   accumulator before changing its type.     */
	/***********************************************/
	clear_acc();

	accum.type = F_FLOAT;
	accum.float_val = val;
}

/**********************************************************************/
/*   Function to return accumulator float value.		      */
/**********************************************************************/
double
acc_get_fval()
{
	return accum.float_val;
}
/**********************************************************************/
/*   Function to return accumulator integer value.		      */
/**********************************************************************/
long
acc_get_ival()
{
	return accum.int_val;
}

/**********************************************************************/
/*   Function  to  return  pointer to string in accumulator, or NULL  */
/*   if we dont have a string there.				      */
/**********************************************************************/
char *
acc_get_sval()
{
	switch (accum.type) {
	  case F_STR:
	  	return accum.str_val;
	  case F_RSTR:
	  	return accum.ref_val->r_ptr;
	  case F_LIT:
	  	return accum.lit_ptr;
	  default:
	  	return NULL;
	  }
	return NULL;
}

/**********************************************************************/
/*   Function to retrieve pointer to list stored in the accumulator.  */
/**********************************************************************/
ref_t *
acc_get_list()
{
	switch (accum.type) {
	  case F_LIST:
	  case F_RLIST:
		return accum.ref_val;
	  default:
		return NULL;
	  }
}
/**********************************************************************/
/*   Return   pointer   to   the   reference  value  stored  in  the  */
/*   accumulator.						      */
/**********************************************************************/
ref_t *
acc_get_ref()
{
	return accum.ref_val;
}
/**********************************************************************/
/*   Function to return type of variable stored in accumulator.	      */
/**********************************************************************/
OPCODE
acc_get_type()
{
	return accum.type;
}
/**********************************************************************/
/*   Trace assignments to accumulator.				      */
/**********************************************************************/
void
acc_trace()
{	extern int dflag;
/*static acc_t	old_acc;*/
	/***********************************************/
	/*   Only   trace  accumulator  if  debugging  */
	/*   turned on and its value has changed.      */
	/***********************************************/
	if (dflag == 0)
		return;
/*	if (memcmp(&accum, &old_acc, sizeof accum) == 0)
		return;
	old_acc = accum;*/

	switch (accum.type) {
	  case F_INT:
	  	trace_ilog("  iACC=%ld\n", accum.int_val);
		break;
	  case F_FLOAT:
	  	trace_ilog("  fACC=%g\n", accum.float_val);
		break;
	  case F_LIT:
		trace_ilog("  litACC='%s'\n", c_string(accum.lit_ptr));
		break;
	  case F_STR:
		trace_ilog("  sACC='%s'\n", c_string(accum.str_val));
		break;
	  case F_RSTR:
		trace_ilog("  rsACC='%s'\n", c_string(accum.ref_val->r_ptr));
		break;
	  case F_RLIST:
	  case F_LIST:
		trace_ilog("  lACC=");
		trace_list((LIST *) accum.ref_val->r_ptr);
		break;
	  case F_NULL:
	  	trace_ilog("  NULL\n");
		break;
	  default:
	  	panic("What accum type ?");
	  }
}
