#include <stdio.h>
#include <string.h>
#include <malloc.h>

void *vmalloc();
void vfree();
void vverify();
void *vrealloc();
void *vcalloc();
void vdump();

/* maximum number of buffers simultaneously in use */
#define MAXMALLOCS 1000
#define KP 0xaa     /* memory fill pattern */
#define KPW 2	    /* memory fill pattern width */

static void trace();
static void dumpbuf();

static int nummallocs = 0;
static char s[80];
struct mtype {
    unsigned char *addr;
    unsigned size;
    };
static struct mtype m[MAXMALLOCS];

/*---------------------------------- vverify --------------------------------*/

#ifdef __LINE__
void vverify(f,l)
char *f;
int l;
#else
void vverify(id)
char *id;
#endif
{
    int x;
    unsigned int c;
    /* verify the entire malloc heap */
    for (x=0; x<nummallocs; ++x)
	if (m[x].addr != NULL)
		for (c=0; c<KPW; ++c) {
		if (*(m[x].addr+c) != KP ||
		    *(m[x].addr + m[x].size + KPW + c) != KP) {
#ifdef __LINE__
#  ifdef MSDOS
#    ifdef __TURBOC__
		    sprintf(s, "%s.%d ERROR: malloc area %p corrupted\n",
			    f, l, m[x].addr);
#    else
		    sprintf(s, "%s.%d ERROR: malloc area %p corrupted\n",
			    f, l, (void far *) m[x].addr);
#    endif
#  else
		    sprintf(s, "%s.%d ERROR: malloc area %04.4lx corrupted\n",
			    f, l, m[x].addr);
#  endif
#else
#  ifdef MSDOS
#    ifdef __TURBOC__
		    sprintf(s, "ERROR: malloc area %p corrupted\n", m[x].addr);
#    else
		    sprintf(s, "ERROR: malloc area %p corrupted\n",
			    (void far *) m[x].addr);
#    endif
#  else
		    sprintf(s, "ERROR: malloc area %04.4lx corrupted\n",
			    m[x].addr);
#  endif
#endif
		    trace(s);
		    fputs(s, stderr);
		    dumpbuf(x);
		    exit(1);
		    }
		}
}

/*------------------------------- vmalloc -----------------------------------*/

#ifdef __LINE__
void *vmalloc(size, f, l)
unsigned int size;
char *f;
int l;
#else
void *vmalloc(size)
unsigned int size;
#endif
{
    char *buffer;
    int c, x;
#ifdef __LINE__
    vverify(f,l);
#else
    vverify("vmalloc");
#endif
    if ((buffer = calloc(size + KPW + KPW, 1)) == NULL) {
#ifdef __LINE__
	sprintf(s, "%s.%d ERROR: calloc returned NULL\n", f, l);
#else
	sprintf(s, "ERROR: calloc returned NULL\n");
#endif
	fputs(s, stderr);
	trace(s);
	exit(3);
	}
#ifdef __LINE__
#  ifdef MSDOS
#    ifdef __TURBOC__
    sprintf(s,"%s.%d %p:vmalloc  size = %u\n", f, l,
	    buffer, size);
#    else
    sprintf(s,"%s.%d %p:vmalloc  size = %u\n", f, l,
	    (void far *)buffer, size);
#    endif
#  else
    sprintf(s,"%s.%d %04.4lx:vmalloc  size = %ld\n", f, l,
	    (long)buffer, (long)size);
#  endif
#else
#  ifdef MSDOS
#    ifdef __TURBOC__
    sprintf(s,"%p:vmalloc  size = %u\n", buffer, size);
#    else
    sprintf(s,"%p:vmalloc  size = %u\n", (void far *)buffer, size);
#    endif
#  else
    sprintf(s,"%s.%d %04.4lx:vmalloc  size = %ld\n", f, l,
	    (long)buffer, (long)size);
#  endif
#endif
    trace(s);
    /* find a place for an entry in m */
    for (x=0; x<MAXMALLOCS && m[x].addr!=NULL; ++x)
	;
    if (x == MAXMALLOCS) {
#ifdef __LINE__
	sprintf(s, "%s.%d ERROR: too many mallocs\n", f, l);
#else
	sprintf(s, "ERROR: too many mallocs\n");
#endif
	fprintf(stderr, s);
	trace(s);
	exit(1);
	}
    m[x].addr = buffer;
    m[x].size = size;
    if (x == nummallocs)
	++nummallocs;
    for (c = 0; c < KPW; ++c) {
	*(m[x].addr + c) = KP;
	*(m[x].addr + m[x].size + KPW + c) = KP;
	}
	return(buffer + KPW);
}

/*-------------------------------- vcalloc ----------------------------------*/

#ifdef __LINE__
void *vcalloc(n, size, f, l)
char *f;
int l;
#else
void *vcalloc(n, size)
#endif
unsigned n, size;
{
    unsigned long t;
    t = ((KPW + KPW + 15 + ((unsigned long) n * (unsigned long) size))) >> 4;
    if ((t << 4) > 65535L) {
#ifdef __LINE__
	sprintf(s, "%s.%d ERROR: too big to calloc\n", f, l);
#else
	sprintf(s, "ERROR: too big to calloc\n");
#endif
	fprintf(stderr, s);
	trace(s);
	exit(1);
	}
#ifdef __LINE__
    return(memset(vmalloc(n*size, f, l), 0, n*size));
#else
    return(memset(vmalloc(n*size), 0, n*size));
#endif
}

/*--------------------------------- vfree -----------------------------------*/

#ifdef __LINE__
void vfree(buffer, f, l)
char *f;
int l;
#else
void vfree(buffer)
#endif
char *buffer;
{
    char *b;
    int x;
    b = buffer - KPW;
#ifdef __LINE__
    vverify(f,l);
#else
    vverify("free");
#endif
    for (x=0; x<nummallocs && m[x].addr != b; ++x)
	;
    if (x == nummallocs) {
#ifdef __LINE__
#  ifdef MSDOS
#    ifdef __TURBOC__
	sprintf(s, "%s.%d ERROR: location %p to free not in list\n", f, l, b);
#    else
	sprintf(s, "%s.%d ERROR: location %p to free not in list\n",
		f, l, (void far *) b);
#    endif
#  else
	sprintf(s, "%s.%d ERROR: location %04.4lx to free not in list\n",
		f, l, b);
#  endif
#else
#  ifdef MSDOS
#    ifdef __TURBOC__
	sprintf(s, "ERROR: location %p to free not in list\n", f, l, b);
#    else
	sprintf(s, "ERROR: location %p to free not in list\n",
		f, l, (void far *) b);
#    endif
#  else
	sprintf(s, "ERROR: location %04.4lx to free not in list\n", b);
#  endif
#endif
	fprintf(stderr, s);
	trace(s);
	exit(1);
	}
#ifdef __LINE__
#  ifdef MSDOS
#    ifdef __TURBOC__
    sprintf(s,"%s.%d %p:vfree\n", f, l, b);
#    else
    sprintf(s,"%s.%d %p:vfree\n", f, l, (void far *)b);
#    endif
#  else
    sprintf(s,"%s.%d %04.4lx:vfree\n", f, l, (long)b);
#  endif
#else
#  ifdef MSDOS
#    ifdef __TURBOC__
    sprintf(s,"%p:vfree\n", b);
#    else
    sprintf(s,"%p:vfree\n", (void far *)b);
#    endif
#  else
    sprintf(s,"%s.%d %04.4lx:vfree\n", (long)b);
#  endif
#endif
    trace(s);
    free(b);
    m[x].addr = NULL;
    if (x == nummallocs - 1)
	--nummallocs;
}

/*-------------------------------- vrealloc ---------------------------------*/

#ifdef __LINE__
void *vrealloc(buffer, size, f, l)
char *f;
int l;
#else
void *vrealloc(buffer, size)
#endif
char *buffer;
unsigned int size;
{
    char *b, *b2;
    int x;
    unsigned int c;
	b = buffer - KPW;
#ifdef __LINE__
    vverify(f,l);
#else
    vverify("vrealloc");
#endif
    for (x=0; x<nummallocs && m[x].addr!=b; ++x)
	;
    if (x == nummallocs) {
#ifdef __LINE__
#  ifdef MSDOS
#    ifdef __TURBOC__
	sprintf(s, "%s.%d ERROR: location %p to realloc not in list\n", f, l, b);
#    else
	sprintf(s, "%s.%d ERROR: location %p to realloc not in list\n",
		f, b, (void far *) b);
#    endif
#  else
	sprintf(s, "%s.%d ERROR: location %04.4lx to realloc not in list\n",
		f, l, b);
#  endif
#else
#  ifdef MSDOS
#    ifdef __TURBOC__
	sprintf(s, "ERROR: location %p to realloc not in list\n", b);
#    else
	sprintf(s, "ERROR: location %p to realloc not in list\n",
		(void far *) b);
#    endif
#  else
	sprintf(s, "ERROR: location %04.4lx to realloc not in list\n", b);
#  endif
#endif
	fprintf(stderr, s);
	trace(s);
	exit(3);
	}
#ifdef __LINE__
#  ifdef MSDOS
#    ifdef __TURBOC__
    sprintf(s,"%s.%d %p:vrealloc  size = %u\n", f, l, buffer, size);
#    else
    sprintf(s,"%s.%d %p:vrealloc  size = %u\n", f, l,
	    (void far *)b, size);
#    endif
#  else
    sprintf(s,"%s.%d %04.4lx:vrealloc  size = %ld\n", f, l,
	    (long)b, (long)size);
#  endif
#else
#  ifdef MSDOS
#    ifdef __TURBOC__
    sprintf(s,"%p:vrealloc  size = %u\n", b, size);
#    else
    sprintf(s,"%p:vrealloc  size = %u\n", (void far *)b, size);
#    endif
#  else
    sprintf(s,"%s.%d %04.4lx:vrealloc  size = %ld\n", f, l,
	    (long)b, (long)size);
#  endif
#endif
    trace(s);
    for (c=0; c<KPW; ++c)
	*(m[x].addr + m[x].size + KPW + c) = 0;
    if ((b2 = realloc(b, size + KPW + KPW)) == NULL) {
#ifdef __LINE__
	sprintf(s, "%s.%d ERROR: realloc returned NULL\n", f, l);
#else
	sprintf(s, "ERROR: realloc returned NULL\n");
#endif
	fputs(s, stderr);
	trace(s);
	exit(3);
	}
    m[x].addr = b2;
    m[x].size = size;
    for (c=0; c<KPW; ++c)
	*(m[x].addr + m[x].size + KPW + c) = KP;
    return(b2 + KPW);
}

/*--------------------------------- vdump -----------------------------------*/

#ifdef __LINE__
void vdump(f, l)
char *f;
int l;
#else
void vdump(id)
char *id;
#endif
{
    int x;
    /* dump the entire malloc heap to the vmalloc file */
#ifdef __LINE__
    sprintf(s, "========== Dump of malloc heap (%s.%d) ==========\n", f, l);
#else
    sprintf(s, "========== Dump of malloc heap (%s) ==========\n", id);
#endif
    trace(s);
    for (x=0; x<nummallocs; ++x)
	if (m[x].addr != NULL) {
#ifdef MSDOS
#  ifdef __TURBOC__
	    sprintf(s, "===== Malloc buffer addr : %p\n", m[x].addr);
#  else
	    sprintf(s, "===== Malloc buffer addr : %p\n",
		    (void far *) m[x].addr);
#  endif
#else
	    sprintf(s, "===== Malloc buffer addr : %04.4x\n", (long)m[x].addr);
#endif
	    trace(s);
#ifdef MSDOS
	    sprintf(s, "===== Malloc buffer size : %u\n",
#else
	    sprintf(s, "===== Malloc buffer size : %04x\n",
#endif
		    m[x].size + KPW + KPW);
	    trace(s);
	    dumpbuf(x);
	    }
}


static void trace(x)
char *x;
{
    static FILE *out = NULL;
    if (out == NULL) {
	unlink("vmalloc");
	out = fopen("vmalloc", "w");
	setbuf(out, NULL);
	}
    fputs(x, out);
}

static void dumpbuf(x)
int x;
{
    unsigned char *c;
    c = (unsigned char *)m[x].addr - 2;
    /* dump malloc buffer to the vmalloc file */
    while (c <= m[x].addr + m[x].size + KPW + KPW - 1) {
#ifdef MSDOS
#  ifdef __TURBOC__
	sprintf(s, "%p : %02x ", c, *c);
#  else
	sprintf(s, "%p : %02x ", (void far *)c, *c);
#  endif
#else
	sprintf(s, "%04.4lx : %02x ", (long)c, *c);
#endif
	if (c == m[x].addr)
	    strcat(s, "<= leading known pattern");
	if (c == m[x].addr + KPW)
	    strcat(s, "<= address of malloc buffer");
	if (c == m[x].addr + m[x].size + KPW)
	    strcat(s, "<= trailing known pattern");
	strcat(s, "\n");
	trace(s);
	++c;
	}
}

