/***********************************************************
Copyright 1993 by Ove Kalkan

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of Digital or MIT not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/

/*
 * Icon.c - Icon widget
 *
 */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xos.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/IconP.h>
#include <X11/Xmu/Converters.h>
#include <X11/Xmu/Drawing.h>
#include <stdio.h>
#include <ctype.h>

#define streq(a,b) (strcmp( (a), (b) ) == 0)

#define MULTI_LINE_LABEL 32767

#ifdef CRAY
#define WORD64
#endif

/****************************************************************
 *
 * Full class record constant
 *
 ****************************************************************/

/* Private Data */

#define offset(field) XtOffsetOf(IconRec, field)
static XtResource resources[] = {
    {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
	offset(icon.foreground), XtRString, XtDefaultForeground},
    {XtNfont,  XtCFont, XtRFontStruct, sizeof(XFontStruct *),
	offset(icon.font),XtRString, XtDefaultFont},
    {XtNicon,  XtCIcon, XtRString, sizeof(String),
	offset(icon.icon), XtRString, NULL},
    {XtNencoding, XtCEncoding, XtRUnsignedChar, sizeof(unsigned char),
	offset(icon.encoding), XtRImmediate, (XtPointer)XawTextEncoding8bit},
    {XtNjustify, XtCJustify, XtRJustify, sizeof(XtJustify),
	offset(icon.justify), XtRImmediate, (XtPointer)XtJustifyCenter},
    {XtNinternalWidth, XtCWidth, XtRDimension,  sizeof(Dimension),
	offset(icon.internal_width), XtRImmediate, (XtPointer)4},
    {XtNinternalHeight, XtCHeight, XtRDimension, sizeof(Dimension),
	offset(icon.internal_height), XtRImmediate, (XtPointer)2},
    {XtNleftBitmap, XtCLeftBitmap, XtRBitmap, sizeof(Pixmap),
       offset(icon.left_bitmap), XtRImmediate, (XtPointer) None},
    {XtNbitmap, XtCPixmap, XtRBitmap, sizeof(Pixmap),
	offset(icon.pixmap), XtRImmediate, (XtPointer)None},
    {XtNresize, XtCResize, XtRBoolean, sizeof(Boolean),
	offset(icon.resize), XtRImmediate, (XtPointer)True},
};
#undef offset

static void Initialize();
static void Resize();
static void Redisplay();
static Boolean SetValues();
static void ClassInitialize();
static void Destroy();
static XtGeometryResult QueryGeometry();

IconClassRec iconClassRec = {
  {
/* core_class fields */	
#define superclass		(&simpleClassRec)
    /* superclass	  	*/	(WidgetClass) superclass,
    /* class_name	  	*/	"Icon",
    /* widget_size	  	*/	sizeof(IconRec),
    /* class_initialize   	*/	ClassInitialize,
    /* class_part_initialize	*/	NULL,
    /* class_inited       	*/	FALSE,
    /* initialize	  	*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize		  	*/	XtInheritRealize,
    /* actions		  	*/	NULL,
    /* num_actions	  	*/	0,
    /* resources	  	*/	resources,
    /* num_resources	  	*/	XtNumber(resources),
    /* xrm_class	  	*/	NULLQUARK,
    /* compress_motion	  	*/	TRUE,
    /* compress_exposure  	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest	  	*/	FALSE,
    /* destroy		  	*/	Destroy,
    /* resize		  	*/	Resize,
    /* expose		  	*/	Redisplay,
    /* set_values	  	*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus	 	*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private   	*/	NULL,
    /* tm_table		   	*/	NULL,
    /* query_geometry		*/	QueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
/* Simple class fields initialization */
  {
    /* change_sensitive		*/	XtInheritChangeSensitive
  }
};
WidgetClass iconWidgetClass = (WidgetClass)&iconClassRec;
/****************************************************************
 *
 * Private Procedures
 *
 ****************************************************************/

static void ClassInitialize()
{
    XawInitializeWidgetSet();
    XtAddConverter( XtRString, XtRJustify, XmuCvtStringToJustify, NULL, 0 );
}

#ifndef WORD64

#define TXT16 XChar2b

#else

#define TXT16 char

static XChar2b *buf2b;
static int buf2blen = 0;

_XawIconWidth16(fs, str, n)
    XFontStruct *fs;
    char *str;
    int	n;
{
    int i;
    XChar2b *ptr;

    if (n > buf2blen) {
	buf2b = (XChar2b *)XtRealloc((char *)buf2b, n * sizeof(XChar2b));
	buf2blen = n;
    }
    for (ptr = buf2b, i = n; --i >= 0; ptr++) {
	ptr->byte1 = *str++;
	ptr->byte2 = *str++;
    }
    return XTextWidth16(fs, buf2b, n);
}

_XawIconDraw16(dpy, d, gc, x, y, str, n)
    Display *dpy;
    Drawable d;
    GC gc;
    int x, y;
    char *str;
    int n;
{
    int i;
    XChar2b *ptr;

    if (n > buf2blen) {
	buf2b = (XChar2b *)XtRealloc((char *)buf2b, n * sizeof(XChar2b));
	buf2blen = n;
    }
    for (ptr = buf2b, i = n; --i >= 0; ptr++) {
	ptr->byte1 = *str++;
	ptr->byte2 = *str++;
    }
    XDrawString16(dpy, d, gc, x, y, buf2b, n);
}

#define XTextWidth16 _XawIconWidth16
#define XDrawString16 _XawIconDraw16

#endif /* WORD64 */

/*
 * Calculate width and height of displayed text in pixels
 */

static void SetTextWidthAndHeight(lw)
    IconWidget lw;
{
    register XFontStruct	*fs = lw->icon.font;
    char *nl;

    if (lw->icon.pixmap != None) {
	Window root;
	int x, y;
	unsigned int width, height, bw, depth;
	if (XGetGeometry(XtDisplay(lw), lw->icon.pixmap, &root, &x, &y,
			 &width, &height, &bw, &depth)) {
	    lw->icon.icon_height = height;
	    lw->icon.icon_width = width;
	    lw->icon.icon_len = depth;
	    return;
	}
    }

    lw->icon.icon_height = fs->max_bounds.ascent + fs->max_bounds.descent;
    if (lw->icon.icon == NULL) {
	lw->icon.icon_len = 0;
	lw->icon.icon_width = 0;
    }
    else if ((nl = index(lw->icon.icon, '\n')) != NULL) {
	char *icon;
	lw->icon.icon_len = MULTI_LINE_LABEL;
	lw->icon.icon_width = 0;
	for (icon = lw->icon.icon; nl != NULL; nl = index(icon, '\n')) {
	    int width;

	    if (lw->icon.encoding)
		width = XTextWidth16(fs, (TXT16*)icon, (int)(nl - icon)/2);
	    else
		width = XTextWidth(fs, icon, (int)(nl - icon));
	    if (width > (int)lw->icon.icon_width)
		lw->icon.icon_width = width;
	    icon = nl + 1;
	    if (*icon)
		lw->icon.icon_height +=
		    fs->max_bounds.ascent + fs->max_bounds.descent;
	}
	if (*icon) {
	    int width = XTextWidth(fs, icon, strlen(icon));

	    if (lw->icon.encoding)
		width = XTextWidth16(fs, (TXT16*)icon, (int)strlen(icon)/2);
	    else
		width = XTextWidth(fs, icon, strlen(icon));
	    if (width > (int) lw->icon.icon_width)
		lw->icon.icon_width = width;
	}
    } else {
	lw->icon.icon_len = strlen(lw->icon.icon);
	if (lw->icon.encoding)
	    lw->icon.icon_width =
		XTextWidth16(fs, (TXT16*)lw->icon.icon,
			     (int) lw->icon.icon_len/2);
	else
	    lw->icon.icon_width =
		XTextWidth(fs, lw->icon.icon, (int) lw->icon.icon_len);
    }
}

static void GetnormalGC(lw)
    IconWidget lw;
{
    XGCValues	values;

    values.foreground	= lw->icon.foreground;
    values.background	= lw->core.background_pixel;
    values.font		= lw->icon.font->fid;
    values.graphics_exposures = False;

    lw->icon.normal_GC = XtGetGC(
	(Widget)lw,
	(unsigned) GCForeground | GCBackground | GCFont | GCGraphicsExposures,
	&values);
}

static void GetgrayGC(lw)
    IconWidget lw;
{
    XGCValues	values;

    values.foreground = lw->icon.foreground;
    values.background = lw->core.background_pixel;
    values.font	      = lw->icon.font->fid;
    values.fill_style = FillTiled;
    values.tile       = XmuCreateStippledPixmap(XtScreen((Widget)lw),
						lw->icon.foreground, 
						lw->core.background_pixel,
						lw->core.depth);
    values.graphics_exposures = False;

    lw->icon.stipple = values.tile;
    lw->icon.gray_GC = XtGetGC((Widget)lw, 
				(unsigned) GCForeground | GCBackground |
					   GCFont | GCTile | GCFillStyle |
					   GCGraphicsExposures,
				&values);
}

static void compute_bitmap_offsets (lw)
    IconWidget lw;
{
    /*
     * icon will be displayed at (internal_width, internal_height + lbm_y)
     */
    if (lw->icon.lbm_height != 0) {
	lw->icon.lbm_y = (((int) lw->core.height) -
			   ((int) lw->icon.internal_height * 2) -
			   ((int) lw->icon.lbm_height)) / 2;
    } else {
	lw->icon.lbm_y = 0;
    }
}


static void set_bitmap_info (lw)
    IconWidget lw;
{
    Window root;
    int x, y;
    unsigned int bw, depth;

    if (!(lw->icon.left_bitmap &&
	  XGetGeometry (XtDisplay(lw), lw->icon.left_bitmap, &root, &x, &y,
			&lw->icon.lbm_width, &lw->icon.lbm_height,
			&bw, &depth))) {
	lw->icon.lbm_width = lw->icon.lbm_height = 0;
    }
    compute_bitmap_offsets (lw);
}



/* ARGSUSED */
static void Initialize(request, new)
 Widget request, new;
{
    IconWidget lw = (IconWidget) new;

    if (lw->icon.icon == NULL) 
        lw->icon.icon = XtNewString(lw->core.name);
    else {
        lw->icon.icon = XtNewString(lw->icon.icon);
    }

    GetnormalGC(lw);
    GetgrayGC(lw);

    SetTextWidthAndHeight(lw);

    if (lw->core.height == 0)
        lw->core.height = lw->icon.icon_height + 2*lw->icon.internal_height;

    set_bitmap_info (lw);		/* need core.height */

    if (lw->core.width == 0)		/* need icon.lbm_width */
        lw->core.width = (lw->icon.icon_width + 2 * lw->icon.internal_width
			  + LEFT_OFFSET(lw));

    lw->icon.icon_x = lw->icon.icon_y = 0;
    (*XtClass(new)->core_class.resize) ((Widget)lw);

} /* Initialize */

/*
 * Repaint the widget window
 */

/* ARGSUSED */
static void Redisplay(w, event, region)
    Widget w;
    XEvent *event;
    Region region;
{
   IconWidget lw = (IconWidget) w;
   GC gc;

   if (region != NULL) {
       int x = lw->icon.icon_x;
       unsigned int width = lw->icon.icon_width;
       if (lw->icon.lbm_width) {
	   if (lw->icon.icon_x > (x = lw->icon.internal_width))
	       width += lw->icon.icon_x - x;
       }
       if (XRectInRegion(region, x, lw->icon.icon_y,
			 width, lw->icon.icon_height) == RectangleOut)
	   return;
   }

   gc = XtIsSensitive((Widget)lw) ? lw->icon.normal_GC : lw->icon.gray_GC;
#ifdef notdef
   if (region != NULL) XSetRegion(XtDisplay(w), gc, region);
#endif /*notdef*/
   if (lw->icon.pixmap == None) {
       int len = lw->icon.icon_len;
       char *icon = lw->icon.icon;
       Position y = lw->icon.icon_y + lw->icon.font->max_bounds.ascent;

       /* display left bitmap */
       if (lw->icon.left_bitmap && lw->icon.lbm_width != 0) {
	   XCopyPlane (XtDisplay(w), lw->icon.left_bitmap, XtWindow(w), gc,
		       0, 0, lw->icon.lbm_width, lw->icon.lbm_height,
		       (int) lw->icon.internal_width, 
		       (int) lw->icon.internal_height + lw->icon.lbm_y, 
		       (unsigned long) 1L);
       }

       if (len == MULTI_LINE_LABEL) {
	   char *nl;
	   while ((nl = index(icon, '\n')) != NULL) {
	       if (lw->icon.encoding)
		   XDrawString16(XtDisplay(w), XtWindow(w), gc,
				 lw->icon.icon_x, y,
				 (TXT16*)icon, (int)(nl - icon)/2);
	       else
		   XDrawString(XtDisplay(w), XtWindow(w), gc,
			       lw->icon.icon_x, y, icon, (int)(nl - icon));
	       y += lw->icon.font->max_bounds.ascent + lw->icon.font->max_bounds.descent;
	       icon = nl + 1;
	   }
	   len = strlen(icon);
       }
       if (len) {
	   if (lw->icon.encoding)
	       XDrawString16(XtDisplay(w), XtWindow(w), gc,
			     lw->icon.icon_x, y, (TXT16*)icon, len/2);
	   else
	       XDrawString(XtDisplay(w), XtWindow(w), gc,
			   lw->icon.icon_x, y, icon, len);
       }
   } else if (lw->icon.icon_len == 1) { /* depth */
       XCopyPlane(XtDisplay(w), lw->icon.pixmap, XtWindow(w), gc,
		  0, 0, lw->icon.icon_width, lw->icon.icon_height,
		  lw->icon.icon_x, lw->icon.icon_y, 1L);
   } else {
       XCopyArea(XtDisplay(w), lw->icon.pixmap, XtWindow(w), gc,
		 0, 0, lw->icon.icon_width, lw->icon.icon_height,
		 lw->icon.icon_x, lw->icon.icon_y);
   }
#ifdef notdef
   if (region != NULL) XSetClipMask(XtDisplay(w), gc, (Pixmap)None);
#endif /* notdef */
}

static void _Reposition(lw, width, height, dx, dy)
    register IconWidget lw;
    Dimension width, height;
    Position *dx, *dy;
{
    Position newPos;
    Position leftedge = lw->icon.internal_width + LEFT_OFFSET(lw);

    switch (lw->icon.justify) {

	case XtJustifyLeft   :
	    newPos = leftedge;
	    break;

	case XtJustifyRight  :
	    newPos = width -
		(lw->icon.icon_width + lw->icon.internal_width);
	    break;

	case XtJustifyCenter :
	default:
	    newPos = (int)(width - lw->icon.icon_width) / 2;
	    break;
    }
    if (newPos < (Position)leftedge)
	newPos = leftedge;
    *dx = newPos - lw->icon.icon_x;
    lw->icon.icon_x = newPos;
    *dy = (newPos = (int)(height - lw->icon.icon_height) / 2)
	  - lw->icon.icon_y;
    lw->icon.icon_y = newPos;
    return;
}

static void Resize(w)
    Widget w;
{
    IconWidget lw = (IconWidget)w;
    Position dx, dy;
    _Reposition(lw, w->core.width, w->core.height, &dx, &dy);
    compute_bitmap_offsets (lw);
}

/*
 * Set specified arguments into widget
 */

#define PIXMAP 0
#define WIDTH 1
#define HEIGHT 2
#define NUM_CHECKS 3

static Boolean SetValues(current, request, new, args, num_args)
    Widget current, request, new;
    ArgList args;
    Cardinal *num_args;
{
    IconWidget curlw = (IconWidget) current;
    IconWidget reqlw = (IconWidget) request;
    IconWidget newlw = (IconWidget) new;
    int i;
    Boolean was_resized = False, redisplay = False, checks[NUM_CHECKS];

    for (i = 0; i < NUM_CHECKS; i++)
	checks[i] = FALSE;

    for (i = 0; i < *num_args; i++) {
	if (streq(XtNbitmap, args[i].name))
	    checks[PIXMAP] = TRUE;
	if (streq(XtNwidth, args[i].name))
	    checks[WIDTH] = TRUE;
	if (streq(XtNheight, args[i].name))
	    checks[HEIGHT] = TRUE;
    }

    if (newlw->icon.icon == NULL) {
	newlw->icon.icon = newlw->core.name;
    }

    /*
     * resize on bitmap change
     */
    if (curlw->icon.left_bitmap != newlw->icon.left_bitmap) {
	was_resized = True;
    }

    if (curlw->icon.encoding != newlw->icon.encoding)
	was_resized = True;

    if (curlw->icon.icon != newlw->icon.icon) {
        if (curlw->icon.icon != curlw->core.name)
	    XtFree( (char *)curlw->icon.icon );

	if (newlw->icon.icon != newlw->core.name) {
	    newlw->icon.icon = XtNewString( newlw->icon.icon );
	}
	was_resized = True;
    }

    if (was_resized || (curlw->icon.font != newlw->icon.font) ||
	(curlw->icon.justify != newlw->icon.justify) || checks[PIXMAP]) {

	SetTextWidthAndHeight(newlw);
	was_resized = True;
    }

    /* recalculate the window size if something has changed. */
    if (newlw->icon.resize && was_resized) {
	if ((curlw->core.height == reqlw->core.height) && !checks[HEIGHT])
	    newlw->core.height = (newlw->icon.icon_height + 
				  2 * newlw->icon.internal_height);

	set_bitmap_info (newlw);

	if ((curlw->core.width == reqlw->core.width) && !checks[WIDTH])
	    newlw->core.width = (newlw->icon.icon_width +
				 LEFT_OFFSET(newlw) +
				 2 * newlw->icon.internal_width);
    }

    if (curlw->icon.foreground != newlw->icon.foreground
	|| curlw->core.background_pixel != newlw->core.background_pixel
	|| curlw->icon.font->fid != newlw->icon.font->fid) {

	XtReleaseGC(new, curlw->icon.normal_GC);
	XtReleaseGC(new, curlw->icon.gray_GC);
	XmuReleaseStippledPixmap( XtScreen(current), curlw->icon.stipple );
	GetnormalGC(newlw);
	GetgrayGC(newlw);
	redisplay = True;
    }

    if ((curlw->icon.internal_width != newlw->icon.internal_width)
        || (curlw->icon.internal_height != newlw->icon.internal_height)
	|| was_resized) {
	/* Resize() will be called if geometry changes succeed */
	Position dx, dy;
	_Reposition(newlw, curlw->core.width, curlw->core.height, &dx, &dy);
    }

    return was_resized || redisplay ||
	   XtIsSensitive(current) != XtIsSensitive(new);
}

static void Destroy(w)
    Widget w;
{
    IconWidget lw = (IconWidget)w;

    XtFree( lw->icon.icon );
    XtReleaseGC( w, lw->icon.normal_GC );
    XtReleaseGC( w, lw->icon.gray_GC);
    XmuReleaseStippledPixmap( XtScreen(w), lw->icon.stipple );
}


static XtGeometryResult QueryGeometry(w, intended, preferred)
    Widget w;
    XtWidgetGeometry *intended, *preferred;
{
    register IconWidget lw = (IconWidget)w;

    preferred->request_mode = CWWidth | CWHeight;
    preferred->width = (lw->label.label_width + 2 * lw->label.internal_width +
			LEFT_OFFSET(lw));
    preferred->height = lw->label.label_height + 2*lw->label.internal_height;
    if (  ((intended->request_mode & (CWWidth | CWHeight))
	   	== (CWWidth | CWHeight)) &&
	  intended->width == preferred->width &&
	  intended->height == preferred->height)
	return XtGeometryYes;
    else if (preferred->width == w->core.width &&
	     preferred->height == w->core.height)
	return XtGeometryNo;
    else
	return XtGeometryAlmost;
}
