#include "copyright.h"

/*
 *  Include file dependencies:
 */

#include <stdio.h>
#ifndef COHERENT
#include <stdlib.h>
#include <stddef.h>
#endif
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/keysym.h>

#include "score.h"
#include "presents.h"
#include "special.h"
#include "audio.h"
#include "mess.h"
#include "ball.h"
#include "gun.h"
#include "sfx.h"
#include "init.h"
#include "blocks.h"
#include "level.h"
#include "bonus.h"
#include "stage.h"
#include "paddle.h"
#include "intro.h"
#include "inst.h"
#include "highscore.h"
#include "keys.h"

#include "main.h"

/*
 *  Internal macro definitions:
 */

#define PADDLE_ANIMATE_DELAY	5
#define BONUS_SEED				2000	

/*
 *  Internal type declarations:
 */

#if NeedFunctionPrototypes
static KeySym 	GetKeySym(XEvent event);
static void 	handleGameMode(Display *display);
static void 	handleEventLoop(Display *display);
static void 	ToggleGamePaused(Display *display, Window window);
static void 	SetGamePaused(Display *display);
static void 	handleGameStates(Display *display);
static void 	sleepSync(Display *display, unsigned long ms);
#else
static void 	sleepSync();
static void 	handleGameStates();
static KeySym 	GetKeySym();
static void 	handleGameMode();
static void 	handleEventLoop();
static void 	ToggleGamePaused();
static void 	SetGamePaused();
#endif

/*
 *  Internal variable declarations:
 */


int paddleMotion = 0;
int paddleDx = 0;
int speedLevel = 5;
int frame, gameActive;
int mode;
static int iconified = False;
long speed;
static int userDelay = 1;
static int paddleControl;
static time_t pauseStartTime;
time_t pausedTime;

#if NeedFunctionPrototypes
void SetUserSpeed(int delay)
#else
void SetUserSpeed(delay)
	int delay;
#endif
{
	int temp;

	/* Set an entire game speedup or slowdown speed */
	temp = (long) (speed / userDelay);
	userDelay = delay;
	speed = (long) (temp * userDelay);
	speedLevel = 10 - delay;
}

#if NeedFunctionPrototypes
int GetPaddleControlMode(void)
#else
int GetPaddleControlMode()
#endif
{
	/* Get the paddle control mode */
	return paddleControl;
}

#if NeedFunctionPrototypes
void SetPaddleControlMode(int type)
#else
void SetPaddleControlMode(type)
	int type;
#endif
{
	/* Set the paddle control mode to the new mode */
	paddleControl = type;
}

#if NeedFunctionPrototypes
void SetGameSpeed(int delay)
#else
void SetGameSpeed(delay)
	int delay;
#endif
{
	/* This is the speed used in the sleeping routine */
	if (delay >= 0)
		speed = (long) (delay * userDelay);
}

#if NeedFunctionPrototypes
static void sleepSync(Display *display, unsigned long ms)
#else
static void sleepSync(display, ms)
	Display *display;
	unsigned long ms;
#endif
{
    struct timeval st, et;
    long SyncTime;

    gettimeofday(&st, NULL);
    XSync(display, False);
    gettimeofday(&et, NULL);

    SyncTime = (((et.tv_sec - st.tv_sec) * 1000) +
               ((et.tv_usec - st.tv_usec) / 1000) );

    if ((ms) > ((1000 / 60) + SyncTime))
        ms_sleep(ms - SyncTime);
}

#if NeedFunctionPrototypes
static KeySym GetKeySym(XEvent event)
#else
static KeySym GetKeySym(event)
	XEvent event;
#endif
{
	int count;
	char key;
	KeySym keysym;
	XComposeStatus compose;

	/* Lookup a keysym using the event key */
	count = XLookupString(&event.xkey, &key, 1, &keysym, &compose);

	return keysym;
}

#if NeedFunctionPrototypes
int paddleIsMoving(void)
#else
int paddleIsMoving()
#endif
{
	/* Returns direction of paddle 1 right -1 left 0 stopped */
	return paddleMotion;
}

#if NeedFunctionPrototypes
void handlePaddleMoving(Display *display)
#else
void handlePaddleMoving(display)
	Display *display;
#endif
{
	static oldx = 0;
	int rx, ry, x, y;
	unsigned int mask;
	Window root, child;

	if (paddleControl == CONTROL_KEYS)
	{
		switch (paddleMotion)
		{
			case 1:		/* Move the paddle to the right 1 increment */
				MovePaddle(display, playWindow, 
					PADDLE_RIGHT, currentPaddleSize, 0);
				break;

			case -1:		/* Move the paddle to the left 1 increment */
				MovePaddle(display, playWindow, 
					PADDLE_LEFT, currentPaddleSize, 0);
				break;

			default:
				break;
		}
	} else if (paddleControl == CONTROL_MOUSE)
	{
		/* Obtain the position of the pointer in the play window */
		if (XQueryPointer(display, playWindow, &root, &child, 
			&rx, &ry, &x, &y, &mask) == True)
		{
			/* Has the pointer moved since our last poll */
			if (x != oldx)
			{
				paddleDx = x - oldx;

				/* Move the paddle to the position of the mouse pointer */
				MovePaddle(display, playWindow, 
					PADDLE_NONE, currentPaddleSize, x);
				oldx = x;

				/* Adjust the paddle motion variable so the ball moves when in
				 * the BALL_READY state and BALL_CREATE state.
				 */
				if (x > oldx)
					paddleMotion = 1;
				else
					paddleMotion = -1;
			}
			else
			{
				/* Reset to no motion */
				paddleMotion = 0;
				paddleDx = 0;
			}
		}
	}
}


#if NeedFunctionPrototypes
static void ToggleGamePaused(Display *display, Window window)
#else
static void ToggleGamePaused(display, window)
	Display *display;
	Window window;
#endif
{
	if (mode == MODE_PAUSE)
	{
		/* Finished pause resume game */
		mode = MODE_GAME;
		SetCurrentMessage(display, messWindow, "- Play ball -", False);
		
		/* How many seconds were we paused for? */
		pausedTime += (time(NULL) - pauseStartTime);

 		XSelectInput(display, mainWindow, 
			KeyPressMask | KeyReleaseMask | ButtonPressMask |
   			ButtonReleaseMask | ExposureMask | StructureNotifyMask);

 		GrabPointer(display, window);
	}
	else 
		SetGamePaused(display);
}

#if NeedFunctionPrototypes
static void SetGamePaused(Display *display)
#else
static void SetGamePaused(display)
	Display *display;
#endif
{
	if (mode == MODE_GAME)
	{
		/* Set game to paused mode */
		mode = MODE_PAUSE;
		SetCurrentMessage(display, messWindow, 
			"- Game paused -", False);
		
		/* we need to keep track of how long we were paused so that later
		 * in the highscore thing I can take that off the time.
		 */
		pauseStartTime = time(NULL);

		XSelectInput(display, mainWindow, 
			KeyPressMask | ExposureMask | StructureNotifyMask);

		UnGrabPointer(display);
	}
}

#if NeedFunctionPrototypes
void handleIconify(Display *display, XUnmapEvent *event)
#else
void handleIconify(display, event)
	Display *display;
	XUnmapEvent *event;
#endif
{
	ToggleGamePaused(display, mainWindow);
}

#if NeedFunctionPrototypes
void SelectiveRedraw(Display *display)
#else
void SelectiveRedraw(display)
	Display *display;
#endif
{
	switch (mode)
	{
		case MODE_GAME:
		case MODE_PAUSE:
			RedrawPlayWindow(display, playWindow);
			break;

		case MODE_INTRO:
			RedrawIntroduction(display, playWindow);
			break;

		case MODE_INSTRUCT:
			RedrawInstructions(display, playWindow);
			break;

		case MODE_KEYS:
			RedrawKeys(display, playWindow);
			break;

		case MODE_BONUS:
			RedrawBonus(display, mainWindow);
			break;

		case MODE_HIGHSCORE:
			RedrawHighScore(display, playWindow);
			break;

		default:
			break;
	}

	/* Redisplay the message and the level/score info */
	RedrawLevelInfo(display, levelWindow);
	DisplayCurrentMessage(display, messWindow);

	/* To be sure - to be sure */
	XFlush(display);
}

#if NeedFunctionPrototypes
void handleExposure(Display *display, XEvent event)
#else
void handleExposure(display, event)
	Display *display;
	XEvent event;
#endif
{
	/* Only redraw window once so wait until all expose events have sent
	 * and then redraw all that we need to redraw based on current game
	 * mode.
	 */
	if (event.xexpose.count == 0)
		SelectiveRedraw(display);
}

#if NeedFunctionPrototypes
void handleMouseButtons(Display *display, XEvent event, int Down)
#else
void handleMouseButtons(display, event, Down)
	Display *display;
	XEvent event;
	int Down;
#endif
{
	if (Down == True)
	{
		/* Button pressed down */
		switch(event.xbutton.button)
		{
			/* Shoot a bullet */
			case Button1:
			case Button2: 
			case Button3:
				/* If we are playing the game and a ball needs to be started
				 * then start it otherwise shoot a bullet.
				 */
				if (mode == MODE_GAME)
					if (ActivateWaitingBall() == False)
						shootBullet(display, playWindow);
				break;
		}
	}
}

#if NeedFunctionPrototypes
void handleKeyPress(Display *display, KeySym keysym, int Pressed)
#else
void handleKeyPress(display, keysym, Pressed)
	Display *display;
	KeySym keysym;
	int Pressed;
#endif
{
	int temp;

	if (Pressed == False)
	{
		/* key was released */
		paddleMotion = 0;
	}
	else
	{
		/* Switch on the key just pressed down */
		switch (keysym)
		{
			case XK_Left:
			case XK_j:
			case XK_J:
				/* Set paddle to move left */
				if (mode == MODE_GAME)
					paddleMotion = -1;
				break;
	
			case XK_k:
			case XK_K:
				/* Shoot a bullet if available */
				if (mode == MODE_GAME)
					if (ActivateWaitingBall() == False)
						shootBullet(display, playWindow);
				break;
	
			case XK_Right:
			case XK_l:
			case XK_L:
				/* Set paddle to move right */
				if (mode == MODE_GAME)
					paddleMotion = 1;
				break;

			case XK_Escape:
				if (mode == MODE_GAME)
				{
					/* Abort game and return to intros */
					SetGameSpeed(FAST_SPEED);
					ResetIntroduction();
					mode = MODE_INTRO;

					SetCurrentMessage(display, messWindow, 
						"Game aborted.", True);
				}
				break;

            case XK_minus:
                if (mode == MODE_GAME)
                {
                    SkipToNextLevel(display, playWindow);
                    SetCurrentMessage(display, messWindow,
                        "Skipping to next level ...", True);
                }
                break;

			case XK_space:
				if (mode == MODE_INTRO || mode == MODE_HIGHSCORE 
				  || mode == MODE_INSTRUCT || mode == MODE_KEYS)
				{
					SetGameSpeed(FAST_SPEED);
					gameActive = False;
					mode = MODE_GAME;
				}

				if (mode == MODE_BONUS)
					SetBonusWait(BONUS_FINISH, frame);
				else if (mode == MODE_PRESENTS)	
					QuickFinish(display, mainWindow);
				break;


			case XK_H:	/* Personal highscores */
				if (mode == MODE_INTRO || mode == MODE_INSTRUCT 
					|| mode == MODE_KEYS || mode == MODE_HIGHSCORE)
				{
					SetGameSpeed(FAST_SPEED);
					ResetHighScore(PERSONAL);
					mode = MODE_HIGHSCORE;

					/* Play a bit of sound */
					if (noSound == False)
						playSoundFile("toggle", 50);
					
				}
				break;

			case XK_h:	/* Global highscores */
				if (mode == MODE_INTRO || mode == MODE_INSTRUCT 
					|| mode == MODE_KEYS || mode == MODE_HIGHSCORE)
				{
					SetGameSpeed(FAST_SPEED);
					ResetHighScore(GLOBAL);
					mode = MODE_HIGHSCORE;

					/* Play a bit of sound */
					if (noSound == False)
						playSoundFile("toggle", 50);
				}
				break;

			case XK_c:
			case XK_C:
				/* Cycle through the introduction screens if note in a game */
				if (mode == MODE_INTRO)
				{
					SetGameSpeed(FAST_SPEED);
					ResetInstructions();
					mode = MODE_INSTRUCT;
				} else if (mode == MODE_INSTRUCT)
				{
					SetGameSpeed(FAST_SPEED);
					ResetKeys();
					mode = MODE_KEYS;
				} else if (mode == MODE_KEYS)
				{
					SetGameSpeed(FAST_SPEED);
					ResetHighScore(GLOBAL);
					mode = MODE_HIGHSCORE;
				} else if (mode == MODE_HIGHSCORE)
				{
					SetGameSpeed(FAST_SPEED);
					ResetIntroduction();
					mode = MODE_INTRO;
				} 
				break;

			case XK_a:
			case XK_A:
				if (noSound == False)
				{
					/* Try and turn audio off */
					FreeAudioSystem();

					noSound = True;
					SetCurrentMessage(display, messWindow, 
						"- Audio OFF -", True);
				}
				else
				{
					/* Try and turn audio on */
					if (SetUpAudioSystem(display) == False)
					{
						/* Unable to turn audio on */
						noSound = True;
						SetCurrentMessage(display, messWindow, 
							"- Audio unavailable -", True);
					}
					else
					{
						/* Audio is now active */
						noSound = False;
						SetCurrentMessage(display, messWindow, 
							"- Audio ON -", True);
					}
				}
				break;

			case XK_s:
			case XK_S:
				if (mode == MODE_INTRO || mode == MODE_INSTRUCT 
					|| mode == MODE_KEYS || mode == MODE_HIGHSCORE)
				{
					/* toggle the special effects system */
					if (getSpecialEffects(display) == True)
					{
						/* Turn off special effects */
						useSpecialEffects(False);

						SetCurrentMessage(display, messWindow, 
							"- SFX OFF -", True);
					}
					else
					{
						/* Cannot use sfx on this display */
						if (getSpecialEffects(display) == -1)
						{
							SetCurrentMessage(display, messWindow, 
								"- SFX Unavailable -", True);
						}
						else
						{
							/* Try and turn on special effects */
							useSpecialEffects(True);

							SetCurrentMessage(display, messWindow, 
								"- SFX ON -", True);
						}
					}
				}
				break;

			case XK_g:
			case XK_G:
				/* Toggle game mode */
				if (GetPaddleControlMode() == CONTROL_KEYS)
				{
					SetCurrentMessage(display, messWindow, 
						"Control: Mouse", True);
					SetPaddleControlMode(CONTROL_MOUSE);
				}
				else
				{
					SetCurrentMessage(display, messWindow, 
						"Control: Keys", True);
					SetPaddleControlMode(CONTROL_KEYS);
				}

				/* Play a bit of sound */
				if (noSound == False)
					playSoundFile("toggle", 50);
					
				/* If looking at the keys screen then update the text
				 * indicating which control mode you are using.
				 */
				if (mode == MODE_KEYS)
					RedrawKeys(display, playWindow);

				break;

			case XK_p:
			case XK_P:
				ToggleGamePaused(display, mainWindow);
				break;

			case XK_i:
			case XK_I:
				/* Iconify the window quickly */
				XIconifyWindow(display, mainWindow, 0);
				break;

			case XK_d:	
			case XK_D:	
				/* Kill the ball - why I don't know */
				if (mode == MODE_GAME)
				{
					if ((temp = GetAnActiveBall()) >= 0)
					{
						/* Erase and reset ball to new one */
						ClearBallNow(display, playWindow, temp);
					}

				}
				break;

			case XK_1:	/* Set speed to speed 1 */
				SetUserSpeed(9);
				SetCurrentMessage(display, messWindow, "Warp 1", True);
				if (noSound == False) playSoundFile("tone1", 50);
				break;

			case XK_2:	/* Set speed to speed 2 */
				SetUserSpeed(8);
				SetCurrentMessage(display, messWindow, "Warp 2", True);
				if (noSound == False) playSoundFile("tone2", 50);
				break;

			case XK_3:	/* Set speed to speed 3 */
				SetUserSpeed(7);
				SetCurrentMessage(display, messWindow, "Warp 3", True);
				if (noSound == False) playSoundFile("tone3", 50);
				break;

			case XK_4:	/* Set speed to speed 4 */
				SetUserSpeed(6);
				SetCurrentMessage(display, messWindow, "Warp 4", True);
				if (noSound == False) playSoundFile("tone4", 50);
				break;

			case XK_5:	/* Set speed to speed 5 */
				SetUserSpeed(5);
				SetCurrentMessage(display, messWindow, "Warp 5", True);
				if (noSound == False) playSoundFile("tone5", 50);
				break;

			case XK_6:	/* Set speed to speed 6 */
				SetUserSpeed(4);
				SetCurrentMessage(display, messWindow, "Warp 6", True);
				if (noSound == False) playSoundFile("tone6", 50);
				break;

			case XK_7:	/* Set speed to speed 7 */
				SetUserSpeed(3);
				SetCurrentMessage(display, messWindow, "Warp 7", True);
				if (noSound == False) playSoundFile("tone7", 50);
				break;

			case XK_8:	/* Set speed to speed 8 */
				SetUserSpeed(2);
				SetCurrentMessage(display, messWindow, "Warp 8", True);
				if (noSound == False) playSoundFile("tone8", 50);
				break;

			case XK_9:	/* Set speed to speed 9 */
				SetUserSpeed(1);
				SetCurrentMessage(display, messWindow, "Warp 9", True);
				if (noSound == False) playSoundFile("tone9", 50);
				break;

			case XK_Q:
			case XK_q:
				if (mode == MODE_GAME || mode == MODE_BONUS)
				{
					/* Save out scores when quitting */
					UpdateHighScores();
				}

				/* Shut down and exit game */
				ShutDown(display, 0, "Thank you for playing XBoing.");
				break;

			default:
				/* Do nothing */
				break;
		}
	}
}

#if NeedFunctionPrototypes
static void handleGameMode(Display *display)
#else
static void handleGameMode(display)
	Display *display;
#endif
{
	static int bonusRow = 0;
	static int bonusCol = 0;
	static int nextBonusFrame = 0;

	/* If we are going to play then setup first level */
	if (gameActive == False)
	{
		/* Choose a random velocity for the ball */

		/* Always start at level 1 or level specified */
		SetLevelNumber(GetStartingLevel());

		/* Set some important variables */
		livesLeft 			= 3;
		score 				= 0;
		nextBonusFrame 		= 0;
		currentPaddleSize 	= PADDLE_HUGE;
		pausedTime			= 0;
		bonusBlock 			= False;

		/* Setup the stage and load 1st level */
		SetupStage(display, playWindow);

		/* Start game play */
		gameActive = True;

		/* Keep track of the game duration - shown in highscores */
		gameTime = time(NULL);
	}

	/* If we need to move the paddle then do so */
	if ((frame % PADDLE_ANIMATE_DELAY) == 0)
		handlePaddleMoving(display);

	if (mode == MODE_GAME)
	{
		HandleBallMode(display, playWindow);

		/* Add bonus coin block at random intervals */
		if (nextBonusFrame == 0 && bonusBlock == False)
			nextBonusFrame = frame + (rand() % BONUS_SEED);

		/* Do we need to add a bonus coin? */
		if (nextBonusFrame <= frame && bonusBlock == False)
		{
			/* Add the bonus block now - different types */
			switch (rand() % 10)
			{
				case 0: case 1: 
				case 2: case 3: 
				case 4: case 5: 
				case 6: case 7:
					/* Add a normal bonus block */
					AddBonusBlock(display, playWindow, &bonusRow, &bonusCol, 
						BONUS_BLK);
					break;

				case 8:
					/* Add the x2 bonus block */
					AddBonusBlock(display, playWindow, &bonusRow, &bonusCol, 
						BONUSX2_BLK);
					break;

				case 9:
					/* Add the x4 bonus block */
					AddBonusBlock(display, playWindow, &bonusRow, &bonusCol, 
						BONUSX4_BLK);
					break;

			}

			nextBonusFrame = 0;
		}
	}

	HandleBulletMode(display, playWindow);

	/* If any blocks need exploding then do so */
	ExplodeBlocksPending(display, playWindow);

	/* So blocks need animation all the time so do it */
	HandlePendingAnimations(display, playWindow);

	/* See if the level is finished and update level info if needed */
	if (mode == MODE_GAME)
		CheckGameRules(display, playWindow);
}

#if NeedFunctionPrototypes
static void handleGameStates(Display *display)
#else
static void handleGameStates(display)
	Display *display;
#endif
{
	/* Update the message window if any new messages come along */
	DisplayCurrentMessage(display, messWindow);

	/* In game effects */
	switch (currentSfxMode())
	{
		case SFX_SHAKE:
			/* Something exploded or bumped the screen */
			WindowShakeEffect(display, playWindow);
			break;

		case SFX_NONE:
		default:
			break;
	}

	switch (mode)
	{
		case MODE_GAME:
			handleGameMode(display);
			break;

		case MODE_PRESENTS:
			Presents(display, mainWindow);
			break;

		case MODE_BONUS:
			DoBonus(display, mainWindow);
			break;

		case MODE_INTRO:
			Introduction(display, playWindow);
			break;

		case MODE_INSTRUCT:
			Instructions(display, playWindow);
			break;

		case MODE_KEYS:
			Keys(display, playWindow);
			break;

		case MODE_HIGHSCORE:
			HighScore(display, playWindow);
			break;

		case MODE_PAUSE:
			break;
	}

	/* Flush the display */
	XFlush(display);
}

#if NeedFunctionPrototypes
static void handleEventLoop(Display *display)
#else
static void handleEventLoop(display)
	Display *display;
#endif
{
	XEvent event;
	int pending;
	KeySym keysym;

	pending = frame = 0;

	/* Initial mode for game is Introduction */
	mode = MODE_PRESENTS;

	/* No special effects yet */
	changeSfxMode(SFX_NONE);

	/* Flush all events until app is fully mapped */
    do
	{
		/* handle audio device events if they exist */
		audioDeviceEvents();

		/* Get the next event */
		XNextEvent(display, &event);
	}
	while (event.type != MapNotify);

	/* Grab the pointer to the main window */
	GrabPointer(display, mainWindow);

	/* Loop forever and ever */
	while (True)
	{
		/* handle and audio device events if supported */
		audioDeviceEvents();

		/* Sleep a bit if not iconified */
		if (iconified == False)
			sleepSync(display, speed);

		/* See if any events are waiting for me to handle */
		if (iconified == False && mode != MODE_PAUSE)
		{
			/* Get an event but don't wait if none arrives */
			pending = XPending(display);
			frame++;
		}
		else
		{
			/* Wait here for an event and then get the number waiting */
			XPeekEvent(display, &event);
			pending = XPending(display);
		}
		
		/* Handle any events pending */
		while (pending > 0)
		{
			/* Get the next X event thanks */
			XNextEvent(display, &event);

			switch(event.type)
			{
				case UnmapNotify:
					/* Turn off just all events except the mapping ones */
    				XSelectInput(display, mainWindow, StructureNotifyMask);
					handleIconify(display, (XUnmapEvent *) &event);
					iconified = True;
					break;

				case MapNotify:
					/* Turn back on all the events that are needed */
    				XSelectInput(display, mainWindow, 
						KeyPressMask | KeyReleaseMask | ButtonPressMask |
       					ButtonReleaseMask | ExposureMask | StructureNotifyMask);

					SelectiveRedraw(display);
 					GrabPointer(display, mainWindow);
					iconified = False;

					break;

				case ButtonRelease:
					handleMouseButtons(display, event, False);
					break;

				case ButtonPress:
					handleMouseButtons(display, event, True);
					break;

				case KeyRelease:
					keysym = GetKeySym(event);
					handleKeyPress(display, keysym, False);
					break;

				case KeyPress:
					keysym = GetKeySym(event);
					handleKeyPress(display, keysym, True);
					break;

				case Expose:
					handleExposure(display, event);
					break;

				default:
					break;
			}

			/* Decrement the number of pending events */
			pending--;
		}

		/* handle all game states and animations */
		if (iconified == False) 
			handleGameStates(display);
	}

	/* NOT REACHED */
}

#if NeedFunctionPrototypes
void main(int argc, char **argv)
#else
void main(argc, argv)
	int argc;
	char **argv;
#endif
{
	static Display *display;

	/* Initialise everything and return display */
	display = InitialiseGame(argv, argc);

	SetGameSpeed(FAST_SPEED);
	gameActive = False;
	iconified = False;

	/* main event loop */
	handleEventLoop(display);

	/* NOTREACHED */
}
