#include "copyright.h"

/*
 *  Include file dependencies:
 */

#include <stdio.h>
#ifndef COHERENT
#include <stdlib.h>
#include <stddef.h>
#endif
#include <ctype.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <pwd.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <xpm.h>

#include "error.h"
#include "misc.h"
#include "main.h"
#include "audio.h"
#include "special.h"
#include "init.h"
#include "inst.h"
#include "stage.h"
#include "blocks.h"
#include "ball.h"
#include "score.h"
#include "paddle.h"
#include "level.h"
#include "mess.h"
#include "intro.h"

#include "bitmaps/highscores.xpm"

#include "highscore.h"

/*
 * cc has problems with htonl() in netinet/in.h, remove this lines
 * after the problem is fixed!!!
 *
 * Udo (udo@umunk.GUN.de)
 */

#ifdef COHERENT
#undef htonl
#define htonl(x) (x)
#undef ntohl
#define ntohl(x) (x)
#endif

/*
 *  Internal macro definitions:
 */

#define GAP				13
#define NUM_HIGHSCORES	10

#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

/*
 *  Internal type declarations:
 */

#if NeedFunctionPrototypes
static void SetHighScoreWait(int newMode, int waitFrame);
static void InitialiseHighScores(void);
static void SortHighScores(void);
static char *GetHomeDir(void);
static void DeleteScore(int index);
static int LockUnlock(int cmd);
#else
static int LockUnlock();
static void DeleteScore();
static char *GetHomeDir();
static void SetHighScoreWait();
static void InitialiseHighScores();
static void SortHighScores();
#endif

/*
 *  Internal variable declarations:
 */

static int nextFrame = 0;
static int endFrame = 0;
enum HighScoreStates HighScoreState;
static Pixmap titlePixmap, titlePixmapM;
static int waitingFrame, waitMode;
static int sparkley = 0;
static int sindex = 0;
static int si = 0;
static int scoreType = GLOBAL;

highScoreEntry highScores[NUM_HIGHSCORES];

#if NeedFunctionPrototypes
void SetUpHighScore(Display *display, Window window, Colormap colormap)
#else
void SetUpHighScore(display, window, colormap)
	Display *display;
	Window window;
	Colormap colormap;
#endif
{
	XpmAttributes   attributes;
	int             XpmErrorStatus;

	attributes.valuemask = XpmColormap;
	attributes.colormap = colormap;

	/* Load the highscore title pixmap */
	XpmErrorStatus = XpmCreatePixmapFromData(display, window, highscores_xpm,
		&titlePixmap, &titlePixmapM, &attributes);
	HandleXPMError(display, XpmErrorStatus, "InitialiseHighScore()");

    /* Free the xpm pixmap attributes */
	XpmFreeAttributes(&attributes);

	/* Setup the high score table */
	InitialiseHighScores();
	ResetHighScore(GLOBAL);
}

#if NeedFunctionPrototypes
static void DoTitle(Display *display, Window window)
#else
static void DoTitle(display, window)
	Display *display;
	Window window;
#endif
{
	char string[80];

	/* Clear */
    DrawStageBackground(display, window, BACKGROUND_0);

	/* Draw the highscore title */
	RenderShape(display, window, titlePixmap, titlePixmapM,
		59, 20, 377, 37, False);

	/* Let the dudes know how to start the game */
	strcpy(string, "Press <Space> to start the game");
	DrawShadowCentredText(display, window, textFont,
		string, PLAY_HEIGHT - 40, tann, PLAY_WIDTH);

	/* Set the message window to have the other display toggle key */
	if (scoreType == GLOBAL)
		SetCurrentMessage(display, messWindow, 
			"<H> - Personal Best", False);
	else
		SetCurrentMessage(display, messWindow, 
			"<h> - Roll of Honour", False);

	SetHighScoreWait(HIGHSCORE_SHOW, frame + 10);
}

#if NeedFunctionPrototypes
static void DoHighScores(Display *display, Window window)
#else
static void DoHighScores(display, window)
	Display *display;
	Window window;
#endif
{
	int i, len, plen;
	int xr = 30;
	int ym = 75;
	int xs = xr + 30;
	int xl = xs + 80;
	int xt = xl + 35;
	int xg = xt + 55;
	int xn = xg + 90;
	int y = 180;
	char string[80];
	char string2[80];
	char *p;

	/* Read the high score table */
	if (ReadHighScoreTable(GLOBAL) == False)
		InitialiseHighScores();

	/* Draw the boing master at top of the list */
	strcpy(string, "Boing Master");
	DrawShadowCentredText(display, window, textFont, 
		string, ym, red, PLAY_WIDTH);
	ym += textFont->ascent + GAP;

	/* Render the boing master's name */
	strcpy(string, highScores[0].name);
	DrawShadowCentredText(display, window, titleFont, 
		string, ym, yellow, PLAY_WIDTH);
	ym += textFont->ascent + GAP * 2;

	/* Read the high score table */
	if (ReadHighScoreTable(scoreType) == False)
		InitialiseHighScores();

	/* Explain that this is the roll of honour */
	if (scoreType == GLOBAL)
		strcpy(string, "- The Roll of Honour -");
	else
		strcpy(string, "- Personal Best -");
	DrawShadowCentredText(display, window, textFont, 
		string, ym, green, PLAY_WIDTH);
	ym += textFont->ascent + GAP;

	/* Draw the titles for the highscore table */
	strcpy(string, "#");
	len = strlen(string);
	DrawText(display, window, xr+2, y+2, textFont, black, string, len);
	DrawText(display, window, xr, y, textFont, yellow, string, len);

	strcpy(string, "Score");
	len = strlen(string);
	DrawText(display, window, xs+2, y+2, textFont, black, string, len);
	DrawText(display, window, xs, y, textFont, yellow, string, len);

	strcpy(string, "L");
	len = strlen(string);
	DrawText(display, window, xl+2, y+2, textFont, black, string, len);
	DrawText(display, window, xl, y, textFont, yellow, string, len);

	strcpy(string, "Time");
	len = strlen(string);
	DrawText(display, window, xt+2, y+2, textFont, black, string, len);
	DrawText(display, window, xt, y, textFont, yellow, string, len);

	strcpy(string, "Date");
	len = strlen(string);
	DrawText(display, window, xg+2, y+2, textFont, black, string, len);
	DrawText(display, window, xg, y, textFont, yellow, string, len);

	strcpy(string, "Player");
	len = strlen(string);
	DrawText(display, window, xn+2, y+2, textFont, black, string, len);
	DrawText(display, window, xn, y, textFont, yellow, string, len);

	y += textFont->ascent + GAP / 2;

	/* Draw the line above the table */
	DrawLine(display, window, 22, y+2, PLAY_WIDTH - 18, y+2, black, 3);
	DrawLine(display, window, 20, y, PLAY_WIDTH - 20, y, white, 3);

	y += textFont->ascent;

	/* Draw the scores into the table */
	for (i = 0; i < NUM_HIGHSCORES; i++)
	{
		if (ntohl(highScores[i].score) > (u_long) 0)
		{
			/* Draw the rank */
			sprintf(string, "%d", i+1);
			if (ntohl(highScores[i].score) != score)
				DrawShadowText(display, window, textFont, string, xr, y, tann);
			else
				DrawShadowText(display, window, textFont, string, xr, y, green);

			/* Draw the score */
			sprintf(string, "%ld", ntohl(highScores[i].score));
			if (highScores[i].score != score)
				DrawShadowText(display, window, textFont, string, xs, y, red);
			else
				DrawShadowText(display, window, textFont, string, xs, y, green);

			/* Write out the level reached */
			sprintf(string, "%ld", ntohl(highScores[i].level));
			DrawShadowText(display, window, textFont, string, xl, y, green);

			/* Game duration in minutes and seconds */
			sprintf(string, "%d'%d\"",
				highScores[i].gameTime / 60, highScores[i].gameTime % 60);
			if (ntohl(highScores[i].score) != score)
				DrawShadowText(display, window, textFont, string, xt, y, tann);
			else
				DrawShadowText(display, window, textFont, string, xt, y, green);

			/* Construct the date for the highscore entry */
#ifndef COHERENT
			strftime(string, 10, "%d %b %y", localtime(&highScores[i].time));
#else
			{
				struct tm *localtm;

				localtm=localtime(&highScores[i].time);

				sprintf(string,"%d %d %d",localtm->tm_mday,localtm->tm_mon+1,localtm->tm_year);
			}
#endif
			string[9] = '\0';	/* Just to be sure */
			if (ntohl(highScores[i].score) != score)
				DrawShadowText(display, window, textFont, string, xg, y, white);
			else	
				DrawShadowText(display, window, textFont, string, xg, y, green);

			/* Name of the boing master */
			strcpy(string, highScores[i].name);
			plen = XTextWidth(textFont, string, len);

			/* Only use the first name if too big for screen */
			if ((plen + xn) > PLAY_WIDTH)
			{
				/* Find the first space and null terminate there */
				p = strchr(string, ' ');
				*p = '\0';
				strcpy(string2, string);

				/* Draw a much smaller version of your name */
				if (ntohl(highScores[i].score) != score)
					DrawShadowText(display, window, textFont, string2, xn, y, 
						yellow);
				else
					DrawShadowText(display, window, textFont, string2, xn, y, 
						green);
			}
			else
			{
				/* Write out users name */
				if (ntohl(highScores[i].score) != score)
					DrawShadowText(display, window, textFont, string, xn, y, 
						yellow);
				else
					DrawShadowText(display, window, textFont, string, xn, y, 
						green);
			}

		}
		else
		{
			/* This bit is for when the table entry is blank */
			sprintf(string, "%d", i+1);
			DrawShadowText(display, window, textFont, string, xr, y, tann);

			/* Draw dashes for blank entries */
			strcpy(string, "--");
			DrawShadowText(display, window, textFont, string, xs, y, red);
			DrawShadowText(display, window, textFont, string, xl, y, green);
			DrawShadowText(display, window, textFont, string, xt, y, tann);
			DrawShadowText(display, window, textFont, string, xg, y, white);
			DrawShadowText(display, window, textFont, string, xn, y, yellow);
		}

		y += textFont->ascent + GAP;
	}

	/* Draw the line above the table */
	DrawLine(display, window, 22, y+2, PLAY_WIDTH - 18, y+2, black, 3);
	DrawLine(display, window, 20, y, PLAY_WIDTH - 20, y, white, 3);

	SetHighScoreWait(HIGHSCORE_SPARKLE, frame + 2);
}

#if NeedFunctionPrototypes
static void DoSparkle(Display *display, Window window)
#else
static void DoSparkle(display, window)
	Display *display;
	Window window;
#endif
{
	static Pixmap store;	/* backing store for the sparkle */
	static int x = 6;		/* X position of the sparkle */

	if (!store)
	{
		/* Create some backing store for the sparkle star */
		store = XCreatePixmap(display, window, 20, 20,
			DefaultDepth(display, XDefaultScreen(display)));
	}

	if (frame == endFrame)
	{
		/* Bug out of the sparkle and goto next sequence */
		si = 0;
		sindex = 0;
		sparkley = 180 + textFont->ascent + 20;

		/* End the sparkle and now set up for finish */
		SetHighScoreWait(HIGHSCORE_FINISH, frame + 1000);
		return;
	}

	if (sindex == 0)
		XCopyArea(display, window, store, gc, x, sparkley, 20, 20, 0, 0);

	if (frame == nextFrame)
	{
		/* Draw the sparkle frame */
		RenderShape(display, window, stars[sindex], starsM[sindex],
			x, sparkley, 20, 20, False);

		sindex++;
		nextFrame = frame + 15;

		/* Last frame of sparkle so reset */
		if (sindex == 11)
		{
			XCopyArea(display, store, window, gc, 0, 0, 20, 20, x, sparkley);

			sindex = 0;
			nextFrame = frame + 100;
			sparkley += textFont->ascent + GAP;

			si++;

			if ((ntohl(highScores[si].score) <= (u_long) 0) || (si == 10))
			{
				si = 0;
				sindex = 0;
				sparkley = 180 + textFont->ascent + 20;
			}
		}
	}
}



#if NeedFunctionPrototypes
static void SetHighScoreWait(int newMode, int waitFrame)
#else
static void SetHighScoreWait(newMode, waitFrame)
	int newMode;
	int waitFrame;
#endif
{
	waitingFrame = waitFrame;
	waitMode = newMode;
	HighScoreState = HIGHSCORE_WAIT;
}

#if NeedFunctionPrototypes
void DoHighScoreWait(void)
#else
void DoHighScoreWait()
#endif
{
	/* Wait for the end frame then change mode */
	if (frame == waitingFrame)
		HighScoreState = waitMode;
}

#if NeedFunctionPrototypes
static void DoFinish(Display *display, Window window)
#else
static void DoFinish(display, window)
	Display *display;
	Window window;
#endif
{
	mode = MODE_INTRO;
	HighScoreState = HIGHSCORE_TITLE;
	ResetIntroduction();

	if (noSound == False)
		playSoundFile("weeek", 100);

	SetGameSpeed(FAST_SPEED);
}

#if NeedFunctionPrototypes
void HighScore(Display *display, Window window)
#else
void HighScore(display, window)
	Display *display;
	Window window;
#endif
{
	/* Switch on the highscore table state */
	switch (HighScoreState)
	{
		case HIGHSCORE_TITLE:
			DoTitle(display, window);
			break;

		case HIGHSCORE_SHOW:
			DoHighScores(display, window);
			break;

		case HIGHSCORE_SPARKLE:
			DoSparkle(display, window);
			if ((frame % FLASH) == 0)
				RandomDrawSpecials(display);
			break;

		case HIGHSCORE_FINISH:
			DoFinish(display, window);
			break;

		case HIGHSCORE_WAIT:
			DoHighScoreWait();
			break;

		default:
			break;
	}
}

#if NeedFunctionPrototypes
void CommandlineHighscorePrint(void)
#else
void CommandlineHighscorePrint()
#endif
{
	char string[11];
	int i;

	InitialiseHighScores();

	/* Must have table initialised with scores */
	if (ReadHighScoreTable(GLOBAL) == False)
		InitialiseHighScores();

	/* Print out a pretty title message for scores */
	fprintf(stdout, "XBoing Roll of Honour\n\n");
	fprintf(stdout, "Rank\tScore\t  Level\tTime\tDate       Name\n");
	fprintf(stdout, 
		"----------------------------------------------------------------\n");

	/* Zoom through the highscore table from the top */
	for (i = 0; i < NUM_HIGHSCORES; i++)
	{
#ifndef COHERENT
		strftime(string, 10, "%d %b %y", localtime(&highScores[i].time));
#else
		{
			struct tm *localtm;

			localtm=localtime(&highScores[i].time);

			sprintf(string,"%d %d %d",localtm->tm_mday,localtm->tm_mon+1,localtm->tm_year);
		}
#endif

		if (ntohl(highScores[i].score) > 0)
		{
			/* Print out the actual record */
			fprintf(stdout, "%d\t%ld\t  %ld\t%d'%d\"\t%s  %s\n", 
				i + 1, ntohl(highScores[i].score), 
				ntohl(highScores[i].level),
				highScores[i].gameTime / 60, highScores[i].gameTime % 60, 
				string, highScores[i].name);
		}
	}

	/* Print a trailing line to make the table look good */
	fprintf(stdout, 
		"----------------------------------------------------------------\n");
	fflush(stdout);

	/* Now display the personal highscore table */

	InitialiseHighScores();

	/* Must have table initialised with scores */
	if (ReadHighScoreTable(PERSONAL) == False)
		InitialiseHighScores();

	/* Print out a pretty title message for scores */
	fprintf(stdout, "\nPersonal Best\n\n");
	fprintf(stdout, "Rank\tScore\t  Level\tTime\tDate       Name\n");
	fprintf(stdout, 
		"----------------------------------------------------------------\n");

	/* Zoom through the highscore table from the top */
	for (i = 0; i < NUM_HIGHSCORES; i++)
	{
#ifndef COHERENT
		strftime(string, 10, "%d %b %y", localtime(&highScores[i].time));
#else
		{
			struct tm *localtm;

			localtm=localtime(&highScores[i].time);

			sprintf(string,"%d %d %d",localtm->tm_mday,localtm->tm_mon+1,localtm->tm_year);
		}
#endif

		if (ntohl(highScores[i].score) > 0)
		{
			/* Print out the actual record */
			fprintf(stdout, "%d\t%ld\t  %ld\t%d'%d\"\t%s  %s\n", 
				i + 1, ntohl(highScores[i].score), 
				ntohl(highScores[i].level),
				highScores[i].gameTime / 60, highScores[i].gameTime % 60, 
				string, highScores[i].name);
		}
	}

	/* Print a trailing line to make the table look good */
	fprintf(stdout, 
		"----------------------------------------------------------------\n");
	fflush(stdout);
}

#if NeedFunctionPrototypes
int GetHighScoreRanking(u_long score)
#else
int GetHighScoreRanking(score)
	u_long score;
#endif
{
	int i;

	/* Must have table initialised with scores */
	if (ReadHighScoreTable(GLOBAL) == False)
		InitialiseHighScores();

	/* Zoom through the highscore table from the top */
	for (i = 0; i < NUM_HIGHSCORES; i++)
	{
		/* Is my score better than theirs */
		if (score >= ntohl(highScores[i].score))
			return (i + 1);
	}

	/* Not even in highscore table yet! */
	return -1;
}

#if NeedFunctionPrototypes
static void ShiftScoresDown(int index, u_long score, u_long level, 
	time_t gameTime, char *name)
#else
static void ShiftScoresDown(index, score, level, gameTime, name)
	int index;
	u_long score;
	u_long level;
	time_t gameTime;
	char *name;
#endif
{
	/* This function will shift all score below the index down
	 * towards the end and kill off the last dude. Sorry mate.
	 */
	int i;

	/* Move all the scores down one notch */
	for (i = NUM_HIGHSCORES-1; i > index; i--)
	{
		/* Shift the scores down one notch */
		highScores[i].score 	= highScores[i-1].score;
		highScores[i].level 	= highScores[i-1].level;
		highScores[i].time 		= highScores[i-1].time;
		highScores[i].gameTime 	= highScores[i-1].gameTime;
		strcpy(highScores[i].name, highScores[i-1].name);
	}

	/* Add our new high score to the high score table */
	highScores[index].score 	= htonl(score);
	highScores[index].level 	= htonl(level);
	highScores[index].gameTime 	= gameTime;
	highScores[index].time 		= time(NULL);
	strcpy(highScores[index].name, name);
}

#if NeedFunctionPrototypes
char *getUsersFullName(void)
#else
char *getUsersFullName()
#endif
{
	struct passwd   *pass;
	char *comma;
    char *cp1, *cp2;
    static char fullname[80];

	/* Get user information from password file */
	if (!(pass = getpwuid(getuid())))
		return("Anonymous?");		/* Unknown user oops. */

	/* find a comma indicating further info after name */
	comma = index(pass->pw_gecos, ',');

	/* NULL out the comma */
	if (comma) *comma = '\0';

    cp1 = pass->pw_gecos;
    cp2 = fullname;

	/* Search through the gecos field looking for an '&' which on very
     * old UNIX systems is supposed to be the users user name with the
     * first letter uppercased.
	 */
    while(*cp1)
    {
		/* Look for the '&' symbol */
    	if(*cp1 != '&')
        	*cp2++ = *cp1++;
        else
        {
			/* A ha. Now copy the users name to be in place of '&' */
            strcpy(cp2, pass->pw_name);
			
			/* Convert the first letter to uppercase. */
            if(islower(*cp2)) 
				*cp2 = toupper(*cp2);

			/* Continue with the remaining string */
            while(*cp2) cp2++;
            	cp1++;
        }
    }

	/* Return their name without any trailing stuff */
	return(fullname);
}

#if NeedFunctionPrototypes
static void DeleteScore(int index)
#else
static void DeleteScore(index)
	int index;
#endif
{
	/* Delete the given score and shift all others up to fill in the hole */
	int i;

	/* Move all the scores down one notch */
	for (i = index; i < NUM_HIGHSCORES-1; i++)
	{
		/* Shift the scores up one notch */
		highScores[i].score 	= highScores[i+1].score;
		highScores[i].level 	= highScores[i+1].level;
		highScores[i].time 		= highScores[i+1].time;
		highScores[i].gameTime 	= highScores[i+1].gameTime;
		strcpy(highScores[i].name, highScores[i+1].name);
	}

	highScores[i].score 	= htonl((u_long)0);
	highScores[i].level 	= htonl((u_long)1);
	highScores[i].gameTime 	= 0;
	highScores[i].time 		= time(NULL);
	strcpy(highScores[i].name, "To be announced!");
}

#if NeedFunctionPrototypes
int CheckAndAddScoreToHighScore(u_long score, u_long level, time_t gameTime,
	int type)
#else
int CheckAndAddScoreToHighScore(score, level, gameTime, type)
	u_long score;
	u_long level;
	time_t gameTime;
	int type;
#endif
{
	int i;
	int id = -1;
	char name[80];

	/* Lock the file for me only */
	if (type == GLOBAL)
		id = LockUnlock(F_LOCK);

	/* Read in the lastest scores */
	if (ReadHighScoreTable(type) == False)
		InitialiseHighScores();

	/* Speed up by obtaining users name */
	strcpy(name, getUsersFullName());

	if (type == GLOBAL)
	{
		/* Go through the highscore table */
		for (i = 0; i < NUM_HIGHSCORES; i++)
		{
			if (strcmp(highScores[i].name, name) == 0)
			{
				/* Can the last score be added to the highscores */
				if (score > ntohl(highScores[i].score))
				{
					/* Delete and move up all old scores */
					DeleteScore(i);

					break;
				}
				else
				{
					/* Don't add as score is smaller */
					return False;
				}
			}
		}	/* for */

		/* Now add the new score into the table */
		for (i = 0; i < NUM_HIGHSCORES; i++)
		{
			/* Can the last game be added to the highscores */
			if (score > ntohl(highScores[i].score))
			{
				ShiftScoresDown(i, score, level, gameTime, name);

				/* Add to the highscore by writing it out */
				(void) WriteHighScoreTable(type);

				/* Unlock the file now thanks */
				if (id != -1) LockUnlock(F_ULOCK);

				/* Yes - it was placed in the highscore */
				return True;
			}
		}

		/* Unlock the file now thanks */
		if (id != -1) LockUnlock(F_ULOCK);

		/* Not even a highscore - loser! */
		return False;
	}
	else	/* Type == PERSONAL */
	{
		/* Go through the highscore table */
		for (i = 0; i < NUM_HIGHSCORES; i++)
		{
			/* Can the last game be added to the highscores */
			if (score > ntohl(highScores[i].score))
			{
				ShiftScoresDown(i, score, level, gameTime, name);

				/* Add to the highscore by writing it out */
				(void) WriteHighScoreTable(type);

				/* Yes - it was placed in the highscore */
				return True;
			}
		}

		/* Not even a highscore - loser! */
		return False;
	}
}

#if NeedFunctionPrototypes
static void SortHighScores(void)
#else
static void SortHighScores()
#endif
{
	int i, j;
	highScoreEntry tempHighScore;

	/* The old bubble sort strikes again :-) */
	for (i = 0; i < NUM_HIGHSCORES - 1; ++i)
	{
		for (j = NUM_HIGHSCORES - 1; j > i; --j)
		{
			/* Who has the higher score */
			if (highScores[j-1].score < highScores[j].score)
			{
				/* Swap the entries around - use memcpy in future */
				tempHighScore.score 		= highScores[j-1].score;
				tempHighScore.level 		= highScores[j-1].level;
				tempHighScore.gameTime 		= highScores[j-1].gameTime;
				strcpy(tempHighScore.name, highScores[j-1].name);

				highScores[j-1].score 		= highScores[j].score;
				highScores[j-1].level 		= highScores[j].level;
				highScores[j-1].gameTime 	= highScores[j].gameTime;
				strcpy(highScores[j-1].name, highScores[j].name);

				highScores[j].score 		= tempHighScore.score;
				highScores[j].level 		= tempHighScore.level;
				highScores[j].gameTime 		= tempHighScore.gameTime;
				strcpy(highScores[j].name, tempHighScore.name);
			}
		}
	}
}

#if NeedFunctionPrototypes
static void InitialiseHighScores(void)
#else
static void InitialiseHighScores()
#endif
{
	int i;

	/* Loop down table clearing everything out */
	for (i = 0; i < NUM_HIGHSCORES; i++)
	{
		/* Null out all entries */
		highScores[i].score = htonl((u_long) 0);
		highScores[i].level = htonl((u_long) 1);
		highScores[i].gameTime = 0;
		highScores[i].time = time(NULL);
		strcpy(highScores[i].name, "To be announced!");
	}
}

#if NeedFunctionPrototypes
static char *GetHomeDir(void)
#else
static char *GetHomeDir()
#endif
{
    int uid;
    struct passwd *pw;
    register char *ptr;
	static char dest[MAXPATHLEN];

    /* This function will return the home directory of the user
     * by either using the HOME environment variable or constructing
     * it through the USER environment variable.
     */

    if ((ptr = getenv("HOME")) != NULL)
        (void) strcpy(dest, ptr);
    else
    {
        /* HOME variable is not present so get USER var */
        if ((ptr = getenv("USER")) != NULL)
            pw = getpwnam(ptr);
        else
        {
            /* Obtain user id etc. */
            uid = getuid();
            pw = getpwuid(uid);
        }

        if (pw)
            (void) strcpy(dest, pw->pw_dir);
        else
            *dest = '\0';
    }

    /* This will be NULL on error or the actual path */
    return dest;
}

#if NeedFunctionPrototypes
int ReadHighScoreTable(int type)
#else
int ReadHighScoreTable(type)
	int type;
#endif
{
	/* Read the high score table into memory structure */
	FILE *hsfp;
	int i;
	char filename[MAXPATHLEN];
	char *str;

	/* Are we going to use the global or personal highscore file */
	if (type == GLOBAL)
	{
		/* Use the environment variable if it exists */
		if ((str = getenv("XBOING_SCORE_FILE")) != NULL)
			strcpy(filename, str);
		else
			strcpy(filename, HIGH_SCORE_FILE);
	}
	else
		sprintf(filename, "%s/.xboing-scores", GetHomeDir());

	/* Open the high score file */
    if ((hsfp = fopen(filename, "r")) == NULL)
	{
		/* Cannot open the high score file */
		ErrorMessage("Warning: Cannot open high score file for reading.");
		return False;
	}

	/* Read all high score entries */
	for (i = 0; i < NUM_HIGHSCORES; i++)
	{
		/* Read the highscore entry */
    	if (fread(&highScores[i], sizeof(highScoreEntry), 1, hsfp) != 1)
		{
			if (fclose(hsfp) < 0)
				ErrorMessage("Warning: Cannot close high score file.");
			return False;
		}
	}

	/* Close the high score file */
	if (fclose(hsfp) < 0)
		ErrorMessage("Warning: Cannot close high score file.");

	return True;
}


#if NeedFunctionPrototypes
int WriteHighScoreTable(int type)
#else
int WriteHighScoreTable(type)
	int type;
#endif
{
	/* write the high score table to the high score file */
	FILE *hsfp;
	int i;
	char filename[MAXPATHLEN];
	char *str;

	/* Make sure the table is sorted */
	SortHighScores();

	/* Are we going to use the global or personal highscore file */
	if (type == GLOBAL)
	{
		/* Use the environment variable if it exists */
		if ((str = getenv("XBOING_SCORE_FILE")) != NULL)
			strcpy(filename, str);
		else
			strcpy(filename, HIGH_SCORE_FILE);
	}	
	else
		sprintf(filename, "%s/.xboing-scores", GetHomeDir());

	/* Open the high score file */
    if ((hsfp = fopen(filename, "w+")) == NULL)
	{
		/* Cannot open the high score file */
		ErrorMessage("Warning: Cannot open high score file for writing.");
		return False;
	}

	/* Write out all high score entries */
	for (i = 0; i < NUM_HIGHSCORES; i++)
	{
		/* Write the highscore entry */
    	if (fwrite(&highScores[i], sizeof(highScoreEntry), 1, hsfp) != 1)
		{
			if (fclose(hsfp) < 0)
				ErrorMessage("Warning: Cannot close high score file.");
			return False;
		}
	}
	
	/* Make sure that the permissions on the file are rw for everyone */
#ifndef COHERENT
	fchmod(fileno(hsfp), 0666);
#else
	chmod(filename, 0666);
#endif
	
	/* Close the high score file */
	if (fclose(hsfp) < 0)
		ErrorMessage("Warning: Cannot close high score file.");

	return True;
}

#if NeedFunctionPrototypes
void RedrawHighScore(Display *display, Window window)
#else
void RedrawHighScore(display, window)
	Display *display;
	Window window;
#endif
{
	/* Draw the title screen for highscore table */
	DoTitle(display, window);
}

#if NeedFunctionPrototypes
void FreeHighScore(Display *display)
#else
void FreeHighScore(display)
	Display *display;
#endif
{
	/* Free up those memory leaks thanks */
	if (titlePixmap) 	XFreePixmap(display, titlePixmap);
	if (titlePixmapM) 	XFreePixmap(display, titlePixmapM);
}

#if NeedFunctionPrototypes
void ResetHighScore(int type)
#else
void ResetHighScore(type)
	int type;
#endif
{
	HighScoreState = HIGHSCORE_TITLE;
	nextFrame = frame + 100;
	endFrame = frame + 4000;

	/* Reset the sparkles for the names */
	sparkley = 180 + textFont->ascent + 20;
	sindex = 0;
	si = 0;
	scoreType = type;
}

#if NeedFunctionPrototypes
static int LockUnlock(int cmd)
#else
static int LockUnlock(cmd)
	int cmd;
#endif
{
	int inter;
	char filename[1024];
	char *str;

	/* Use the environment variable if it exists */
	if ((str = getenv("XBOING_SCORE_FILE")) != NULL)
		strcpy(filename, str);
	else
		strcpy(filename, HIGH_SCORE_FILE);

	/* Open the highscore file for both read & write */
	inter = open(filename, O_RDWR);

	/* Ok - if successful then lock or unlock the file */
	if (inter != -1) lockf(inter, cmd, sizeof(highScores));

	/* Return success */
	return inter;
}
