#include "copyright.h"

/* SUN Audio format - Reads .au files. */

/*
 *  Include file dependencies:
 */

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <ctype.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/fcntl.h>

#ifdef AUDIO_AVAILABLE
#include <libaudio.h>
#include <audio_device.h>
#endif

#include "error.h"

#include "audio.h"

/*
 *  Internal macro definitions:
 */

#define	MAX_GAIN				(100)		/* maximum gain */
#define	SAMPLE_RATE_THRESHOLD	(.01)
#define BUFFER_SIZE				(1024 * 128)	/* 128K yup */

/*
 *  Internal type declarations:
 */

#if NeedFunctionPrototypes
static void flushAudioDevice(void);
static int reconfig(void);
#else
static void flushAudioDevice();
static int reconfig();
#endif

/*
 *  Internal variable declarations:
 */



#ifdef AUDIO_AVAILABLE
Audio_hdr		Dev_hdr;			/* audio header for device */
Audio_hdr		File_hdr;			/* audio header for file */
#endif

unsigned char	buf[BUFFER_SIZE];	/* size should depend on sample_rate */
double			Savevol;			/* saved volume level */
char			*Audio_dev = "/dev/audio";
int				Audio_fd = -1;		/* file descriptor for audio device */
int 			err;
struct stat     st;
double          vol;
int             cnt;
char			errorString[255];

#if NeedFunctionPrototypes
int SetUpAudioSystem(void)
#else
int SetUpAudioSystem()
#endif
{
#ifdef AUDIO_AVAILABLE
	/* Validate and open the audio device */
	if (stat(Audio_dev, &st) < 0)
	{
		/* No audio device so barf */
		sprintf(errorString, 
			"Error: Cannot stat audio device %s.", Audio_dev);
		ErrorMessage(errorString);
		return False;
	}

	/* Check that the audio device is a character device */
	if (!S_ISCHR(st.st_mode)) 
	{
		/* Not a character device which is not right */
		sprintf(errorString, "Error: %s is not an audio device.", Audio_dev);
		ErrorMessage(errorString);
		return False;
	}

	/* Try it quickly, first */
	Audio_fd = open(Audio_dev, O_WRONLY | O_NDELAY);
	if ((Audio_fd < 0) && (errno == EBUSY)) 
	{
		/* The audio is in use so barf */
		sprintf(errorString, "Error: %s audio device is busy.", Audio_dev);
		ErrorMessage(errorString);
		return False;
	}

	/* Ok so we cannot open it for some reason */
	if (Audio_fd < 0) 
	{
		sprintf(errorString, "Error: Cannot open audio device %s.", Audio_dev);
		ErrorMessage(errorString);
		return False;
	}

	/* Get the device output encoding configuration */
	if (audio_get_play_config(Audio_fd, &Dev_hdr) != AUDIO_SUCCESS) 
	{
		sprintf(errorString, "Error: %s is not an audio device.", Audio_dev);
		ErrorMessage(errorString);
		return False;
	}

	/* Get the previous volume so we can reset it later */
	(void) audio_get_play_gain(Audio_fd, &Savevol);

	/* Clear any active audio for the new sound */
	flushAudioDevice();

	/* Success in opening audio device */
	return True;
#endif
}

#if NeedFunctionPrototypes
void FreeAudioSystem(void)
#else
void FreeAudioSystem()
#endif
{
#ifdef AUDIO_AVAILABLE
	/* Clear any active audio for the new sound */
	flushAudioDevice();

	/*
	 * Though drain is implicit on close(), it's performed here
	 * for the sake of completeness, and to ensure that the volume
	 * is reset after all output is complete.
	 */
	(void) audio_drain(Audio_fd, FALSE);

	/* Reset the volume to the saved volume */
	(void) audio_set_play_gain(Audio_fd, &Savevol);
	
	/* Close the audio device thanks */
	(void) close(Audio_fd);
#endif
}

#if NeedFunctionPrototypes
static void flushAudioDevice(void)
#else
static void flushAudioDevice()
#endif
{
#ifdef AUDIO_AVAILABLE
	/* Flush any audio activity */
	(void) audio_flush_play(Audio_fd);
#endif
}

#if NeedFunctionPrototypes
void setNewVolume(unsigned int Volume)
#else
void setNewVolume(Volume)
	unsigned int Volume;
#endif
{
#ifdef AUDIO_AVAILABLE
	double oldVol = (Savevol * MAX_GAIN) / 100;
	double modVol = Volume * oldVol;;

	/* volume is double value between 0 and 1 */
	vol = modVol / (double) MAX_GAIN;
	err = audio_set_play_gain(Audio_fd, &vol);

	/* Check if the volume was set correctly */
	if (err != AUDIO_SUCCESS) 
	{
		/* Nup - issue an error message */
		sprintf(errorString, "Error: Unable to set output volume for %s\n",
		    Audio_dev);
		ShutDown(1, errorString);
	}
#endif
}

#if NeedFunctionPrototypes
void playSoundFile(char *filename, int volume)
#else
void playSoundFile(filename, volume)
	char *filename;
	int volume;
#endif
{
#ifdef AUDIO_AVAILABLE
	int ifd;
	char soundfile[1024];

	/* Clear any active audio for the new sound */
	flushAudioDevice();

	/* Set to the required volume */
 	setNewVolume((unsigned) volume);

	/* Construct the sounds file path */
	sprintf(soundfile, "%s/%s.au", SOUNDS_DIR, filename);

	/* Open the sound file for reading */
	if ((ifd = open(soundfile, O_RDONLY, 0)) < 0) 
	{
		/* Issue an error about not opening sound file */
		sprintf(errorString, 
			"Warning: Unable to open sound file %s.", soundfile);
		ErrorMessage(errorString);
		return;
	}

	/* Check to make sure this is an audio file */
	err = audio_read_filehdr(ifd, &File_hdr, (char *)NULL, 0);
	if (err != AUDIO_SUCCESS) 
	{
		/* Cannot understand the sound file so close file and return */
		sprintf(errorString, 
			"Warning: Unable to parse sound file %s.", soundfile);
		ErrorMessage(errorString);
		goto closeinput;
	}

	/* Check the device configuration */
	if (audio_cmp_hdr(&Dev_hdr, &File_hdr) != 0) 
	{
		/*
		 * The device does not match the input file.
		 * Wait for any old output to drain, then attempt
		 * to reconfigure the audio device to match the
		 * input data.
		 */
		if (audio_drain(Audio_fd, FALSE) != AUDIO_SUCCESS) 
		{
			/* Cannot drain the audio device so barf */
			sprintf(errorString, "Error: Unable to drain the audio device.");
			ShutDown(1, errorString);
		}

		if (!reconfig()) goto closeinput;
	}

	/* At this point, we're all ready to copy the data. */
	while ((cnt = read(ifd, (char *)buf, BUFFER_SIZE)) >= 0) 
	{
		/* If input EOF, write an eof marker */
		err = write(Audio_fd, (char *)buf, cnt);
		if (err != cnt) 
		{
			/* Did we succeed in writing all the sound out */
			sprintf(errorString, 
				"Warning: Problem while writing to audio device.");
			ErrorMessage(errorString);
			break;
		}

		if (cnt == 0) break;
	}

	if (cnt < 0) 
	{
		/* Some error - while reading - notify user */
		sprintf(errorString, 
			"Warning: Problem while reading soundfile %s", soundfile);
		ErrorMessage(errorString);
	}

closeinput:
	/* Close the sound file */
	(void) close(ifd);
#endif
}


#if NeedFunctionPrototypes
static int reconfig(void)
#else
static int reconfig()
#endif
{
#ifdef AUDIO_AVAILABLE
	/*
	 * Try to reconfigure the audio device to match the file encoding.
	 * If this fails, we should attempt to make the input data match the
	 * device encoding.  For now, we give up on this file.
	 *
	 * Returns TRUE if successful.  Returns FALSE if not.
	 */

	char	msg[AUDIO_MAX_ENCODE_INFO];

	Dev_hdr = File_hdr;
	err = audio_set_play_config(Audio_fd, &Dev_hdr);

	switch (err) 
	{
		case AUDIO_SUCCESS:
			return (TRUE);

		case AUDIO_ERR_NOEFFECT:
			/*
			 * Couldn't change the device.
			 * Check to see if we're nearly compatible.
			 * audio_cmp_hdr() returns >0 if only sample rate difference.
			 */
			if (audio_cmp_hdr(&Dev_hdr, &File_hdr) > 0) 
			{
				double	ratio;
	
				ratio = (double) abs((int)
				    (Dev_hdr.sample_rate - File_hdr.sample_rate)) /
				    (double) File_hdr.sample_rate;
	
				if (ratio <= SAMPLE_RATE_THRESHOLD) 
					return (TRUE);
	
				return (FALSE);
			}
	
			(void) audio_enc_to_str(&File_hdr, msg);
	
			return (FALSE);
	
		default:
			sprintf(errorString, 
				"Error: I/O error (set config) audio device.");
			ShutDown(1, errorString);
	}
#endif
}
