/* This program is Copyright (c) 1991 David Allen.  It may be freely
   distributed as long as you leave my name and copyright notice on it.
   I'd really like your comments and feedback; send e-mail to
   allen@viewlogic.com, or send us-mail to David Allen, 10 O'Moore
   Avenue, Maynard, MA 01754. */

/* This file contains the wind computation routines. */

#include "const.h"
#include "clim.h"
#define MAXPRESS 255

/* Input arrays are pr, from pressure.c, and l, from main.c.  The array
   hl is used as temporary storage for highs and lows, while p is used
   to store the smoothed pressure map.  The output array, wd, contains
   an edge map with wind directions. */

extern unsigned char pr[MAXB][MAXX][MAXY], l[MAXX][MAXY];
unsigned char wd[MAXB][MAXX][MAXY];
static char hl[2][MAXX][MAXY];
static unsigned char p[MAXX][MAXY];

/* The externs below are parameters defined in main.c.  The other two ints
   are parameters described in params.doc. */

extern int XSIZE, YSIZE, BSIZE, PRINTMODE;
int BARSEP = 16, PRINTWIND = 0;


windpar (s) char *s; {
   /* This function is called by mainpar() in main.c; it simply checks to
   see if the parameter is defined in this file.  If so, it returns true. */
   if (CMP ("BARSEP")) getlng (&BARSEP, M_WIND);
   else if (CMP ("PRINTWIND")) getlng (&PRINTWIND, M_WIND);
   else return (0);
   return (1); }


windcomp () {
   /* This is the main function in this file; it calls getpress() to create
   a smoothed pressure map, then getwind() to put isobars (wind lines) on
   the output map.  The last step makes sure that contradictory winds are
   removed, such as N and S winds in the same square. */

   register int i, j, x, buf;

   for (buf=0; buf<BSIZE; buf++) {
      status (M_WIND, buf); getpress (buf); getwind (buf);
      for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++) {
         x = wd[buf][i][j];
         if (x & N) x &= (~S); if (x & E) x &= (~W);
         wd[buf][i][j] = x; } }

   if (PRINTWIND) for (buf=0; buf<BSIZE; buf++) {
      if (PRINTMODE == PRINTMODE_GREY) 
         putmat ("WIND", buf, PRINTMODE_SHORT, l, wd[buf]);
      else putmat ("WIND", buf, PRINTMODE_SHORT, wd[buf], 0); } }


windraw (n) int n; { draw (DRAW_LAND, LINE_DIAG, l, wd[n]); }
   /* This function calls draw with the right arguments to display wind */


getpress (buf) int buf; {
   /* This function takes the high and low markings from pressure.c and creates
   a smoothed function.  Highs turn into MAXPRESS and lows turn into 0. */

   register int i, j;

   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++) {
      /* Zero out the arrays to be used */
      wd[buf][i][j] = 0; hl[0][i][j] = 0; hl[1][i][j] = 0;

      /* Fill hl[0] with the low pressure zones, and hl[1] with highs */
      if      (pr[buf][i][j] == PR_LOW)  hl[0][i][j] = -1;
      else if (pr[buf][i][j] == PR_HIGH) hl[1][i][j] = -1;
      else if (pr[buf][i][j] == PR_HEQ)  hl[0][i][j] = -1; }

   /* Set each square in hl[0] to the distance from that square to the */
   /* nearest low, and each square in hl[1] to the distance to a high. */
   range (hl[0]); range (hl[1]);

   /* The final pressure, in array p, is zero if a low is there and */
   /* MAXPRESS if a high is there.  Otherwise, the pressure in a square is */
   /* proportional to the ratio of (distance from the square to the nearest */
   /* low) to (total of distance from nearest high and nearest low).  This */
   /* gives a smooth curve between the extremes. */
   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++) {
      if (hl[1][i][j] == -1) p[i][j] = MAXPRESS;
      else if (hl[0][i][j] == -1) p[i][j] = 0;
      else p[i][j] = (MAXPRESS*hl[0][i][j]) / (hl[0][i][j] + hl[1][i][j]); } }


getwind (buf) int buf; {
   /* This function draws isobars around the pressure map created above.  These
   isobars are the directions of wind flow.  The isobars are given a direction
   depending on whether the square is above or below the heat equator; north of
   the heat equator, the winds blow counterclockwise out from a low, while
   south of it, the opposite is true. */

   register int i, j; int a, b, e, bar;

   /* Step from 0 to MAXPRESS by BARSEP; bar is the pressure for which this */
   /* isobar will be drawn. */
   for (bar=BARSEP; bar<=MAXPRESS; bar+=BARSEP) {
      checkmouse ();
      for (i=0; i<XSIZE; i++) for (e=0, j=0; j<YSIZE; j++) {

         /* Set e if this square is south of the heat equator */
         a = p[i][j]; if (pr[buf][i][j] == 3) e = 1;

         /* Provided the square is not at the top of the array, compare the */
         /* pressure here to the pressure one square up.  This gives the */
         /* direction of the wind in terms of east / west flow. */
         if (j) {
            b = p[i][j-1];
            if ((a < bar) && (b >= bar)) wd[buf][i][j] |= (e ? E : W);
            if ((a >= bar) && (b < bar)) wd[buf][i][j] |= (e ? W : E); }

         /* Compare the pressure here to the pressure one square to the left */
         /* (including wraparound); this gives the wind direction in terms */
         /* of north / south flow. */
         b = i ? p[i-1][j] : p[XSIZE-1][j];
         if ((a < bar) && (b >= bar)) wd[buf][i][j] |= (e ? N : S);
         if ((a >= bar) && (b < bar)) wd[buf][i][j] |= (e ? S : N); } } }
