/* This program is Copyright (c) 1991 David Allen.  It may be freely
   distributed as long as you leave my name and copyright notice on it.
   I'd really like your comments and feedback; send e-mail to
   allen@viewlogic.com, or send us-mail to David Allen, 10 O'Moore Ave,
   Maynard, MA 01754. */

/* This is the file containing all of the important functions except for
   trysplit (), which splits a continent into pieces.  Also, all of the main
   arrays are declared here, even a couple that are only used by functions in
   tec2.c.  The array declarations are first, followed by the sequencing
   function onestep () and some miscellaneous routines including the text
   output routines; initialization routines and the routines that do all
   the interesting stuff are last. */

#include "const.h"
#include "tec.h"

/* These are the parameters and their default values; each is defined in
   params.doc */ 
int XSIZE = 100,	YSIZE = 100,	MAXSTEP = 100;
int MAXBUMP = 50,	BUMPTOL = 50,	UNDERSCAN = 0;
int HYDROPCT = 70,	DRAWEVERY = 0,	PRINTMODE = PRINTMODE_NONE;
int ZINIT = 22,		ZSUBSUME = 16,	ZCOAST = 16;
int ZSHELF = 8,		ZMOUNTAIN = 48, ZERODE = 16;
int RIFTPCT = 40,	DOERODE = 1,	ERODERND = 4;
int MAXCTRTRY = 50,	RIFTDIST = 5,	BENDEVERY = 6;
int BENDBY = 50,	SPEEDRNG = 300,	SPEEDBASE = 200;

double MR [] = { 1.0,1.0,1.0,0.7,0.4,0.1,-0.2,-0.5,-0.8,-1.0 };
int MAXLIFE = 10; /* Length of MR vector */
int change[1]; /* needed for fileio */

/* The following arrays are global and most are used by functions in both
   source files.  The two main ones are m and t.  Each is set up to be two
   2-d arrays, where each array is the size of the whole world.  M is the
   map; elements in m are indices of plates, showing which squares are
   covered by which plate.  T is the topography; elements in t are altitudes. */
 
char m[2][MAXX][MAXY]; unsigned char t[2][MAXX][MAXY];
 
/* Several arrays are used by the binary blob segmenter, segment() in tec2.c.
   These include r, which is used to store fragment indices; many fragments
   make up one region during a segmentation.  Kid is a lookup table; fragment
   k belongs to region kid[k] after a segmentation is finished.  Karea[k]
   is the area of fragment k. */
 
char r[MAXX][MAXY], kid[MAXFRAG]; int karea [MAXFRAG];
 
/* The merge routine gets information from the move routine; when the move
   routine puts a square of one plate on top of another plate, that information
   is recorded in the merge matrix mm. */
 
char mm[MAXPLATE][MAXPLATE];
 
/* The erosion routine needs an array to store delta information; during an
   erosion, the increases or decreases in elevation are summed in e and then
   applied all at once to the topography. */
 
char e[MAXX][MAXY];
 
/* Several routines need temporary storage for areas and plate identifiers. */
 
int tarea[MAXPLATE]; int ids[MAXPLATE];
 
/* The plates in use are stored in this data structure.  Dx,dy are the
   values to move by THIS STEP ONLY; odx,ody are the permanent move
   values; rx,ry are the remainder x and y values used by newdxy() to
   determine dx,dy; age is the age of the plate, in steps; area is the
   area of the plate, in squares; id is the value in the m array which
   corresponds to this plate; next is a pointer to the next occupied
   element of the plate array. */
 
struct plate p [MAXPLATE];
 
/* The linked list header for available plates and used plates are global,
   as is the step counter.  */
 
int pavail, phead, step;
 
 
onestep () {
   /* This is the sequencing routine called by main once per step.
   It just calls the important subfunctions in order:
   - trysplit   finds a plate to break up, and computes new velocities
   - newdxy     computes the deltas to move each plate this step
   - move       moves the plates
   - merge      determines results when plates rub together
   - erode      erodes the terrain, adding or subtracting altitude
   - draw       draw the resulting array once every DRAWEVERY steps
   The m and t arrays are double-buffered in the sense that operations go
   from m[0] to m[1] or vice-versa; src and dest determine which is which. */
 
   int src, dest;
 
   src = step % 2; dest = 1 - src;
   if (rnd (100) < RIFTPCT) trysplit (src);
   newdxy ();
   move (src, dest);
   merge (dest);
   if (DOERODE) erode (dest);
   draw (DRAW_TEC, LINE_NONE, t[dest], 0);

   if (DRAWEVERY) if (step && !(step % DRAWEVERY)) tecst (dest);
   if (!DRAWEVERY && (step == MAXSTEP - 1)) tecst (dest); }
 
 
palloc () {
   /* Allocate a plate from the array and return its index.  All the fields
   of the plate are initialized to 0, except `next'.  That field is used to
   link together the plate structures in use.  */
 
   int x;

   if (!pavail) panic ("No more objects");
   x = pavail; pavail = p[x].next;
   p[x].next = phead; phead = x;
   p[x].area = 0; p[x].age = 0;
   p[x].rx = 0; p[x].ry = 0;
   p[x].odx = 0; p[x].ody = 0;
   p[x].dx = 0; p[x].dy = 0;
   return (x); }
 
 
pfree (n) int n; {
   /* Return a plate array element to the pool of available elements.
   To check for infinite loops, the variable guard is incremented
   at each operation; if the number of operations exceeds the maximum
   possible number, the program panics. */
 
   int i, guard = 0;
 
   if (phead == n) phead = p[n].next;
   else {
      for (i=phead; p[i].next!=n; i=p[i].next)
         if (++guard > MAXPLATE) panic ("Infinite loop in pfree");
      p[i].next = p[n].next; }
   p[n].next = pavail; pavail = n; }
 

mainpar (s) char *s; { 
   if      (CMP ("XSIZE"))     getdim  (&XSIZE, 0);
   else if (CMP ("YSIZE"))     getdim  (&YSIZE, 0);
   else if (CMP ("MOVERATE"))  getdvec (&MAXLIFE, MR, 0);
   else if (CMP ("MAXSTEP"))   getlng  (&MAXSTEP, 0);
   else if (CMP ("MAXBUMP"))   getlng  (&MAXBUMP, 0);
   else if (CMP ("BUMPTOL"))   getlng  (&BUMPTOL, 0);
   else if (CMP ("DRAWEVERY")) getlng  (&DRAWEVERY, 0);
   else if (CMP ("PRINTMODE")) getlng  (&PRINTMODE, 0);
   else if (CMP ("HYDROPCT"))  getlng  (&HYDROPCT, 0);
   else if (CMP ("ZINIT"))     getlng  (&ZINIT, 0);
   else if (CMP ("ZSUBSUME"))  getlng  (&ZSUBSUME, 0);
   else if (CMP ("ZCOAST"))    getlng  (&ZCOAST, 0);
   else if (CMP ("ZSHELF"))    getlng  (&ZSHELF, 0);
   else if (CMP ("ZMOUNTAIN")) getlng  (&ZMOUNTAIN, 0);
   else if (CMP ("ZERODE"))    getlng  (&ZERODE, 0);
   else if (CMP ("RIFTPCT"))   getlng  (&RIFTPCT, 0);
   else if (CMP ("DOERODE"))   getlng  (&DOERODE, 0);
   else if (CMP ("ERODERND"))  getlng  (&ERODERND, 0);
   else if (CMP ("MAXCTRTRY")) getlng  (&MAXCTRTRY, 0);
   else if (CMP ("RIFTDIST"))  getlng  (&RIFTDIST, 0);
   else if (CMP ("BENDEVERY")) getlng  (&BENDEVERY, 0);
   else if (CMP ("BENDBY"))    getlng  (&BENDBY, 0);
   else if (CMP ("SPEEDBASE")) getlng  (&SPEEDBASE, 0);
   else if (CMP ("SPEEDRNG"))  getlng  (&SPEEDRNG, 0);
   else if (CMP ("UNDERSCAN")) getlng  (&UNDERSCAN, 0);
   else return (0);
   return (1); }


tecst (src) int src; {
   /* This function is called whenever map output is called for.  It looks
   at the parameter `printmode' to decide between long text, simple text,
   and PostScript output formats.  Note that the default for this
   function is no output at all, corresponding to PRINTMODE_NONE.  If only
   one output map is desired, then move the coastline up or down to meet the
   desired hydrographic percentage. */

   register int i, j, zcoast; int hist[256], goal; unsigned char sk[MAXX][MAXY];

   if (!PRINTMODE) return (0);
   if (!DRAWEVERY) {
      /* Create a histogram of the output array */
      for (i=0; i<256; i++) hist[i] = 0;
      for (i=0; i<XSIZE; i++)
         for (j=0; j<YSIZE; j++) hist[t[src][i][j]]++;

      /* Starting from the highest altitude, move down until number of */
      /* squares above water is slightly greater than the exact goal */
      goal = XSIZE * YSIZE;
      goal = (goal * (100 - HYDROPCT)) / 100;
      for (zcoast=255, i=0; zcoast>0; zcoast--)
         if ((i += hist[zcoast]) > goal) break;

      /* If the new coast level is zero, then there wasn't enough land */
      /* to meet the goal, even going right down to the ocean floor.  The */
      /* only possible result is to panic since the goal can't be met. */
      if (!zcoast) panic ("Scaled till oceans dried up");
      ZCOAST = zcoast; }

   if (PRINTMODE != PRINTMODE_SHORT) putmat ("LAND", -1, PRINTMODE, t[src], 0);
   else {
      for (i=0; i<XSIZE; i++) for (j=0; j<YSIZE; j++) {
         if (t[src][i][j] < ZCOAST) sk[i][j] = 0;
         else if (t[src][i][j] > ZMOUNTAIN) sk[i][j] = 2;
         else sk[i][j] = 1; }
      putmat ("LAND", -1, PRINTMODE, sk, 0); }
   return (0); }


double greyscale (x) int x; {
   /* Called by the PostScript print routine, this function simply computes
   the intensity from 0-1 corresponding to the altitude 0-255 */
   if (x < ZCOAST) return ((float) 0);
   return (1.0 - ((x > 128) ? 128 : x) / 128.0); }


init (s) char *s; {
   /* This is the catchall function that initializes everything.  First,
   it calls getparams() in fileio.c to allow the user to set parameters.  Next,
   it links together the plates onto the free list and starts the used list
   at empty.  The first plate is created by a fractal technique and then
   improved.  Finally, the fractal is copied to the data array and drawn.
   There are two kinds of improvement done here.  First, islands are
   eliminated by segmenting the blob and erasing all the regions except
   for the biggest.  Second, oceans inside the blob (holes) are eliminated
   by segmenting the _ocean_ and filling in all regions except the biggest. */
 
   int besti, x; register int i, j;
 
   if (s) if (*s) getparams (s); fileinit ();
 
   for (i=1; i<MAXPLATE; i++) p[i].next = i + 1;
   p[MAXPLATE-1].next = 0;
   pavail = 1; phead = 0;
 
   /* Allocate a plate structure for the first plate and make a blob */
   x = palloc (); makefrac (0, x);

   /* Segment m[0] looking for x, set besti to the largest region, */
   /* and zero out all the other regions.  This eliminates islands. */
   besti = singlefy (0, x);
   if (besti > 0) for (i=1; i<XSIZE; i++) for (j=1; j<YSIZE; j++)
      if (kid[r[i][j]] != besti) m[0][i][j] = 0;
 
   /* Segment m[0] looking for 0 (ocean), set besti to the largest region, */
   /* and fill in all the other regions.  This eliminates holes in the blob. */
   besti = singlefy (0, 0);
   if (besti > 0) for (i=1; i<XSIZE; i++) for (j=1; j<YSIZE; j++)
      if (kid[r[i][j]] != besti) m[0][i][j] = x;
 
   /* Fill the topo structure with the blob shape while finding its area */
   for (i=0; i<XSIZE; i++) for (j=0; j<YSIZE; j++)
      if (m[0][i][j]) { t[0][i][j] = ZINIT; p[x].area++; }
 
   /* Draw the blob */
   if (DRAWEVERY) draw (DRAW_TEC, LINE_NONE, t[0], 0); }
