/* This program is Copyright (c) 1991 David Allen.  It may be freely
   distributed as long as you leave my name and copyright notice on it.
   I'd really like your comments and feedback; send e-mail to
   allen@viewlogic.com, or send us-mail to David Allen, 10 O'Moore
   Avenue, Maynard, MA 01754. */

/* This file contains the routines that compute local temperatures */

#include "const.h"
#include "clim.h"

/* Some private defines.  FREEZING is 0 degrees C in Kelvin; DEG2RAD is the
   conversion factor from angular degrees to radians. */

#define FREEZING 273.0
#define PI 3.14159
#define DEG2RAD (PI / 180)

/* The input array is l, from main.c; lm is used by heatdraw().  The output
   array is ts, containing temperatures.  Array t is an unscaled copy of the
   temperatures; tmin and tmax are used for scaling, and tscale is the
   computed scale factor.  To convert the unscaled temperatures to degrees K,
   divide by TEMPSCALE. */

extern unsigned char l[MAXX][MAXY], lm[MAXX][MAXY];
unsigned char ts[MAXB][MAXX][MAXY];
int tt[MAXB][MAXX][MAXY], tmax, tmin;
double tscale; int TEMPSCALE = 10;

/* These parameters are defined in main.c */

extern int BSIZE, XSIZE, YSIZE, PRINTMODE;

/* These parameters are used here, and are described in params.doc */

int PRINTEMP = 0;
double TILT = 23.0, ECCENT = 0.0, ECCPHASE = 0.0;
double LCOS = 45.0, LCONST = 275.0, LTILT = 1.0, LSMOOTH = 0.6, LDIV = 180.0;
double OCOS = 30.0, OCONST = 275.0, OTILT = 0.2, OSMOOTH = 0.2, ODIV = 250.0;


heatpar (s) char *s; {
   /* This function is called by mainpar() in main.c; it simply tests input
   parameter names to see if they are defined in this file.  Each of the
   above ints are defined in this file.  If the input string matches here,
   the function returns true. */
   if      (CMP ("TILT"))     getdbl  (&TILT,     M_HEAT);
   else if (CMP ("ECCENT"))   getdbl  (&ECCENT,   M_HEAT);
   else if (CMP ("ECCPHASE")) getdbl  (&ECCPHASE, M_HEAT);

   else if (CMP ("LCOS"))     getdbl  (&LCOS,     M_HEAT);
   else if (CMP ("LCONST"))   getdbl  (&LCONST,   M_HEAT);
   else if (CMP ("LTILT"))    getdbl  (&LTILT,    M_HEAT);
   else if (CMP ("LSMOOTH"))  getdbl  (&LSMOOTH,  M_HEAT);
   else if (CMP ("LDIV"))     getdbl  (&LDIV,     M_HEAT);

   else if (CMP ("OCOS"))     getdbl  (&OCOS,     M_HEAT);
   else if (CMP ("OCONST"))   getdbl  (&OCONST,   M_HEAT);
   else if (CMP ("OTILT"))    getdbl  (&OTILT,    M_HEAT);
   else if (CMP ("OSMOOTH"))  getdbl  (&OSMOOTH,  M_HEAT);
   else if (CMP ("ODIV"))     getdbl  (&ODIV,     M_HEAT);

   else if (CMP ("PRINTEMP")) getlng  (&PRINTEMP, M_HEAT);
   else return (0);
   return (1); }


heatlut (x, s) int x; char *s; {
   /* After the heatcomp routine is finished, a scale factor is computed
   for converting degrees K to 0..255; this routine converts back.  Functions
   in the machine-dependent code can call here to print map keys.  The caller
   must provide a char buffer; a string containing degrees F is put there. */

   double temp;

   temp = (((double) x / tscale) + (double) tmin) / TEMPSCALE;
   temp = (temp - FREEZING) * 1.8 + 32;
   sprintf (s, "%6.1f", temp); }


heatcomp () {
   /* This is the main routine for computing temperatures.  After getheat()
   is called to do all the work, this routine takes the ints from t and
   finds the smallest and largest values.  These are used to compute a scale
   factor, tscale; the arrays ts are then filled with scaled values.  Finally,
   putmat() from main.c is called if needed to print results. */

   register int i, j, buf; char s[20];

   getheat (); tmin = 32000; tmax = 0;

   usermessage ("Scaling heat");
   /* Find minimum and maximum across all buffers */
   for (buf=0; buf<BSIZE; buf++) {
      checkmouse ();
      for (i=0; i<XSIZE; i++) for (j=0; j<YSIZE; j++) {
         if (tt[buf][i][j] < tmin) tmin = tt[buf][i][j];
         if (tt[buf][i][j] > tmax) tmax = tt[buf][i][j]; } }

   /* Compute scale; for every buffer, fill ts from t */
   tscale = 254.0 / ((double) (tmax - tmin));
   for (buf=0; buf<BSIZE; buf++) {
      for (i=0; i<XSIZE; i++) for (j=0; j<YSIZE; j++)
         ts[buf][i][j] = (tt[buf][i][j] - tmin) * tscale;
      checkmouse (); }

   if (PRINTEMP) {
      if (PRINTMODE != PRINTMODE_GREY) {
         printf ("(KEY-IN-FAREN (\n");
         for (i=0; i<8; i++) { heatlut (16 * i, s); printf (" %s", s); }
         printf ("\n");
         for (i=8; i<16; i++) { heatlut (16 * i, s); printf (" %s", s); }
         printf ("))\n"); }
      for (i=0; i<BSIZE; i++)
         putmat ("TEMPERATURE", i, PRINTMODE_SCALE, ts[i], lm); } }


heatdraw (n) int n; { draw (DRAW_GREY, LINE_CORN, ts[n], lm); }
   /* This function calls draw() with the right arguments to display heat */


getheat () {
   /* This function does all the work for computing temperatures.  The outermost
   loop goes through each row of the output array once, computing all buffers
   at the same time.  The loop has two inner loops: first, tland and tsea are
   filled with the temperatures found far inland and far at sea for each buffer.
   In the second loop, the weight function for each point in the latitude line
   is computed and the temperature is found for each buffer. */

   register int buf, i, j;
   double lat, lscl, sscl, x, fact, theta, delth, phase;
   double tland[MAXB], tsea[MAXB];

   lscl = DEG2RAD * 180.0 / (90.0 + LTILT * TILT);
   sscl = DEG2RAD * 180.0 / (90.0 + OTILT * TILT);
   delth = 2.0 * PI / (double) BSIZE;
   for (j=0; j<YSIZE; j++) {
      status (M_HEAT, j); checkmouse ();
      lat = 90.0 - 180.0 * (double) j / (double) YSIZE;
      for (buf=0, theta=0; buf<BSIZE; buf++, theta+=delth) {
         phase = theta + ECCPHASE; if (phase > 2.0 * PI) phase -= (2 * PI);
         fact = (1.0 + ECCENT * cos (phase)) * TEMPSCALE;
         x = (lat + cos (theta) * TILT * LTILT) * lscl;
         tland[buf] = (LCONST + LCOS * cos (x)) * fact;
         x = (lat + cos (theta) * TILT * OTILT) * sscl;
         tsea[buf]  = (OCONST + OCOS * cos (x)) * fact; }
      for (i=0; i<XSIZE; i++) {
         if (!l[i][j]) x = OSMOOTH + (countland (i, j) / ODIV);
         else x = LSMOOTH + (countland (i, j) / LDIV);
         for (buf=0; buf<BSIZE; buf++)
            tt[buf][i][j] = tsea[buf] + (tland[buf] - tsea[buf]) * x; } } }


countland (x, y) int x, y; {
   /* Called by getheat() for each square, this function looks in a 11 wide
   by 5 high box and counts the number of land squares found there.  It
   compensates for y values off the map, and wraps x values around.   The
   answer is returned. */

   register int sum=0; int jmin, jmax, j1, i0, i1;

   jmin = y - 2; if (jmin < 0) jmin = 0;
   jmax = y + 2; if (jmax >= YSIZE) jmax = YSIZE-1;
   for (j1=jmin; j1<=jmax; j1++) for (i0=-5; i0<6; i0++) {
      i1 = i0 + x; if (i1 < 0) i1 += XSIZE;
      if (i1 >= XSIZE) i1 -= XSIZE;
      sum += l[i1][j1]; }
   return (sum); }
