/* This program is Copyright (c) 1991 David Allen.  It may be freely
   distributed as long as you leave my name and copyright notice on it.
   I'd really like your comments and feedback; send e-mail to
   allen@viewlogic.com, or send us-mail to David Allen, 10 O'Moore
   Avenue, Maynard, MA 01754. */

/* This file contains all of the file input and output routines; putmat()
   is the output routine.  The function getparam() is called to start the
   parameter file input; the routines to read in the parameters follow. */

#include "const.h"
#include "clim.h"
#include <stdio.h>

/* The input is handled by one function, gettoken(), which returns one
   of the following codes each time it is called. */
#define TOK_EOF 0
#define TOK_OPEN 1
#define TOK_CLOSE 2
#define TOK_WORD 3


/* These parameters are needed by every source file; they are defined
   in main.c and described in params.doc */

extern int XSIZE, YSIZE, BSIZE, ZCOAST, PRINTMODE;

/* The file pointer is used to read the parameters from a file.  Linecount
   is incremented each time a newline is read from the file, so error
   messages can refer to a line number.  Getbuf is the buffer each word
   is read into.  Getfname is a private copy of the filename used, so it
   can be reread if needed.   The two lookup tables are used by putmat();
   scalelut is set up by init().  Change is set by any of the parameter
   input routines which change a parameter. */
static FILE *fp;
static int linecount = 1;
static char getbuf [256];
char scalelut[256], shortlut[] = "0123456789ABCDEF";
extern int change[];


usermessage (s) char *s; { fprintf (stderr, "%s\n", s); }


fileinit () { int i; for (i=0; i<256; i++) scalelut[i] = shortlut[i>>4]; }


putmat (s, buf, mode, cra, lra)
   char *s; int buf, mode; unsigned char cra[MAXX][MAXY], lra[MAXX][MAXY]; {
   register int i, j, k;

   if (mode == PRINTMODE_CLIM) { psprint (cra, lra, 1); return (0); }
   else if (PRINTMODE == PRINTMODE_GREY) {
      psprint (cra, lra, 0); return (0); }

   if (buf > -1) printf ("(%s %d (\n", s, buf);
   else printf ("(%s (\n", s);

   if (PRINTMODE == PRINTMODE_LONG)
      for (j=0, k=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++) {
         printf ("%4d", cra[i][j]); if (++k == 15) { printf ("\n"); k = 0; } }
   else if (mode == PRINTMODE_SHORT) for (j=0; j<YSIZE; j++) {
      for (i=0; i<XSIZE; i++) printf ("%c", shortlut[cra[i][j]]);
      if (j < YSIZE-1) printf ("\n"); }
   else if (mode == PRINTMODE_SCALE) for (j=0; j<YSIZE; j++) {
      for (i=0; i<XSIZE; i++) printf ("%c", scalelut[cra[i][j]]);
      if (j < YSIZE-1) printf ("\n"); }
   printf ("))\n"); return (0); }


getparams (s) char *s; {
   /* This function is called either by init() or picktype(), in main.c.
   In either case, the argument is a string which is supposed to be either
   "-", in which case stdin is opened, or a filename.  If the file doesn't
   exist, the program exits via panic().  Once the file is open, the loop
   expects to find paren-delimited pairs of (name value).  Value could be
   a vector, like (name (val1 val2)).  After a name is found, params(),
   above, is called to recognize the parameter.  If there is a syntax
   error or an unrecognized parameter, the program exits via panic().
   Each computation source file has its own parameter function; if all of
   them fail the parameter name is unknown. */

   int x;

   if (!strcmp (s, "-")) fp = stdin;
   else if (!(fp = fopen (s, "r"))) panic ("Can't find input file");
   while ((x = gettoken (getbuf)) != TOK_EOF) {
      if (x != TOK_OPEN) geterr ("expected open paren");
      if (gettoken (getbuf) != TOK_WORD) geterr ("expected param name");
      if (!mainpar  (getbuf)) geterr ("unknown parameter name");
      if (gettoken  (getbuf) != TOK_CLOSE) geterr ("expected close paren"); }
   fclose (fp); }


gettoken (s) char *s; {
   /* This is the only function which actually reads from the file.  It
   maintains a one-character private unget buffer.  It ignores initial
   whitespace, but counts newlines in order to maintain an accurate linecount.
   Open or close parentheses count as tokens, and so does any amount of text
   with no white space or parens.  The return code indicates whether an
   open or close paren, end of file, or a word was found.  If a word was
   found, it is copied into the string pointed to by s.  When a word is found,
   the character which terminated it (whitespace, EOF, or paren) is put into
   the unget buffer to be read the next time gettoken() is called. */

   static char buf = 0; char c;

   white: if (buf) { c = buf; buf = 0; } else c = getc (fp);
   switch (c) {
      case '\n': linecount++;
      case '\t':
      case ' ' : goto white;
      case EOF : return (TOK_EOF);
      case '(' : return (TOK_OPEN);
      case ')' : return (TOK_CLOSE); }
   text: if ((c==' ')||(c=='\t')||(c=='\n')||(c==')')||(c=='(')||(c==EOF)) {
      buf = c; *s = 0; return (TOK_WORD); }
   else { *s++ = c; c = getc (fp); goto text; } }


geterr (s) char *s; {
   /* Use panic() to print a nicely formatted message, containing the input
   file line number, describing why the program just spat up. */
   sprintf (getbuf, "Par error: %s on line %d", s, linecount);
   fclose (fp);
   panic (getbuf); }


getlng (x, chg) int *x, chg; { int n;
   /* Called after reading a name associated with a single int, this
   function just reads a int from the input file and uses atoi() to
   convert it to a int, stored at the address passed in.  Note that any
   text, including a string or a real, will be read in; no type checking
   is performed.  If the value is different from the previous value of the
   parameter, change[chg] is set to one. */

   if (gettoken (getbuf) != TOK_WORD) geterr ("expected int");
   n = atoi (getbuf); if (n != *x) change[chg] = 1; *x = n; }


getdbl (x, chg) double *x; int chg; { double n;
   /* This function reads in a double format number, like getlng above.  If
   the value is different from the previous value, change[chg] is set to 1. */

   if (gettoken (getbuf) != TOK_WORD) geterr ("expected double");
   n = atof (getbuf); if (n != *x) change[chg] = 1; *x = n; }


getmat (dim1, dim2, chg, m)
   int *dim1, *dim2, chg; unsigned char m[MAXX][MAXY]; {
   /* This function expects to find a vector of strings, where each string
   contains only 0-9, A-Z.  The characters are converted into the numbers
   0..36 and stored in the character array whose address is passed into the
   function.  If one of the strings is too long, or if there are too many
   strings, the function spits up.  Before exiting, the function sets the
   array limits passed in; dim1 is set to the length of the longest string,
   while dim2 is set to the number of strings.  If any of the characters,
   or either dimension, is changed, then change[chg] is set to 1. */

   int x, i = 0, j = 0, imax = -1; char c;

   if (gettoken (getbuf) != TOK_OPEN) geterr ("expected open paren");
   while ((x = gettoken (getbuf)) == TOK_WORD) {
      if (j == *dim2) geterr ("matrix too high");
      for (i=0; getbuf[i]; i++) {
         if ((c = getbuf[i] - '0') > 9) c = 10 + getbuf[i] - 'A';
         if (c != m[i][j]) change[chg] = 1; m[i][j] = c; }
      if (i > *dim1) geterr ("string too long");
      if (i > imax) imax = i; j++; }
   if (x != TOK_CLOSE) geterr ("expected close paren");
   if ((*dim1 != imax) || (*dim2 != j)) change[chg] = 1;
   *dim1 = imax; *dim2 = j; }


getlvec (dim, v, chg) int *dim, *v, chg; {
   /* This function expects to find a list of ints, starting with an open
   paren and ending with a close paren.  The parameter dim should contain the
   compiled-in array limit; if the input file contains more than this number
   of entries, the function spits up.  No type checking is done; atoi() is
   used to convert any text to an int.  On exit, the dimension is set to
   the right value.  If either the dimension, or any of the elements, have
   changed from the previous values, change[chg] is set to one. */

   int x, i = 0, n;

   if (gettoken (getbuf) != TOK_OPEN) geterr ("expected open paren");
   while ((x = gettoken (getbuf)) == TOK_WORD) {
      if (i == *dim) geterr ("vector too long");
      n = atoi (getbuf);
      if (n != v[i]) change[chg] = 1; v[i++] = n; }
   if (x != TOK_CLOSE) geterr ("expected close paren");
   if (i != *dim) change[chg] = 1; *dim = i; }


getdim (x, chg) int *x, chg; { int y;
   /* Called right after reading a name associated with an array bound,
   this function reads one int from the input file; if the value is
   greater than the default value assigned above, the user is trying
   to exceed a compiled-in limit.  That's an error. */

   if (gettoken (getbuf) != TOK_WORD) geterr ("expected int");
   y = atoi (getbuf);
   if (y > *x) geterr ("dimension exceeds reserved space");
   *x = y; change[chg] = 1; }


getdvec (dim, v, chg) int *dim; double *v; int chg; {
   /* Called right after reading a name associated with a one-dimensional
   array of doubles, this function expects to find a list of doubles
   starting with an open paren and ended by a closing paren.  The parameter
   dim contains the compiled-in array limit; if the input file contains
   more than this number of entries, the function complains.  No type checking
   is done; atof() is used to convert any text to a double.  On exit, the
   dimension is set to the correct value. */

   int x, i = 0;

   if (gettoken (getbuf) != TOK_OPEN)
      geterr ("expected open paren");
   while ((x = gettoken (getbuf)) == TOK_WORD) {
      if (i == *dim) geterr ("vector is too long");
      v[i++] = atof (getbuf); }
   if (x != TOK_CLOSE) geterr ("expected close paren");
   *dim = i; change[chg] = 1; } 


#define D ((double) 648 / XSIZE)
#define XX(x) (72 + ((x) * D))
#define YY(y) (72 + ((y) * D))

static char *psdef[] = {
   "/b { setgray moveto d 0 rlineto 0 d rlineto d neg 0 rlineto fill",
   "/v { moveto d 0 rlineto stroke",
   "/h { moveto 0 d rlineto stroke", 
   "/x { moveto d d rlineto d neg 0 rmoveto d d neg rlineto stroke",
   "/p { moveto e 0 rmoveto 0 d rlineto e neg dup rmoveto d 0 rlineto stroke",
   0 };
static double climlut[] = { 0.00, 0.00, 0.00, 0.95, 0.95, 0.85, 0.70, 0.55,
   0.40, 0.70, 0.70 };
extern double greyscale ();


psprint (cra, lra, doclim)
   unsigned char cra[MAXX][MAXY], lra[MAXX][MAXY]; int doclim; {
   register int i, j, k; double z;
   static int firstpage = 1;

   if (firstpage) {
      printf ("%%!PS-Adobe-1.0\n/d %4.2f def /e %4.2f def\n", D, D/2);
      firstpage = 0;
      for (i=0; psdef[i]; i++) printf ("%s } bind def\n", psdef[i]); }
   printf ("%6.2f %6.2f moveto\n",         XX(-0.25),      YY(-0.25));
   printf ("%6.2f %6.2f lineto\n",         XX(YSIZE+0.25), YY(-0.25));
   printf ("%6.2f %6.2f lineto\n",         XX(YSIZE+0.25), YY(XSIZE+0.25));
   printf ("%6.2f %6.2f lineto\n",         XX(-0.25),      YY(XSIZE+0.25));
   printf ("%6.2f %6.2f lineto\nstroke\n", XX(-0.25),      YY(-0.25));

   if (doclim) {
      for (j=0; j<YSIZE;j++) for (i=0; i<XSIZE; i++) {
         z = climlut[cra[i][j]];
         if (z > 0.0) printf ("%6.2f %6.2f %5.4f b\n", XX(j), YY(i), z); }
      printf ("0 setgray\n");
      for (j=0; j<YSIZE;j++) for (i=0; i<XSIZE; i++) {
         if (cra[i][j] == C_JUNGLE)
            printf ("%6.2f %6.2f p\n", XX(j), YY(i));
         if (cra[i][j] == C_SWAMP)
            printf ("%6.2f %6.2f x\n", XX(j), YY(i)); } }

   else for (j=0; j<YSIZE;j++) for (i=0; i<XSIZE; i++) {
      z = greyscale (cra[i][j]);
      if (z > 0.0) printf ("%6.2f %6.2f %5.4f b\n", XX(j), YY(i), z); }

   printf ("0 setgray\n");
   if (lra) for (j=0; j<YSIZE;j++) for (i=0; i<XSIZE; i++) {
      if (lra[i][j] & LINE_0V) printf ("%6.2f %6.2f v\n", XX(j), YY(i));
      if (lra[i][j] & LINE_0H) printf ("%6.2f %6.2f h\n", XX(j), YY(i)); }

   printf ("1 setgray\n");
   if (lra) for (j=0; j<YSIZE;j++) for (i=0; i<XSIZE; i++) {
      if (lra[i][j] & LINE_1V) printf ("%6.2f %6.2f v\n", XX(j), YY(i));
      if (lra[i][j] & LINE_1H) printf ("%6.2f %6.2f h\n", XX(j), YY(i)); }

   printf ("\nshowpage\n"); }
