/* This program is Copyright (c) 1991 David Allen.  It may be freely
   distributed as long as you leave my name and copyright notice on it.
   I'd really like your comments and feedback; send e-mail to
   allen@viewlogic.com, or send us-mail to David Allen, 10 O'Moore
   Avenue, Maynard, MA 01754. */

/* This file contains all of the general-purpose routines which are not
   machine specific: init, mainpar (called by fileio), onestep (called
   by the machine-specific control routine), and the range-finding
   function called by several computation routines. */

#include "const.h"
#include "clim.h"

/* L holds the land values everybody starts with; lm is an edge map with
   continents outlined in black and mountains outlined in white. */
unsigned char l[MAXX][MAXY], lm[MAXX][MAXY];


/* These are the parameters used by all of the climate functions.  They are
   described in params.doc. */
int XSIZE = MAXX, YSIZE = MAXY, BSIZE = MAXB;
int PRINTMODE = PRINTMODE_SHORT, MAXRANGE = 15;
int MAXSTEP = 10000, ZCOAST = 0;

int change[] = { 1, 1, 1, 1, 1, 1 }, step = 0;
extern int picktype;


init (s) char *s; {
   if (s && *s) getparams (s);
   fileinit ();
   heatcomp (); presscomp (); windcomp ();
   raincomp (); climcomp (); }


onestep () {
   switch (picktype) {
      case M_HEAT:  heatdraw  (step % MAXB); break;
      case M_PRESS: pressdraw (step % MAXB); break;
      case M_WIND:  windraw   (step % MAXB); break;
      case M_RAIN:  raindraw  (step % MAXB); break;
      case M_CLIM:  climdraw  ();            break; } }
   

mainpar (s) char *s; {
   /* This function is called by getparams() in fileio.c; it looks at each
   parameter name read from the input file.  If it recognizes the name,
   the right function is called to set that parameter.  The function
   returns true if it recognizes the parameter name, and false otherwise. */

   if      (CMP ("LAND"))       getland ();
   else if (CMP ("BSIZE"))      getlng  (&BSIZE,     M_MAIN);
   else if (CMP ("MAXRANGE"))   getlng  (&MAXRANGE,  M_MAIN);
   else if (CMP ("PRINTMODE"))  getlng  (&PRINTMODE, M_MAIN);
   else if (heatpar  (s)) { }
   else if (presspar (s)) { }
   else if (windpar  (s)) { }
   else if (rainpar  (s)) { }
   else if (climpar  (s)) { }
   else return (0);
   return (1); }


getland () { register int i, j, x;
   /* This function is called by mainpar, above, when the LAND parameter
   is encountered.  It just calls getmat in fileio.c to do the work, but
   then it also creates an edge map lm; this is used by a couple of the
   drawing routines as background. */

   getmat  (&XSIZE, &YSIZE, M_MAIN, l);

   for (j=0; j<YSIZE; j++) for (i=0, x=0; i<XSIZE; i++, x=0) {
      /* Draw a white line if a mountain is present */
      if (i) if ((l[i][j] == 2) != (l[i-1][j] == 2)) x |= LINE_1V;
      if (j) if ((l[i][j] == 2) != (l[i][j-1] == 2)) x |= LINE_1H;

      /* Draw a black line if a coast is present */
      if (i) if ((l[i][j] > 0) != (l[i-1][j] > 0))   x |= LINE_0V;
      if (j) if ((l[i][j] > 0) != (l[i][j-1] > 0))   x |= LINE_0H;

      /* If both black and white lines are present, white wins */
      if ((x & LINE_0V) && (x & LINE_1V)) x &= (~LINE_0V);
      if ((x & LINE_0H) && (x & LINE_1H)) x &= (~LINE_0H);
      lm[i][j] = x; } }


range (rr) char rr[MAXX][MAXY]; {
   /* This function is called by a number of climate routines.  It takes an
   input array with blobs of -1's on a background of 0's.  The function winds
   up replacing each 0 with the distance from that square to the nearest -1.
   The function onerange() does all the work, but it will not compute ranges
   greater than MAXRANGE.  Therefore, after onerange() is called, any remaining
   0 values must be replaced with MAXRANGE, indicating that that square is
   "very far" from any -1 value. */

   register int i, j;

   onerange (rr); checkmouse ();
   for (j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++)
      if (!rr[i][j]) rr[i][j] = MAXRANGE; }


onerange (rr) char rr[MAXX][MAXY]; {
   /* This routine consists of a loop.  Each time through the loop, every
   square is checked.  If the square is zero, it has not yet been updated.
   In that case, look to see if any adjacent squares were previously updated
   (or if they were initialized to -1).  If so, set the square to the current
   distance value, which happens to be identical to the outer loop variable.
   If, after one loop iteration, no squares have been updated, the matrix
   must be completely updated.  Stop.  To keep down run-time, a maximum
   distance value, MAXRANGE, is used as the terminating loop value. */

   register int i, j, x, k, keepgo;
   for (k=1; k<MAXRANGE; k++) {
      checkmouse ();
      for (keepgo=0, j=0; j<YSIZE; j++) for (i=0; i<XSIZE; i++)
         if (!rr[i][j]) {
            keepgo = 1;
            x = rr[i ? i-1 : XSIZE-1][j]; if (x && (x != k)) rr[i][j] = k;
            x = rr[(i<XSIZE-1) ? i+1 : 0][j]; if (x && (x != k)) rr[i][j] = k;
            if (j<YSIZE-1) { x = rr[i][j+1]; if (x && (x != k)) rr[i][j] = k; }
            if (j) { x = rr[i][j-1]; if (x && (x != k)) rr[i][j] = k; } }
      if (!keepgo) return (0); } return (0); }


status (n, i) int n, i; {
   static char *title[] = { "", "Heat","Pressure","Wind","Rain","Climate" };
   char t[256];
   sprintf (t, "%s buffer %d", title[n], i); usermessage (t); }


double greyscale (x) int x; {
   /* This function just returns the 0..1 equivalent of 0..255 */
   return ((float) ((255 - x) / 320.0) + 0.1); }
