/****************************************************************************
 * This module is based on Twm, but has been siginificantly modified 
 * by Rob Nation (nation@rocket.sanders.lockheed.com)
 ****************************************************************************/
/*****************************************************************************/
/**       Copyright 1988 by Evans & Sutherland Computer Corporation,        **/
/**                          Salt Lake City, Utah                           **/
/**  Portions Copyright 1989 by the Massachusetts Institute of Technology   **/
/**                        Cambridge, Massachusetts                         **/
/**                                                                         **/
/**                           All Rights Reserved                           **/
/**                                                                         **/
/**    Permission to use, copy, modify, and distribute this software and    **/
/**    its documentation  for  any  purpose  and  without  fee is hereby    **/
/**    granted, provided that the above copyright notice appear  in  all    **/
/**    copies and that both  that  copyright  notice  and  this  permis-    **/
/**    sion  notice appear in supporting  documentation,  and  that  the    **/
/**    names of Evans & Sutherland and M.I.T. not be used in advertising    **/
/**    in publicity pertaining to distribution of the  software  without    **/
/**    specific, written prior permission.                                  **/
/**                                                                         **/
/**    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD    **/
/**    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-    **/
/**    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR    **/
/**    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-    **/
/**    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA    **/
/**    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER    **/
/**    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE    **/
/**    OR PERFORMANCE OF THIS SOFTWARE.                                     **/
/*****************************************************************************/


/***********************************************************************
 * fvwm - "Feeble Virtual Window Manager"
 ***********************************************************************/

#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <string.h>
#include <sys/wait.h>
#include "fvwm.h"
#include "menus.h"
#include "misc.h"
#include "screen.h"
#include <X11/Xproto.h>
#include <X11/Xatom.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif /* SHAPE */

ScreenInfo Scr;		        /* structures for the screen */
Display *dpy;			/* which display are we talking to */

static int CatchRedirectError();	/* for settting RedirectError */
static int FvwmErrorHandler();
void newhandler(int sig);
void CreateCursors();
void NoisyExit();

XContext FvwmContext;		/* context for fvwm windows */

XClassHint NoClass;		/* for applications with no class */

int JunkX = 0, JunkY = 0;
Window JunkRoot, JunkChild;		/* junk window */
unsigned int JunkWidth, JunkHeight, JunkBW, JunkDepth, JunkMask;

/* assorted gray bitmaps for decorative borders */
#define g_width 2
#define g_height 2
static char g_bits[] = {0x02, 0x01};

#define l_g_width 4
#define l_g_height 2
static char l_g_bits[] = {0x08, 0x02};
Bool debugging = False,PPosOverride;

#ifndef NO_PAGER
extern Window Pager_w;
#endif

char **g_argv;

#ifdef SHAPE
int ShapeEventBase, ShapeErrorBase;
#endif

#define USAGE "Fvwm Version 0.975\n\nusage:  fvwm [-d dpy] [-debug]\n"

/***********************************************************************
 *
 *  Procedure:
 *	main - start of fvwm
 *
 ***********************************************************************
 */

void main(int argc, char **argv)
{
    char *display_name = NULL;
    unsigned long valuemask;	/* mask for create windows */
    XSetWindowAttributes attributes;	/* attributes for create windows */
    extern XEvent Event;
    void InternUsefulAtoms ();
    void InitVariables();
    void enterAlarm();
    int i;

    for (i = 1; i < argc; i++) 
      {
	if (strncasecmp(argv[i],"-debug",6)==0)
	  debugging = True;
	else if (strncasecmp(argv[i],"-d",2)==0)
	  {
	    if (++i >= argc)
	      fprintf (stderr,USAGE);	      
	    display_name = argv[i];
	  }
	else 
	  fprintf (stderr,USAGE);
      }

    g_argv = argv;

    newhandler (SIGINT);
    newhandler (SIGHUP);
    newhandler (SIGQUIT);
    newhandler (SIGTERM);

    if (signal (SIGSEGV, SIG_IGN) != SIG_IGN)
      signal (SIGSEGV, NoisyExit);

    signal(SIGALRM,enterAlarm);
    /* catch dying processes so they don't become zombies */
    /* Massive reduction in zombie zapping complexity for fvwm-0.97 */
    signal (SIGCHLD,SIG_IGN);
    if (!(dpy = XOpenDisplay(display_name))) 
      {
	fprintf (stderr, "fvwm:  can't open display \"%s\"\n",
		 XDisplayName(display_name));
	exit (1);
      }
    
    if (fcntl(ConnectionNumber(dpy), F_SETFD, 1) == -1) 
      {
	fprintf(stderr,"fvwm: can't mark dpy conn close-on-exec\n");
	exit (1);
      }
    
    Scr.screen = DefaultScreen(dpy);

#ifdef SHAPE
    XShapeQueryExtension(dpy, &ShapeEventBase, &ShapeErrorBase);
#endif /* SHAPE */

    InternUsefulAtoms ();

    /* Make sure property priority colors is empty */
    XChangeProperty (dpy, RootWindow(dpy,Scr.screen), _XA_MIT_PRIORITY_COLORS,
		     XA_CARDINAL, 32, PropModeReplace, NULL, 0);

    XSetErrorHandler(CatchRedirectError);
    XSelectInput(dpy, RootWindow (dpy, Scr.screen),
		 ColormapChangeMask | EnterWindowMask | PropertyChangeMask | 
		 SubstructureRedirectMask | KeyPressMask |
		 ButtonPressMask | ButtonReleaseMask | PointerMotionMask);
    XSync(dpy, 0);

    XSetErrorHandler(FvwmErrorHandler);
    CreateCursors();
    InitVariables();

    Scr.gray = XCreateBitmapFromData(dpy,Scr.Root,g_bits,g_width,g_height);
    Scr.light_gray = 
      XCreateBitmapFromData(dpy,Scr.Root,l_g_bits,l_g_width, l_g_height);

    /* read config file, set up menus, colors, fonts */
    MakeMenus();

    if(Scr.d_depth<2)
      {
	Scr.gray_pixmap = 
	XCreatePixmapFromBitmapData(dpy,Scr.Root,g_bits, g_width,g_height,
				    Scr.StdColors.fore,Scr.StdColors.back,
				    Scr.d_depth);	
	Scr.light_gray_pixmap = 
	XCreatePixmapFromBitmapData(dpy,Scr.Root,l_g_bits,l_g_width,l_g_height,
				    Scr.StdColors.fore,Scr.StdColors.back,
				    Scr.d_depth);
      }

    XSetInputFocus (dpy, Scr.NoFocusWin, RevertToParent, CurrentTime);
    Scr.TitleHeight=Scr.WindowFont.font->ascent+Scr.WindowFont.font->descent+3;

    XGrabServer(dpy);
    XSync(dpy, 0);
    if(debugging)
      XSynchronize(dpy,1);

    CaptureAllWindows();

    Scr.SizeStringWidth = XTextWidth (Scr.StdFont.font,
				      " 8888 x 8888 ", 13);
    attributes.border_pixel = Scr.StdColors.fore;
    attributes.background_pixel = Scr.StdColors.back;
    attributes.bit_gravity = NorthWestGravity;
    valuemask = (CWBorderPixel | CWBackPixel | CWBitGravity);
    Scr.SizeWindow = XCreateWindow (dpy, Scr.Root, 0, 0, 
				     (unsigned int)(Scr.SizeStringWidth +
						    SIZE_HINDENT*2),
				     (unsigned int) (Scr.StdFont.height +
						     SIZE_VINDENT*2),
				     (unsigned int) 0, 0,
				     (unsigned int) CopyFromParent,
				     (Visual *) CopyFromParent,
				     valuemask, &attributes);
    XUngrabServer(dpy);
    HandleEvents();
    return;
}

/***********************************************************************
 *
 *  Procedure:
 *      CaptureAllWindows
 *
 *   Decorates all windows at start-up
 *
 ***********************************************************************/

void CaptureAllWindows()
{
  int i,j;
  unsigned int nchildren;
  Window root, parent, *children;
  FvwmWindow *Tmp_win;

  PPosOverride = TRUE;
  XQueryTree(dpy, Scr.Root, &root, &parent, &children, &nchildren);
  /*
   * weed out icon windows
   */
  for (i = 0; i < nchildren; i++) 
    {
      if (children[i]) 
	{
	  XWMHints *wmhintsp = XGetWMHints (dpy, children[i]);
	  if (wmhintsp) 
	    {
	      if (wmhintsp->flags & IconWindowHint) 
		{
		  for (j = 0; j < nchildren; j++) 
		    {
		      if (children[j] == wmhintsp->icon_window) 
			{
			  children[j] = None;
			  break;
			}
		    }
		}
	      XFree ((char *) wmhintsp);
	    }
	}
    }
  /*
   * map all of the non-override windows
   */
  
  for (i = 0; i < nchildren; i++)
    {
      if (children[i] && MappedNotOverride(children[i]))
	{
	  XUnmapWindow(dpy, children[i]);
	  Event.xmaprequest.window = children[i];
	  HandleMapRequest ();
	}
    }
  
  XFree(children);
  /* after the windows already on the screen are in place,
   * don't use PPosition */
  PPosOverride = FALSE;
}

/***********************************************************************
 *
 *  Procedure:
 *	MappedNotOverride - checks to see if we should really
 *		put a fvwm frame on the window
 *
 *  Returned Value:
 *	TRUE	- go ahead and frame the window
 *	FALSE	- don't frame the window
 *
 *  Inputs:
 *	w	- the window to check
 *
 ***********************************************************************/
int MappedNotOverride(Window w)
{
  XWindowAttributes wa;
  
  XGetWindowAttributes(dpy, w, &wa);
  return ((wa.map_state != IsUnmapped) && (wa.override_redirect != True));
}


/***********************************************************************
 *
 *  Procedure:
 *	InternUsefulAtoms:
 *            Dont really know what it does
 *
 ***********************************************************************
 */
Atom _XA_MIT_PRIORITY_COLORS;
Atom _XA_WM_CHANGE_STATE;
Atom _XA_WM_STATE;
Atom _XA_WM_COLORMAP_WINDOWS;
Atom _XA_WM_PROTOCOLS;
Atom _XA_WM_TAKE_FOCUS;
Atom _XA_WM_SAVE_YOURSELF;
Atom _XA_WM_DELETE_WINDOW;

void InternUsefulAtoms ()
{
  /* 
   * Create priority colors if necessary.
   */
  _XA_MIT_PRIORITY_COLORS = XInternAtom(dpy, "_MIT_PRIORITY_COLORS", False);   
  _XA_WM_CHANGE_STATE = XInternAtom (dpy, "WM_CHANGE_STATE", False);
  _XA_WM_STATE = XInternAtom (dpy, "WM_STATE", False);
  _XA_WM_COLORMAP_WINDOWS = XInternAtom (dpy, "WM_COLORMAP_WINDOWS", False);
  _XA_WM_PROTOCOLS = XInternAtom (dpy, "WM_PROTOCOLS", False);
  _XA_WM_TAKE_FOCUS = XInternAtom (dpy, "WM_TAKE_FOCUS", False);
  _XA_WM_SAVE_YOURSELF = XInternAtom (dpy, "WM_SAVE_YOURSELF", False);
  _XA_WM_DELETE_WINDOW = XInternAtom (dpy, "WM_DELETE_WINDOW", False);
  return;
}

/***********************************************************************
 *
 *  Procedure:
 *	newhandler: Installs new signal handler
 *
 ***********************************************************************
 */
void newhandler(int sig)
{
  if (signal (sig, SIG_IGN) != SIG_IGN)
    signal (sig, SigDone);
}

/***********************************************************************
 *
 *  Procedure:
 *	CreateCursors - Loads fvwm cursors
 *
 ***********************************************************************
 */
void CreateCursors()
{
  /* define cursors */
  Scr.FvwmCursors[POSITION] = XCreateFontCursor(dpy,XC_top_left_corner);
  Scr.FvwmCursors[FRAME] = XCreateFontCursor(dpy, XC_top_left_arrow);
  Scr.FvwmCursors[SYS] = XCreateFontCursor(dpy, XC_hand2);
  Scr.FvwmCursors[TITLE_CURSOR] = XCreateFontCursor(dpy, XC_top_left_arrow);
  Scr.FvwmCursors[MOVE] = XCreateFontCursor(dpy, XC_fleur);
  Scr.FvwmCursors[MENU] = XCreateFontCursor(dpy, XC_sb_left_arrow);
  Scr.FvwmCursors[WAIT] = XCreateFontCursor(dpy, XC_watch);
  Scr.FvwmCursors[SELECT] = XCreateFontCursor(dpy, XC_dot);
  Scr.FvwmCursors[DESTROY] = XCreateFontCursor(dpy, XC_pirate);
  Scr.FvwmCursors[LEFT] = XCreateFontCursor(dpy, XC_left_side);
  Scr.FvwmCursors[RIGHT] = XCreateFontCursor(dpy, XC_right_side);
  Scr.FvwmCursors[TOP] = XCreateFontCursor(dpy, XC_top_side);
  Scr.FvwmCursors[BOTTOM] = XCreateFontCursor(dpy, XC_bottom_side);
}

/***********************************************************************
 *
 *  Procedure:
 *	InitVariables - initialize fvwm variables
 *
 ***********************************************************************
 */

void InitVariables()
{
  XSetWindowAttributes attributes;
  int i;

  FvwmContext = XUniqueContext();
  NoClass.res_name = NoName;
  NoClass.res_class = NoName;

  Scr.d_depth = DefaultDepth(dpy, 0);
  Scr.d_visual = DefaultVisual(dpy, 0);
  Scr.Root = RootWindow(dpy, 0);
  Scr.FvwmRoot.w = Scr.Root;
  Scr.FvwmRoot.next = 0;
  Scr.root_pushes = 0;
  Scr.pushed_window = &Scr.FvwmRoot;

  Scr.MyDisplayWidth = DisplayWidth(dpy, 0);
  Scr.MyDisplayHeight = DisplayHeight(dpy, 0);
    
  Scr.BorderWidth = BW;
  Scr.NoBorderWidth = NOFRAME_BW;
  Scr.BoundaryWidth = BOUNDARY_WIDTH;
  Scr.CornerWidth = CORNER_WIDTH;
  Scr.Focus = NULL;
  
  Scr.StdFont.font = NULL;
  Scr.StdFont.name = "fixed";
  Scr.WindowFont.name = "fixed";

  Scr.VScale = 32;
#ifndef NON_VIRTUAL  
  Scr.VxMax = 2*Scr.MyDisplayWidth;
  Scr.VyMax = 2*Scr.MyDisplayHeight;
#else
  Scr.VxMax = 0;
  Scr.VyMax = 0;
#endif
  Scr.Vx = Scr.Vy = 0;

  Scr.EdgeScrollX = Scr.EdgeScrollY = 32;
  Scr.ClickTime = 150;
  Scr.AutoRaiseDelay = -750;
  Scr.CenterOnCirculate = FALSE;
  Scr.DontMoveOff = FALSE;
  Scr.RandomPlacement = False;
#ifndef NO_ICONS
  Scr.SuppressIcons = False;
#else
  Scr.SuppressIcons = True;
#endif
  Scr.randomx = Scr.randomy = 0;
  Scr.DecorateTransients = False;
  Scr.ClickToFocus = False;
  Scr.buttons2grab = 7;

  /* create a window which will accept the keyboard focus when no other 
     windows have it */
  attributes.event_mask = KeyPressMask;
  attributes.override_redirect = True;
  Scr.NoFocusWin=XCreateWindow(dpy,Scr.Root, -10, -10, 10, 10, 0, 0,InputOnly,
			       CopyFromParent,CWEventMask|CWOverrideRedirect,
			       &attributes);
  XMapWindow(dpy, Scr.NoFocusWin);


  Scr.left_button_styles[0][0] = -55;
  Scr.left_button_styles[1][0] = -25;
  Scr.left_button_styles[0][1] = -35;
  Scr.left_button_styles[1][1] = -10;
  Scr.left_button_styles[0][2] = 1;
  Scr.left_button_styles[1][2] = 1;
  Scr.left_button_styles[0][3] = 35;
  Scr.left_button_styles[1][3] = 10;
  Scr.left_button_styles[0][4] = 56;
  Scr.left_button_styles[1][4] = 25;

  Scr.right_button_styles[0][0] = -28;
  Scr.right_button_styles[1][0] = -28;
  Scr.right_button_styles[0][1] = -50;
  Scr.right_button_styles[1][1] = -50;
  Scr.right_button_styles[0][2] = 1;
  Scr.right_button_styles[1][2] = 1;
  Scr.right_button_styles[0][3] = 50;
  Scr.right_button_styles[1][3] = 50;
  Scr.right_button_styles[0][4] = 28;
  Scr.right_button_styles[1][4] = 28;

  return;
}



/***********************************************************************
 *
 *  Procedure:
 *	Reborder - Removes fvwm border windows
 *
 ***********************************************************************
 */

void Reborder()
{
  FvwmWindow *tmp;			/* temp fvwm window structure */
  Window dumwin;
  int dstx,dsty;

  /* put a border back around all windows */

  XGrabServer (dpy);
  
  InstallWindowColormaps (0, &Scr.FvwmRoot);	/* force reinstall */
  for (tmp = Scr.FvwmRoot.next; tmp != NULL; tmp = tmp->next)
    {
      RestoreWithdrawnLocation (tmp); 
      XMapWindow (dpy, tmp->w);
    }
  XUngrabServer (dpy);
  XSetInputFocus (dpy, PointerRoot, RevertToPointerRoot,CurrentTime);
}

/***********************************************************************
 *
 *  Procedure: NoisyExit
 *	Print error messages and die. (segmentation violation)
 *
 ***********************************************************************
 */
void NoisyExit()
{
  XErrorEvent event;
  fprintf(stderr,"fvwm: Seg Fault\n");
  event.error_code = 0;
  event.request_code = 0;
  FvwmErrorHandler(dpy, &event);

  /* Attempt to do a re-start of fvwm */
  Done(0);
}




/***********************************************************************
 *
 *  Procedure:
 *	Done - cleanup and exit fvwm
 *
 ***********************************************************************
 */
SIGNAL_T SigDone()
{
  Done(0);
  SIGNAL_RETURN;
}

void Done(int restart, char *command)
{

#ifndef NON_VIRTUAL
  MoveViewport(0,0,True);
#endif

  Reborder ();
  if(restart)
    execvp(command,g_argv);
  else
    {
      XCloseDisplay(dpy);
      exit(0);
    }
}

/***********************************************************************
 *
 *  Procedure:
 *	CatchRedirectError - Figures out if there's another WM running
 *
 ***********************************************************************
 */
static int CatchRedirectError(Display *dpy, XErrorEvent event)
{
  fprintf (stderr, "fvwm: another window manager is running\n");
  exit(1);
  return 0;
}


/***********************************************************************
 *
 *  Procedure:
 *	FvwmErrorHandler - displays info on internal errors
 *
 ************************************************************************/
static int FvwmErrorHandler(Display *dpy, XErrorEvent *event)
{
  extern int move_on, resize_on,menu_on;
  extern Window last_event_window;
  extern FvwmWindow *last_event_Fvwmwindow;
  extern int last_event_type;

  /* some errors are acceptable, mostly they're caused by 
   * trying to update a lost  window */
  if((event->error_code == BadWindow)||
     (event->request_code == X_GetGeometry)||
     (event->error_code == BadDrawable))
    return 0 ;

  fprintf(stderr,"fvwm: internal error\n");
  fprintf(stderr,"      Request %d, Error %d\n", event->request_code,
	  event->error_code);
  fprintf(stderr,"      EventType: %d",last_event_type);
  fprintf(stderr,"\n");
  if(menu_on)
    fprintf(stderr,"   Menu on\n");
  if(resize_on)
    fprintf(stderr,"   Resizing\n");
  if(move_on)
    fprintf(stderr,"   Moving\n");
  return 0;
}

#ifdef COHERENT
/*
 * Copyright (c) 1987 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#if defined(LIBC_SCCS) && !defined(lint)
static char sccsid[] = "@(#)strcasecmp.c	5.6 (Berkeley) 6/27/88";
#endif /* LIBC_SCCS and not lint */

#ifdef atarist
#include <sys/types.h>
#else
#define u_char unsigned char
#endif

/*
 * This array is designed for mapping upper and lower case letter
 * together for a case independent comparison.  The mappings are
 * based upon ascii character sequences.
 */
static u_char charmap[] = {
	'\000', '\001', '\002', '\003', '\004', '\005', '\006', '\007',
	'\010', '\011', '\012', '\013', '\014', '\015', '\016', '\017',
	'\020', '\021', '\022', '\023', '\024', '\025', '\026', '\027',
	'\030', '\031', '\032', '\033', '\034', '\035', '\036', '\037',
	'\040', '\041', '\042', '\043', '\044', '\045', '\046', '\047',
	'\050', '\051', '\052', '\053', '\054', '\055', '\056', '\057',
	'\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067',
	'\070', '\071', '\072', '\073', '\074', '\075', '\076', '\077',
	'\100', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
	'\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
	'\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
	'\170', '\171', '\172', '\133', '\134', '\135', '\136', '\137',
	'\140', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
	'\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
	'\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
	'\170', '\171', '\172', '\173', '\174', '\175', '\176', '\177',
	'\200', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
	'\210', '\211', '\212', '\213', '\214', '\215', '\216', '\217',
	'\220', '\221', '\222', '\223', '\224', '\225', '\226', '\227',
	'\230', '\231', '\232', '\233', '\234', '\235', '\236', '\237',
	'\240', '\241', '\242', '\243', '\244', '\245', '\246', '\247',
	'\250', '\251', '\252', '\253', '\254', '\255', '\256', '\257',
	'\260', '\261', '\262', '\263', '\264', '\265', '\266', '\267',
	'\270', '\271', '\272', '\273', '\274', '\275', '\276', '\277',
	'\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
	'\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
	'\360', '\361', '\362', '\363', '\364', '\365', '\366', '\327',
	'\370', '\371', '\372', '\373', '\374', '\375', '\376', '\337',
	'\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
	'\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
	'\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
	'\370', '\371', '\372', '\373', '\374', '\375', '\376', '\377',
};

int
strcasecmp(s1, s2)
	const char *s1, *s2;
{
	register u_char	*cm = charmap,
			*us1 = (u_char *)s1,
			*us2 = (u_char *)s2;

	while (cm[*us1] == cm[*us2++])
		if (*us1++ == '\0')
			return(0);
	return(cm[*us1] - cm[*--us2]);
}

int
strncasecmp(s1, s2, n)
	const char *s1, *s2;
	register size_t n;
{
	register u_char	*cm = charmap,
			*us1 = (u_char *)s1,
			*us2 = (u_char *)s2;

	while ((long)(--n) >= 0 && cm[*us1] == cm[*us2++])
		if (*us1++ == '\0')
			return(0);
	return((long)n < 0 ? 0 : cm[*us1] - cm[*--us2]);
}
#endif
