/****************************************************************************
 * This module is mostly all new
 * by Rob Nation (nation@rocket.sanders.lockheed.com)
 * Copyright 1993 Robert Nation. No restrictions are placed on this code,
 * as long as the copyright notice is preserved
 ****************************************************************************/


/***************************************************************************
 * 
 * Configure.c: reads the .fvwmrc or system.fvwmrc file, interprets it,
 * and sets up menus, bindings, colors, and fonts as specified
 *
 ***************************************************************************/

#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <X11/Xos.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>
#include "fvwm.h"
#include "menus.h"
#include "misc.h"
#include "parse.h"
#include "screen.h"


char *white = "white";
char *black = "black";
char *Stdback;
char *Stdfore;
char *Hiback;
char *Hifore;
#ifndef NO_PAGER
char *Pagerback;
#endif

void     GetColors();
Pixel    GetShadow(Pixel);
Pixel    GetHilite(Pixel);
Pixel    GetColor(char *);
void	 ParsePopupEntry(char *,FILE *, char **);
void     ParseMouseEntry(char *,FILE *, char **);
void     ParseKeyEntry(char *, FILE *, char **);
void     AddToList(char *text,FILE *,name_list **);
void     AddToList2(char *text,FILE *,name_list **);
MenuRoot *NewMenuRoot(char *name);
void     AddFuncKey (char *, int, int, int, char *, int, int, MenuRoot *);
char     *stripcpy(char *);
char     *stripcpy2(char *,int);
char     *stripcpy3(char *, Bool);
#ifndef NO_PAGER
void     initialize_pager(int x, int y);
#endif
void     bad_binding(int num);
void     no_popup(char *ptr);
void     nofont(char *name);
void nocolor(char *note, char *name);
int contexts;
int mods,func,func_val_1,func_val_2;

#ifndef NO_PAGER
int pager_x,pager_y;
Window Pager_w = (Window)0;
Bool pager = FALSE;
#endif

#define MAXPOPUPS 32

unsigned PopupCount = 0;
MenuRoot *PopupTable[MAXPOPUPS];

struct config
{
  char *keyword;
  void (*action)();
  char **arg;
};

void assign_string(char *text, FILE *fd, char **arg);
void EdgeFix(char *text, FILE *fd, char **arg);
void ButtonStyle(char *text, FILE *fd, char **arg);
void SetFlag(char *text, FILE *fd, Bool *arg);
void SetCursor(char *text, FILE *fd, char  **arg);
#ifndef NO_PAGER
void ActivatePager(char *text, FILE *fd, char **arg);
#endif
void SetOneInt(char *text, FILE *fd, int *arg);
void SetSize(char *text, FILE *fd, char **arg);
void set_func(char *, FILE *, int);
void match_string(struct config *, char *, char *, FILE *);

struct config main_config[] =
{
  {"Font",              assign_string,  &Scr.StdFont.name},
  {"WindowFont",        assign_string,  &Scr.WindowFont.name},
  {"StdForeColor",      assign_string,  &Stdfore},
  {"StdBackColor",      assign_string,  &Stdback},
  {"HiForeColor",       assign_string,  &Hifore},
  {"HiBackColor",       assign_string,  &Hiback},
#ifndef NO_PAGER
  {"PagerBackColor",    assign_string,  &Pagerback},
#endif
  {"NoTitle",           AddToList,      (char **)&Scr.NoTitle},
  {"NoBorder",          AddToList,      (char **)&Scr.NoBorder},
  {"Sticky",            AddToList,      (char **)&Scr.Sticky},
  {"StaysOnTop",        AddToList,      (char **)&Scr.OnTop},
  {"CirculateSkip",     AddToList,      (char **)&Scr.CirculateSkip},
  {"Icon",              AddToList2,     (char **)&Scr.Icons},
  {"EdgeScroll",        EdgeFix,        (char **)0},
  {"ButtonStyle",       ButtonStyle,    (char **)0},
  {"RandomPlacement",   SetFlag,        (char **)&Scr.RandomPlacement},
  {"SuppressIcons",     SetFlag,        (char **)&Scr.SuppressIcons},
  {"DontMoveOff",       SetFlag,        (char **)&Scr.DontMoveOff},
  {"DecorateTransients",SetFlag,        (char **)&Scr.DecorateTransients},
  {"CenterOnCirculate", SetFlag,        (char **)&Scr.CenterOnCirculate},
  {"AutoPlaceIcons",    SetFlag,        (char **)&Scr.AutoPlaceIcons},
  {"AutoRaise",         SetOneInt,      (char **)&Scr.AutoRaiseDelay},
#ifndef NO_PAGER
  {"Pager",             ActivatePager,  (char **)1},
#endif
#ifndef NON_VIRTUAL
  {"DeskTopScale",      SetOneInt,      (char **)&Scr.VScale},
  {"BoundaryWidth",     SetOneInt,      (char **)&Scr.BoundaryWidth},
  {"ClickTime",         SetOneInt,      (char **)&Scr.ClickTime},
  {"DeskTopSize",       SetSize,        (char **)1},
#endif
  {"Mouse",             ParseMouseEntry,(char **)1},
  {"Popup",             ParsePopupEntry,(char **)1},
  {"Function",          ParsePopupEntry,(char **)1},
  {"Key",               ParseKeyEntry,  (char **)1},
  {"ClickToFocus",      SetFlag,        (char **)&Scr.ClickToFocus},
  {"Cursor",            SetCursor,      (char **)0},  
  {"",                  0,              (char **)0}
};

struct config func_config[] =
{
  {"Nop",          set_func,(char **)F_NOP},
  {"Title",        set_func,(char **)F_TITLE},
  {"Beep",         set_func,(char **)F_BEEP},
  {"Quit",         set_func,(char **)F_QUIT},
  {"Refresh",      set_func,(char **)F_REFRESH},
  {"Move",         set_func,(char **)F_MOVE},
  {"Iconify",      set_func,(char **)F_ICONIFY},
  {"Maximize",     set_func,(char **)F_MAXIMIZE},
  {"Resize",       set_func,(char **)F_RESIZE},
  {"RaiseLower",   set_func,(char **)F_RAISELOWER},
  {"Raise",        set_func,(char **)F_RAISE},
  {"Lower",        set_func,(char **)F_LOWER},
  {"Delete",       set_func,(char **)F_DELETE},
  {"Destroy",      set_func,(char **)F_DESTROY},
  {"PopUp",        set_func,(char **)F_POPUP},
  {"Function",     set_func,(char **)F_FUNCTION},
  {"Scroll",       set_func,(char **)F_SCROLL},
  {"CursorMove",   set_func,(char **)F_MOVECURSOR},
  {"Stick",        set_func,(char **)F_STICK},
  {"CirculateUp",  set_func,(char **)F_CIRCULATE_UP},
  {"CirculateDown",set_func,(char **)F_CIRCULATE_DOWN},
#ifndef NON_VIRTUAL
  {"GotoPage",     set_func,(char **)F_GOTO_PAGE},
#endif
  {"Exec",         set_func,(char **)F_EXEC},
  {"Restart",      set_func,(char **)F_RESTART},
#ifndef NO_WINDOWLIST
  {"WindowList",   set_func,(char **)F_WINDOWLIST},
#endif
  {"",                    0,(char **)0}
};
  
struct charstring 
{
  char key;
  int  value;
};


/* The keys musat be in lower case! */
struct charstring win_contexts[]=
{
  {'w',C_WINDOW},
  {'t',C_TITLE},
  {'i',C_ICON},
  {'r',C_ROOT},
  {'f',C_FRAME},
  {'s',C_SIDEBAR},
  {'1',C_L1},
  {'2',C_R1},
  {'3',C_L2},
  {'4',C_R2},
  {'5',C_L3},
  {'6',C_R3},
  {'7',C_L4},
  {'8',C_R4},
  {'9',C_L5},
  {'0',C_R5},
  {'a',C_WINDOW|C_TITLE|C_ICON|C_ROOT|C_FRAME|C_SIDEBAR|
     C_L1|C_L2|C_L3|C_L4|C_L5|C_R1|C_R2|C_R3|C_R4|C_R5},
  {0,0}
};

/* The keys musat be in lower case! */
struct charstring key_modifiers[]=
{
  {'s',ShiftMask},
  {'c',ControlMask},
  {'m',Mod1Mask},
  {'a',AnyModifier},
  {'n',0},
  {0,0}
};

void     find_context(char *, int *, struct charstring *);
char *orig_tline;

/*****************************************************************************
 * 
 * This routine is responsible for reading and parsing the config file
 *
 ****************************************************************************/
void MakeMenus()
{
  char *system_file = FVWMRC;
  char *home_name = "/.fvwmrc";
  char *home_file;
  FILE *fd = (FILE *)0;
  char line[256],*tline;
  char *Home;			/* the HOME environment variable */
  int HomeLen;			/* length of Home */
  MouseButton *mbtmp;
  
  Stdback = white;
  Stdfore = black;
  Hiback = white;
  Hifore = black;
#ifndef NO_PAGER
  Pagerback = white;
#endif

  /* initialize some lists */
  Scr.MouseButtonRoot = NULL;
  Scr.FuncKeyRoot.next = NULL;
  Scr.NoTitle = NULL;
  Scr.OnTop = NULL;
  Scr.Icons = NULL;
  Scr.CirculateSkip = NULL;
  
  Scr.DefaultIcon = NULL;

  /* find the home directory to look in */
  Home = getenv("HOME");
  if (Home == NULL)
    Home = "./";
  HomeLen = strlen(Home);

  home_file = safemalloc(HomeLen+strlen(home_name)+1);
  strcpy(home_file,Home);
  strcat(home_file,home_name);

  fd = fopen(home_file,"r");
  if(fd == (FILE *)0)
    fd = fopen(system_file,"r");
  if(fd == (FILE *)0)
    {
      fprintf(stderr,"fvwm: can't open %s or %s\n",
	      system_file,home_file);
      exit(1);
    }
  free(home_file);

  tline = fgets(line,(sizeof line)-1,fd);
  orig_tline = tline;
  while(tline != (char *)0)
    {
      while(isspace(*tline))tline++;
      if((strlen(&tline[0])>1)&&(tline[0]!='#'))
	match_string(main_config,tline,"error in config:",fd);
      tline = fgets(line,255,fd);
      orig_tline = tline;
    }

  GetColors();
#ifndef NO_PAGER
  if(pager)initialize_pager(pager_x,pager_y);
#endif
  return;
}

/*****************************************************************************
 * 
 * Copies a text string from the config file to a specified location
 *
 ****************************************************************************/
void assign_string(char *text, FILE *fd, char **arg)
{
  *arg = stripcpy(text);
}

/*****************************************************************************
 * 
 * Sets the Scroll-on-edge size 
 *
 ****************************************************************************/
void EdgeFix(char *text, FILE *fd, char **arg)
{
  switch (sscanf(text,"%d %d",&Scr.EdgeScrollX,&Scr.EdgeScrollY)) 
    {
    case 1:
      Scr.EdgeScrollY=Scr.EdgeScrollX;
      /* Fallthrough */
    case 2:
      Scr.EdgeScrollX=Scr.EdgeScrollX*Scr.MyDisplayWidth/100;
      Scr.EdgeScrollY=Scr.EdgeScrollY*Scr.MyDisplayHeight/100;
      break;
    default:
      Scr.EdgeScrollX=Scr.MyDisplayWidth/5;
      Scr.EdgeScrollY=Scr.MyDisplayHeight/5;
    }
}



/*****************************************************************************
 * 
 * Changes a button decoration style
 *
 ****************************************************************************/
void ButtonStyle(char *text, FILE *fd, char **arg)
{
  int button,w,h,n,num;

  num = sscanf(text," %d %dx%d",&button, &w, &h);
  if((num != 3)||(button>9)||(button<0))
    {
      fprintf(stderr,"fvwm: Bad button style in line %s",orig_tline);    
      return;
    }
  n=button/2;
  if((n*2) == button)
    {
      n = n - 1;
      if(n<0)n=4;
      Scr.right_button_styles[0][n] = w;
      Scr.right_button_styles[1][n] = h;
    }
  else
    {
      Scr.left_button_styles[0][n] = w;
      Scr.left_button_styles[1][n] = h;
    }
}

/*****************************************************************************
 * 
 * Changes a cursor def.
 *
 ****************************************************************************/
void SetCursor(char *text, FILE *fd, char **arg)
{
  int num,cursor_num,cursor_style;

  num = sscanf(text,"%d %d",&cursor_num, &cursor_style);
  if((num != 2)||((num==2) && (cursor_num >= MAX_CURSORS))||(cursor_num<0))
    {
      fprintf(stderr,"fvwm: Bad cursor in line %s",orig_tline);    
      return;
    }
  Scr.FvwmCursors[cursor_num] = XCreateFontCursor(dpy,cursor_style);
}

/*****************************************************************************
 * 
 * Sets a boolean flag to true
 *
 ****************************************************************************/
void SetFlag(char *text, FILE *fd, Bool *arg)
{
  *arg = TRUE;
}

/*****************************************************************************
 * 
 * Activates the Pager, and reads its location
 *
 ****************************************************************************/
#ifndef NO_PAGER
void ActivatePager(char *text, FILE *fd, char **arg)
{
  if(sscanf(text,"%d %d",&pager_x,&pager_y)!=2)
    {
      pager_x = -1;
      pager_y = -1;
    }
  pager = TRUE;
}
#endif


/*****************************************************************************
 * 
 * Reads an integer from the config file to set a specified variable
 *
 ****************************************************************************/
void SetOneInt(char *text, FILE *fd, int *arg)
{
  sscanf(text,"%d",arg);
}

/*****************************************************************************
 * 
 * Reads an 2 integers from the config file to set the desktop size
 *
 ****************************************************************************/
void SetSize(char *text, FILE *fd, char **arg)
{
  sscanf(text,"%dx%d",&Scr.VxMax,&Scr.VyMax);
  Scr.VxMax = Scr.VxMax*Scr.MyDisplayWidth - Scr.MyDisplayWidth;
  Scr.VyMax = Scr.VyMax*Scr.MyDisplayHeight - Scr.MyDisplayHeight;
}


/****************************************************************************
 *
 * This routine computes the shadow color from the background color
 *
 ****************************************************************************/
Pixel GetShadow(Pixel background) 
{
  XColor bg_color;
  XWindowAttributes attributes;
  
  XGetWindowAttributes(dpy,Scr.Root,&attributes);
  
  bg_color.pixel = background;
  XQueryColor(dpy,attributes.colormap,&bg_color);
  
  bg_color.red = (unsigned short)((bg_color.red*50)/100);
  bg_color.green = (unsigned short)((bg_color.green*50)/100);
  bg_color.blue = (unsigned short)((bg_color.blue*50)/100);
  
  if(!XAllocColor(dpy,attributes.colormap,&bg_color))
    nocolor("alloc shadow","");
  
  return bg_color.pixel;
}

/****************************************************************************
 *
 * This routine computes the hilight color from the background color
 *
 ****************************************************************************/
Pixel GetHilite(Pixel background) 
{
  XColor bg_color, white_p;
  XWindowAttributes attributes;
  
  XGetWindowAttributes(dpy,Scr.Root,&attributes);
  
  bg_color.pixel = background;
  XQueryColor(dpy,attributes.colormap,&bg_color);

  white_p.pixel = GetColor(white);
  XQueryColor(dpy,attributes.colormap,&white_p);
  
#define min(a,b) (((a)<(b)) ? (a) : (b))
#define max(a,b) (((a)>(b)) ? (a) : (b))
  
  bg_color.red = max((white_p.red/5), bg_color.red);
  bg_color.green = max((white_p.green/5), bg_color.green);
  bg_color.blue = max((white_p.blue/5), bg_color.blue);
  
  bg_color.red = min(white_p.red, (bg_color.red*140)/100);
  bg_color.green = min(white_p.green, (bg_color.green*140)/100);
  bg_color.blue = min(white_p.blue, (bg_color.blue*140)/100);
  
  if(!XAllocColor(dpy,attributes.colormap,&bg_color))
    nocolor("alloc hilight","");
  
  return bg_color.pixel;
}

/****************************************************************************
 *
 * This routine loads all needed colors, and fonts,
 * and creates the GC's
 *
 ***************************************************************************/
#ifndef NO_PAGER
Pixel PagerBackColor;
#endif
void GetColors()
{
  static int have_em = 0;

  if(have_em) return;

  have_em = 1;

  /* setup default colors */
  if(Scr.d_depth < 2)
    {
      /* black and white - override user choices */
      Scr.StdColors.back = GetColor(white);
      Scr.StdColors.fore = GetColor(black); 
      Scr.HiColors.back  = GetColor(white);
      Scr.HiColors.fore  = GetColor(black); 
      Scr.StdRelief.back = GetColor(black);
      Scr.StdRelief.fore = GetColor(white);
      Scr.HiRelief.back  = GetColor(black);
      Scr.HiRelief.fore  = GetColor(white);
#ifndef NO_PAGER
      PagerBackColor     = GetColor(white);
#endif
    }
  else
    {
      /* color - accept user choices */
      Scr.StdColors.back = GetColor(Stdback);
      Scr.StdColors.fore = GetColor(Stdfore); 
      Scr.HiColors.back  =  GetColor(Hiback);
      Scr.HiColors.fore  = GetColor(Hifore); 
      Scr.StdRelief.back = GetShadow(Scr.StdColors.back);
      Scr.StdRelief.fore = GetHilite(Scr.StdColors.back);
      Scr.HiRelief.back  = GetShadow(Scr.HiColors.back);
      Scr.HiRelief.fore  = GetHilite(Scr.HiColors.back);
#ifndef NO_PAGER
      PagerBackColor     = GetColor(Pagerback);
#endif
    }

  /* load the font */
  if ((Scr.StdFont.font = XLoadQueryFont(dpy, Scr.StdFont.name)) == NULL)
    {
      nofont(Scr.StdFont.name);
      if ((Scr.StdFont.font = XLoadQueryFont(dpy, "fixed")) == NULL)
	exit(1);
    }
  Scr.StdFont.height = Scr.StdFont.font->ascent + Scr.StdFont.font->descent;
  Scr.StdFont.y = Scr.StdFont.font->ascent;
  Scr.EntryHeight = Scr.StdFont.height + HEIGHT_EXTRA;

  /* load the window-title font */
  if ((Scr.WindowFont.font = XLoadQueryFont(dpy, Scr.WindowFont.name)) == NULL)
    {
      nofont(Scr.WindowFont.name);
      if ((Scr.WindowFont.font = XLoadQueryFont(dpy, "fixed")) == NULL)
	exit(1);
    }
  Scr.WindowFont.height=
    Scr.WindowFont.font->ascent+Scr.WindowFont.font->descent;
  Scr.WindowFont.y = Scr.WindowFont.font->ascent;

  /* create graphics contexts */
  CreateGCs();

  return;
}

/****************************************************************************
 * 
 *  Prints an error message for font loading
 *
 ****************************************************************************/ 
void nofont(char *name)
{
  fprintf (stderr, "fvwm: can't get font %s\n", name);
}

/****************************************************************************
 * 
 *  Processes a menu body definition
 *
 ****************************************************************************/ 
MenuRoot *ParseMenuBody(char *name,FILE *fd)
{
  MenuRoot *mr;
  char newline[256];
  register char *pline;
  char *ptr;

  pline = fgets(newline,(sizeof newline)-1,fd);
  orig_tline = pline;
  if (pline == NULL)
    return 0;
  mr = NewMenuRoot(name);
  GetColors();

  while(isspace(*pline))pline++;
  while((pline != (char *)0)
      &&(strncasecmp("End",pline,3)!=0))
    {
      if((*pline!='#')&&(*pline != 0))
	{
	  char *ptr2 = 0;
	  match_string(func_config,pline, "bad menu body function:",fd);
	  if((func == F_EXEC)||(func == F_POPUP)||(func == F_RESTART)
	     ||(func == F_FUNCTION))
	    ptr2=stripcpy3(pline,True);
	  else
	    ptr2=stripcpy3(pline,False);

	  if(ptr2 != NULL)
	    {
	      sscanf(ptr2,"%d %d",&func_val_1,&func_val_2);
	    }
	  AddToMenu(mr, stripcpy2(pline,1), stripcpy2(pline,2),
		    ptr2, func,func_val_1,func_val_2);
	}
      
      pline = fgets(newline,(sizeof newline)-1,fd);
      if(pline == (char *)0)return;
      orig_tline = pline;

      while(isspace(*pline))pline++;
    }
  MakeMenu(mr);

  return mr;
}

/****************************************************************************
 * 
 *  Parses a popup definition 
 *
 ****************************************************************************/ 
void ParsePopupEntry(char *tline,FILE *fd, char **junk)
{
  MenuRoot *mr=0;
  mr = ParseMenuBody(stripcpy2(tline,0),fd);
  if (PopupCount < MAXPOPUPS)
    {
      PopupTable[PopupCount] = mr;
      PopupCount++;
    }
  else
    {
      fprintf(stderr,"Popup %s ignored, you have more than %u\n",
	      mr->name,MAXPOPUPS);
      free(mr);
    }
}

/****************************************************************************
 * 
 *  Parses a mouse binding
 *
 ****************************************************************************/ 
void ParseMouseEntry(char *tline,FILE *fd, char **junk)
{
  char context[256],modifiers[256],function[256],*ptr;
  MenuRoot *mr=0;
  MenuItem *mi=0;
  MouseButton *temp;
  int button,i,j;

  func_val_1 = 0;
  func_val_2 = 0;

  sscanf(tline,"%d %s %s %s %d %d",&button,context,modifiers,function,
	 &func_val_1,&func_val_2);
  
  find_context(context,&contexts,win_contexts);
  if(contexts &C_LALL)
    {
      /* check for nr_left_buttons */
      i=0;
      j=(contexts &C_LALL)/C_L1;
      while(j>0)
	{
	  i++;
	  j=j>>1;
	}
      if(Scr.nr_left_buttons <i)
	Scr.nr_left_buttons = i;
    }
  if(contexts & C_RALL)
     {
       /* check for nr_right_buttons */
       i=0;
       j=(contexts&C_RALL)/C_R1;
       while(j>0)
	 {
	   i++;
	   j=j>>1;
	 }
       if(Scr.nr_right_buttons <i)
	 Scr.nr_right_buttons = i;
     }
  find_context(modifiers,&mods,key_modifiers);
  if((contexts & C_WINDOW)&&(((mods==0)||mods == AnyModifier)))
    {
      Scr.buttons2grab &= ~(1<<(button-1));
    }

  func = F_NOP;
  match_string(func_config,function,"bad mouse function:",fd);
  if((func == F_POPUP)||(func == F_FUNCTION))
    {
      unsigned i,len;
      ptr = stripcpy2(tline,0);
      if(ptr != NULL)
	len = strlen(ptr);
      else
	len = 0;
      for (i = 0; i < PopupCount; i++)
	if (strncasecmp(PopupTable[i]->name,ptr,len) == 0)
	  {
	    mr = PopupTable[i];
	    break;
	  }
      if (!mr)
	{
	  no_popup(ptr);
	  func = F_NOP;
	}
    }
  else if((func == F_EXEC)||(func == F_RESTART))
    {
      mi = (MenuItem *)safemalloc(sizeof(MenuItem));
      
      mi->next = (MenuItem *)NULL;
      mi->prev = (MenuItem *)NULL;
      mi->item_num = 0;
      mi->item = stripcpy2(tline,0);
      mi->action = stripcpy3(tline,True);
      mi->state = 0;
      mi->func = func;
      mi->strlen = strlen(mi->item);
      mi->val1 = 0;
      mi->val2 = 0;
    }
  
  temp = Scr.MouseButtonRoot;
  Scr.MouseButtonRoot = (MouseButton *)safemalloc(sizeof(MouseButton));
  Scr.MouseButtonRoot->func = func;
  Scr.MouseButtonRoot->menu = mr;
  Scr.MouseButtonRoot->item = mi;
  Scr.MouseButtonRoot->Button = button;
  Scr.MouseButtonRoot->Context = contexts;
  Scr.MouseButtonRoot->Modifier = mods;
  Scr.MouseButtonRoot->NextButton = temp;
  Scr.MouseButtonRoot->val1 = func_val_1;
  Scr.MouseButtonRoot->val2 = func_val_2;
  return;
}

void no_popup(char *ptr)
{
  fprintf(stderr,"Popup '%s' not defined in line %s",ptr,orig_tline);
}


/****************************************************************************
 * 
 *  Processes a line with a key binding
 *
 ****************************************************************************/ 
void ParseKeyEntry(char *tline, FILE *fd,char **junk)
{
  char context[256],modifiers[256],function[256],*ptr;
  char name[256];
  MenuRoot *mr = 0;

  func_val_1 = 0;
  func_val_2 = 0;
  sscanf(tline,"%s %s %s %s %d %d",name,context,modifiers,function,
	 &func_val_1,&func_val_2);
  find_context(context,&contexts,win_contexts);
  find_context(modifiers,&mods,key_modifiers);
  match_string(func_config,function,"bad key function:",fd);

  if((func == F_EXEC)||(func == F_RESTART))
    {
      ptr = stripcpy3(tline,True);
    }
  else if((func == F_POPUP)||(func == F_FUNCTION))
    {
      unsigned i,len;
      ptr = stripcpy2(tline,0);
      if(ptr != (char *)0)
	{
	  len = strlen(ptr);
	  for (i = 0; i < PopupCount; i++)
	    if (strncasecmp(PopupTable[i]->name,ptr,len) == 0)
	      {
		mr = PopupTable[i];
		break;
	      }
	}
      if (!mr)
	{
	  no_popup(ptr);
	  func = F_NOP;
	}
    }

  AddFuncKey(name,contexts,mods,func,ptr,func_val_1,func_val_2,mr);
}

/****************************************************************************
 * 
 * Sets menu/keybinding/mousebinding function to specified value
 *
 ****************************************************************************/ 
void set_func(char *text, FILE *fd, int value)
{
  func = value;
}

/****************************************************************************
 * 
 * Turns a  string context of context or modifier values into an array of 
 * true/false values (bits)
 *
 ****************************************************************************/ 
void find_context(char *string, int *output, struct charstring *table)
{
  int i=0,j=0;
  Bool matched;
  char tmp1;

  *output=0;
  i=0;
  while(i<strlen(string))
    {
      j=0;
      matched = FALSE;
      while((!matched)&&(table[j].key != 0))
	{
	  /* in some BSD implementations, tolower(c) is not defined
	   * unless isupper(c) is true */
	  tmp1=string[i];
	  if(isupper(tmp1))
	    tmp1=tolower(tmp1);
	  /* end of ugly BSD patch */

	  if(tmp1 == table[j].key)
	    {
	      *output |= table[j].value;
	      matched = TRUE;
	    }
	  j++;
	}
      if(!matched)
	fprintf(stderr,"fvwm: bad entry %c in line %s",
		string[i],orig_tline);
      i++;
    }
  return;
}

/****************************************************************************
 * 
 * Matches text from config to a table of strings, calls routine
 * indicated in table.
 *
 ****************************************************************************/ 
void match_string(struct config *table, char *text, char *error_msg, FILE *fd)
{
  int j;
  Bool matched;

  j=0;
  matched = FALSE;
  while((!matched)&&(strlen(table[j].keyword)>0))
    {
      if(strncasecmp(text,table[j].keyword,strlen(table[j].keyword))==0)
	{
	  matched=TRUE;
	  /* found key word */
	  table[j].action(&text[strlen(table[j].keyword)],
				fd,table[j].arg);
	}
      else
	j++;
    }
  if(!matched)
    fprintf(stderr,"fvwm: %s %s in line %s",error_msg,text,orig_tline);
}

  
  

/****************************************************************************
 * 
 * Generates the window for a menu
 *
 ****************************************************************************/ 
void MakeMenu(MenuRoot *mr)
{
  MenuItem *cur;
  unsigned long valuemask;
  XSetWindowAttributes attributes;
  int y;
  
  /* lets first size the window accordingly */
  mr->width += 40;
  
  /* allow two pixels for top border */
  for (y=2, cur = mr->first; cur != NULL; cur = cur->next)
  {
    cur->y_offset = y;
    if(cur->func==F_TITLE)
      {
	/* Title */
	if((cur == mr->first)||(cur->next == NULL))
	  cur->y_height=Scr.EntryHeight-HEIGHT_EXTRA+1+(HEIGHT_EXTRA_TITLE>>1);
	else
	  cur->y_height = Scr.EntryHeight -HEIGHT_EXTRA +1+ HEIGHT_EXTRA_TITLE;
      }
    else if(cur->func==F_NOP && *cur->item==0)
      /* Separator */
      cur->y_height = HEIGHT_SEPARATOR;
    else
      /* Normal text entry */
      cur->y_height = Scr.EntryHeight;
    y += cur->y_height;
    if(mr->width2 == 0)
      {
	cur->x = (mr->width - XTextWidth(Scr.StdFont.font, cur->item,
				       cur->strlen))>>1;
	cur->x2 = cur->x;
      }
    else
      {
	cur->x = 20;
	cur->x2 = mr->width -14;
      }
  }
  mr->in_use = 0;
  mr->height = y+2;
   
  valuemask = (CWBackPixel | CWEventMask | CWCursor);
  attributes.background_pixel = Scr.StdColors.back;
  attributes.event_mask = (ExposureMask | EnterWindowMask);
  attributes.cursor = Scr.FvwmCursors[MENU];
  mr->width = mr->width + mr->width2;
  mr->w = XCreateWindow (dpy, Scr.Root, 0, 0, (unsigned int) (mr->width),
			 (unsigned int) mr->height, (unsigned int) 0,
			 CopyFromParent, (unsigned int) CopyFromParent,
			 (Visual *) CopyFromParent,
			 valuemask, &attributes);
  
  
  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	AddToMenu - add an item to a root menu
 *
 *  Returned Value:
 *	(MenuItem *)
 *
 *  Inputs:
 *	menu	- pointer to the root menu to add the item
 *	item	- the text to appear in the menu
 *	action	- the string to possibly execute
 *	func	- the numeric function
 *
 ***********************************************************************/
void AddToMenu(MenuRoot *menu, char *item, char *item2, char *action,int func, 
	       int func_val_1,int func_val_2)
{
  MenuItem *tmp;
  int width;

  if(item == NULL)
     return;
  tmp = (MenuItem *)safemalloc(sizeof(MenuItem));
  if (menu->first == NULL)
    {
      menu->first = tmp;
      tmp->prev = NULL;
    }
  else
    {
      menu->last->next = tmp;
      tmp->prev = menu->last;
    }
  menu->last = tmp;
  
  tmp->item = item;
  if(item != (char *)0)
    tmp->strlen = strlen(item);
  else
    tmp->strlen = 0;

  tmp->item2 = item2;
  if(item2 != (char *)0)
    tmp->strlen2 = strlen(item2);
  else
    tmp->strlen2 = 0;

  tmp->menu = 0;

  if((func == F_POPUP)||(func == F_FUNCTION))
    {
      unsigned i,len;
      if(action != (char *)0)
	{
	  len = strlen(action);
	  if(len >0)
	    {
	      for (i = 0; i < PopupCount; i++)
		if (strncasecmp(PopupTable[i]->name,action,len) == 0)
		  {
		    tmp->menu = PopupTable[i];
		    break;
		  }
	    }
	}
      if(tmp->menu == (MenuRoot *)0)
	{
	  no_popup(action);
	  func = F_NOP;
	}
    }
  tmp->action = action;
  tmp->next = NULL;
  tmp->state = 0;
  tmp->func = func;
  tmp->val1 = func_val_1;
  tmp->val2 = func_val_2;

  width = XTextWidth(Scr.StdFont.font, item, tmp->strlen);
  if (width <= 0)
    width = 1;
  if (width > menu->width)
    menu->width = width;

  width = XTextWidth(Scr.StdFont.font, item2, tmp->strlen2);
  if (width < 0)
    width = 0;
  if (width > menu->width2)
    menu->width2 = width;
  if((width==0)&&(tmp->strlen2>0))
    menu->width2 = 1;
  
  tmp->item_num = menu->items++;
}

/***********************************************************************
 *
 *  Procedure:
 *	NewMenuRoot - create a new menu root
 *
 *  Returned Value:
 *	(MenuRoot *)
 *
 *  Inputs:
 *	name	- the name of the menu root
 *
 ***********************************************************************/
MenuRoot *NewMenuRoot(char *name)
{
  MenuRoot *tmp;
  
  tmp = (MenuRoot *) safemalloc(sizeof(MenuRoot));
  tmp->name = name;
  tmp->first = NULL;
  tmp->last = NULL;
  tmp->items = 0;
  tmp->width = 0;
  tmp->width2 = 0;
  tmp->w = None;
  return (tmp);
}



/***********************************************************************
 *
 *  Procedure:
 *	AddFuncKey - add a function key to the list
 *
 *  Inputs:
 *	name	- the name of the key
 *	cont	- the context to look for the key press in
 *	mods	- modifier keys that need to be pressed
 *	func	- the function to perform
 *	action	- the action string associated with the function (if any)
 *
 ***********************************************************************/
void AddFuncKey (char *name, int cont, int mods, int func,  char *action,
		 int val1, int val2,MenuRoot *mr)
{
  FuncKey *tmp;
  KeySym keysym;
  KeyCode keycode;

  /*
   * Don't let a 0 keycode go through, since that means AnyKey to the
   * XGrabKey call in GrabKeys().
   */
  if ((keysym = XStringToKeysym(name)) == NoSymbol ||
      (keycode = XKeysymToKeycode(dpy, keysym)) == 0)
    return;
  
  tmp = (FuncKey *) safemalloc(sizeof(FuncKey));
  tmp->next = Scr.FuncKeyRoot.next;
  Scr.FuncKeyRoot.next = tmp;
  
  tmp->name = name;
  tmp->keycode = keycode;
  tmp->cont = cont;
  tmp->mods = mods;
  tmp->func = func;
  tmp->action = action;
  tmp->val1 = val1;
  tmp->val2 = val2;
  tmp->menu = mr;
  return;
}

/****************************************************************************
 * 
 * Loads a single color
 *
 ****************************************************************************/ 
Pixel GetColor(char *name)
{
  XColor color, junkcolor;
  XWindowAttributes attributes;

  XGetWindowAttributes(dpy,Scr.Root,&attributes);
  color.pixel = 0;
   if (!XParseColor (dpy, attributes.colormap, name, &color)) 
     {
       nocolor("parse",name);
     }
   else if(!XAllocColor (dpy, attributes.colormap, &color)) 
     {
       nocolor("alloc",name);
     }
  return color.pixel;
}

void nocolor(char *note, char *name)
{
  fprintf(stderr,"fvwm: can't %s color %s\n", note,name);
}
/****************************************************************************
 * 
 * Copies a string into a new, malloc'ed string
 * Strips leading spaces and trailing spaces and new lines
 *
 ****************************************************************************/ 
char *stripcpy(char *source)
{
  char *tmp,*ptr;
  int len;

  while(isspace(*source))
    source++;
  len = strlen(source);
  tmp = source + len -1;
  while(((isspace(*tmp))||(*tmp == '\n'))&&(tmp >=source))
    {
      tmp--;
      len--;
    }
  ptr = safemalloc(len+1);
  strncpy(ptr,source,len);
  ptr[len]=0;
  return ptr;
}
  


/****************************************************************************
 * 
 * Copies a string into a new, malloc'ed string
 * Strips all data before the first quote and after the second
 *
 ****************************************************************************/
char *stripcpy2(char *source, int tab_sensitive)
{
  char *ptr;
  int count;
  while((*source != '"')&&(*source != 0))
    source++;
  if(*source == 0)
    {
      bad_binding(2);
      return 0;
    }
  source++;
  ptr = source;
  count = 0;
  if(!tab_sensitive)
    while((*ptr!='"')&&(*ptr != 0))
      {
	ptr++;  
	count++;
      }
  else if(tab_sensitive==1)
    while((*ptr!='"')&&(*ptr != 0)&&(*ptr!='\t'))
      {
	ptr++;  
	count++;
      }
  else if(tab_sensitive==2)
    {
      while((*ptr!='"')&&(*ptr != 0)&&(*ptr!='\t'))
	{
	  source++;
	  ptr++;
	}
      while((*ptr!='"')&&(*ptr != 0))
	{
	  ptr++;
	  count++;
	}
    }
  ptr = safemalloc(count+1);
  strncpy(ptr,source,count);
  ptr[count]=0;
  return ptr;
}


/****************************************************************************
 * 
 * Copies a string into a new, malloc'ed string
 * Strips all data before the second quote. and strips trailing spaces and
 * new lines
 *
 ****************************************************************************/
char *stripcpy3(char *source,Bool Warn)
{
  while((*source != '"')&&(*source != 0))
    source++;
  if(*source != 0)
    source++;
  while((*source != '"')&&(*source != 0))
    source++;
  if(*source == 0)
    {
      if(Warn)bad_binding(3);
      return 0;
    }
  source++;
  return stripcpy(source);
}
  
void bad_binding(int num)
{
  fprintf(stderr,"fvwm: bad binding in line %s",orig_tline);
  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	CreateGCs - open fonts and create all the needed GC's.  I only
 *		    want to do this once, hence the first_time flag.
 *
 ***********************************************************************/
void CreateGCs()
{
  XGCValues gcv;
  unsigned long gcm;
  
  /* create GC's */
  gcm = GCFunction|GCLineWidth|GCForeground|GCSubwindowMode; 
  gcv.function = GXxor;
  gcv.line_width = 0;
  gcv.foreground = 3;
  gcv.subwindow_mode = IncludeInferiors;
  Scr.DrawGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);

  gcm = GCFunction|GCPlaneMask|GCGraphicsExposures|GCLineWidth|GCForeground|
    GCBackground|GCFont;
  gcv.function = GXcopy;
  gcv.plane_mask = AllPlanes;
  gcv.foreground = Scr.StdColors.fore;
  gcv.background = Scr.StdColors.back;
  gcv.font =  Scr.StdFont.font->fid;
  /*
   * Prevent GraphicsExpose and NoExpose events.  We'd only get NoExpose
   * events anyway;  they cause BadWindow errors from XGetWindowAttributes
   * call in FindScreenInfo (events.c) (since drawable is a pixmap).
   */
  gcv.graphics_exposures = False;
  
  Scr.NormalGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);

  /* Normal GC for selected window border */
  gcv.font = Scr.WindowFont.font->fid;
  Scr.NormalWindowGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);

  /* Hilite GC for selected window border */
  gcv.foreground = Scr.HiColors.fore;
  gcv.background = Scr.HiColors.back;
  Scr.HiliteGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);

  gcv.font = Scr.StdFont.font->fid;
  /* Stippled GCs for mono screens */
  gcm = GCFunction|GCPlaneMask|GCGraphicsExposures|GCLineWidth|GCForeground|
    GCBackground|GCFont|GCFillStyle|GCStipple;
  gcv.stipple = Scr.gray;	  
  gcv.foreground = Scr.HiColors.back;
  gcv.background = Scr.HiColors.fore;
  if(Scr.d_depth < 2)
    gcv.fill_style = FillOpaqueStippled;  
  else
    gcv.fill_style = FillSolid;
  Scr.HiStippleGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);  

  gcv.stipple = Scr.light_gray;	  
  if(Scr.d_depth < 2)
    {
      gcv.foreground = Scr.StdColors.fore;
      gcv.background = Scr.StdColors.back;
      gcv.fill_style = FillOpaqueStippled;  
    }
  else
    {
      gcv.foreground = Scr.StdColors.back;
      gcv.background = Scr.StdColors.fore;
      gcv.fill_style = FillOpaqueStippled;  
      gcv.fill_style = FillSolid;
    }
  Scr.NormalStippleGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);  

  gcm = GCFunction|GCPlaneMask|GCGraphicsExposures|GCLineWidth|GCForeground|
    GCBackground|GCFont;
  gcv.foreground = Scr.HiRelief.fore;
  gcv.background = Scr.HiRelief.back;
  gcv.fill_style = FillSolid;
  Scr.HiReliefGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);  

  gcv.foreground = Scr.HiRelief.back;
  gcv.background = Scr.HiRelief.fore;
  Scr.HiShadowGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);  

  gcv.foreground = Scr.StdRelief.fore;
  gcv.background = Scr.StdRelief.back;
  Scr.StdReliefGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);  

  gcv.foreground = Scr.StdRelief.back;
  gcv.background = Scr.StdRelief.fore;
  Scr.StdShadowGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);  

  
}

/***********************************************************************
 *
 *  Procedure:
 *	safemalloc - mallocs specified space or exits if there's a 
 *		     problem
 *
 ***********************************************************************/
char *safemalloc(int length)
{
  char *ptr;
  
  ptr = malloc(length);
  if(ptr == (char *)0)
    {
      fprintf(stderr,"fvwm: malloc failed\n");
      exit(1);
    }
  return ptr;
}


/***********************************************************************
 *
 *  Procedure:
 *	AddToList - add a window name to the no title list
 *
 *  Inputs:
 *	name	- a pointer to the name of the window 
 *
 ***********************************************************************/
void AddToList(char *text, FILE *fd, name_list **list)
{
  name_list *nptr;
  char *name;
  
  name = stripcpy(text);
  nptr = (name_list *)safemalloc(sizeof(name_list));
  nptr->next = *list;
  nptr->name = name;
  nptr->value = (char *)0;
  *list = nptr;
}    

void AddToList2(char *text, FILE *fd, name_list **list)
{
  name_list *nptr;
  char *name;
  char *value;

  name = stripcpy2(text,0);
  value = stripcpy3(text,True);
  if((strlen(name)==0)&&(list == &Scr.Icons))
    Scr.DefaultIcon = value;
  else
    {
      nptr = (name_list *)safemalloc(sizeof(name_list));
      nptr->next = *list;
      nptr->name = name;
      nptr->value = value;
      *list = nptr;
    }
}    

/***********************************************************************
 *
 *  Procedure:
 *	Initialize_pager - creates the pager window, if needed
 *
 *  Inputs:
 *	x,y location of the window
 *
 ***********************************************************************/
#ifndef NO_PAGER
char *pager_name = "Fvwm Pager";
XSizeHints sizehints = {
  PMinSize | PResizeInc | PBaseSize | PWinGravity,
  0, 0, 100, 100,	                /* x, y, width and height */
  1, 1,		                        /* Min width and height */
  0, 0,		                        /* Max width and height */
  1, 1,	                         	/* Width and height increments */
  0, 0, 0, 0,	                        /* Aspect ratio - not used */
  1, 1,                 		/* base size */
  NorthWestGravity                      /* gravity */
};

void initialize_pager(int x, int y)
{
  XTextProperty name;
  int width,height,window_x,window_y;
  unsigned long valuemask;
  XSetWindowAttributes attributes;
  FILE *fd = 0;

  width = (Scr.VxMax + Scr.MyDisplayWidth)/Scr.VScale;
  height = (Scr.VyMax + Scr.MyDisplayHeight)/Scr.VScale;
  
  if(x >=0)
    window_x = x;
  else
    {
      sizehints.win_gravity = NorthEastGravity;
      window_x = Scr.MyDisplayWidth - width + x;
    }

  if(y >=0)
    window_y = y;
  else
    {
      window_y = Scr.MyDisplayHeight - height + y;
      if(x<0)
	sizehints.win_gravity = SouthEastGravity;
      else
	sizehints.win_gravity = SouthWestGravity;
    }
  valuemask = (CWBackPixel | CWBorderPixel | CWEventMask|CWCursor);
  attributes.background_pixel = PagerBackColor;
  attributes.border_pixel = Scr.StdColors.fore;
  attributes.cursor = Scr.FvwmCursors[FRAME];
  attributes.event_mask = (ExposureMask | EnterWindowMask|ButtonReleaseMask|
			   ButtonMotionMask);
  sizehints.width = width;
  sizehints.height = height;
  sizehints.x = window_x;
  sizehints.y = window_y;

  Pager_w = XCreateWindow (dpy, Scr.Root, window_x, window_y, width,
			    height, (unsigned int) 1,
			    CopyFromParent, (unsigned int) CopyFromParent,
			    (Visual *) CopyFromParent,
			    valuemask, &attributes);
  XSetWMNormalHints(dpy,Pager_w,&sizehints);
  XStringListToTextProperty(&pager_name,1,&name);
  XSetWMName(dpy,Pager_w,&name);
  XSetWMIconName(dpy,Pager_w,&name);
  XFree(name.value);

  XMapRaised(dpy, Pager_w);
  AddToList(pager_name, fd, &Scr.Sticky);  
}
#endif

