/*
 *      PAGE.C
 *      UTREE pager routines.
 *      3.04-um klin, Sat May 23 17:04:45 1992, Initial version
 *
 *      Copyright (c) 1991/92 by Peter Klingebiel & UNIX Magazin Muenchen.
 *      For copying and distribution information see the file COPYRIGHT.
 */
#ifndef lint
static char sccsid[] = "@(#) utree 3.04-um (klin) May 23 1992 page.c";
#endif  /* !lint */

#include "defs.h"

/* ---- Local variables and definitions ------------------------------- */

#ifdef  BUFSIZ
# define FBUFLEN BUFSIZ         /* Format buffer length                 */
#else   /* !BUFSIZ */
# define FBUFLEN 512
#endif  /* BUFSIZ */

#define BCOL    0               /* Column for position bar              */
#define CCOL    1               /* Column for cursor positioning        */
#define LCOL    2               /* Start column for line string         */
#define MAXCOL  (columns-2)     /* Last column on screen                */
#define K_IGN  '?'              /* Ignore return value                  */

LOCAL plist *proot;             /* Root of page lines                   */
LOCAL plist *cplist;            /* Current line on page                 */
LOCAL plist *cplast;            /* Last current line on page            */
LOCAL plist *tplist;            /* Top line on screen                   */
LOCAL plist *tplast;            /* Last top line on scree               */
LOCAL int pflag;                /* Screen flag                          */
LOCAL int npage;                /* Number of lines on page              */
LOCAL int nline;                /* Screen lines                         */
LOCAL int lline;                /* Last line of page                    */
LOCAL int psel;                 /* Select enabled                       */

#define XDLEN   16              /* Buffer size for hexdump line         */
LOCAL char *xdhex = "0123456789ABCDEF";

/* ---- Functions and procedures -------------------------------------- */

/*
 *      INTERNAL USED ROUTINES
 */

/* Display line for page line pp */
LOCAL VOID showline(pp, f)
  register plist *pp;
  register int f;
{
  if(pp && PLROW(pp) >= firstline && PLROW(pp) <= lastline) {
#ifdef  USEXTERMMOUSE
    if(usefont && usemouse && videomode && pp == cplist) {
      (void) setgraphic(GC_ON);
      putcxy(CCOL, PLROW(pp), GC_CD);
      (void) setgraphic(GC_OFF);
    }
    else
#endif  /* USEXTERMMOUSE */
      putcxy(CCOL, PLROW(pp), pp == cplist ? '>' : ' ');
    if(psel && pp == cplist && videomode)
      setvideo(DA_REVERSE);
    if(f)
      if(putsxy(LCOL, PLROW(pp), PLINE(pp)) < columns)
	clearline();
    if(psel && pp == cplist && videomode)
      setvideo(DA_NORMAL);
  }

} /* showline() */

/* Display the page lines  from line f to line t */
LOCAL VOID showpage(f, t, c)
  register int f, t, c;
{
  register plist *pp;

  /* Search for first line to print ... */
  for(pp = tplist; pp && PLROW(pp) < f; pp = (plist *) PNEXT(pp))
    ;
  /* ... and print out from f to t */
  for( ; pp && PLROW(pp) <= t; pp = (plist *) PNEXT(pp))
    showline(pp, 1);
  /* Clear to end of tree window */
  if(c && pp && PLROW(pp) < lastline)
    clearwindow(PLROW(pp), lastline);

} /* showpage() */

/* Display page position bar */
LOCAL VOID showpbar()
{
  static int bar = 0;
  register plist *pp;
  register int f, t, i;

  if(npage > nline) {           /* More page lines than screen lines */
    pp = tplist;
    while(PNEXT(pp) && PLROW(pp) < lastline)
      pp = (plist *) PNEXT(pp);
    f = ((PLNUM(tplist) + 1) * nline) / npage + firstline;
    t = ((PLNUM(pp)     + 1) * nline) / npage + firstline;
    if(f <= firstline)
      f = PPREV(tplist) ? firstline + 1 : firstline;
    if(t >= lastline)
      t = PNEXT(pp)     ? lastline  - 1 : lastline;
    i = firstline;
#ifdef  USEXTERMMOUSE
    if(usefont && usemouse && videomode) {
      (void) setgraphic(GC_ON);
      while(i < f) {
	putcxy(BCOL, i, i > firstline ? GC_SB : GC_PU);
	++i;
      }
      if(i == firstline)
	putcxy(BCOL, i++, GC_TO);
      else
	putcxy(BCOL, i++, GC_SF);
      while(i < t)
	putcxy(BCOL, i++, GC_SF);
      if(i == lastline)
	putcxy(BCOL, i++, GC_BO);
      else
	putcxy(BCOL, i++, GC_SF);
      while(i <= lastline) {
	putcxy(BCOL, i, i < lastline ? GC_SB : GC_PD);
	++i;
      }
      (void) setgraphic(GC_OFF);
      bar = 1;
      return;
    }
#endif  /* USEXTERMMOUSE */
    while(i < f)
      putcxy(BCOL, i++, ' ');
    if(videomode && (videocap & VA_REVERSE)) {
      setvideo(DA_HALFREV);
      putcxy(BCOL, i++, ' ');
      while(i < t)
	putcxy(BCOL, i++, ' ');
      putcxy(BCOL, i++, ' ');
      setvideo(DA_NORMAL);
    }
    else {
      (void) setgraphic(GC_ON);
      putcxy(BCOL, i++, GC_TT);
      while(i < t)
	putcxy(BCOL, i++, GC_VB);
      putcxy(BCOL, i++, GC_BT);
      (void) setgraphic(GC_OFF);
    }
    while(i <= lastline)
      putcxy(BCOL, i++, ' ');
    bar = 1;
  }
  else if(bar) {
    for(i = firstline; i <= lastline; i++)
      putcxy(BCOL, i, ' ');
    bar = 0;
  }

} /* showpbar() */

/* Update page screen */
LOCAL VOID updatepage(h, e, f)
  register char *h, *e;
  register int f;
{
  register int n;

  if(keypressed())                      /* There are chars in input buffer */
    return;

  if(pflag == SF_FULL) {                /* Full screen update */
    clearscreen();
    cplast = tplast = PNULL;
  }
  if(pflag & SF_HELP)                   /* Help line */
    puthelp("%s", h);
  if(pflag & SF_TREE) {                 /* Page */
    n = tplast ? PLNUM(tplast) - PLNUM(tplist) : 0;
    if(CANSCROLL && n < 0 && n > -nline) {
      (void) windowup(firstline, lastline, -n);
      showpage(lastline + n + 1, lastline, 0);
      showline(cplast, 1);
    }
    else if(CANSCROLL && n > 0 && n < nline) {
      (void) windowdown(firstline, lastline, n);
      showpage(firstline, firstline + n - 1, 0);
      showline(cplast, 1);
    }
    else
      showpage(firstline, lastline, pflag != SF_FULL);
    pflag |= SF_PBAR;
  }
  else if(pflag & SF_LAST)              /* Last directory */
    showline(cplast, f);
  if(pflag & SF_PBAR)                   /* Page position bar */
    showpbar();
  if(pflag & SF_LIST) {                 /* Current line on page */
    showline(cplist, f);
    pflag |= SF_ECHO;
  }
  if(pflag & SF_ECHO)                   /* Echo line */
    (void) putecho("%s %d/%d)", e, PLLNO(cplist), lline);
  /* Position to current line, set variables and return */
  (void) cursorxy(CCOL, PLROW(cplist));
  cplast = cplist;
  tplast = tplist;
  pflag = 0;

} /* updatepage() */

/* Move up or down on page */
LOCAL int goline(dir)
  register int dir;
{
  register int i;

  /* At beginning or end of page */
  if((dir < 0 && PPREV(cplist) == GNULL) || (dir > 0 && PNEXT(cplist) == GNULL))
    return(0);
  if(dir < 0) {                 /* Move backward */
    cplist = (plist *) PPREV(cplist);
    /* Out of screen boundaries */
    if(PLROW(cplist) <= firstline && cplist != proot) {
      tplist = cplist;
      for(i = nline / 2; i > 0 && PPREV(tplist); i--)
	tplist = (plist *) PPREV(tplist);
      pflag |= SF_TREE;
    }
    else
      pflag |= SF_LAST;
  }
  else {                        /* Move forward */
    cplist = (plist *) PNEXT(cplist);
    /* Out of screen boundaries */
    if(PLROW(cplist) > lastline || (PLROW(cplist) == lastline && PNEXT(cplist))) {
      tplist = cplist;
      for(i = nline / 2; i > 0 &&  PNEXT(tplist); i--)
	tplist = (plist *) PNEXT(tplist);
      for(i = nline; PPREV(tplist) && i > 0; i--)
	tplist = (plist *) PPREV(tplist);
      pflag |= SF_TREE;
    }
    else
      pflag |= SF_LAST;
  }
  pflag |= SF_LIST;
  return(1);

} /* goline() */

/* Go page up or down */
LOCAL int movepage(dir)
  register int dir;
{
  register int l;

  if(dir < 0 && PPREV(cplist)) {        /* Page up */
    for(l = nline; l > 0 && goline(-1); l--)
      ;
    return(1);
  }
  else if(dir > 0 && PNEXT(cplist)) {   /* Page down */
    for(l = nline; l > 0 && goline(1) ; l--)
      ;
    return(1);
  }
  return(0);

} /* movepage() */

/* Go to beginning or end of page */
LOCAL int gohome(dir)
  register int dir;
{
  if(dir < 0 && PPREV(cplist)) {        /* Beginning */
    while(goline(-1))
      ;
    return(1);
  }
  else if(dir > 0 && PNEXT(cplist)) {   /* End */
    while(goline(1))
      ;
    return(1);
  }
  return(0);                            /* Not possible */

} /* gohome() */

/* Scroll page */
LOCAL int scrollpage(dir)
  register int dir;
{
  register plist *pp;
  register int i;

  /* Is scrolling possible? */
  if((dir < 0 && PPREV(tplist) == GNULL) || (dir > 0 && PNEXT(cplist) == GNULL))
    return(0);
  if(dir < 0) {                 /* Scroll down */
    tplist = (plist *) PPREV(tplist);
    if(CANSCROLL) {
      (void) windowdown(firstline, lastline, 1);
      showline(tplist, 1);
      pflag |= SF_MOVE|SF_PBAR;
    }
    else
      pflag |= SF_TREE|SF_LIST|SF_PBAR;
  }
  else {                        /* Scroll up */
    for(pp = tplist, i = nline; i >= 0 ; i--)
      if(((pp = (plist *) PNEXT(pp))) == PNULL)
	return(0);
    tplist =  (plist *) PNEXT(tplist);
    if(CANSCROLL) {
      (void) windowup(firstline, lastline, 1);
      for(pp = tplist; PLROW(pp) < lastline; pp = (plist *) PNEXT(pp))
	;
      showline(pp, 1);
      pflag |= SF_MOVE|SF_PBAR;
    }
    else
      pflag |= SF_TREE|SF_LIST|SF_PBAR;
  }
  if(PLROW(cplist) < firstline)         /* Change current directory */
    (void) goline(1);                   /* if out of screen         */
  else if(PLROW(cplist) > lastline)
    (void) goline(-1);
  return(1);

} /* scrollpage() */

#if     defined(SIGWINCH) && defined(TIOCGWINSZ)
/* Refresh page screen after screen size changes */
LOCAL int refreshpage()
{
  register plist *pp;

  (void) refreshfile(0);        /* Refresh file screen */
  (void) refreshtree(0);        /* Refresh tree screen */
  nline = lastline - firstline;
  pp = cplist;
  cplist = tplist = proot;
  while(cplist != pp && goline(1))
    ;
  pflag = SF_FULL;
  return(K_IGN);

} /* refreshpage() */
#endif  /* SIGWINCH && TIOCGWINSZ */

#ifdef  USEXTERMMOUSE
/* Get and interprete mouse event */
LOCAL int pagemouse(m, x)
  register char *m;
  register int x;
{
  register plist *pp;
  register int l, c;

  c = K_MOUSE;
  if(mousey == echoline)                        /* Return */
    c = mouseb ? K_MOUSE : K_DEL;
  else if(mousey == helpline) {                 /* Select from menu */
    if(mousex >= x && mousex < (int) strlen(m))
      c = m[mousex];
    else if(mouseb && mousex < x)               /* Refresh */
      c = K_REFR;
  }
  else if(mousex == 0) {                        /* On "scrollbar" */
    if(npage > nline) {
      if(mousey == firstline)                   /* Scroll down/Beginning */
	c = mouseb ? K_HOME : K_DOWN;
      else {
	l = npage > nline ? lastline : firstline + npage - 1;
	if(mousey >= l)                         /* Scroll up/End */
	  c = mouseb ? K_END : K_UP;
	else if(mouseb == 0)                    /* Next page */
	  c = K_NPAG;
	else                                    /* Prev page */
	  c = K_PPAG;
      }
    }
  }
  else {                                        /* Select line on page */
    if(psel && mousey == PLROW(cplist))         /* Select current line */
      c = mouseb ? K_MOUSE : K_SEL;
    for(pp = tplist; pp && PLROW(pp) != mousey; pp = (plist *) PNEXT(pp))
      ;
    if(pp) {                                    /* Move to line */
      if(mousey < PLROW(cplist)) {              /* Move up */
	while(pp != cplist)
	  (void) goline(-1);
      }
      else {                                    /* Move down */
	while(pp != cplist)
	  (void) goline(1);
      }
    }
  }
  return(c);

} /* pagemouse() */
#endif  /* USEXTERMMOUSE */

/* Goto a line on page */
LOCAL int gotopage()
{
  char inp[PATLEN];
  register int c, l;

  pflag |= SF_ECHO|SF_HELP;
  puthelp("GOTO LINE: Give line number (CR:quit)");
  c = putecho("Line number:");
  if((c = getline(inp, sizeof(inp), c, 0, NULL, GNULL, 0)) < RV_NUL)
    return(c);
  else if(c == RV_NUL)
    return(K_IGN);
  l = atoi(inp);                /* Linenumber to goto */
  if(l > PLLNO(cplist))         /* Move forward */
    while(l > PLLNO(cplist) && goline(1))
      ;
  else if(l < PLLNO(cplist)) {  /* Move backward */
    while(l <= PLLNO(cplist) && goline(-1))
      ;
    if(l > 1)
      (void) goline(1);
  }
  return(K_IGN);

} /* gotopage() */

/* Search forward/backward on page */
LOCAL int searchpage(dir)
  register int dir;
{
  static char pat[PATLEN] = { '\0' };
  char inp[PATLEN+2];
  register plist *pp;
  register int c;

  pflag |= SF_ECHO|SF_HELP;
  puthelp("SEARCH: Give pattern (CR:%s)", pat[0] ? pat : "quit");
  c = putecho("Search %s:", dir > 0 ? "forward" : "backward");
  if((c = getline(inp, sizeof(pat), c, 0, NULL, GNULL, 0)) == RV_OK)
    (void) strcpy(pat, inp);
  else if(c < RV_NUL)
    return(c);
  else if(c == RV_NUL && pat[0] == '\0')
    return(K_IGN);

  (void) sprintf(inp, "*%s*", pat);     /* Build search pattern */
  if(dir > 0) {                         /* Search forward */
    for(pp = (plist *) PNEXT(cplist); pp; pp = (plist *) PNEXT(pp))
      if(match(PLINE(pp), inp))
	break;
  }
  else {                                /* Search backward */
    for(pp = (plist *) PPREV(cplist); pp; pp = (plist *) PPREV(pp))
      if(match(PLINE(pp), inp))
	break;
  }
  if(pp) {                              /* Found: position to line */
    while(pp != cplist)
      (void) goline(dir);
    return(K_IGN);
  }
  c = errequest(pat, "Not found");
  pflag |= SF_LIST;
  return(c < RV_NUL ? c : K_IGN);

} /* searchpage() */

/* Allocate a new line and insert into page list */
LOCAL plist *insertpage(l, f)
  register int l, f;
{
  static plist *lp = PNULL;
  register plist *pp;

  pp = (plist *) ualloc(1, sizeof(plist));
  PNEXT(pp) = PPREV(pp) = GNULL;
  PLINE(pp) = ualloc(l, sizeof(char));
  if(proot) {
    PNEXT(lp) = (glist *) pp;
    PPREV(pp) = (glist *) lp;
    PLNUM(pp) = PLNUM(lp) + 1;
    PLLNO(pp) = f ? PLLNO(lp) + 1 : PLLNO(lp);
    lline = PLLNO(pp);
    lp = pp;
    ++npage;
  }
  else {
    PLLNO(pp) = 1;
    proot = lp = pp;
    npage = lline = 1;
  }
  return(pp);

} /* insertpage() */

/*
 *      READ ASCII/HEXDUMP ROUTINES
 */

/* Get line length */
LOCAL int linelen(s)
  register char *s;
{
  register int i, l;

  for(i = l = 0; s[i]; i++) {
    if(s[i] == '\t') {
      do
	++l;
      while(l & 0x07);
    }
    else
      ++l;
  }
  return(l);

} /* linelen() */

/* Check for input */
LOCAL int checkinput()
{
  register int c;

  if( !keypressed())    /* No input pending */
    return(RV_OK);
  switch(c = getkey()) {
    default:            /* Ignore and continue */
      c = RV_OK;
      break;
    case K_BRK:         /* Break */
    case K_CANC:
    case K_EOF:
      c = RV_INT;
      break;
    case K_SEL:         /* Stop */
      c = RV_NUL;
    break;
#ifdef  USEXTERMMOUSE
    case K_MOUSE:       /* Break or stop */
      c = mouseb ? RV_INT : RV_NUL;
      break;
#endif  /* USEXTERMMOUSE */
  }
  return(c);

} /* checkinput() */

/* Copy buffer b2 to b1 and compare buffers */
LOCAL int copycompare(b1, b2)
  register unsigned char *b1, *b2;
{
  register int i, f;

  for(i = f = 0; i < XDLEN; i++) {
    if(b1[i] != b2[i])
      ++f;
    b1[i] = b2[i];
  }
  return(f);

} /* copycompare) */

/* Read a hexdump file into page */
LOCAL int readdump(fp)
  register FILE *fp;
{
  char line[INPLEN];
  unsigned char buf[XDLEN], obuf[XDLEN];
  register long nr;
  register int c, n, i, j, l, s;

  c = RV_OK;
  i = l = 0;
  nr = 0;
  while((n = fread(buf, sizeof(unsigned char), sizeof(buf), fp)) > 0) {
    if((++l % lines) == 0 && (c = checkinput()) <= RV_NUL)
      return(c);
    if(copycompare(obuf, buf) || nr == 0 || n < XDLEN) {
      s = 0;
      (void) sprintf(line, "%8ld:  ", nr);
      j = strlen(line);
      for(i = 0; i < XDLEN; i++) {
	if(i < n) {
	  line[j++] = xdhex[buf[i]>>4];
	  line[j++] = xdhex[buf[i]&0x0f];
	}
	else {
	  line[j++] = ' ';
	  line[j++] = ' ';
	}
	if(((i+1) % 4) == 0)
	  line[j++] = ' ';
	if(((i+1) % 8) == 0)
	  line[j++] = ' ';
      }
      line[j++] = '[';
      for(i = 0; i < XDLEN; i++)
	line[j++] = (i < n && buf[i] > ' ' && buf[i] <= '~') ? buf[i] : ' ';
      line[j++] = ']';
      line[j] = '\0';
      addpage(line);
    }
    else {
      if( !s) {
	(void) sprintf(line, "%35s", "*** same ***");
	addpage(line);
      }
      s = 1;
    }
    nr += n;
  }
  if(s) {
    (void) sprintf(line, "%8ld:", nr);
    addpage(line);
  }
  return(c);

} /* readdump() */

/* Read an ascii file into page */
LOCAL int readfile(fp)
  register FILE *fp;
{
  char buf[INPLEN];
  register int c, i;

  c = RV_OK;
  i = 0;
  while(fgets(buf, sizeof(buf), fp)) {
    addpage(buf);
    if((++i % lines) == 0 && (c = checkinput()) <= RV_NUL)
      break;
  }
  return(c);

} /* readfile() */

/*
 *      COMMON USED PAGER ROUTINES
 */

/* Initialize pages */
GLOBL VOID initpage()
{
  register plist *gp, *pp;

  if(proot) {                   /* Free last used page */
    gp = pp = proot;
    while(pp) {
      gp = (plist *) PNEXT(pp);
      if(PLINE(pp))
	ufree(PLINE(pp));
      ufree(pp);
      pp = gp;
    }
  }
  proot = PNULL;                /* Reset variables */
  npage = lline = 0;

} /* initpage() */

/* Insert one line into page */
GLOBL int addpage(s)
  register char *s;
{
  register plist *pp;
  register int i, j, f, sl, pl;

  f = 1;
  sl = linelen(s);
  do {
    pl = sl > MAXCOL ? MAXCOL : sl;
    pp = insertpage(pl+1, f);
    for(i = j = 0; i < pl && s[j] && s[j] != '\n'; j++) {
      if(s[j] == '\t') {
	do {
	  PCHAR(pp, i) = ' ';
	  ++i;
	} while(i & 0x07 && i < pl);
      }
      else {
	PCHAR(pp, i) = s[j];
	++i;
      }
    }
    PCHAR(pp, i) = '\0';
    f = 0;
    sl -= pl;
    if(sl > 0)
      s = &s[i];
  } while(sl > 0);

} /* addpage() */

/* Insert line into page. Return length */
/*VARARGS1*/
#if     !defined(BSD) || defined(HASVSPRINTF)
GLOBL int putpage(x, va_alist)
  register int x;
  va_dcl
{
  va_list ap;
  register char *fmt;
#else   /* BSD && !HASVSPRINTF */
/*VARARGS2*/
GLOBL int putpage(x, fmt, p1, p2, p3, p4, p5, p6)
  int x;
  char *fmt, *p1, *p2, *p3, *p4, *p5, *p6;
{
#endif  /* !BSD || HASVSPRINTF */
  char buf[FBUFLEN];
  static plist *pp;
  register int l, i;

  if(x == 0)                    /* Create and insert a new line on page */
    pp = insertpage(MAXCOL+1, 1);
  else {                        /* Add to last line on page */
    l = strlen(PLINE(pp));
    if(l < x)                   /* Fill up with spaces */
      for(i = l; i < x && i < MAXCOL; i++)
	PCHAR(pp, i) = ' ';
  }

#if     !defined(BSD) || defined(HASVSPRINTF)
  va_start(ap);                 /* Format line */
  fmt = va_arg(ap, char *);
  (void) vsprintf(buf, fmt, ap);
  va_end(ap);
#else   /* BSD && !HASVSPRINTF */
  (void) sprintf(buf, fmt, p1, p2, p3, p4, p5, p6);
#endif  /* !BSD || HASVSPRINTF */

  for(i = 0, l = x; buf[i] && buf[i] != '\n' && l < MAXCOL; i++) {
    if(buf[i] == '\t') {
      do {
	PCHAR(pp, l) = ' ';
	++l;
      } while(l & 0x07 && l < MAXCOL);
    }
    else {
      PCHAR(pp, l) = buf[i];
      ++l;
    }
  }
  if(l >= MAXCOL)
    PCHAR(pp, MAXCOL) = '\0';
  else
    PCHAR(pp, l) = '\0';
  return(l);

} /* putpage() */

/* Display current page */
GLOBL int pagemenu(s, elin, ind, ext, lst)
  register char *s, *elin;
  register int *ind, ext, lst;
{
  char hlin[INPLEN];
  register int c;
#ifdef  USEXTERMMOUSE
  char mlin[INPLEN];
  register int mx;
#endif  /* USEXTERMMOUSE */

  if(proot == PNULL)
    return(RV_NUL);

  if(ind) {
    psel = 1;
    cursorset(CF_VISIBLE);
  }
  else {
    psel = 0;
    cursorset(CF_INVISIBLE);
  }
  if(ext)
    (void) sprintf(hlin, "%s (C:continue  G:goline  F:searchforw  B:searchback  Q:quit)", s);
  else if(ind)
    (void) sprintf(hlin, "%s (CR:select  SP:continue  Q:quit)", s);
  else
    (void) sprintf(hlin, "%s (C:continue  Q:quit)", s);
#ifdef  USEXTERMMOUSE
  mx = strlen(s);
  if(ext)
    (void) sprintf(mlin, "%s??cccccccccc??gggggggg??ffffffffffff??bbbbbbbbbbbb??qqqqqq?", s);
  else if(ind)
    (void) sprintf(mlin, "%s??sssssssss??ccccccccccc??qqqqqq?", s);
  else
    (void) sprintf(mlin, "%s??cccccccccc??qqqqqq?", s);
#endif  /* USEXTERMMOUSE */
  cplist = tplist = proot;      /* Init page variables */
  cplast = tplast = PNULL;
  pflag = SF_FULL;
  nline = lastline - firstline;

  if(lst)                       /* Position to last line on page */
    (void) gohome(1);

  /* Page menu loop */
  do {
    if(pflag)
      updatepage(hlin, elin, ind ? 1 : 0);
#ifdef  UTCLOCK
    if(VARSET(V_CL))
      clockon();
#endif  /* UTCLOCK */
    c = getkey();               /* Get command */
#ifdef  UTCLOCK
    if(VARSET(V_CL))
      clockoff();
#endif  /* UTCLOCK */
#ifdef  USEXTERMMOUSE
    if(c == K_MOUSE)            /* Handle mouse event */
      c = pagemouse(mlin, mx);
#endif  /* USEXTERMMOUSE */
    switch(c) {
      default:                  /* Ignore */
	bell(VARSET(V_BL));
#ifdef  USEXTERMMOUSE
      case K_MOUSE:             /* Ignore mouse event */
#endif  /* USEXTERMMOUSE */
      case K_IGN:               /* Ignore return value */
	break;
      case K_SIZE:              /* Screen size changed */
#if     defined(SIGWINCH) && defined(TIOCGWINSZ)
	c = RV_SIZ;
#else   /* !SIGWINCH || !TIOCGWINSZ */
	c = K_IGN;
#endif  /* SIGWINCH && TIOCGWINSZ */
	/*Fall thru*/
      case K_REFR:              /* Refresh */
	pflag = SF_FULL;
	break;
      case K_SEL:               /* Scroll up or select */
	if( !ind) {             /* For more fans */
	  if( !scrollpage(1))
	    bell(VARSET(V_BL));
	  break;
	}
      case K_INS:               /* Select */
      case 's':
      case 'S':
	if(ind)
	  *ind = PLNUM(cplist);
	c = RV_OK;
	break;
      case 'g':                 /* Goto line */
      case 'G':
	if( !ind)
	  c = gotopage();
	break;
      case 'f':                 /* Search forward */
      case 'F':
	if( !ind)
	  c = searchpage(1);
	break;
      case 'b':                 /* Search backward */
      case 'B':
	if( !ind)
	  c = searchpage(-1);
	break;
      case ' ':                 /* Quit or next page */
	if( !ind) {             /* Fore more fans */
	  if( !movepage(1))
	    bell(VARSET(V_BL));
	  break;
	}
      case K_DEL:               /* Quit */
      case 'c':
      case 'C':
	if(ind)
	  *ind = PLNUM(cplist);
	c = RV_NUL;
	break;
      case K_BRK:               /* Interrupt */
      case 'q':
      case 'Q':
	if(ind)
	  *ind = -1;
	c = RV_INT;
	break;
      case K_EOF:
	if(ind)
	  *ind = -1;
	c = RV_END;
	break;
      case K_PREV:              /* Previous */
      case K_BACK:
      case 'k':                 /* For vi fans */
	if( !goline(-1))
	  bell(VARSET(V_BL));
	break;
      case K_NEXT:              /* Next */
      case K_FORW:
      case 'j':                 /* For vi fans */
	if( !goline(1))
	  bell(VARSET(V_BL));
	break;
      case K_UP:                /* Scroll up */
	if( !scrollpage(1))
	  bell(VARSET(V_BL));
	break;
      case K_DOWN:              /* Scroll down */
	if( !scrollpage(-1))
	  bell(VARSET(V_BL));
	break;
      case K_PPAG:              /* Page up */
	if( !movepage(-1))
	  bell(VARSET(V_BL));
	break;
      case K_NPAG:              /* Page down */
	if( !movepage(1))
	  bell(VARSET(V_BL));
	break;
      case K_HOME:              /* Begin */
	if( !gohome(-1))
	  bell(VARSET(V_BL));
	break;
      case K_END:               /* End */
	if( !gohome(1))
	  bell(VARSET(V_BL));
	break;
    }
#if     defined(SIGWINCH) && defined(TIOCGWINSZ)
    /* Refresh screen after screen resize */
    if(c == RV_SIZ)
      c = refreshpage();
#endif  /* SIGWINCH && TIOCGWINSZ */
  } while(c > RV_OK);

  cursorset(CF_INVISIBLE);
  return(c);

} /* pagemenu() */

/* Page a file fil */
GLOBL int callpager(who, tit, nam, etl, fil, xd)
  register char *who, *tit, *nam, *etl, *fil;
  register int xd;
{
  char buf[EXECLEN];
  register FILE *fp;
  register int c;
  int st;

  if( !VARSET(V_UP)) {          /* Use external pager */
    if(xd)                      /* Hexdump */
      (void) sprintf(buf, "%s %s %s|%s %s", VARVAL(V_XD), VARVAL(V_XDO), fil,
					    VARVAL(V_PG), VARVAL(V_XDO));
    else                        /* Ascii view */
      (void) sprintf(buf, "%s %s %s", VARVAL(V_PG), VARVAL(V_PGO), fil);
    (void) callsystem(buf, 1, 1);
    return(hitakey("Done (Hit a key)", lines-1, DA_REVERSE));
  }
  pagerpid = fork();            /* Create a new pager process */
  if(pagerpid == 0) {           /* Child: pager */
    if(fp = fopen(fil, "r")) {
      initpage();               /* Build output page */
      puthelp("Hit BREAK to abort, CR to stop");
      (void) putecho("Reading %s. Wait a moment ...", nam);
      flushout();
      c = xd ? readdump(fp) : readfile(fp);
      (void) fclose(fp);
      if(c >= RV_NUL) {         /* Not stopped */
	(void) sprintf(buf, "%s %s (%s", tit, nam, etl);
	c = pagemenu(who, buf, (int *) 0, 1, 0);
      }
    }
    else
      c = errequest(nam, "Cannot open file");
    _exit(32 + c);              /* Return coded return value c */
  }
  else if(pagerpid < 0)         /* Error in fork */
    c = errequest(who, "Cannot fork");
  else {                        /* Parent */
    do {
      c = (int) wait(&st);      /* Wait for termination of pager */
#if     defined(SIGWINCH) && defined(TIOCGWINSZ)
      if(c < 0 && sizechange) {  /* Signal SIGWINCH breaks wait: Ignore! */
	sizechange = c = 0;
	(void) refreshpage();
      }
#endif  /* SIGWINCH && TIOCGWINSZ */
    } while(c != pagerpid && c != -1);
    c = (st >> 8) - 32;
    treeflag = fileflag = SF_FULL;
  }
  pagerpid = 0;
  return(c);

} /* callpager() */
