/* $Id: procuucico.c,v 3.3 1991/09/01 14:02:27 piggy Rel $
 * Process /usr/spool/uucp/.Log/uucico/
 *
 *   Copyright (C) 1991  Lele Gaifax (piggy@idea.sublink.org)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * SYNOPSIS:
 * int ProcUucico()
 *
 */

#include <stdio.h>
#include <string.h>
#include "hdbstat.h"

#if defined(DIRENT)
#include <dirent.h>
typedef struct dirent DIRECT;
#else
#if defined(NDIR)
#include <sys/ndir.h>
typedef struct direct DIRECT;
#else
error:"You need the directory package!"
#endif
#endif

#define DNULL ((DIRECT *) NULL)

#ifdef __GNUC__

static inline int
strequ (str1, str2)
     const char * str1;
     const char * str2;
{
  return (strcmp (str1, str2) == 0);
}

#else

#define strequ(str1, str2)  (strcmp(str1, (str2)) == 0)

#endif

time_t StatStartingTime = -1L;

int
ProcUucico ()
{
  DIR *dirfp;
  DIRECT *direntry;
  char dirname[LPNMAX];

  sprintf (dirname, "%s/%s", PrefixPath, UUCICO_DIR);

  if ((dirfp = opendir (dirname)) == (DIR *) NULL)
    {
      perror ("ProcUucico(opendir)");
      return (ERROR);
    }

  if (VerboseOutput)
    fputs ("\nProcessing uucico logs:", stderr);

  while ((direntry = readdir (dirfp)) != DNULL)
    {
      FILE *CicoFp;
      char PathName[LPNMAX];
      char System[15];
      static void ProcUucicoEntry PROTO( (FILE *, char *) );

      if (*direntry->d_name == '.')
	continue;

#if defined(DIRENT)
      sprintf (System, "%s", direntry->d_name);
#else
      sprintf (System, "%.*s", direntry->d_namlen, direntry->d_name);
#endif

      if (JustSomeSystem && (CheckSystemEsist(System) == (sysrep_t *) NULL))
        continue;

      sprintf (PathName, "%s/%s/%s", PrefixPath, UUCICO_DIR, System);
      if ((CicoFp = fopen (PathName, "r")) == (FILE *) NULL)
	{
	  perror ("ProcUucico(fopen)");
	  return ERROR;
	}
      ProcUucicoEntry (CicoFp, System);
      fclose(CicoFp);
    }
  return OK;
}


#define NUMELEM(array) (sizeof(array) / sizeof(array[0]))
typedef enum
{
  I_DONT_KNOW = -1,
  MSG_CAN_NOT_CALL,
  MSG_CONN_FAILED,
  MSG_FAILED,
  MSG_INPUT_FAILURE,
  MSG_LOGIN_FAILED,
  MSG_OK,
  MSG_SUCCEEDED,
  MSG_CAUGHT,
  MSG_REQUEST
} mesg_id_t;

static mesg_id_t
DecodeMsg (msg)
     char *msg;
{
  register idx;

  static struct
  {
    char *mesg;
    mesg_id_t mesg_id;
  } HandledMsg[] =
  {
    {
      "CAN NOT CALL ", MSG_CAN_NOT_CALL
    },
    {
      "CAUGHT ", MSG_CAUGHT
    },
    {
      "CONN FAILED ", MSG_CONN_FAILED
    },
    {
      "FAILED ", MSG_FAILED
    },
    {
      "IN SEND/SLAVE MODE ", MSG_INPUT_FAILURE
    },
    {
      "LOGIN FAILED ", MSG_LOGIN_FAILED
    },
    {
      "OK ", MSG_OK
    },
    {
      "REQUEST ", MSG_REQUEST
    },
    {
      "SUCCEEDED ", MSG_SUCCEEDED
    }
  };

  for (idx = 0; idx < NUMELEM (HandledMsg) &&
       *msg >= *(HandledMsg[idx].mesg); idx++)
    {
      if (strequ (msg, HandledMsg[idx].mesg))
	return HandledMsg[idx].mesg_id;
    }
  return I_DONT_KNOW;
}

/*
 * This function is a little bit intricated: in my opinion it is the weak
 * point of TUA; it tries to be smart in calculating the timings of the
 * connections, and to keep track of the errors, but is seems that
 * sometimes some garbage is putted in by uucico, and that fools the
 * "algorithm".
 * In a few words, it reads a line at a time from a uucico log, an tries
 * to understand what's going on, computing the time spent "speaking"
 * with the remote, until an error is found or the end of the
 * conversation is reached.
 */
static void
ProcUucicoEntry (fp, sys)
     FILE *fp;
     char *sys;

{
  char line[256];
  sysrep_t *sr = InsertSys (sys);
  time_t timestamp, StartTime = -1L, first_line_timestamp = -1L;

#ifndef DIFFTIME_MISSING
  extern double difftime PROTO( (time_t, time_t) );
#else
#define difftime(time2, time1) (double)((time2) - (time1))
#endif

  char *Command;
  int LineNumber = 0;

  if (VerboseOutput)
    fprintf (stderr, "\n%s:\n", sys);

  while (fgets (line, 256, fp) != NULL)
    {
      if (VerboseOutput && (++LineNumber % 50) == 0)
	fprintf (stderr, "\r%6d", LineNumber);
      (void) strtok (line, "("); /* Skip user and system */

      timestamp = get_date (strtok (NULL, ","));
      
      if (first_line_timestamp == -1L)
	first_line_timestamp = timestamp;

      /*
       * Skip the process id and the progress number.
       */
      (void) strtok (NULL, " "); 
      Command = strtok (NULL, "(");

      switch (DecodeMsg (Command))
	{
	case MSG_CAN_NOT_CALL:
	case MSG_CONN_FAILED:
	  sr->CallsFAIL++;
	  sr->Calls++;
	  break;

        case MSG_INPUT_FAILURE:
          sr->CallsSTOPPED++;
          break;
          
	case MSG_FAILED:        
          /*
           * It can be either "FAILED (conversation complete)" or "FAILED
           * (CAN'T READ...)"  In the latter case, it's not a fatal
           * error, and so the conversation keeps going.
           */
          if (strequ (strtok (NULL, ")"), "conversation complete"))
            {
              if (StartTime != -1L) /* Maybe after a CAUGHT? */
                {
                  sr->TimeConnect += (float) difftime (timestamp, StartTime);
                }
              else
                {
                  sr->CallsFAIL++;
                  sr->Calls++;
                }
              StartTime = -1L;
            }
	  break;

	case MSG_LOGIN_FAILED:
	  sr->CallsFAIL++;
	  sr->CallsOut--;
	  break;

	case MSG_OK:
	  if (strequ (strtok (NULL, ") "), "startup"))
	    {
              sr->LastConnection = timestamp;
	      sr->CallsOK++;
	      if (StartTime == -1L)
		{
		  /*
                   * It is an inbound calls (it didn't find a SUCCEEDED yet) 
                   */
		  sr->Calls++;
		  sr->CallsIn++;
		  StartTime = timestamp;
		}
	    }
	  else
	    {
	      /* This is a line like "OK (convers..complete PORTNAME ...)" */

	      sr->TimeConnect += (float) difftime (timestamp, StartTime);
	      StartTime = -1L;
	    }
	  break;

	case MSG_SUCCEEDED:
	  StartTime = timestamp;
	  sr->Calls++;
	  sr->CallsOut++;
	  break;

	case MSG_CAUGHT:
#if 0          
	  if (StartTime != -1L)
	    sr->TimeConnect += (float) difftime (timestamp, StartTime);
	  StartTime = -1L;
#endif
          sr->CallsSTOPPED++;
	  break;

	case MSG_REQUEST:	
          /*
           * Check for "remote sys can't create temp file" or "remote
           * access to path/file denied"
           */
	  if (strequ (strtok (NULL, ") "), "remote") &&
              strequ (strtok (NULL, ") "), "sys"))
            {
              sr->TimeConnect += (float) difftime (timestamp, StartTime);
              StartTime = -1L;
            }
	  break;

	case I_DONT_KNOW:
	  break;
	}

      /*
       * TO DO: COMPLETE THE MESSAGE LIST
       * ================================
       *
       * Through a short analysis of various uucico logs, I
       * think I taked care of all the foundamental ones, but I
       * cannot be sure, because I couldn't find a complete list of
       * them.
       */
    }
  if (VerboseOutput)
    fprintf (stderr, "\r%6d\n", LineNumber);

  if (StatStartingTime > first_line_timestamp || StatStartingTime == -1L)
    StatStartingTime = first_line_timestamp;
}
