/* $Id: prochist.c,v 3.3 1991/09/01 14:02:21 piggy Rel $
 * Process the history file, where it keeps the old summary from "The Epoc".
 *
 *   Copyright (C) 1991  Lele Gaifax (piggy@idea.sublink.org)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * SYNOPSIS:
 * int ReadHistory()
 *	Read the history file, and initialize the trees.
 * int WriteHistory(int reset)
 *	Write back the updated stuff. If reset is TRUE, then you have to
 *      reset the dates.
 *
 */

#include <stdio.h>

#ifdef USG
#include <string.h>
#else
#include <strings.h>
#endif

#include "hdbstat.h"

#define NotZero(rec)	\
    (rec.FilesOut || rec.FilesIn || rec.BytesOut || rec.BytesIn || \
	rec.TimeOut || rec.TimeIn)

char TheEpoc[30] =
{0};
static char TheLastOne[30] =
{0};

/*
 * The history file is kept in ascii form, so anybody can easily alter
 * it. The first line is the date we began to play, the second one the
 * date of the last update. Then, in alphabetic order, the systems and
 * all the field of the syshist structure.
 */

/*
 * Read a line from file, skipping comments, that begins with a '#'
 */

static char *
ReadALine (file, where, lenght)
     FILE *file;
     char where[];
     int lenght;
{
  char *status;
  char buffer[256];
  do
    {
      status = fgets (buffer, 256, file);
  } while (status != NULL && buffer[0] == '#');
  strncpy (where, buffer, lenght);
  return status;
}

int
ReadHistory ()
{
  extern int access ();
  char pathname[LPNMAX];
  char buffer[256];
  FILE *FileHist;

  sprintf (pathname, "%s/%s", PrefixPath, HISTORY);
  if (access (pathname, 0))
    return (OK);
  if ((FileHist = fopen (pathname, "r")) == (FILE *) NULL)
    {
      perror("ReadHistory(fopen)");
      return ERROR;
    }
  if (ReadALine (FileHist, TheEpoc, 30) == NULL)
    {
      perror("ReadHistory(ReadLine1)");
      return ERROR;
    }
  if (ReadALine (FileHist, TheLastOne, 30) == NULL)
    {
      perror("ReadHistory(ReadLine2)");
      return ERROR;
    }

  while (ReadALine (FileHist, buffer, 256) != NULL && *buffer != '%')
    {
      sysrep_t *sr;
      int Fin, Fout;
      float Bin, Bout, Tin, Tout;
      char sys[20];

      if (sscanf (buffer, "%s%d%d%f%f%f%f", sys, &Fout, &Fin, &Bout, &Bin,
		  &Tout, &Tin) != 7)
        {
          perror("ReadHistory(sscanf1)");
  	  return ERROR;
	}

      if (JustSomeSystem &&
          (CheckSystemEsist(sys) == (sysrep_t *) NULL))
        continue;

      sr = InsertSys (sys);
      sr->History.FilesOut = Fout;
      sr->History.FilesIn = Fin;
      sr->History.BytesOut = Bout;
      sr->History.BytesIn = Bin;
      sr->History.TimeOut = Tout;
      sr->History.TimeIn = Tin;
    }
  if (*buffer == '%')		/* If there is a line beginning with '%'
                                 * then there is the monthly data too */
    {
      while (ReadALine (FileHist, buffer, 256) != NULL)
	{
	  sysrep_t *sr;
	  char sys[20];
	  float monthly_data[12];
	  int idx, last_month;

	  if (sscanf (buffer, "%s%f%f%f%f%f%f%f%f%f%f%f%f%d",
		      sys, &monthly_data[0], &monthly_data[1],
		      &monthly_data[2], &monthly_data[3],
		      &monthly_data[4], &monthly_data[5],
		      &monthly_data[6], &monthly_data[7],
		      &monthly_data[8], &monthly_data[9],
		      &monthly_data[10], &monthly_data[11],
		      &last_month) != 14)
	    {
              perror("ReadHistory(sscanf2)");
              return ERROR;
	    }

          if (JustSomeSystem &&
              (CheckSystemEsist(sys) == (sysrep_t *) NULL))
            continue;

	  sr = InsertSys (sys);
	  for (idx = 0; idx <= 11; idx++)
	    sr->History.MonthlyActivity[idx] = monthly_data[idx];
	  sr->History.LastMonthProcessed = last_month;
	}
    }
  else
    return ERROR;
  fclose (FileHist);
  return (OK);
}

static FILE *FileHist;

static void
WriteSysHist (sys)
     sysrep_t sys;
{
  if (NotZero (sys) || NotZero (sys.History))
    {
      fprintf (FileHist, "%s %d %d %f %f %f %f\n",
	       sys.System,
	       sys.History.FilesOut + sys.FilesOut,
	       sys.History.FilesIn + sys.FilesIn,
	       sys.History.BytesOut + sys.BytesOut,
	       sys.History.BytesIn + sys.BytesIn,
	       sys.History.TimeOut + sys.TimeOut,
	       sys.History.TimeIn + sys.TimeIn);
    }
}

static void
WriteSysHistResetting (sys)
     sysrep_t sys;
{
  if (NotZero (sys))
    fprintf (FileHist, "%s %d %d %f %f %f %f\n",
	     sys.System, sys.FilesOut, sys.FilesIn, sys.BytesOut,
	     sys.BytesIn, sys.TimeOut, sys.TimeIn);
}

static void
WriteMonthlyData (sys)
     sysrep_t sys;
{
  if (NotZero (sys) || NotZero (sys.History))
    {
      int idx;
      fprintf (FileHist, "%s", sys.System);
      for (idx = 0; idx <= 11; idx++)
	fprintf (FileHist, " %.2f", sys.History.MonthlyActivity[idx]);
      fprintf (FileHist, " %d\n", sys.History.LastMonthProcessed);
    }
}

int
WriteHistory (reset)
     int reset;
{
  char pathname[LPNMAX];

  sprintf (pathname, "%s/%s", PrefixPath, HISTORY);
  if ((FileHist = fopen (pathname, "w")) == (FILE *) NULL)
    {
      perror("WriteHistory(fopen)");
      return ERROR;
    }
  /*
   * If I have to reset the history, the initial date become the
   * date of the last update, while this one become the current
   * date. Otherwise update only the last one.
   */

  fputs ("#\n\
# This is the history file for tua, created automatically.\n\
# It should't be edited by hand. Anyway (;-)), there must be no empty lines,\n\
# and the number of fields must be 7 for the historical section, 14 for the\n\
# monthly one, separated with blanks.\n\
# After the historical data, there is a line with a single '%' char at the\n\
# beginning, and next the monthly summary, system by system.\n\
#\n\
# system-name  fileout  filein  byteout  bytein  timeout  timein\n\
#\n", FileHist);
  if (reset)
    {
      fprintf (FileHist, "%s", (*TheLastOne ? TheLastOne : Since));
      if (!*TheLastOne)
	fprintf (FileHist, "\n");
      fprintf (FileHist, "%s\n", CurrentTime);
    }
  else
    {
      fprintf (FileHist, "%s", (*TheEpoc ? TheEpoc : Since));
      if (!*TheEpoc)
	fprintf (FileHist, "\n");
      fprintf (FileHist, "%s\n", CurrentTime);
    }
  EnquirySys (reset ? WriteSysHistResetting : WriteSysHist);
  fputs ("%\n", FileHist);
  EnquirySys (WriteMonthlyData);
  fclose (FileHist);
  return OK;
}
