/* editor.c - Editor source file */

/*{{{  includes*/
#include <sys/types.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <access.h>
#include <bool.h>

#include "short.h"

#include "globals.h"
#define EDITOR_G
#include "origami.h"
#include "macros.h"
/*}}}  */

#define left_cursor(y);   gotoxy(1, (y));
#define left_scroll_region(y);  (W_oy=(y), W_dy=SCREEN_LEN)
#define pre_pre_open_fold(q)  {check_fold(q); \
          Line_after_fold->prec = End_of_fold; \
          Start_of_fold->next = Start_of_fold->fold; \
          Start_of_fold->foldline = START_OPEN_FOLD;}

/*{{{  delete_dsp_line*/
void delete_dsp_line(level) int level;
{
  left_cursor(level);
  clreol();
}
/*}}}  */
/*{{{  up_a_bit*/
void up_a_bit(from) int from;
{
  left_scroll_region((from));
  if (from!=TITLE_LINE-1)
  {
    gotoxy(1,W_oy); DelLine();
    gotoxy(1,TITLE_LINE-1); insLine();
  }
  gotoxy(1,W_oy);full_window();
}
/*}}}  */
/*{{{  down_a_bit*/
void down_a_bit(from) int from;
{
  left_scroll_region(from);
  if (from!=TITLE_LINE-1)
  {
    gotoxy(1,W_oy); insLine();
    gotoxy(1,TITLE_LINE); DelLine();
  }
  gotoxy(1,W_oy); full_window();
}
/*}}}  */
/*{{{  create_list*/
#ifdef STD_C
void create_list(void)
#else
void create_list()
#endif /* STD_C */
{
  element *p;

  proc_new(&head);
  head->prec = NULL;
  proc_new(&p);
  proc_new(&tail);
  tail->next = NULL;
  join_links(head, p);
  join_links(p, tail);
  p->foldline = START_ENTER_FILED;
  p->other_end = tail;
  tail->other_end = p;
  tail->foldline = END_FOLD;
  insert_file((element *)NULL, p, tail, TRUE);
  current = p->next;
  real_head = head;
  real_tail = tail;
}
/*}}}  */
/*{{{  line_no*/
#ifdef STD_C
PRIVATE int line_no(element *q)
#else
PRIVATE int line_no(q)
element *q;
#endif /* STD_C */
{
  int i;
  element *p;

  i = 0;
  p = real_head;
  while (p != q) {
    i++;   /*_full*/
    move_on(&p);
  }
  return i;
}
/*}}}  */

PRIVATE uchar on_screen_line;

/*{{{  on_screen*/
#ifdef STD_C
PRIVATE bool on_screen(element *p)
#else
PRIVATE bool on_screen(p)
element *p;
#endif /* STD_C */
{
  element *q, *old_q;
  bool on_scr;
  uchar line;

  line = 0;
  q = screen_start;
  on_scr = FALSE;
  do {
    line++;
    old_q = q;
    if (q == p) {
      on_screen_line = line;
      on_scr = TRUE;
    }
    q = q->next;
  } while (old_q != screen_end);
  on_scr = (on_scr && on_screen_line > 1 && on_screen_line < SCREEN_LEN);
  return on_scr;
}
/*}}}  */
/*{{{  select_forward*/
#ifdef STD_C
PRIVATE bool select_forward(void)
#else
PRIVATE bool select_forward()
#endif /* STD_C */
{
  if (current == tail)
    return FALSE;
  else
    return (current->next->selected);
}
/*}}}  */
/*{{{  select_backward*/
#ifdef STD_C
PRIVATE bool select_backward(void)
#else
PRIVATE bool select_backward()
#endif /* STD_C */
{
  if (current == head->next)
    return FALSE;
  else
    return (current->prec->selected);
}
/*}}}  */
/*{{{  line_refresh*/
#ifdef STD_C
PRIVATE void line_refresh(int level,element *ptr)
#else
PRIVATE void line_refresh(level, ptr)
int level;
element *ptr;
#endif /* STD_C */
{
  char dsp_line[MAX_FIELD_SIZE + 1];

  copyin(dsp_line, ptr, FALSE);
  gotoxy(1, level);
  if (strlen(dsp_line) >= LEN)
  {
    dsp_line[LEN-1]=END_OF_LINE;
    dsp_line[LEN]='\0';
 fputs(dsp_line, stdout);
}
  else
  {
 fputs(dsp_line, stdout);
 clreol();
 }
}
/*}}}  */
/*{{{  write_dsp_line*/
#ifdef STD_C
void write_dsp_line(element *ptr,int level)
#else
void write_dsp_line(ptr, level)
element *ptr;
int level;
#endif /* STD_C */
{
  line_refresh(level, ptr);
}
/*}}}  */

void restore();

/*{{{  whole_screen_up*/
#ifdef STD_C
PRIVATE void whole_screen_up(void)
#else
PRIVATE void whole_screen_up()
#endif /* STD_C */
{
  bool s;

  if (screen_start == tail) return;
  up_a_bit(1);
  screen_start = screen_start->next;
  cursor_level--;
  if (screen_end != tail)
  {
    screen_end = screen_end->next;
    write_dsp_line(screen_end, SCREEN_LEN);
  }
  else screen_end_level--;
  if (dirty) restore(1);
}
/*}}}  */
/*{{{  whole_screen_down*/
#ifdef STD_C
PRIVATE void whole_screen_down(void)
#else
PRIVATE void whole_screen_down()
#endif /* STD_C */
{
  if (screen_start == head->next) return;
  down_a_bit(1);
  screen_start = screen_start->prec;
  cursor_level++;
  write_dsp_line(screen_start, 1);
  if (screen_end_level == SCREEN_LEN)
    screen_end = screen_end->prec;
  else
    screen_end_level++;
  if (dirty) restore(1);
}
/*}}}  */
/*{{{  title*/
#ifdef STD_C
void title(void)
#else
void title()
#endif
{
  char filename[33];
  char statline[98];
  char language[20];
  int tail;

  if (real_tail == real_head) *filename = '\0';
  else line_of(filename, real_tail);
  strcpy(language,language_string[dialect]);
  if (dialect == NO_LANGUAGES-1)
  {
    strcat(language," ");
    strcat(language,dialect_start[dialect]);
    strcat(language," ");
    strcat(language,dialect_end[dialect]);
  }
  sprintf(statline, "=%c Origami %s (%s%s%s) == %s ",
          file_changed ? '*' : '=',
          version,
          keybinding_name,
          language,
          browse_mode ? ",View" : "",
          filename);
  gotoxy(1,TITLE_LINE);
  standout();
  fputs(statline,stdout);
  tail=W_dx-strlen(statline)-2*sg;
  while (tail-->0) putchar('=');
  standend();
}
/*}}}  */
/*{{{  restore*/
#ifdef STD_C
void restore(int from)
#else
void restore(from)
int from;
#endif /* STD_C */
{
  element *p=screen_start;
  int h, FORLIM=screen_end_level;

  for (h = 1; h <= FORLIM; h++) {
    /*{{{  display if under from*/
    if (h >= from) write_dsp_line(p, h);
    /*}}}  */
    p = p->next;
  }
  if (screen_end_level < SCREEN_LEN) {
    left_scroll_region(screen_end_level + 1);
    ClrScr();
    full_window();
    title();
  }
}
/*}}}  */
/*{{{  line_or_move_up*/
#ifdef STD_C
void line_or_move_up(int to_line)
#else
void line_or_move_up(to_line)
int to_line;
#endif /* STD_C */
{
  if (current->prec == head) {
    message("TOP of LIST",TRUE);
    return;
  }
  if (select_on && select_backward()) {
    current->selected = FALSE;
  }
  current = current->prec;
  if (screen_start->prec != head && cursor_level < to_line) whole_screen_down();
  cursor_level--;
  if (select_on) {
    current->selected = TRUE;
  }
}
/*}}}  */
/*{{{  sub_move_up*/
#ifdef STD_C
PRIVATE void sub_move_up(void)
#else
PRIVATE void sub_move_up()
#endif /* STD_C */
{
  if (current->prec == head)
    return;
  if (select_on && select_backward())
    current->selected = FALSE;
  current = current->prec;
  current->selected = select_on;
}
/*}}}  */
/*{{{  sub_move_down*/
#ifdef STD_C
PRIVATE void sub_move_down(void)
#else
PRIVATE void sub_move_down()
#endif /* STD_C */
{
  if (current == tail)
    return;
  if (select_on && select_forward())
    current->selected = FALSE;
  current = current->next;
  current->selected = select_on;
}
/*}}}  */
/*{{{  move_or_line_down*/
#ifdef STD_C
void move_or_line_down(int from_line)
#else
void move_or_line_down(from_line)
int from_line;
#endif /* STD_C */
{
  if (current == tail) {
    message("BOTTOM of LIST",TRUE);
    return;
  }
  if (select_on && select_forward()) {
    current->selected = FALSE;
  }
  current = current->next;
  if (cursor_level > from_line && screen_end != tail) whole_screen_up();
  cursor_level++;
  if (select_on) {
    current->selected = TRUE;
  }
}
/*}}}  */
/*{{{  restore_to_end*/
#ifdef STD_C
PRIVATE void restore_to_end(int from)
#else
PRIVATE void restore_to_end(from)
int from;
#endif /* STD_C */
{  /*if select_on then set select separately*/
  screen_end = current;
  screen_end_level = cursor_level;
  while (screen_end_level < SCREEN_LEN && screen_end != tail) {
    screen_end_level++;
    screen_end = screen_end->next;
  }
  restore(from);
}
/*}}}  */
/*{{{  to_start*/
#ifdef STD_C
PRIVATE int to_start(int m)
#else
PRIVATE int to_start(m)
int m;
#endif /* STD_C */
{
  int dist_to_start;
  element *p;

  dist_to_start = 1;
  p = current;
  while (p != head->next && dist_to_start <= m) {
    dist_to_start++;
    p = p->prec;
  }
  return dist_to_start;
}
/*}}}  */
/*{{{  to_bottom*/
#ifdef STD_C
int to_bottom(int m)
#else
int to_bottom(m)
int m;
#endif /* STD_C */
{
  int dist_to_end;
  element *p;

  dist_to_end = 0;
  p = current;
  while (p != tail && dist_to_end <= m) {
    dist_to_end++;
    p = p->next;
  }
  return dist_to_end;
}
/*}}}  */
/*{{{  restore_element*/
#ifdef STD_C
void restore_element(uchar close_line)
#else
void restore_element(close_line)
uchar close_line;
#endif /* STD_C */
{  /*if select_on then set select separately*/
  int level, dist_to_end;
  int dist_to_start;

  dist_to_start = to_start((int)(SCREEN_LEN - 1));
  dist_to_end = to_bottom((int)(SCREEN_LEN - 2));
  cursor_level = close_line;
  if (cursor_level < 2)
    cursor_level = 2;
  if (cursor_level >= SCREEN_LEN)
    cursor_level = SCREEN_LEN - 1;
  if (dist_to_start < cursor_level)
    cursor_level = dist_to_start;
  while (cursor_level + dist_to_end < SCREEN_LEN && dist_to_start > cursor_level)
    cursor_level++;
  screen_start = current;
  for (level = cursor_level - 1; level >= 1; level--)
    screen_start = screen_start->prec;
  restore_to_end(1);
}
/*}}}  */
/*{{{  restore_or_restore_to_end*/
#ifdef STD_C
PRIVATE void restore_or_restore_to_end(void)
#else
PRIVATE void restore_or_restore_to_end()
#endif /* STD_C */
{
  element *p;

  if (screen_start != head->next) {
    restore_element(4);
    return;
  }
  p = head->next;
  cursor_level = 1;
  while (p != current) {
    cursor_level++;
    p = p->next;
  }
  restore_to_end(cursor_level);
}
/*}}}  */
/*{{{  pre_open_fold*/
#ifdef STD_C
void pre_open_fold(element *q)
#else
void pre_open_fold(q)
element *q;
#endif /* STD_C */
{
  element *p;
  int i;

  pre_pre_open_fold(q);
  p = Start_of_fold;
  i = total_indent(p);
  p->UU.U1.indent = i;
  do {
    p = p->next;
    p->UU.U1.indent += i;
  } while (p != End_of_fold);
}
/*}}}  */
/*{{{  pre_enter_fold*/
#ifdef STD_C
PRIVATE void pre_enter_fold(element *cur)
#else
PRIVATE void pre_enter_fold(cur)
element *cur;
#endif /* STD_C */
{   /*rent*/
  element *WITH;

  check_fold(cur);
  Line_after_fold->prec = End_of_fold;
  enter_depth++;
  WITH = Start_of_fold;
  enter_depth_spaces += total_indent(Start_of_fold);
  WITH->next = WITH->fold;
  WITH->foldline = START_ENTER_FOLD;
  WITH->fold = head;
  head = WITH->prec;
  End_of_fold->fold = tail;
  tail = End_of_fold;
  /*save indent*/
  End_of_fold->UU.U1.fold_indent = Start_of_fold->UU.U1.indent;
  Start_of_fold->UU.U1.indent = 0;
}
/*}}}  */
/*{{{  enter_folds_to*/
#ifdef STD_C
PRIVATE void enter_folds_to(element *qq)
#else
PRIVATE void enter_folds_to(qq)
element *qq;
#endif /* STD_C */
{
  element *q, *old_q, *save_q;

  do {
    save_q = NULL;
    q = qq;
    while (q != head) {
      old_q = q;
      q = q->prec;
      if (q->foldline == START_FOLD && q->fold == old_q)
  save_q = q;
    }
    if (save_q != NULL)
      pre_enter_fold(save_q);
  } while (save_q != NULL);
}
/*}}}  */
/*{{{  close_and_open_folds*/
#ifdef STD_C
PRIVATE void close_and_open_folds(element *qq)
#else
PRIVATE void close_and_open_folds(qq)
element *qq;
#endif /* STD_C */
{
  element *q, *old_q;

  q = qq;
  while (q != head) {
    old_q = q;
    q = q->prec;
    if (q->foldline == START_FOLD && q->fold == old_q) pre_open_fold(q);
    if (q->foldline == END_FOLD) {
      if (q->other_end->foldline == START_OPEN_FOLD) {
        q = q->other_end;
        close_fold_at(q);
      }
    }
  }
  q = qq;
  while (q != tail) {
    q = q->next;
    if (q->foldline == START_OPEN_FOLD)
      close_fold_at(q);
  }
}
/*}}}  */
/*{{{  pre_find_element*/
#ifdef STD_C
PRIVATE element *pre_find_element(int new_line_no)
#else
PRIVATE element *pre_find_element(new_line_no)
int new_line_no;
#endif /* STD_C */
{
  int current_line_no;
  element *p;

  current_line_no = 0;
  p = head->next;
  while (current_line_no < new_line_no && p != tail) {
    current_line_no++;
    move_on(&p);
  }
  return p;
}
/*}}}  */
/*{{{  find_element*/
#ifdef STD_C
void find_element(int new_line_no,uchar close_line)
#else
void find_element(new_line_no, close_line)
int new_line_no;
uchar close_line;
#endif /* STD_C */
{
  element *p;

  p = pre_find_element(new_line_no);
  close_and_open_folds(p);
  current = p;
  restore_element(close_line);
}
/*}}}  */
/*{{{  get_search*/
#ifdef STD_C
void get_search(void)
#else
void get_search()
#endif /* STD_C */
{
  char s[sizeof(item_to_look_for)];

  readprompt(s, "Search for", 40);
  if (*s!='\0') strcpy(item_to_look_for, s);
  else fputs(item_to_look_for, stdout);
}
/*}}}  */
/*{{{  strrstr only a hack,suboptimal performance.Shouldn't it be in the library?*/
char *strrstr(s1,s2) char *s1,*s2;
{
  char *f,*g;

  g=s1;
  while ((f=strstr(g,s2))!=NULL) g=f+1;
  return (g==s1 ? NULL : g-1);
}
/*}}}  */
/*{{{  find_item*/
#ifdef STD_C
bool find_item(int *cursor_x_pos, bool backup)
#else
bool find_item(cursor_x_pos,backup) int *cursor_x_pos; bool backup;
#endif
{
  int position, i;
  element *p, *q;
  bool found, on_scr;
  char this_line[MAX_FIELD_SIZE + 1], *f;
  int FORLIM;

  p = current;
  found = FALSE;
  if (backup) {
    /*{{{  search reverse*/
    while (p != head && !found)
    {
      copyin(this_line, p, FALSE);
      if (p == current) this_line[*cursor_x_pos-1]='\0';
      if ((f=strrstr(this_line, item_to_look_for))!=NULL)
      {
        found = TRUE;
        position=f-this_line+1;
      }
      else
      {
        if (p->prec->foldline==START_FOLD && p->prec->next==p)
        p=p->prec->other_end;
        else p=p->prec;
      }
    }
    /*}}}  */
  } else {
    /*{{{  search forward*/
    while (p != tail && !found) {
      copyin(this_line, p, FALSE);
      if (p == current) {
        position=strlen(this_line);
        if (position > *cursor_x_pos) f=strstr(this_line+ *cursor_x_pos,
        item_to_look_for);
        else f=NULL;
      }
      else f=strstr(this_line,item_to_look_for);
      if (f==NULL)
        move_on(&p);
      else {
        found = TRUE;
        position=f-this_line+1;
      }
    }
    /*}}}  */
  }
  if (!found) return FALSE;
  q = current;
  close_and_open_folds(p);
  current = p;
  restore_element(4);
  if (p != q)
  {
    copyin(this_line, p, FALSE);
    if (backup) *cursor_x_pos = strrstr(this_line, item_to_look_for)-this_line+1;
    else *cursor_x_pos = strstr(this_line, item_to_look_for)-this_line+1;
  }
  else *cursor_x_pos = position;
  if (*cursor_x_pos < 1) *cursor_x_pos = 1;
  return TRUE;
}
/*}}}  */
/*{{{  proc_replace*/
#ifdef STD_C
void proc_replace(char *search,char *replace,char *line,int position)
#else
void proc_replace(search, replace, line, position)
char *search, *replace, *line;
int position;
#endif /* STD_C */
{
  int ls, ll, i;
  bool rep;
  char STR[MAX_FIELD_SIZE + 1];

  ls = strlen(search);
  ll = strlen(line);
  if (ll < position + ls - 1)
    return;
  rep = TRUE;
  for (i = 1; i <= ls; i++)
    rep = (rep && line[position + i - 2] == search[i - 1]);
  if (rep) {
    strcpy(STR,line);
    strcpy(STR+position-1,replace);
    strcat(STR,line+position+ls-1);
    strcpy(line, STR);
  }
}
/*}}}  */
/*{{{  top*/
#ifdef STD_C
void top(void)
#else
void top()
#endif /* STD_C */
{
  int dist_to_top, far_;

  far_ = cursor_level + 4;
  dist_to_top = to_start(far_);
  if (dist_to_top < far_) {
    while (current != head->next)
      move_up();
    return;
  }
  if (select_on) {
    while (current != head->next)
      sub_move_up();
  } else
    current = head->next;
  restore_element(4);
}
/*}}}  */
/*{{{  bottom*/
#ifdef STD_C
void bottom(void)
#else
void bottom()
#endif /* STD_C */
{
  int dist_to_bottom, far_;

  far_ = SCREEN_LEN - cursor_level + 4;
  dist_to_bottom = to_bottom(far_);
  if (dist_to_bottom < far_) {
    while (current != tail)
      move_down();
    return;
  }
  if (select_on) {
    while (current != tail)
      sub_move_down();
  }
  else
    current = tail;
  restore_element(4);
}
/*}}}  */
/*{{{  copy_line*/
#ifdef STD_C
bool copy_a_line(element **to_ptr)
#else
bool copy_a_line(to_ptr)
element **to_ptr;
#endif /* STD_C */
{
  element *p, *q, *last, *fold_ptr;
  bool filed;

  filed = FALSE;
  p = current;
  while (!filed && p != current->next) {
    if (p->foldline == START_FILED) filed = TRUE;
    move_on(&p);
  }
  if (filed) {
    message("cannot copy filed folds",TRUE);
    return (FALSE);
  }
  p = current;
  last = NULL;
  fold_ptr = NULL;
  while (p != current->next) {
    proc_new(&q);
    if (p == current) *to_ptr = q;
    *q = *p;
    if (p->strng) {
        q->strng=malloc(strlen(p->strng)+1);
        strcpy(q->strng, p->strng);
    }
    q->prec = last;
    if (last != NULL) {
      last->next = q;
      if (last->foldline == START_FOLD) last->fold = q;
      if (last->foldline == END_FOLD) {
        q->prec = last->other_end;
        q->prec->next = q;
      }
    }
    if (q->foldline == START_FOLD) {
      q->other_end = fold_ptr;
      fold_ptr = q;
    }
    move_on(&p);
    if (q->foldline == END_FOLD) {
      q->other_end = fold_ptr;
      fold_ptr = fold_ptr->other_end;
      q->other_end->other_end = q;
    }
    last = q;
  }
  return (!filed);
}
/*}}}  */
/*{{{  insert_link_before*/
#ifdef STD_C
PRIVATE void insert_link_before(element *ptr_to_new)
#else
PRIVATE void insert_link_before(ptr_to_new)
element *ptr_to_new;
#endif /* STD_C */
{
  element *ptr_to_prec;

  ptr_to_new->UU.U1.indent = insert_indent_of(current);
  ptr_to_prec = current->prec;
  if (screen_start == current)
    screen_start = ptr_to_new;
  join_links(ptr_to_prec, ptr_to_new);
  join_links(ptr_to_new, current);
  current = ptr_to_new;
}
/*}}}  */
/*{{{  undelete_before*/
#ifdef STD_C
void undelete_before(element *ptr_to_new)
#else
void undelete_before(ptr_to_new)
element *ptr_to_new;
#endif /* STD_C */
{
  bool s;

  insert_link_before(ptr_to_new);
  if (cursor_level < SCREEN_LEN) {
    down_a_bit(cursor_level);
  }
  else delete_dsp_line(SCREEN_LEN);
  write_dsp_line(current, cursor_level);
  if (screen_end_level == SCREEN_LEN)
    screen_end = screen_end->prec;
  else
    screen_end_level++;
  if (cursor_level > SCREEN_LEN - 2 && screen_end != tail)
    whole_screen_up();
  if (dirty) restore(cursor_level);
}
/*}}}  */
/*{{{  undelete_pick_before*/
#ifdef STD_C
void undelete_pick_before(void)
#else
void undelete_pick_before()
#endif /* STD_C */
{
  element *p, *sf, *ef;

  proc_new(&ef);
  join_links(pick_tail, ef);
  p = pick_head->next;
  proc_new(&sf);
  join_links(pick_head, sf);
  join_links(sf, p);
  sf->foldline = START_FOLD;
  sf->fold = sf->next;
  sf->other_end = ef;
  ef->foldline = END_FOLD;
  ef->other_end = sf;
  undelete_before(sf);
  pick_tail = pick_head;
}
/*}}}  */
/*{{{  undelete_after*/
#ifdef STD_C
void undelete_after(element *ptr_to_new)
#else
void undelete_after(ptr_to_new)
element *ptr_to_new;
#endif /* STD_C */
{
  element *ptr_to_next;


  /*note current<>tail*/
  ptr_to_next = current->next;
  join_links(current, ptr_to_new);
  join_links(ptr_to_new, ptr_to_next);
  if (cursor_level == SCREEN_LEN) {
    screen_end = screen_end->prec;
    whole_screen_up();
  }
  else {
    if (cursor_level == SCREEN_LEN - 1) {
      delete_dsp_line(SCREEN_LEN);
      write_dsp_line(ptr_to_new, SCREEN_LEN);
      if (screen_end_level == SCREEN_LEN) {
        screen_end = screen_end->prec;
        whole_screen_up();
      }
      else screen_end_level++;
    }
    else {
      if (screen_end_level == SCREEN_LEN) screen_end = screen_end->prec;
      else screen_end_level++;
      down_a_bit(cursor_level + 1);
      write_dsp_line(ptr_to_new, cursor_level + 1);
    }
  }
  move_down();
}
/*}}}  */
/*{{{  erase_current_link*/
#ifdef STD_C
PRIVATE void erase_current_link(void)
#else
PRIVATE void erase_current_link()
#endif /* STD_C */
{
  element *prec_ptr, *p;

  if (current == tail)
    return;
  p = current;
  prec_ptr = p->prec;
  current = p->next;
  proc_dispose(p);
  join_links(prec_ptr, current);
}
/*}}}  */
/*{{{  pre_remove_line*/
#ifdef STD_C
PRIVATE void pre_remove_line(element **p)
#else
PRIVATE void pre_remove_line(p)
element **p;
#endif /* STD_C */
{
  element *prec_ptr;

  *p = current;
  prec_ptr = current->prec;
  if (screen_start == current)
    screen_start = current->next;
  current = current->next;
  join_links(prec_ptr, current);
}
/*}}}  */
/*{{{  remove_line*/
#ifdef STD_C
void remove_line(element **p)
#else
void remove_line(p)
element **p;
#endif /* STD_C */
{
  bool s;

  if (!(current->foldline == NOT_FOLD || filed_or_fold(current)))
    return;
  if (current == tail)
    return;
  if (*p != NULL)
    proc_dispose(*p);
  pre_remove_line(p);
  if (cursor_level < SCREEN_LEN) {
    up_a_bit(cursor_level);
  }
  else {
    delete_dsp_line(SCREEN_LEN);
  }
  if (screen_end != tail) {
    screen_end = screen_end->next;
    write_dsp_line(screen_end, SCREEN_LEN);
  }
  else {
    screen_end_level--;
    whole_screen_down();
  }
  if (dirty) restore(cursor_level);
}
/*}}}  */
/*{{{  start_make_fold*/
#ifdef STD_C
void start_make_fold(int ind)
#else
void start_make_fold(ind)
int ind;
#endif /* STD_C */
{
  element *p;
  char STR1[MAX_FIELD_SIZE + 1];
  char STR2[162];

  if (entered(current))
    return;
  proc_new(&p);
  p->fold_close_line = cursor_level;
  makefold_indent = ind - 1;
  if (makefold_indent < current->UU.U1.indent)
    makefold_indent = current->UU.U1.indent;
  sprintf(STR2, "%s%s",
    spaces(STR1, (int)(makefold_indent - current->UU.U1.indent)),
    fold_open_str);
  copy_line_to_parts(STR2, &p);
  undelete_before(p);
  select_on = TRUE;
  select_ptr = current;
  current->selected = TRUE;
  move_down();
}
/*}}}  */
/*{{{  make_fold*/
#ifdef STD_C
void make_fold(int *cursor_x_pos)
#else
void make_fold(cursor_x_pos)
int *cursor_x_pos;
#endif /* STD_C */
{
  element *p;
  int open_count, ind, i;
  bool indented_enough;
  int FORLIM;
  element *WITH;
  char STR1[MAX_FIELD_SIZE + 1];
  char STR2[MAX_FIELD_SIZE + 1];

  if (!select_backward()) {
    message("cannot fold backwards",TRUE);
    return;
  }
  p = select_ptr->next;
  indented_enough = TRUE;
  open_count = 0;
  while (p != current && open_count >= 0 && indented_enough) {
    if (open_count == 0) {   /*check indentation >= makefold_indent*/
      if (p->foldline == START_OPEN_FOLD) indented_enough = (makefold_indent <= p->UU.U1.indent);
      if (filed_or_fold(p)) indented_enough = (makefold_indent <= total_indent(p));
      if (p->foldline == NOT_FOLD) {
        copy_parts_to_line(p, current_dsp_line);
        FORLIM = makefold_indent - p->UU.U1.indent;
        for (i = 0; i < FORLIM; i++) {
    if(current_dsp_line[i] == '\0')
      break; /* blank lines are ok */
   indented_enough = (indented_enough && current_dsp_line[i] == ' ');
  }
      }
    }
    /* check not across open fold boundaries,
     * ie no of open folds=no of end folds
     */
    if (p->foldline == START_OPEN_FOLD) open_count++;
    if (p->foldline == END_FOLD) open_count--;
    p = p->next;
  }
  if (!(indented_enough && open_count == 0)) {
    message("cannot fold",TRUE);
    return;
  }
  p = select_ptr->next;
  ind = makefold_indent - select_ptr->UU.U1.indent;
  while (p != current) {
    WITH = p;
    if (WITH->foldline == START_OPEN_FOLD) close_fold_at(p);
    if (filed_or_fold(p)) WITH->UU.U1.fold_indent -= ind;
    if (WITH->foldline == NOT_FOLD) copy_line_to_parts(copy_to_end(STR1, line_of(STR2, p), ind + 1), &p);
    WITH->UU.U1.indent = 0;
    p = p->next;
  }
  current->selected = FALSE;
  proc_new(&p);
  insert_link_before(p);
  p = select_ptr;
  while (p->selected) {
    p->selected = FALSE;
    p = p->next;
  }
  End_of_fold = current;
  Line_after_fold = End_of_fold->next;
  Start_of_fold = select_ptr;
  Start_of_fold->fold = Start_of_fold->next;
  Start_of_fold->other_end = End_of_fold;
  End_of_fold->other_end = Start_of_fold;
  Line_after_fold->prec = Start_of_fold;
  Start_of_fold->next = Line_after_fold;
  Start_of_fold->foldline = START_FOLD;
  Start_of_fold->UU.U1.indent = makefold_indent - ind;
  Start_of_fold->UU.U1.fold_indent = ind;
  End_of_fold->foldline = END_FOLD;
  End_of_fold->UU.U1.indent = 0;
  current = Start_of_fold;
  select_on = FALSE;
  copy_line_to_parts(null_dsp_line, &End_of_fold);
  copy_line_to_parts(null_dsp_line, &Start_of_fold);
  *cursor_x_pos = total_indent(current) + 6;
  if (on_screen(current))
    current->fold_close_line = on_screen_line;
  restore_element(current->fold_close_line);
  /* if on_screen and enough lines to fill screen
   * then restore_from(on_screen_line);
   */
}
/*}}}  */
/*{{{  open_fold*/
#ifdef STD_C
void open_fold(void)
#else
void open_fold()
#endif /* STD_C */
{
  if (current->foldline != START_FOLD) {
    proc_error(0);
    return;
  }
  current->fold_close_line = cursor_level;
  pre_open_fold(current);
  restore_to_end(cursor_level);
}
/*}}}  */
/*{{{  close_fold*/
#ifdef STD_C
void close_fold(void)
#else
void close_fold()
#endif /* STD_C */
{
  element *p;

  p = current;
  if (p->foldline == END_FOLD)
    p = p->other_end;
  while (p != head && p->foldline != START_OPEN_FOLD) {
    if (p->foldline == END_FOLD)
      p = p->other_end->prec;
    else
      p = p->prec;
  }
  if (p != head) {
    current = p;
    close_fold_at(p);
    restore_or_restore_to_end();
    return;
  }
  if (tail != real_tail)
    message("use exit fold !",TRUE);
  else
    message("at top",TRUE);
}
/*}}}  */
/*{{{  pre_exit_fold*/
#ifdef STD_C
PRIVATE void pre_exit_fold(void)
#else
PRIVATE void pre_exit_fold()
#endif /* STD_C */
{
  element *p, *WITH;

  enter_depth--;
  Start_of_fold = head->next;
  End_of_fold = tail;
  Line_after_fold = tail->next;
  tail = End_of_fold->fold;
  WITH = Start_of_fold;
  WITH->foldline = START_FOLD;
  head = WITH->fold;
  WITH->fold = WITH->next;
  WITH->next = Line_after_fold;
  Line_after_fold->prec = Start_of_fold;
  /*restore indent*/
  WITH->UU.U1.indent = End_of_fold->UU.U1.fold_indent;
  enter_depth_spaces -= total_indent(Start_of_fold);
  current = Start_of_fold;
  check_fold(current);
  p = Start_of_fold->fold;
  while (p->foldline != END_FOLD) {
    if (p->foldline == START_OPEN_FOLD)
      close_fold_at(p);
    p = p->next;
  }
}
/*}}}  */
/*{{{  delete_list*/
#ifdef STD_C
void delete_list(element *from_ptr,element *to_ptr)
#else
void delete_list(from_ptr, to_ptr)
element *from_ptr, *to_ptr;
#endif /* STD_C */
{
  element *p, *q;

  q = from_ptr;   /*head^.next^.next*/
  while (q != to_ptr) {   /*tail*/
    p = q;
    q = q->next;
    proc_dispose(p);
  }
  proc_dispose(to_ptr);
}
/*}}}  */
/*{{{  total_save*/
#ifdef STD_C
void total_save(void)
#else
void total_save()
#endif /* STD_C */
{
  while (tail != real_tail)
    pre_exit_fold();
  if (file_changed) {
    proc_exit(real_tail, real_head->next);
    file_changed = FALSE;
  }
}
/*}}}  */

PRIVATE int head_line_no, cur_line_no;

/*{{{  enter_fold*/
#ifdef STD_C
void enter_fold(void)
#else
void enter_fold()
#endif /* STD_C */
{
  element *p;
  char filename[256];

  if (current->foldline == START_FOLD) {
    current->fold_close_line = cursor_level;
    pre_enter_fold(current);
    current = current->next;
    restore_element(4);
    return;
  }
  if (current->foldline != START_FILED) {
    proc_error(0);
    return;
  }
  if (!normal_att(current)) {
    message("not text fold",TRUE);
    return;
  }
  if (current->fold != current->other_end) {
    message("cannot enter listing of filed fold",TRUE);
    return;
  }
  (void) line_of(filename, current->other_end);
  if (access(filename,AREAD) == -1)
  {
    message("file not found",TRUE);
    return;
  }
  head_line_no = line_no(head->next);
  cur_line_no = line_no(current);
  p = current;
  total_save();
  real_tail->UU.U0.int1 = head_line_no;
  real_tail->UU.U0.int2 = cur_line_no;
  real_tail->fold_close_line = cursor_level;
  current = p;
  current->foldline = START_FOLD;   /*to allow copy*/
  /*always TRUE*/
  if (copy_a_line(&p)) current = p;
  delete_list(head->next->next, tail->prec);   /*except for tail/head*/
  current->foldline = START_ENTER_FILED;
  real_tail = current->fold;
  current->next = real_tail;
  head = real_head->next;
  join_links(head, current);
  real_head = head;
  join_links(real_tail, tail);
  tail = real_tail;   /*ie file_ptr*/
  insert_file(tail, current, tail, TRUE);
  current = current->next;
  restore_element(4);
  title();
}
/*}}}  */
/*{{{  exit_fold*/
#ifdef STD_C
void exit_fold(void)
#else
void exit_fold()
#endif /* STD_C */
{
  element *WITH;

  if (tail != real_tail) {
    pre_exit_fold();
    restore_element(current->fold_close_line);
    return;
  }
  if (real_tail->next == NULL) {
    message("at top level already!",TRUE);
    return;
  }
  total_save();
  real_tail = tail->next;
  delete_list(head->next, tail);
  real_head = head->prec;
  WITH = real_tail;
  head_line_no = WITH->UU.U0.int1;
  WITH->UU.U0.int1 = 0;
  cur_line_no = WITH->UU.U0.int2 - head_line_no;
  WITH->UU.U0.int2 = 0;
  tail = real_tail;
  head = real_head;
  current = head->next;   /*ie file_ptr*/
  insert_file(tail, current, tail, TRUE);
  current = pre_find_element(head_line_no);
  enter_folds_to(current);
  find_element(cur_line_no, real_tail->fold_close_line);
  title();
}
/*}}}  */
/*{{{  remove_fold*/
#ifdef STD_C
void remove_fold(void)
#else
void remove_fold()
#endif /* STD_C */
{
  element *new_, *p;
  int ind, f_ind;
  element *WITH;
  char STR1[MAX_FIELD_SIZE + 1];
  char STR2[MAX_FIELD_SIZE + 1];
  char STR3[256];

  if (current->foldline != START_FOLD) {
    proc_error(0);
    return;
  }
  ind = current->UU.U1.indent;
  f_ind = current->UU.U1.fold_indent;
  pre_pre_open_fold(current);
  new_ = Start_of_fold->next;
  p = new_;
  while (p != End_of_fold) {
    WITH = p;
    WITH->UU.U1.indent = ind;
    if (WITH->foldline == NOT_FOLD) {
      sprintf(STR3, "%s%s", spaces(STR1, f_ind), line_of(STR2, p));
      copy_line_to_parts(STR3, &p);
    }
    else WITH->UU.U1.fold_indent += f_ind;
    p = p->next;
  }
  if (new_ == End_of_fold) new_ = Line_after_fold;
  current = End_of_fold;
  erase_current_link();
  current = Start_of_fold;
  erase_current_link();
  current = new_;
  if (screen_start == Start_of_fold) {
    restore_element(4);
    return;
  }
  if (new_ == Line_after_fold)
    restore_or_restore_to_end();
  else
    restore_to_end(cursor_level);
}
/*}}}  */
/*{{{  help*/
#ifdef STD_C
bool help(bool h)
#else
bool help(h) bool h;
#endif
{
  char *help, helpfile[128], line[80];
  TOKEN ch=' ';
  FILE *fp;
  int li;

  help=getenv("HOME");
  strcpy(helpfile,help);
  if (h) {
    strcat(helpfile,"/.origamihelp");
  } else {
    strcat(helpfile,"/");
    strcat(helpfile,keybase);
    strcat(helpfile,"bind");
  }
  if ((fp=fopen(helpfile,"r"))==NULL)
{
 message(h ? "Helpfile is not online" : "Bindingfile is not online" ,TRUE);
return FALSE;
}
  else
{
li=1;
while (fgets(line,80,fp))
{
  gotoxy(1,li); clreol(); fputs(line,stdout);
  if (li++==SCREEN_LEN)
 {
 li=1;
 message("press any key or q to quit",TRUE);
 ch=single_key();
 }
if ((ch=tolower(ch))=='q') break;
}
if (ch!='q' && li>1)
{
while (li<=SCREEN_LEN) { gotoxy(1,li++); clreol(); }
message("press any key"); single_key();
}
fclose(fp);
return TRUE;
}
}
/*}}}  */
