/*
 * This file is part of "mazewar", an interactive
 * multiuser action game for non-BSD-Unix-systems.
 * Copyright (C) 1988 Hans Korneder      korn@altger.uucp
 * Permission is granted to the public
 * to copy all parts of the source, as long as no money
 * is made out of it, and the copyrightmessage is not removed.
 */

#include <signal.h>
#include <curses.h>
#include "globals.h"

char *user_pipe_name;
int user_pipe_fd, daemon_pipe_fd;
int parent_pid, child_pid;
char richtung[4] = { '>','^','<','v' };

catch(s)
int s;
	{
	signal(s,catch);
	}

fehler(text)
char *text;
	{
	standout();
	mvaddstr(23,0,text);
	standend();
	refresh();
	sleep(4);
	}

main()
	{
	catch(SIGALRM);
	catch(SIGPIPE);
	parent_pid = getpid();
	initscr(); noecho(); raw();
	draw_maze();
	if ( init_pipes() ) play_the_game();
	clear(); refresh(); noraw(); echo(); endwin();
	}

draw_maze()
	{
	int x,y;
	erase();
	for(y=0; y<MAZE_ROWS; y++)
		{
		move(y,0);
		for(x=0; x<MAZE_COLS; x++)
			if ( maze[y][x] != ' ' ) addstr("[]");
			else                     addstr("  ");
		}
	mvaddstr(19,0,"a = turn left");
	mvaddstr(20,0,"d = turn right");
	mvaddstr(21,0,"x = turn back");
	mvaddstr(22,0,"s = shoot");
	mvaddstr(23,0,"q = quit");
	mvaddstr(19,20,"w = walk");
	mvaddstr(20,20,"_ = walk back");
	mvaddstr(21,20,"r = reposition");
	refresh();
	}

int init_pipes()
	{
	int n_bytes;
	extern char *mktemp();
	struct anmeldung anm;
	umask(0);
	user_pipe_name = mktemp(MAZE_PIPE);
	if ( mknod(user_pipe_name,010600,0)<0 )
		{
		fehler("MAZE_PIPE cannot be created");
		return 0;
		}
	user_pipe_fd = open(user_pipe_name,2);
	if ( user_pipe_fd<0 )
		{
		fehler("MAZE_PIPE cannot be opened");
		return 0;
		}
	alarm(3);
	daemon_pipe_fd = open(MAZE_DAEMON,1);
	alarm(0);
	if ( daemon_pipe_fd<0 )
		{
		fehler("MAZE_DAEMON_PIPE cannot be opened");
		return 0;
		}

	/* startup message to the daemon */
	anm.an_magic = ANMELD;
	anm.an_pid   = parent_pid;
	anm.an_uid   = getuid();
	strcpy(anm.an_pipe,user_pipe_name);
	alarm(3);
	n_bytes = write(daemon_pipe_fd,&anm,sizeof(anm));
	alarm(0);
	if ( n_bytes<0 )
		{
		fehler("startupmessage to the daemon could not be sent");
		return 0;
		}

	return 1;
	}

play_the_game()
	{
	switch( child_pid=fork() )
		{
		case -1:
			fehler("cannot fork");
			break;
		case 0: /* child */
			child_proc();
			break;
		default: /* parent */
			parent_proc();
			break;
		}
	}

child_proc()
	{
	struct sp_anzeige spa;
	close(user_pipe_fd);
	user_pipe_fd = open(user_pipe_name,0);
	while ( read(user_pipe_fd,&spa,sizeof(spa))>0 )
		switch ( spa.sp_magic )
			{
			case BILD_NEU:
				clearok(stdscr,TRUE);
				refresh();
				break;
			case SP_ANZ:
				if ( spa.sp_flag & 1 ) /* within the game */
					mvprintw(19+spa.sp_lfd_nr%5,40+(spa.sp_lfd_nr/5)*20,
					"%c: %-8.8s %6d", spa.sp_lfd_nr+'A', spa.sp_name,
					spa.sp_score);
				else /* Aus dem Spiel raus */
					mvaddstr(19+spa.sp_lfd_nr%5,40+(spa.sp_lfd_nr/5)*20,
					"                  ");
				if ( spa.sp_flag & 2 ) /* visible */
					{
					move(spa.sp_y_pos,spa.sp_x_pos*2);
					if ( spa.sp_pid==parent_pid )
						addch(richtung[spa.sp_richtg]);
					else
						addch(spa.sp_lfd_nr + 'A');
					addch(richtung[spa.sp_richtg]);
					}
				else
					mvaddstr(spa.sp_y_pos,spa.sp_x_pos*2,"  ");
				refresh();
				break;
			}
	exit(0);
	}

parent_proc()
	{
	long now;
	int code, status;
	struct bewegung bew;
	struct sp_anzeige spa;
	static long last_shot = 0;
	extern long time();

	bew.be_magic = BEWEGUNG;
	bew.be_pid   = parent_pid;
	spa.sp_magic = BILD_NEU;
	do	switch( code=getch() )
		{
		case 'a': case 'A':
			bew.be_code = 'A';
			if ( write(daemon_pipe_fd,&bew,sizeof(bew) )<0 ) code=EXIT;
			break;
		case 'd': case 'D':
			bew.be_code = 'D';
			if ( write(daemon_pipe_fd,&bew,sizeof(bew) )<0 ) code=EXIT;
			break;
		case 'x': case 'X':
			bew.be_code = 'X';
			if ( write(daemon_pipe_fd,&bew,sizeof(bew) )<0 ) code=EXIT;
			break;
		case 'w': case 'W':
			bew.be_code = 'W';
			if ( write(daemon_pipe_fd,&bew,sizeof(bew) )<0 ) code=EXIT;
			break;
		case ' ': case '_':
			bew.be_code = ' ';
			if ( write(daemon_pipe_fd,&bew,sizeof(bew) )<0 ) code=EXIT;
			break;
		case 's': case 'S':
			/* allow for 1 shot per second */
			now = time((long *)0);
			if ( last_shot == now ) break;
			last_shot = now;
			bew.be_code = 'S';
			if ( write(daemon_pipe_fd,&bew,sizeof(bew) )<0 ) code=EXIT;
			break;
		case 'r': case 'R':
			bew.be_code = 'R';
			if ( write(daemon_pipe_fd,&bew,sizeof(bew) )<0 ) code=EXIT;
			break;
		case 'Q': case 'q': case 127: case 4: /* ^D */
			code        = EXIT;
			bew.be_code = EXIT;
			if ( write(daemon_pipe_fd,&bew,sizeof(bew) )<0 ) code=EXIT;
			break;
		case 12: /* ^L */ case 022: /* ^R */
			if ( write(user_pipe_fd  ,&spa,sizeof(spa) )<0 ) code=EXIT;
			break;
		}
		while ( code!=EXIT );
	close(user_pipe_fd);
	while ( wait(&status) != child_pid ) ;
	}

msg(t)char*t;{write(2,t,strlen(t));}
/* ENDE */
