/**************************************************************
 *
 *	CRISP - Custom Reduced Instruction Set Programmers Editor
 *
 *	(C) Paul Fox, 1989
 *
 *    Please See COPYRIGHT notice.
 *
 **************************************************************/
# include        "list.h"

static	WINDOW	blank_window;
extern cmap_t	*base_cmap;

int	win_before PROTO((WINDOW *, WINDOW *));
void	attach_buffer PROTO((void));
void	sort_windows PROTO((void));
void	create_common_window PROTO((int, char *));
void	win_append PROTO((WINDOW *));
void	w_title PROTO((WINDOW *, char *, char *));
void	set_buffer_bottom PROTO((WINDOW *)); 
void	win_modify PROTO((int));
void	wwin_modify PROTO((WINDOW *, int));
WINDOW	*new_window PROTO((void));

u_int16	win_num = 0;
int	num_popup_wins;
int	border_flag = TRUE;
int	w_layer = 0;

void
top_of_window()
{
	u_dot();
	win_modify(WFMOVE);
	curwp->w_line = curwp->w_top_line;
	curwp->w_col = 1;
	win_modify(WFMOVE);
}
void
end_of_window()
{	int	line = curwp->w_line;

	u_dot();
	win_modify(WFMOVE);
	curwp->w_line = curwp->w_top_line + curwp->w_h - 1;
	if (curwp->w_line > curbp->b_numlines)
		curwp->w_line = curbp->b_numlines;
	acc_assign_int((long) (line != curwp->w_line));
	win_modify(WFMOVE);
}
void
borders()
{	register WINDOW *wp;
	extern int screen_garbled;
	int	oflag = border_flag;

	acc_assign_int((long) border_flag);
	if (argv[1].l_flags != F_NULL)
		border_flag = argv[1].l_int != 0;
	else
		border_flag = !border_flag;
		
	if (border_flag == oflag)
		return;
	for (wp = wheadp; wp; wp = wp->w_wndp)
		if (wp->w_tiled != W_POPUP) {
			if (border_flag) {
				wp->w_tiled = W_ROOT;
				}
			else {
				wp->w_tiled = 0;
				}
			}
	screen_garbled = TRUE;		
}
/*
 * Split the current window. A window
 * smaller than 3 lines cannot be split.
 * The only other error that is possible is
 * a "malloc" failure allocating the structure
 * for the new window.
 */
WINDOW	*
splitwind()
{
	register WINDOW *wp;

	if (curwp->w_h <= 3) {
		ewprintf("Window would be too small.");
		return NULL;
		}
	wp = new_window();
	curwp->w_flag |= WFHARD;
	*wp = *curwp;

	wp->w_ttitle = wp->w_btitle = NULL;
	++curbp->b_nwnd;
	wp->w_num = win_num++;
	wp->w_bufp  = curbp;
	wp->w_force = 0;
	wp->w_h = curwp->w_h / 2 - 1;
	curwp->w_h -= wp->w_h;
	wp->w_y = curwp->w_y + curwp->w_h;
	curwp->w_h--;
	if (curwp->w_top_line + curwp->w_h < curwp->w_line) {
		curwp->w_top_line = curwp->w_line - (curwp->w_h / 2);
		if (curwp->w_top_line < 1)
			curwp->w_top_line = 1;
		}
	curwp->w_old_line = curwp->w_line;
	wp->w_tiled = W_ROOT;
	win_append(wp);
	w_title(wp, bname(curbp->b_fname), "");

	return wp;
}
void
win_append(win)
WINDOW	*win;
{
	win->w_wndp = wheadp;
	wheadp = win;
	
	/***********************************************/
	/*   Sort  windows  into  order  so that when  */
	/*   we   redraw   screen   we  dont  end  up  */
	/*   obscuring   a   popup  by  a  background  */
	/*   window, etc.			       */
	/***********************************************/
	sort_windows();
	
	/***********************************************/
	/*   Recalculate the corners of windows.       */
	/***********************************************/
	set_corners();
}
/**********************************************************************/
/*   This macro creates a background (tiled) window.		      */
/**********************************************************************/
void
create_tiled_window()
{
	create_common_window(W_ROOT, "");
	if (argv[5].l_flags == F_INT) {
		argv[1] = argv[5];
		attach_buffer();
		}
	acc_assign_int((long) curwp->w_num);
}
/**********************************************************************/
/*   This macro creates a popup window.				      */
/**********************************************************************/
void
create_window()
{
	create_common_window(W_POPUP, get_str(5));
}
/**********************************************************************/
/*   Common code for creating a tiled or popup window.		      */
/**********************************************************************/
void
create_common_window(flag, title)
int	flag;
char	*title;
{	register WINDOW	*wp = new_window();

	/***********************************************/
	/*   If  we  couldn't  create  a  new  window  */
	/*   structure then return an error code.      */
	/***********************************************/
	if (wp == NULL) {
		acc_assign_int((long) -1);
		return;
		}

	wp->w_old_line = wp->w_top_line = wp->w_line = 1;
	wp->w_num = win_num++;
	acc_assign_int((long) wp->w_num);

	wp->w_x = (u_int16) (argv[1].l_int + 1);
/*	wp->w_y = (u_int16) (argv[4].l_int + (flag == W_POPUP ? 1 : 0));*/
	wp->w_y = (u_int16) argv[4].l_int;
	wp->w_h = (u_int16) (argv[2].l_int - argv[4].l_int - 1);
	wp->w_w = (u_int16) (argv[3].l_int - argv[1].l_int - 1);
	if (wp->w_w < strlen(title) + 4)
		wp->w_w = strlen(title) + 4;
	if (wp->w_y >= nrow)
		wp->w_y = nrow -1;
	if (wp->w_y + wp->w_h >= nrow - 2)
		wp->w_h = nrow - wp->w_y - 3;

	wp->w_tiled = flag;
	wp->w_force = TRUE;
	wp->w_popup = flag == W_POPUP;
	wp->w_flag = WFHARD;
	wp->w_prev = curwp;
	wp->w_corner_hints[TL_CORNER] = CORNER_3 | CORNER_6;
	wp->w_corner_hints[TR_CORNER] = CORNER_9 | CORNER_6;
	wp->w_corner_hints[BL_CORNER] = CORNER_12 | CORNER_3;
	wp->w_corner_hints[BR_CORNER] = CORNER_12 | CORNER_9;
	w_title(wp, "", title);

	win_append(wp);
	/***********************************************/
	/*   Keep  count  of  number of popup windows  */
	/*   on  display.  We do this so that some of  */
	/*   the  display  optimization won't be done  */
	/*   if  we  have  a  popup,  e.g.  scrolling  */
	/*   screws  up  if we have a popup on top of  */
	/*   a window.				       */
	/***********************************************/
	if (wp->w_popup)
		num_popup_wins++;
	curwp = wp;
}
void
w_title(wp, top, bottom)
WINDOW	*wp;
char	*top;
char	*bottom;
{
	if (wp->w_ttitle && top)
		chk_free(wp->w_ttitle);
	if (wp->w_btitle && bottom)
		chk_free(wp->w_btitle);
	if (top)
		wp->w_ttitle = strdup(top);
	if (bottom)
		wp->w_btitle = strdup(bottom);
}
void
attach_buffer()
{	extern BUFFER *numberb();
	BUFFER	*bp = numberb((u_int16) argv[1].l_int);
	char	buf[128];

	if (bp == NULL) {
		ewprintf("attach_buffer: no such buffer");
		return;
		}
	if (curwp == NULL)
		return;
	detach_buffer(curwp);
	curwp->w_bufp = bp;
	bp->b_nwnd++;
	curwp->w_old_line = 1;
	curwp->w_top_line = bp->b_top;
	curwp->w_line = bp->b_line;
	curwp->w_col = bp->b_col;
	curwp->w_flag |= WFHARD;
	curbp = bp;

	strcpy(buf, curbp->b_title ? curbp->b_title : curbp->b_fname);
	w_title(curwp, bname(buf), (char *) NULL);
}
void
del_window()
{	WINDOW	*wp;

	if (argv[1].l_flags == F_INT) {
		for (wp = wheadp; wp; wp = wp->w_wndp)
			if (wp->w_num == argv[1].l_int)
				break;
		}
	else
		wp = curwp;
	if (wp == NULL)
		return;
	if (wp == curwp) {
		curwp = &blank_window;
		}
	delete_window(wp);
}
void
delete_window(winp)
WINDOW	*winp;
{
	WINDOW	*wp1;

	if (wheadp == winp)
		wheadp = winp->w_wndp;
	else {
		for (wp1 = wheadp; wp1; wp1 = wp1->w_wndp)
			if (wp1->w_wndp == winp) {
				wp1->w_wndp = winp->w_wndp;
				break;
				}
		/***********************************************/
		/*   If   we  can't  find  the  window  don't  */
		/*   worry  --  the  macros  may have left us  */
		/*   in an inconsistent state.		       */
		/***********************************************/
		if (wp1 == NULL)
			return;
		}
		
	/***********************************************/
	/*   If  we  are  deleting  a  window for the  */
	/*   current  buffer  then  we  need  to save  */
	/*   the line/column information.	       */
	/***********************************************/
	if (winp->w_bufp == curbp && curbp) {
		curbp->b_line  = winp->w_line;
		curbp->b_col = winp->w_col;
		curbp->b_top = winp->w_top_line;
		curbp->b_nwnd--;
		}
	else if (winp->w_bufp) {
		/***********************************************/
		/*   Otherwise decrement the reference count.  */
		/***********************************************/
		winp->w_bufp->b_nwnd--;
		}
	if (winp->w_btitle)
		chk_free(winp->w_btitle);
	if (winp->w_ttitle)
		chk_free(winp->w_ttitle);
	if (winp->w_popup)
		num_popup_wins--;
	chk_free((char *) winp);
	
	sort_windows();
	harden_windows();
	set_corners();
}
void
inq_screen_size()
{
	argv_assign(1, (long) nrow);
	argv_assign(2, (long) ncol);
	acc_assign_int((long) pt.pt_color);
}
void
inq_window_size()
{
	acc_assign_int((long) curwp->w_h);

	argv_assign(1, (long) curwp->w_h);
	argv_assign(2, (long) curwp->w_w);
	argv_assign(3, (long) curwp->w_indent);
}
void	
inq_w()
{
	acc_assign_int(curwp && curwp != &blank_window ? 
		(long) curwp->w_num : -1L);
}

/**********************************************************************/
/*   Function to return information about the current window.	      */
/**********************************************************************/
void
inq_window_info()
{
	/***********************************************/
	/*   Make  sure  we  have a current window to  */
	/*   describe.				       */
	/***********************************************/
	if (curwp == NULL) {
		acc_assign_int((long) -1);
		return;
		}
	argv_assign(1, (long) curwp->w_num);
	argv_assign(2, (long) (curwp->w_bufp ? curwp->w_bufp->b_bufnum : -1));
	argv_assign(3, (long) (curwp->w_x - border_flag));
	argv_assign(4, (long) (curwp->w_y + curwp->w_h + border_flag));
	argv_assign(5, (long) (curwp->w_x + curwp->w_w));
	argv_assign(6, (long) curwp->w_y);
	acc_assign_int((long) curwp->w_popup);
		
}
/*******************************************************************/
/*   This  function  sets  the  current  window to the next tiled  */
/*   window.							   */
/*******************************************************************/
void
next_window()
{	register WINDOW *wp;

	if (argv[1].l_flags == F_INT) {
		for (wp = wheadp; wp; wp = wp->w_wndp)
			if (wp->w_num == argv[1].l_int)
				break;
		}
	else
		wp = curwp;
		
	/***********************************************/
	/*   Allow  us  to keep on working even if no  */
	/*   current window.			       */
	/***********************************************/
	if (wp == NULL || wp == &blank_window) {
		acc_assign_int((long) -1);
		return;
		}
		
	/***********************************************/
	/*   Look  for  next  non  tiled window after  */
	/*   this one.				       */
	/***********************************************/
	while (1) {
		wp = wp->w_wndp;
		if (wp == NULL) {
			wp = wheadp;
			if (wp == NULL) {
				acc_assign_int((long) -1);
				return;
				}
			}
		if (wp == curwp)
			break;
		if (!wp->w_popup)
			break;
		}
	acc_assign_int((long) wp->w_num);
	curwp = wp;
	curbp = curwp->w_bufp;
	set_hooked();
}
void
set_window()
{
	/***********************************************/
	/*   Set  the  HARD  flag for the old and new  */
	/*   windows   so   that   the   title   gets  */
	/*   hilighted properly.		       */
	/***********************************************/
	if (curwp)
		curwp->w_flag |= WFHARD;
	curwp = find_window(argv[1].l_int);
	curwp->w_flag |= WFHARD;
}
WINDOW *
find_window(num)
int	num;
{	register WINDOW *wp;

	for (wp = wheadp; wp; wp = wp->w_wndp)
		if (wp->w_num == num)
			return wp;
	return NULL;
}
WINDOW	*
vsplitwind()
{
	register WINDOW *wp;
	int	w;

	if (curwp->w_w < 16) {
		ewprintf("Window would be too small.");
		return 0;
		}
	wp = new_window();
	*wp = *curwp;
	wp->w_indent = curwp->w_indent = 0;
	wp->w_btitle = wp->w_ttitle = NULL;
	wp->w_tiled = W_ROOT;
	++curbp->b_nwnd;                        /* Displayed twice.     */

	w = wp->w_w;
	wp->w_w /= 2;
	curwp->w_w = (u_int16) (w - wp->w_w - 1);
	wp->w_x = curwp->w_x + curwp->w_w + 1;

	wp->w_num = win_num++;
	wp->w_force = 0;
	wp->w_wndp = wheadp;

	win_append(wp);

	curwp->w_flag |= WFHARD;
	wp->w_flag |= WFHARD;

	w_title(wp, bname(curbp->b_fname), "");
	return wp;
}
void
window_color()
{
} 
void
win_modify(flag)
int	flag;
{
	wwin_modify(curwp, flag);
} 
void
wwin_modify(wp, flag)
register WINDOW *wp;
int	flag;
{	register int line = *cur_line;

	if (wp == NULL || wp->w_bufp != curbp)
		return;
		
	if (flag & WFDELL & wp->w_flag) /* That really is '&' and '&' */
		flag |= WFHARD;
	if (flag == WFMOVE && curbp->b_anchor)
		flag = WFEDIT;
	wp->w_flag |= flag;
	if ((wp->w_flag & WFEDIT) == 0) 
		return;
	if (wp->w_mined == 0 || wp->w_mined > line)
		wp->w_mined = line;
	if (wp->w_maxed < line)
		wp->w_maxed = line;
} 
 
void
set_buffer_bottom(wp)
WINDOW	*wp;
{	int	new_top_line = wp->w_line - wp->w_h + 1;

	if (new_top_line < 1)
		new_top_line = 1;
	if (new_top_line != wp->w_top_line) {
		wp->w_top_line = new_top_line;
		wwin_modify(wp, WFHARD);
		}
}
WINDOW	*
new_window()
{	static WINDOW null_window = {0};
	WINDOW *wp = (WINDOW *) chk_alloc(sizeof (WINDOW));

	*wp = null_window;
	wp->w_layer = w_layer++;
	wp->w_cmap = base_cmap;
	return wp;
}
void
inq_window_buf()
{	register WINDOW *wp;

	if (argv[1].l_flags != F_NULL) {
		for (wp = wheadp; wp; wp = wp->w_wndp)
			if (wp->w_num == (u_int16) argv[1].l_int)
				break;
		}
	else
		wp = curwp;
		
	if (wp && wp->w_bufp)
		acc_assign_int((long) wp->w_bufp->b_bufnum);
	else
		acc_assign_int(-1L);
}
/**********************************************************************/
/*   Macro  to  position  a buffer so that the lein where the cursor  */
/*   is  on  is somewhere within the window, but not necessarily the  */
/*   top line.							      */
/**********************************************************************/
void
set_top_left()
{	WINDOW	*wp;
	BUFFER	*bp;
	
	wp = curwp;
	if (argv[3].l_flags == F_INT)
		wp = find_window((int) argv[3].l_int);
	else if (argv[6].l_flags == F_INT) {
		/***********************************************/
		/*   If   buffer   ID   specified,  then  use  */
		/*   window attached to that buffer.	       */
		/***********************************************/
		for (wp = wheadp; wp; wp = wp->w_wndp)
			if (wp->w_bufp && wp->w_bufp->b_bufnum == (int) argv[6].l_int)
				break;
		if (wp == NULL)
			return;
		}
	if (wp == NULL)
		return;
	if (argv[1].l_flags == F_INT) {
		int new_top_line = argv[1].l_int > curbp->b_numlines ?
				    curbp->b_numlines : argv[1].l_int;
		if (new_top_line < 0)
			new_top_line = 1;
		if (new_top_line != wp->w_top_line) {
			wp->w_top_line = new_top_line;
			wwin_modify(wp, WFHARD);
			}
		}
	if (argv[2].l_flags == F_INT) {
		if (argv[2].l_int > 0) {
			wp->w_indent = argv[2].l_int;
			wwin_modify(wp, WFHARD);
			}
		}
		
	if (wp->w_bufp == NULL)
		return;
		
	/***********************************************/
	/*   If  line  and column position specified,  */
	/*   then attempt to use them.		       */
	/***********************************************/
	bp = wp->w_bufp;
	if (argv[4].l_flags == F_INT && argv[4].l_int >= 1) {
		wp->w_line = (int) argv[4].l_int;
		if (wp->w_line >= bp->b_numlines)
			wp->w_line = bp->b_numlines;
		}
	if (argv[5].l_flags == F_INT && argv[5].l_int >= 1) {
		wp->w_col = (int) argv[5].l_int;
		}
}

/**********************************************************************/
/*   Macro  to  return  the  line number of the line at the top of a  */
/*   window.							      */
/**********************************************************************/
void
inq_top_left()
{	WINDOW	*wp = curwp;

	if (argv[3].l_flags == F_INT)
		wp = find_window((int) argv[3].l_int);
	if (wp == NULL)
		return;
		
	argv_assign(1, (long) wp->w_top_line);
	argv_assign(2, (long) wp->w_indent);
	argv_assign(4, (long) wp->w_line);
	argv_assign(5, (long) wp->w_col);
	argv_assign(6, (long) (wp->w_bufp ? wp->w_bufp->b_bufnum : -1));
}
/**********************************************************************/
/*   Function  to  flag  all  windows as needing hard update after a  */
/*   change which may affect all windows.			      */
/**********************************************************************/
void
harden_windows()
{	register WINDOW *wp;

	for (wp = wheadp; wp; wp = wp->w_wndp)
		wp->w_flag |= WFHARD;
}
/**********************************************************************/
/*   Function  to  sort  the  windows into order on the window list.  */
/*   The  list  is  ordered with windows higher up the screen nearer  */
/*   the  front  of  the list. 'higher' means higher and more to the  */
/*   left.  This  order is assumed by certain bits of code, e.g. the  */
/*   window resizer on a SIGWINCH and by the display algorithms.      */
/**********************************************************************/
void
sort_windows()
{	WINDOW	*new_wheadp = NULL;
	register WINDOW *wp, *wp1;
	register WINDOW	*next_wp;	

	for (wp = wheadp; wp; wp = next_wp) {
		next_wp = wp->w_wndp;
		if (new_wheadp == NULL) {
			new_wheadp = wp;
			wp->w_wndp = NULL;
			continue;
			}
		if (win_before(wp, new_wheadp)) {
			wp->w_wndp = new_wheadp;
			new_wheadp = wp;
			continue;
			}
		for (wp1 = new_wheadp; wp1->w_wndp; wp1 = wp1->w_wndp)
			if (win_before(wp, wp1->w_wndp))
				break;
		wp->w_wndp = wp1->w_wndp;
		wp1->w_wndp = wp;
		}
	wheadp = new_wheadp;
}
/**********************************************************************/
/*   Function  used  to  compare  a  windows  position and determine  */
/*   which window comes first on the screen. Used by sort_windows().  */
/**********************************************************************/
int
win_before(w1, w2)
WINDOW	*w1, *w2;
{
	/***********************************************/
	/*   Popups  have  higher priority than tiled  */
	/*   windows.				       */
	/***********************************************/
	if (w1->w_popup != w2->w_popup)
		return w1->w_popup ? FALSE : TRUE;
		
	/***********************************************/
	/*   Windows  which  start  at the same place  */
	/*   on  the  screen, are sorted by age, i.e.  */
	/*   newest  window  sits  on  top  of  older  */
	/*   window.				       */
	/***********************************************/
	if (w1->w_y == w2->w_y && w1->w_x == w2->w_x)
		return w1->w_num - w2->w_num;
	if (w1->w_y < w2->w_y)
		return TRUE;
	if (w1->w_y == w2->w_y)
		return w1->w_x < w2->w_x;
	return FALSE;
}
/**********************************************************************/
/*   Function  to  set  flags  for  the  display code for background  */
/*   windows so that abutting corners are drawn properly.	      */
/**********************************************************************/
void
set_corners()
{
	register WINDOW *wp, *wp1;
	int	c;

# define	TL(w)	(w->w_y * ncol + w->w_x)
# define	TR(w)	(w->w_y * ncol + w->w_x + w->w_w + 1)
# define	BL(w)	((w->w_y + w->w_h + 1) * ncol + w->w_x)
# define	BR(w)	((w->w_y + w->w_h + 1) * ncol + w->w_x + w->w_w + 1)

	for (wp = wheadp; wp; wp = wp->w_wndp) {
		if (wp->w_popup)
			continue;
			
		/***********************************************/
		/*   Top left corner of window.		       */
		/***********************************************/
		c = TL(wp);
		wp->w_corner_hints[TL_CORNER] = CORNER_3 | CORNER_6;
		for (wp1 = wheadp; wp1; wp1 = wp1->w_wndp) {
			if (wp1 == wp || wp1->w_popup)
				continue;
			if (c == TR(wp1)) {
				wp->w_corner_hints[TL_CORNER] |= CORNER_9;
				}
			else if (c == BL(wp1)) {
				wp->w_corner_hints[TL_CORNER] |= CORNER_12;
				}
			else if (c == BR(wp1)) {
				wp->w_corner_hints[TL_CORNER] |= CORNER_9 | CORNER_12;
				break;
				}
			}
		/***********************************************/
		/*   Top right corner of window.	       */
		/***********************************************/
		c = TR(wp);
		wp->w_corner_hints[TR_CORNER] = CORNER_9 | CORNER_6;
		for (wp1 = wheadp; wp1; wp1 = wp1->w_wndp) {
			if (wp1 == wp || wp1->w_popup)
				continue;
			if (c == BR(wp1)) {
				wp->w_corner_hints[TR_CORNER] |= CORNER_12;
				}
			else if (c == TR(wp1)) {
				wp->w_corner_hints[TR_CORNER] |= CORNER_3;
				}
			else if (c == BL(wp1)) {
				wp->w_corner_hints[TR_CORNER] |= CORNER_12 | CORNER_3;
				break;
				}
			}
		/***********************************************/
		/*   Bottom left corner of window.	       */
		/***********************************************/
		c = BL(wp);
		wp->w_corner_hints[BL_CORNER] = CORNER_12 | CORNER_3;
		for (wp1 = wheadp; wp1; wp1 = wp1->w_wndp) {
			if (wp1 == wp || wp1->w_popup)
				continue;
			if (c == BR(wp1)) {
				wp->w_corner_hints[BL_CORNER] |= CORNER_9;
				}
			else if (c == TR(wp1)) {
				wp->w_corner_hints[BL_CORNER] |= CORNER_6;
				}
			else if (c == TL(wp1)) {
				wp->w_corner_hints[BL_CORNER] |= CORNER_9 | CORNER_6;
				break;
				}
			}
		/***********************************************/
		/*   Bottom right corner of window.	       */
		/***********************************************/
		c = BR(wp);
		wp->w_corner_hints[BR_CORNER] = CORNER_12 | CORNER_9;
		for (wp1 = wheadp; wp1; wp1 = wp1->w_wndp) {
			if (wp1 == wp || wp1->w_popup)
				continue;
			if (c == BL(wp1)) {
				wp->w_corner_hints[BR_CORNER] |= CORNER_3;
				}
			else if (c == TR(wp1)) {
				wp->w_corner_hints[BR_CORNER] |= CORNER_6;
				}
			else if (c == TL(wp1)) {
				wp->w_corner_hints[BR_CORNER] |= CORNER_3 | CORNER_6;
				break;
				}
			}
		}
}
/**********************************************************************/
/*   Function  to  set  the  window  and/or icon name. This function  */
/*   only works if we are running under a windowing system.	      */
/**********************************************************************/
void
set_wm_name()
{	char	*wname = NULL;
	char	*iname = NULL;
	
	if (argv[1].l_flags != F_NULL)
		wname = get_str(1);
	if (argv[2].l_flags != F_NULL)
		iname = get_str(2);
	if (scrfn.scr_set_names)
		(*scrfn.scr_set_names)(wname, iname);
}
