/* Dies soll ein Programm werden, das ein beliebiges Dokument einliest und
 * in ein TeX File umwandelt.
 * Ich habe dabei speziell an das Umwandeln von News-Artikeln oder von beliebigen
 * Readme-Files gedacht.
 * Bei News-artikeln mu natuerlich der Header abgeschnitten werden.
 * Generell muss auf Sonderzeichen geachtet werden, die nicht mit ausgegeben werden
 * koennen.
 * Eingaben erfolgen standartmaessig ueber stdin. Durch Angabe eines Filenamen
 * kann als Option auch aus einem File gelesen werden.
 * Die Ausgabe erfolgt ueber stdout und kann von dort aus weitergeleitet werden.
 * Beispiel : cat a | txt2tex | more > a.tex bewirkt die Umwandlung des textfiles
 * a in ein TeX File. Analog dazu funktioniert txt2tex a > a.tex
 */

#include<stdio.h>
#include<string.h>
#include<unistd.h>
#include<ctype.h>

typedef struct {
	char	*from;
	char	*to;
} Translation;


Translation	Normal_trans[] = {{"&",		"\\&"},
				  {"*",		"$*$"},
				  {"$",		"\\$"},
				  {"%",		"\\%"},
				  {"#",		"\\#"},
				  {"{",		"\\{"},
				  {"}",		"\\}"},
				  {"\\",	"$\\setminus$"},
				  {"_",		"\\_"},
				  {"<",		"$<$"},
				  {">",		"$>$"},
				  {"^",		"$\\uparrow$"},
				  {NULL,	NULL}};

Translation	NHead_trans[] =  {{"Approved:",		""},
				  {"Path:",		""},
				  {"Message-ID:",	""},
				  {"Lines:",		""},
				  {"X-Newsreader:",	""},
				  {"Xref:",		""},
				  {"References:",	""},
				  {"NNTP-Posting-Host:",""},
				  {"Sender:",		""},
				  {"Distribution:",	""},
				  {"Keywords:",		""},
				  {"Reply-To:",		""},
				  {"Nntp-Posting-Host:",""},
				  {"Originator:",	""},
				  {"X-Original-Newsgroups:",""},
				  {"X-Original-Date:",	""},
				  {"Followup-To:"	""},
				  {"Summary:",		""},
				  {"X-Gateway:",	""},
				  {"AREA:",		""},
				  {"X-Anonymously-To:",	""},
				  {"Oganization:",	""},
				  {"Subject:",		"\\\\\n{\\bf Subject:} "},
				  {"Date:",		"\\\\\n{\\bf Date:} "},
				  {"Organization:",	"\\\\\n{\\bf Organization:} "},
				  {"From:",		"\\\\\n{\\bf From:} "},
				  {"Newsgroups:",	"\\\\\n{\\bf Newsgroups:} "},
				  {"X-Realname:",	"\\\\\n{\\bf X-Realname:} "},
				  {NULL,	NULL}};

Translation	News_trans[] = {{"&",		"\\&"},
				  {"*",		"$*$"},
				  {"$",		"\\$"},
				  {"%",		"\\%"},
				  {"#",		"\\#"},
				  {"{",		"\\{"},
				  {"}",		"\\}"},
				  {"\\",	"$\\setminus$"},
				  {"_",		"\\_"},
				  {"<",		"$<$"},
				  {">",		"$>$"},
				  {"^",		"$\\uparrow$"},
				  {NULL,	NULL}};

Translation	Fortran_trans[] = {{"&",	"\\&"},
				  {"\\",	"$\\setminus$"},
				  {"*",		"$*$"},
				  {"$",		"\\$"},
				  {"%",		"\\%"},
				  {"#",		"\\#"},
				  {"<",		"$<$"},
				  {">",		"$>$"},
				  {"^",		"$\\uparrow$"},
				  {"->",	"$\\rightarrow$"},
				  {"\n",	" "},
				  {"\t",	" "},
				  {"{",		"\\{"},
				  {"}",		"\\}"},
				  {"_",		"\\_"},
				  {NULL,	NULL}};

Translation	C_trans[] = {	  {"&",		"\\&"},
				  {"\\",	"$\\setminus$"},
				  {"*",		"$*$"},
				  {"$",		"\\$"},
				  {"%",		"\\%"},
				  {"|",		"$|$"},
				  {"#",		"\\#"},
				  {"<",		"$<$"},
				  {">",		"$>$"},
				  {"^",		"$\\uparrow$"},
				  {"->",	"$\\rightarrow$"},
				  {"\n",	"\\\\\n \\>"},
				  {"{",		"\\{"},
				  {"}",		"\\}"},
				  {"_",		"\\_"},
				  {"--",	"$--$"},
				  {NULL,	NULL}};


Translation	fortran_commands[] = {{"open",	"{\\bf open}"},
				  {"close",	"{\\bf close}"},
				  {"print",	"{\\bf print}"},
				  {"write",	"{\\bf write}"},
				  {"read",	"{\\bf read}"},
				  {"do",	"{\\bf do} "},
				  {"if",	"{\\bf if} "},
				  {"then",	"{\\bf then} "},
				  {"else",	"{\\bf else} "},
				  {"elseif",	"{\\bf elseif} "},
				  {"call",	"{\\bf call} "},
				  {"end",	"{\\bf end} "},
				  {"endif",	"{\\bf endif} "},
				  {".le.",	"{\\bf .le.} "},
				  {".ge.",	"{\\bf .ge.} "},
				  {".eq.",	"{\\bf .eq.} "},
				  {".ne.",	"{\\bf .ne.} "},
				  {".lt.",	"{\\bf .lt.} "},
				  {".gt.",	"{\\bf .gt.} "},
				  {".and.",	"{\\bf .and.} "},
				  {".or.",	"{\\bf .or.} "},
				  {".not.",	"{\\bf .not.} "},
				  {"integer",	"{\\bf integer} "},
				  {"character",	"{\\bf character} "},
				  {"real",	"{\\bf real} "},
				  {"doubleprecision",	"{\\bf doubleprecision} "},
				  {"logical",	"{\\bf logical} "},
				  {"implicit",	"{\\bf implicit} "},
				  {"common",	"{\\bf common} "},
				  {"parameter",	"{\\bf parameter} "},
				  {"return",	"{\\bf return} "},
				  {"stop",	"{\\bf stop} "},
				  {"goto",	"{\\bf goto} "},
				  {"continue",	"{\\bf continue} "},
				  {"program",	"{\\bf program} "},
				  {"subroutine","{\\bf subroutine} "},
				  {"function",	"{\\bf function} "},
				  {"external",	"{\\bf external} "},
				  {"format",	"{\\bf format} "},
				  {"include",	"{\\bf include} "},
				  {"stop",	"{\\bf stop} "},
				  {NULL,NULL}};

Translation	c_commands[] = {  {"while ",	"{\\bf while} "},
				  {"for ",	"{\\bf for} "},
				  {"if ",	"{\\bf if} "},
				  {"while(",	"{\\bf while} ("},
				  {"for(",	"{\\bf for} ("},
				  {"if(",	"{\\bf if} ("},
				  {"do ",	"{\\bf do} "},
				  {"else ",	"{\\bf else} "},
				  {"else\n",	"{\\bf else}\\\\\n \\> "},
				  {"switch ",	"{\\bf switch} "},
				  {"break;",	"{\\bf break};"},
				  {"case ",	"{\\bf case} "},
				  {"default:",	"{\\bf default}:"},
				  {"return ",	"{\\bf return} "},
				  {"return(",	"{\\bf return} ("},
				  {"continue ",	"{\\bf continue} "},
				  {"goto ",	"{\\bf goto} "},
				  {"enum ",	"{\\bf enum} "},
				  {"union ",	"{\\bf union} "},
				  {"struct ",	"{\\bf struct} "},
				  {"typedef ",	"{\\bf typedef} "},
				  {"extern ",	"{\\bf extern} "},
				  {"auto ",	"{\\bf auto} "},
				  {"static ",	"{\\bf static} "},
				  {"register ",	"{\\bf register} "},
				  {"void ",	"{\\bf void} "},
				  {"char ",	"{\\bf char} "},
				  {"short ",	"{\\bf short} "},
				  {"int ",	"{\\bf int} "},
				  {"long ",	"{\\bf long} "},
				  {"float ",	"{\\bf float} "},
				  {"typedef ",	"{\\bf typedef} "},
				  {"typedef\t",	"{\\bf typedef} "},
				  {"double ",	"{\\bf double} "},
				  {"extern\t",	"{\\bf extern} "},
				  {"auto\t",	"{\\bf auto} "},
				  {"static\t",	"{\\bf static} "},
				  {"register\t","{\\bf register} "},
				  {"void\t",	"{\\bf void} "},
				  {"char\t",	"{\\bf char} "},
				  {"short\t",	"{\\bf short} "},
				  {"int\t",	"{\\bf int} "},
				  {"long\t",	"{\\bf long} "},
				  {"float\t",	"{\\bf float} "},
				  {"double\t",	"{\\bf double} "},
				  {"signed\t",	"{\\bf signed} "},
				  {"unsigned\t","{\\bf unsigned} "},
				  {"signed ",	"{\\bf signed} "},
				  {"unsigned ",	"{\\bf unsigned} "},
				  {"sizeof ",	"{\\bf sizeof} "},
				  {"sizeof(",	"{\\bf sizeof}("},
				  {"#include",	"{\\bf \\#include}"},
				  {"#define",	"{\\bf \\#define}"},
				  {"#ifdef",	"{\\bf \\#ifdef}"},
				  {"#ifndef",	"{\\bf \\#ifndef}"},
				  {"#else",	"{\\bf \\#else}"},
				  {"#endif",	"{\\bf \\#endif}"},
				  {"#elif",	"{\\bf \\#elif}"},
				  {"#undef",	"{\\bf \\#undef}"},
				  {"#line",	"{\\bf \\#line}"},
				  {"#error",	"{\\bf \\#error}"},
				  {"#pragma",	"{\\bf \\#pragma}"},
				  {NULL,	NULL}};


Translation *trans 	= Normal_trans;	/* Uebersetzungsvorschrift */

enum modes {NORMAL, C, NEWS, FORTRAN};


char	*filename 	= NULL;
FILE	*fpin 		= stdin;

int	TWOCOLUMNS	= 0;

int	HEADER		= 1;		/* Latexfileheader schreiben wenn 1 */

enum modes MODE		= NORMAL;

/* Variablen fuer die Language-parser */
int	COMMENT		= 0;
int	TEXT		= 0;
int	CHAR		= 0;
int	WAIT		= 0;

int	line		= 1;

int	CONTENTS	= 0;
int	NEWPAGE		= 0;

main(argc,argv)
	int	argc;
	char	**argv;
{	int	i;
	extern	char	*optarg;
	extern	int	optind;
	char	*h;
	int	fid;

	/* Optionen auswerten und evtl. Filenamen holen */
	while ((i = getopt(argc,argv,"tcnfhCN")) != EOF)
		switch (i) {
		case 't':
			TWOCOLUMNS = 1;
			break;
		case 'c':
			trans = C_trans;
			MODE = C;
			break;
		case 'C':
			CONTENTS = 1;
			break;
		case 'N':
			NEWPAGE = 1;
			break;
		case 'f':
			trans = Fortran_trans;
			MODE = FORTRAN;
			break;
		case 'n':
			trans = News_trans;
			MODE = NEWS;
			break;
		case 'h':
			usage();
		}

	/* Mehrspaltig nur im normalen Textmodus */
	if (MODE != NORMAL)
		TWOCOLUMNS = 0;
	/* Den Dokumentkopf schreiben */
	if (HEADER)
		write_header(argc - optind);

	fid = optind;
	do {
		COMMENT		= 0;
		TEXT		= 0;
		CHAR		= 0;
		WAIT		= 0;

		if (fid < argc)
			filename = argv[fid];

		if (filename)	/* Nicht von stdin lesen */
		{	fpin = fopen(filename,"r");	/* Weil File -> File oeffnen */
			if (!fpin)
			{	perror(filename);
				exit(-1);	/* Fehler beim oeffnen des files */
			}
			if (MODE == C || MODE == FORTRAN) {
				char	*c = (filename ? filename : "stdin");
				int	i = 0;
				int	MATCH;

				printf("\\section{");
				while (*c != '\n' && *c != '\0') {
					MATCH = 0;
					i = 0;
					while (Normal_trans[i].from && !MATCH) {
						if (!strncmp(Normal_trans[i].from,c,strlen(Normal_trans[i].from))) {
							printf("%s",Normal_trans[i].to);
							MATCH = 1;
							c += strlen(Normal_trans[i].from);
						}
						i++;
					}
					if (!MATCH)
						printf("%c",*c++);
				}
				printf("}\n");
			}
		}
		if (MODE == C) {
			printf("\\begin{tabbing}\n");
			printf("%% Tabulatorstops setzen, Zeile aber nicht ausgeben\n");
			printf("mmm \\= mm \\= mm \\=");
			printf("mm \\= mm \\= mm \\=");
			printf("mm \\= mm \\= mm \\=");
			printf("mm \\= mm \\= mm \\=");
			printf("mm \\= \\kill\n\n\\> ");
		}

		/* File umwandeln und Ergebnis auf stdout schicken. Von dort kann es
		   weiterbearbeitet werden.
		   Standardmaessig wird von stdin gelesen, falls kein filename angegeben
		   wurde.
		*/
		line = 1;
		convert_file(fpin);

		if (COMMENT || TEXT || WAIT || CHAR) {
			fprintf(stderr,"Warning: In file %s %s region not closed.\n",
				(filename ? filename : "stdin"),
				(COMMENT ? "commentary" : TEXT ? "string constant" : CHAR
				 ? "character" : "backslash"));
			printf("}");
		}
		fid++;

		if (MODE == C) {
			printf("\\end{tabbing}\n");
		}
		/* Neue Seite fuer ein neues File */
		if (argc > optind && (MODE == C || MODE == FORTRAN)) {
			if (NEWPAGE)
				printf(" \\newpage\n");
			else
				printf(" \\vspace{2cm}\n");
		}

		/* Wenn Filename angegeben, dann File wieder schliessen */
		if (filename) {
			fclose(fpin);	/* File wieder schliessen */
		}
	} while (fid < argc);
	fprintf(stdout,"\\end{document}\n");
}

/*
 * konvertieren eines Ascii-textes in ein Roh-TeX Dokument.
 * Probleme treten bei Absaetzen und eigerueckten Zeilen auf, da
 * dort ein Newline eingefuegt wird.
 * Die Routine arbeitet allerdings relativ zuverlaessig mit NetNews-Artikeln
 * und aehnlichen Artikeln. Auch Sourcecodes koennen nun einigermassen sinnvoll
 * bearbeitet werden.
 */
convert_file(fp)
	FILE	*fp;
{	char	dummy[512];	/* Characterbuffer */
	char	outdummy[1024];	/* Ausgabebuffer */
	int	NL = 0;
	int	TAB = 0;
	int	QUOTE = 0;
	int	MAX_RUN;
	int	NOTAB;

	/* Das File einlesen und baerbeiten */
	while (fgets(dummy,512,fp)) {	/* Zeile einlesen */
		char	*od = outdummy;
		char	*id = dummy;
		int	MATCH;
		int	i;

		TAB = 1;
		NOTAB = 1;
		if (MODE == NEWS) {
			if (*id == '>' || *id == '|' || *id == '.' || *id == '-' ||
			    *id == '=' || *id == ':' || *id == '#' || *id == '+') {
				QUOTE = 1;
				NL = 0;
				strncpy(od,"\\\\\n",3);
				od += 3;
			}
			else if (QUOTE) {
				QUOTE = 0;
				NL = 0;
				strncpy(od,"\\\\\n",3);
				od += 3;
			}
			else {
				MATCH = parse_news(&id,&od);
				if (!MATCH && TEXT) {
					TEXT = 0;
					strncpy(od,"\\\\\n",3);
					od += 3;
				}
			}
		}
		else if (MODE == FORTRAN) {	/* Zeilenanfang und Zeilenendeparsen */
			parse_fortran(&id,&od);
		}
		while (*id != '\0') {
			MATCH = 0;
			i = 0;

			/* speziell fuer eine Sprache parsen */
			switch (MODE) {
				case C:
					MATCH = parse_c(&id,&od);
					NOTAB = (1 - MATCH) & NOTAB;
					break;
			}
			/* Sondercharakter herausfiltern */
			while (trans[i].from && !MATCH) {
				if (!strncmp(trans[i].from,id,strlen(trans[i].from))) {
					strncpy (od, trans[i].to, strlen(trans[i].to));
					MATCH = 1;
					if (MODE != NEWS || !COMMENT)
						od += strlen(trans[i].to);
					id += strlen(trans[i].from);
				}
				i++;
			}
			/* Fortransyntax heraussuchen */
			i = 0;
			if (MODE == FORTRAN && !COMMENT) {
				while (fortran_commands[i].from && !MATCH) {
					if (!strncmp(fortran_commands[i].from,id,
					    strlen(fortran_commands[i].from))) {
						strncpy (od, fortran_commands[i].to,
							strlen(fortran_commands[i].to));
						MATCH = 1;
						od += strlen(fortran_commands[i].to);
						id += strlen(fortran_commands[i].from);
					}
					i++;
				}
			}
			/* Es wurde kein besonderes Zeichen gefunden */
			if (!MATCH) {
				if (MODE == C) {
					if (*id == '\t') {
						if (NOTAB) {
							if (COMMENT) {
								strncpy(od,"}\\> {\\it ",9);
								od += 9;
							}
							else {
								strncpy(od,"\\> ",3);
								od += 3;
							}
							id++;
							MATCH = 1;
						}
						else
							*id = ' ';
					}
					else if (!isspace(*id))
						NOTAB = 0;
				}
				if ((MODE == NEWS || MODE==NORMAL) && !COMMENT && !TEXT) {
					if (*id == '\n') {
						NL++;
/*						id++;
*/					}
					else {
						TAB = (isspace(*id) && TAB ? 1 : 0);
						if (NL + TAB > 1) {
							od--;
							strncpy(od,"\\\\\n\n",4);
							od += 4;
						}
/*						if (NL)
							*od++ = '\n';
*/						NL = 0;
					}
				}
				if (!MATCH)
					*od++ = *id++;
			}
		}
		*od = '\0';
		printf("%s", outdummy);
	}	
}

/*
 * Schreiben des TeX-Headers mit den Angaben fuer Dokumentgroesse und Art
 */
write_header(files)
int	files;
{	static char	*txt[]={
			"\\voffset=0pt",
			"\\documentstyle[]{article}",
			"\\pagestyle{plain}\n",
			"\\topmargin -1.5cm",
			"\\oddsidemargin 0.0cm",
			"\\evensidemargin -1.9cm",
			"\\textwidth 16cm",
			"\\textheight 24cm",
			"\\topskip 0cm\n",
			"\\begin{document}",
			NULL};
	char	**cpp;

	if (TWOCOLUMNS)
		txt[1] = "\\documentstyle[twocolumn]{article}";
	if (files && (MODE == C || MODE == FORTRAN))
		txt[2] = "\\pagestyle{headings}\n";
	for(cpp = txt;*cpp;cpp++)
		printf("%s\n",*cpp);
	if (files && CONTENTS && (MODE == C || MODE == FORTRAN))
		printf("\\tableofcontents\n\\newpage");
	HEADER = 0;
}

/*
 * Ausgabe der Usage
 */
usage()
{	static char	*usagetxt[]={
	"usage: txt2tex [-cnthCN][file1][file2] ...",
	" ",
	"   -N          : start newpage after filending",
	"   -C  	: show table of contents (only if c or fortran)",
	"   -c  	: takes C-sources as input",
	"   -t  	: Two-column-mode",
	"   -n  	: format NetNews-articles",
	"   -h  	: Usage",
	"   file	: specify file to read from (default stdin)",
	"Note: Make sure that you use favour options before entering filenames",
	NULL};
	char	**cpp;

	for(cpp = usagetxt;*cpp;cpp++)
		fprintf(stderr,"%s\n",*cpp);
	exit(-1);
}


/*
 * Hier steht der C-Parser, der auf Texte, Kommentare und Befehlswoerter untersucht.
 */

int	parse_c (id, od)
char	**id;
char	**od;
{	int	r = 0;
	int	i = 0;
	int	MARK = 0;
	int	CMARK = 0;

	if (!COMMENT && !TEXT) {		/* Kein Kommentar */
		if (!strncmp(*id,"\"",1) && !CHAR) {
			TEXT = 1;
			strncpy(*od,"\"{\\tt ",6);
			*id += 1;
			*od += 6;
			r = 1;
		}
		else if (!strncmp(*id,"\'",1) && !WAIT) {	/* Text faengt an */
			CHAR = 1 - CHAR;
		}
		else if (!strncmp(*id,"/*",2)) {	/* Kommentaranfangszeichen */
			COMMENT = 1;
			r = 1;
			strncpy(*od,"{\\it /$*$",9);
			*id += 2;
			*od += 9;
			if (!strncmp(*id,"\n",1)) {	/* Bei Zeilenumbruch aufpassen */
				r = 1;
				strncpy(*od,"}",1);
				*od += 1;
				MARK = 1;
			}
		}
		else while (c_commands[i].from && !r) {
			if (!strncmp(c_commands[i].from,*id,strlen(c_commands[i].from))) {
				strncpy (*od, c_commands[i].to, strlen(c_commands[i].to));
				r = 1;
				*od += strlen(c_commands[i].to);
				*id += strlen(c_commands[i].from);
			}
			i++;
		}
	}
	else if (COMMENT) {	/* Kommentar hat schon begonnen */
		if (!strncmp(*id,"*/",2)) {	/* Kommentarendezeichen */
			r = 1;
			COMMENT = 0;
			strncpy(*od,"$*$/} ",6);
			*id += 2;
			*od += 6;
		}
		else if (!strncmp(*id,"\n",1)) {	/* Bei Zeilenumbruch aufpassen */
			r = 1;
			strncpy(*od,"}",1);
			*od += 1;
			MARK = 1;
		}
	}
	else if (TEXT) {
		if (!strncmp(*id,"\"",1) && !WAIT) {	/* Text faengt an */
			TEXT = 0;
			strncpy(*od,"}\"",2);
			*od += 2;
			*id += 1;
			r = 1;
		}
		else if (!strncmp(*id,"\\",1) && !WAIT)
			WAIT = 1;
		else if (!strncmp(*id,"\n",1)) {	/* Bei Zeilenumbruch aufpassen */
			r = 1;
			strncpy(*od,"}",1);
			*od += 1;
			CMARK = 1;
			WAIT = 0;
		}
		else if (!strncmp(*id," ",1)) {	/* Bei Zeilenumbruch aufpassen */
			r = 1;
			strncpy(*od,"$\\sqcup$",8);
			*od += 8;
			*id += 1;
			WAIT = 0;
		}
		else if (!strncmp(*id,"\t",1)) {	/* Bei Zeilenumbruch aufpassen */
			r = 1;
			strncpy(*od,"[Tab]",5);
			*od += 5;
			*id += 1;
			WAIT = 0;
		}
		else
			WAIT = 0;
	}
	if (!strncmp(*id,"\n",1)) {
		line++;
		if (((float) line/5.0 - line/5) == 0.0) {
			char	t[50];

			r = 1;
			sprintf(t," \\\\\n {\\footnotesize %d} \\>\0",line);
			strncpy(*od,t,strlen(t));
			*od += strlen(t);
			*id += 1;
			if (MARK) {
				strncpy(*od,"{\\it ",5);
				*od += 5;
			}
			else if (CMARK) {
				strncpy(*od,"{\\tt ",5);
				*od += 5;
			}
		}
		else {
			if (MARK) {
				strncpy(*od,"\\\\\n\\> {\\it ",11);
				*od += 11;
				*id += 1;
			}
			else if (CMARK) {
				strncpy(*od,"\\\\\n\\> {\\tt ",11);
				*od += 11;
				*id += 1;
			}
			else {
				r = 0;
			}
		}
	}
	return (r);
}


/*
 * News-Parser
 */
int	parse_news(id,od)
char	**id, **od;
{	int	r = 0;
	int	i = 0, MATCH = 0;

	/* Sondercharakter herausfiltern */
	while (NHead_trans[i].from && !MATCH) {
		if (!strncmp(NHead_trans[i].from,*id,strlen(NHead_trans[i].from))) {
			if (strlen(NHead_trans[i].to)) {
				strncpy (*od, NHead_trans[i].to, strlen(NHead_trans[i].to));
				*od += strlen(NHead_trans[i].to);
				*id += strlen(NHead_trans[i].from);
				TEXT = 1;
			}
			else
				*id = "\0";
			MATCH = 1;
			r = 1;
		}
		i++;
	}
	if (!MATCH) {
		if (!strncmp("Article:",*id,8)) {	/* Ein neuer Artikel */
			r = 1;
			strncpy(*od,"\n\\newpage\n\\noindent\n",20);
			*od += 20;
			*id += strlen(*id);
			COMMENT = 0;
			TEXT = 1;
		}
	}
	return(r);
}


/*
 * Hier steht der Fortran-Linienanfangsparser
 * Es werden alle 5 Zeilen die Zeilennummern ausgegeben, Ganzzeilige Kommentare
 * im italic-mode dargestellt, Labels und &'s herausgefiltert sowie die nachfolgenden
 * Zeichen je nach Anzahl der Spaces und Tabs eingerueckt.
 */

int	parse_fortran (id, od)
char	**id;
char	**od;
{	int	r = 0;
	int	i = 0;
	int	MARK = 0;
	int	label;
	char	*e;
	int	SPACES_FOR_TABS = 8;
	int	chars = 0;

	if (COMMENT) {		/* Die Zeile davor war eine Kommentarzeile */
		strncpy(*od,"}",1);
		*od += 1;
	}
	if (line)
		strncpy(*od,"\\\\\n",3);
	*od += 3;
	COMMENT = 0;
	if ((float)(line/5.0 - line/5) == 0.0) {
		char	t[50];

		sprintf(t,"\\makebox[0.5cm][r]{\\footnotesize %d} $|$ \0",line);
		strncpy(*od,t,strlen(t));
		*od += strlen(t);
	}
	else {
		strncpy(*od,"\\hspace*{0.5cm} $|$ ",20);
		*od += 20;
	}
	/* Zeilennummer erhoehen */
	line++;
	/* feststellen ob Zeile mit einem Kommentarzeichen beginnt */
	if (**id == '*' || **id == 'c' || **id == 'C') {
		COMMENT = 1;
		strncpy(*od,"{\\it \\makebox[2ex]{c} \\hspace{0.7cm} ",36);
		*od += 36;
		*id += 1;	/* Kommentarzeichen uebergehen */
		COMMENT = 1;
		return(0);
	}
	else {
		COMMENT = 0;
		strncpy(*od,"\\hspace*{2ex} ",14);
		*od += 14;
	}
	/* Label finden */
	label = (int) strtoul(*id,&e,10);
	if (e != *id) {	/* Es gab einen label */
		char	t[50];

		sprintf(t,"\\makebox[0.5cm][r]{%4d} \0",label);
		strncpy(*od,t,strlen(t));
		*od += strlen(t);
	}
	else {
		strncpy(*od,"\\hspace*{0.5cm} ",16);
		*od += 16;
	}
	*id = e;
	while (isspace((int) *(++(*id))))
		if (**id == ' ')
			chars++;
		else if (**id == '\t')
			chars += SPACES_FOR_TABS;
	if (**id == '&') {
		strncpy(*od,"\\makebox[0.2cm]{\\&} ",20);
		*od += 20;
		chars = -8;
		while (isspace((int) *(++(*id))))
			if (**id == ' ')
				chars++;
			else if (**id == '\t')
				chars += SPACES_FOR_TABS;
	}
	else {
		strncpy(*od,"\\hspace*{0.2cm} ",16);
		*od += 16;
	}
	/* Einruecken */
	if (chars) {
		char	t[50];

		sprintf(t,"\\hspace*{%dex} \0",chars);
		strncpy(*od,t,strlen(t));
		*od += strlen(t);
	}
}
