/****************************************************************************
 * This module is based on Twm, but has been siginificantly modified 
 * by Rob Nation (nation@rocket.sanders.lockheed.com)
 ****************************************************************************/
/*****************************************************************************/
/**       Copyright 1988 by Evans & Sutherland Computer Corporation,        **/
/**                          Salt Lake City, Utah                           **/
/**  Portions Copyright 1989 by the Massachusetts Institute of Technology   **/
/**                        Cambridge, Massachusetts                         **/
/**                                                                         **/
/**                           All Rights Reserved                           **/
/**                                                                         **/
/**    Permission to use, copy, modify, and distribute this software and    **/
/**    its documentation  for  any  purpose  and  without  fee is hereby    **/
/**    granted, provided that the above copyright notice appear  in  all    **/
/**    copies and that both  that  copyright  notice  and  this  permis-    **/
/**    sion  notice appear in supporting  documentation,  and  that  the    **/
/**    names of Evans & Sutherland and M.I.T. not be used in advertising    **/
/**    in publicity pertaining to distribution of the  software  without    **/
/**    specific, written prior permission.                                  **/
/**                                                                         **/
/**    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD    **/
/**    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-    **/
/**    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR    **/
/**    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-    **/
/**    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA    **/
/**    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER    **/
/**    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE    **/
/**    OR PERFORMANCE OF THIS SOFTWARE.                                     **/
/*****************************************************************************/


/***********************************************************************
 *
 * fvwm event handling
 *
 ***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>

#include "fvwm.h"
#include <X11/Xatom.h>
#include "menus.h"
#include "misc.h"
#include "parse.h"
#include "screen.h"
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif /* SHAPE */

unsigned int mods_used = (ShiftMask | ControlMask | Mod1Mask);
extern int menuFromFrameOrWindowOrTitlebar;
void My_XNextEvent(Display *dpy, XEvent *event);

char *Action;
int Context = C_NO_CONTEXT;	/* current button press context */
FvwmWindow *ButtonWindow;	/* button press window structure */
XEvent Event;			/* the current event */
FvwmWindow *Tmp_win;		/* the current fvwm window */

void flush_expose();

int last_event_type=0;
FvwmWindow *last_event_Fvwmwindow=0;
Window last_event_window=0;
FvwmWindow *CurrentWindow;

#ifndef NO_PAGER
extern Window Pager_w;
#endif

#ifdef SHAPE
extern int ShapeEventBase;
void HandleShapeNotify();
#endif /* SHAPE */

/***********************************************************************
 *
 *  Procedure:
 *	DispatchEvent - handle a single X event stored in global var Event
 *
 ***********************************************************************
 */
void DispatchEvent()
{
    Window w = Event.xany.window;

    if (XFindContext (dpy, w, FvwmContext, (caddr_t *) &Tmp_win) == XCNOENT)
      Tmp_win = NULL;
    last_event_type = Event.type;
    last_event_Fvwmwindow = Tmp_win;
    last_event_window = w;

    switch(Event.type)
      {
      case Expose:
	HandleExpose();
	break;
      case DestroyNotify:
	HandleDestroyNotify();
	break;
      case MapRequest:
	HandleMapRequest();
	break;
      case MapNotify:
	HandleMapNotify();
	break;
      case UnmapNotify:
	HandleUnmapNotify();
	break;
      case MotionNotify:
	HandleMotionNotify();
	break;
      case ButtonPress:
	HandleButtonPress();
	break;
      case ButtonRelease:
	HandleButtonRelease();
	break;
      case EnterNotify:
	HandleEnterNotify();
	break;
      case LeaveNotify:
	HandleLeaveNotify();
	break;
      case ConfigureRequest:
	HandleConfigureRequest();
	break;
      case ClientMessage:
	HandleClientMessage();
	break;
      case PropertyNotify:
	HandlePropertyNotify();
	break;
      case KeyPress:
	HandleKeyPress();
	break;
      case VisibilityNotify:
	HandleVisibilityNotify();
	break;
      case ColormapNotify:
	HandleColormapNotify();
	break;
      default:
#ifdef SHAPE
	if(Event.type = (ShapeEventBase + ShapeNotify)) 
	  HandleShapeNotify();
#endif /* SHAPE */

	break;
      }
    last_event_type =0;
    return;
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleEvents - handle X events
 *
 ************************************************************************/
int fd_width, x_fd;
void HandleEvents()
{
  fd_width = sysconf(_SC_OPEN_MAX);
  x_fd = XConnectionNumber(dpy);

  while (TRUE)
    {
      My_XNextEvent(dpy, &Event);
      DispatchEvent ();
      
    }
}

/***********************************************************************
 *
 *  Procedure:
 *	Find the Fvwm context for the Event.
 *
 ************************************************************************/
int GetContext(FvwmWindow *t, XEvent *e)
{
  int Context,i;
  Window w;

  Context = C_NO_CONTEXT;
  w= e->xany.window;

  if((e->type == KeyPress)&&(e->xkey.subwindow != None))
    w = e->xkey.subwindow;
  if((e->type == ButtonPress)&&(e->xbutton.subwindow != None)&&
     (e->xbutton.subwindow == t->w))
    w = e->xbutton.subwindow;
  if ((w == Scr.Root)||(w == Scr.NoFocusWin))
    Context = C_ROOT;
  if (t)
    {
      if (w == t->title_w)
	Context = C_TITLE;
      if (w == t->w)
	Context = C_WINDOW;
      if (w == t->icon_w)
	Context = C_ICON;
      if (w == t->icon_pixmap_w)
	Context = C_ICON;
      if (w == t->frame)
	Context = C_FRAME;
      if ((w == t->left_side_w)||
	  (w == t->right_side_w)||
	  (w == t->top_w)||
	  (w == t->bottom_w))
	Context = C_SIDEBAR;
      for(i=0;i<Scr.nr_left_buttons;i++)
	{
	  if(w == t->left_w[i])
	    Context = (1<<i)*C_L1;
	}
      for(i=0;i<Scr.nr_right_buttons;i++)
	{
	  if(w == t->right_w[i])
	    Context = (1<<i)*C_R1;
	}
    }
  return Context;
}

/***********************************************************************
 *
 *  Procedure:
 *	HandleKeyPress - key press event handler
 *
 ************************************************************************/

void HandleKeyPress()
{
  FuncKey *key;
  unsigned int modifier;
  
  Context = GetContext(Tmp_win,&Event);
  modifier = (Event.xkey.state & mods_used);
  for (key = Scr.FuncKeyRoot.next; key != NULL; key = key->next)
    {
      ButtonWindow = Tmp_win;
      /* Here's a real hack - some systems have two keys with the
       * same keysym and different keycodes. This converts all
       * the cases to one keycode. */
      Event.xkey.keycode = XKeysymToKeycode(dpy,XKeycodeToKeysym(dpy,
			      Event.xkey.keycode,0));
  
      if ((key->keycode == Event.xkey.keycode) &&
	  ((key->mods == modifier)||(key->mods == AnyModifier)) &&
	   (key->cont & Context))
	{
	  ExecuteFunction(key->func, key->action, Event.xany.window,Tmp_win,
			  &Event,Context,key->val1,key->val2,key->menu);
	  return;
	}
    }
  
  /* if we get here, no function key was bound to the key.  Send it
   * to the client if it was in a window we know about.
   */
  if (Tmp_win)
    {
      Event.xkey.window = Tmp_win->w;
      XSendEvent(dpy, Tmp_win->w, False, KeyPressMask, &Event);
    }
}


/**************************************************************************
 * 
 * Releases dynamically allocated space used to store window/icon names
 *
 **************************************************************************/
void free_window_names (FvwmWindow *tmp, Bool nukename, Bool nukeicon)
{
  if (tmp->icon_name == tmp->name) nukename = False;
  
  if (nukename && tmp && tmp->name != NoName) XFree (tmp->name);
  if (nukeicon && tmp && tmp->icon_name != NoName) XFree (tmp->icon_name);
  return;
}




/***********************************************************************
 *
 *  Procedure:
 *	HandlePropertyNotify - property notify event handler
 *
 ***********************************************************************/
#define MAX_NAME_LEN 200L		/* truncate to this many */
#define MAX_ICON_NAME_LEN 200L		/* ditto */

void HandlePropertyNotify()
{
  char *prop = NULL;
  Atom actual = None;
  int actual_format;
  unsigned long nitems, bytesafter;

  if ((!Tmp_win)||(XGetGeometry(dpy, Tmp_win->w, &JunkRoot, &JunkX, &JunkY,
		   &JunkWidth, &JunkHeight, &JunkBW, &JunkDepth) == 0))
      return;
  switch (Event.xproperty.atom) 
    {
    case XA_WM_NAME:
      if (XGetWindowProperty (dpy, Tmp_win->w, Event.xproperty.atom, 0L, 
			      MAX_NAME_LEN, False, XA_STRING, &actual,
			      &actual_format, &nitems, &bytesafter,
			      (unsigned char **) &prop) != Success ||
	  actual == None)
	return;
      if (!prop) prop = NoName;
      free_window_names (Tmp_win, True, False);
      
      Tmp_win->name = prop;
      
      /* fix the name in the title bar */
      if(!(Tmp_win->flags & ICON))
	{
	  if(Scr.Focus == Tmp_win)
	    SetTitleBar(Tmp_win,True);
	  else
	    SetTitleBar(Tmp_win,False);
	}
      
      /*
       * if the icon name is NoName, set the name of the icon to be
       * the same as the window 
       */
      if (Tmp_win->icon_name == NoName) 
	{
	  Tmp_win->icon_name = Tmp_win->name;
	  RedoIconName(Tmp_win);
	}
      break;
      
    case XA_WM_ICON_NAME:
      if (XGetWindowProperty (dpy, Tmp_win->w, Event.xproperty.atom, 0, 
			      MAX_ICON_NAME_LEN, False, XA_STRING, &actual,
			      &actual_format, &nitems, &bytesafter,
			      (unsigned char **) &prop) != Success ||
	  actual == None)
	return;
      if (!prop) prop = NoName;
      free_window_names (Tmp_win, False, True);
      Tmp_win->icon_name = prop;
      RedoIconName(Tmp_win);
      break;
      
    case XA_WM_HINTS:
      if (Tmp_win->wmhints) 
	XFree ((char *) Tmp_win->wmhints);
      Tmp_win->wmhints = XGetWMHints(dpy, Event.xany.window);

      if((Tmp_win->flags&ICON)&&((Tmp_win->wmhints->flags & IconPixmapHint)||
				 (Tmp_win->wmhints->flags & IconWindowHint)))
	{
	  if (!Scr.SuppressIcons)
	    {
	      if(Tmp_win->flags & ICON_OURS)
		{
		  XDestroyWindow(dpy,Tmp_win->icon_w);
		  if(Tmp_win->icon_pixmap_w != None)
		    XDestroyWindow(dpy,Tmp_win->icon_pixmap_w);
		}
	      else 
		XUnmapWindow(dpy,Tmp_win->icon_w);

	    }
	  Tmp_win->flags &= ~ICON;
	  CreateIconWindow(Tmp_win,Tmp_win->icon_x_loc, Tmp_win->icon_y_loc);
	  RaiseWindow(Tmp_win);
	  if (!Scr.SuppressIcons)
	    {
	      XMapWindow(dpy, Tmp_win->icon_w);
	      if(Tmp_win->icon_pixmap_w != None)
		XMapWindow(dpy, Tmp_win->icon_pixmap_w);
	    }
	  Tmp_win->flags |= ICON;
	  DrawIconWindow(Tmp_win);
	}
      break;
      
    case XA_WM_NORMAL_HINTS:
      GetWindowSizeHints (Tmp_win);
      break;
      
    default:
      if (Event.xproperty.atom == _XA_WM_PROTOCOLS) 
	FetchWmProtocols (Tmp_win);
      break;
    }
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleClientMessage - client message event handler
 *
 ************************************************************************/

void HandleClientMessage()
{
  XEvent button;

  if ((Event.xclient.message_type == _XA_WM_CHANGE_STATE)&&
      (Tmp_win)&&(Event.xclient.data.l[0]==IconicState)&&
      !(Tmp_win->flags & ICON))
    {
      XQueryPointer( dpy, Scr.Root, &JunkRoot, &JunkChild,
		    &(button.xmotion.x_root),
		    &(button.xmotion.y_root),
		    &JunkX, &JunkY, &JunkMask);
      button.type = 0;
      ExecuteFunction(F_ICONIFY, NULLSTR, Event.xany.window,
		      Tmp_win, &button, C_FRAME,0,0, (MenuRoot *)0);
    }
}

/***********************************************************************
 *
 *  Procedure:
 *	HandleExpose - expose event handler
 *
 ***********************************************************************/
void HandleExpose()
{
  if (Event.xexpose.count != 0)
    return;

  if (Tmp_win)
    {
#ifndef NO_PAGER
      if(Tmp_win->w == Pager_w)
	ReallyRedrawPager();
#endif
      if ((Event.xany.window == Tmp_win->title_w))
	{
	  if(Scr.Focus == Tmp_win)
	    SetTitleBar(Tmp_win,True);
	  else
	    SetTitleBar(Tmp_win,False);
	}
      else
	{
	  if(Scr.Focus == Tmp_win)
	    SetBorder(Tmp_win,True,True,True);
	  else
	    SetBorder(Tmp_win,False,True,True);
	}
    }
  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleDestroyNotify - DestroyNotify event handler
 *
 ***********************************************************************/
void Destroy(FvwmWindow *Tmp_win)
{ 
  XEvent dummy;
  int i;

  /*
   * Warning, this is also called by HandleUnmapNotify; if it ever needs to
   * look at the event, HandleUnmapNotify will have to mash the UnmapNotify
   * into a DestroyNotify.
   */

  if(!Tmp_win)
    return;

  if(Tmp_win->frame)
    XDestroyWindow(dpy, Tmp_win->frame);

  if ((Tmp_win->icon_w)&&(Tmp_win->flags & PIXMAP_OURS))
    XFreePixmap(dpy, Tmp_win->iconPixmap);
  if ((Tmp_win->icon_w)&&(Tmp_win->flags &ICON_OURS))
    {
      XDestroyWindow(dpy, Tmp_win->icon_w);
      if(Tmp_win->icon_pixmap_w != None)
	XDestroyWindow(dpy, Tmp_win->icon_pixmap_w);
    }
#ifndef NO_PAGER
  if ((Pager_w)&& !(Tmp_win->flags & STICKY))
    {
      ShowCurrentPort();
      XDestroyWindow(dpy, Tmp_win->pager_view);
      ShowCurrentPort();
    }
#endif

  if (Tmp_win == NULL)
    return;

  if (Tmp_win == Scr.Focus)
    {
      if((Scr.ClickToFocus)&&(Tmp_win->next))
	SetBorder (Tmp_win->next, True,False,True);
      else
	XSetInputFocus (dpy, Scr.NoFocusWin, RevertToParent, CurrentTime);
    }
  XDeleteContext(dpy, Tmp_win->w, FvwmContext);

  if (Tmp_win->icon_w)
    {
      XDeleteContext(dpy, Tmp_win->icon_w, FvwmContext);
      if(Tmp_win->icon_pixmap_w != None)
	XDeleteContext(dpy, Tmp_win->icon_pixmap_w, FvwmContext);
    }

  if((Tmp_win->flags & TITLE)||(Tmp_win->flags & BORDER))
    XDeleteContext(dpy, Tmp_win->frame, FvwmContext);
  if (Tmp_win->flags & TITLE)
    {
      XDeleteContext(dpy, Tmp_win->title_w, FvwmContext);
      for(i=0;i<Scr.nr_left_buttons;i++)
	XDeleteContext(dpy, Tmp_win->left_w[i], FvwmContext);
      for(i=0;i<Scr.nr_right_buttons;i++)
	XDeleteContext(dpy, Tmp_win->right_w[i], FvwmContext);
    }
  if (Tmp_win->flags & BORDER)
    {
      XDeleteContext(dpy, Tmp_win->left_side_w, FvwmContext);
      XDeleteContext(dpy, Tmp_win->right_side_w, FvwmContext);
      XDeleteContext(dpy, Tmp_win->bottom_w, FvwmContext);
      XDeleteContext(dpy, Tmp_win->top_w, FvwmContext);
    }

  Tmp_win->prev->next = Tmp_win->next;
  if (Tmp_win->next != NULL)
    Tmp_win->next->prev = Tmp_win->prev;
  free_window_names (Tmp_win, True, True);		
  if (Tmp_win->wmhints)					
    XFree ((char *)Tmp_win->wmhints);
  if (Tmp_win->class.res_name && Tmp_win->class.res_name != NoName)  
    XFree ((char *)Tmp_win->class.res_name);
  if (Tmp_win->class.res_class && Tmp_win->class.res_class != NoName) 
    XFree ((char *)Tmp_win->class.res_class);
  
  free((char *)Tmp_win);
#ifndef NO_PAGER
  RedrawPager();
#endif
  return;
}


void HandleDestroyNotify()
{
  Destroy(Tmp_win);
}




/***********************************************************************
 *
 *  Procedure:
 *	HandleMapRequest - MapRequest event handler
 *
 ************************************************************************/
void HandleMapRequest()
{
  Event.xany.window = Event.xmaprequest.window;
  if(XFindContext(dpy, Event.xany.window, FvwmContext, 
		      (caddr_t *)&Tmp_win)==XCNOENT)
    Tmp_win = NULL;

  XFlush(dpy);

  /* If the window has never been mapped before ... */
  if(!Tmp_win)
    {
      /* Add decorations. */
      Tmp_win = AddWindow(Event.xany.window);
      if (Tmp_win == NULL)
	return;
    }

  /* If it's not merely iconified, and we have hints, use them. */
  if ((!(Tmp_win->flags & ICON)) &&
      Tmp_win->wmhints && (Tmp_win->wmhints->flags & StateHint))
    {
      int state;

      state = Tmp_win->wmhints->initial_state;
      switch (state) 
	{
	case DontCareState:
	case NormalState:
	case InactiveState:
	  XMapWindow(dpy, Tmp_win->w);
	  XMapWindow(dpy, Tmp_win->frame);
	  Tmp_win->flags |= MAPPED;
	  SetMapStateProp(Tmp_win, NormalState);
	  break;
	  
	case IconicState:
	  Iconify(Tmp_win, 0, 0);
	  break;
	}
    }
  /* If no hints, or currently an icon, just "deiconify" */
  else
    {
      DeIconify(Tmp_win);
    }
  KeepOnTop();
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleMapNotify - MapNotify event handler
 *
 ***********************************************************************/
void HandleMapNotify()
{
  if (!Tmp_win)
    return;
  /*
   * Need to do the grab to avoid race condition of having server send
   * MapNotify to client before the frame gets mapped; this is bad because
   * the client would think that the window has a chance of being viewable
   * when it really isn't.
   */
  XGrabServer (dpy);

  if (Tmp_win->icon_w)
    XUnmapWindow(dpy, Tmp_win->icon_w);
  if(Tmp_win->icon_pixmap_w != None)
    XUnmapWindow(dpy, Tmp_win->icon_pixmap_w);
  XMapSubwindows(dpy, Tmp_win->lead_w);
  XMapWindow(dpy, Tmp_win->lead_w);
  if(Scr.Focus == Tmp_win)
    {
      XSetInputFocus (dpy, Tmp_win->w, RevertToParent, CurrentTime);
    }
  XUngrabServer (dpy);
  XFlush (dpy);
  Tmp_win->flags |= MAPPED;
  Tmp_win->flags &= ~ICON;
  KeepOnTop();
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleUnmapNotify - UnmapNotify event handler
 *
 ************************************************************************/
void HandleUnmapNotify()
{
  int dstx, dsty;
  Window dumwin;
  XEvent dummy;

  /*
   * The July 27, 1988 ICCCM spec states that a client wishing to switch
   * to WithdrawnState should send a synthetic UnmapNotify with the
   * event field set to (pseudo-)root, in case the window is already
   * unmapped (which is the case for fvwm for IconicState).  Unfortunately,
   * we looked for the FvwmContext using that field, so try the window
   * field also.
   */
  if (!Tmp_win)
    {
      Event.xany.window = Event.xunmap.window;
      if (XFindContext(dpy, Event.xany.window,
		       FvwmContext, (caddr_t *)&Tmp_win) == XCNOENT)
	Tmp_win = NULL;
    }
  
  if (Tmp_win == Scr.Focus)
    {
      if((Scr.ClickToFocus)&&(Tmp_win->next))
	SetBorder (Tmp_win->next, True,False,True);
      else
	XSetInputFocus (dpy, Scr.NoFocusWin, RevertToParent, CurrentTime);
    }

  if ((!Tmp_win) || (!(Tmp_win->flags & MAPPED)&&!(Tmp_win->flags&ICON)))
    return;

  XGrabServer (dpy);
  if(XCheckTypedWindowEvent (dpy, Event.xunmap.window, DestroyNotify,&dummy)) 
    {
      Destroy(Tmp_win);
      XUngrabServer(dpy);
      return;
    }
      
  /*
   * The program may have unmapped the client window, from either
   * NormalState or IconicState.  Handle the transition to WithdrawnState.
   *
   * We need to reparent the window back to the root (so that fvwm exiting 
   * won't cause it to get mapped) and then throw away all state (pretend 
   * that we've received a DestroyNotify).
   */
  if (XTranslateCoordinates (dpy, Event.xunmap.window, Tmp_win->attr.root,
			     0, 0, &dstx, &dsty, &dumwin)) 
    {
      XEvent ev;
      Bool reparented;
      reparented = XCheckTypedWindowEvent (dpy, Event.xunmap.window, 
						ReparentNotify, &ev);
      SetMapStateProp (Tmp_win, WithdrawnState);
      if (reparented) 
	{
	  if (Tmp_win->old_bw)
	    XSetWindowBorderWidth (dpy, Event.xunmap.window, Tmp_win->old_bw);
	  if((!Scr.SuppressIcons)&&
	     (Tmp_win->wmhints && (Tmp_win->wmhints->flags & IconWindowHint)))
	    XUnmapWindow (dpy, Tmp_win->wmhints->icon_window);
	      
	} 
      else
	{
	  RestoreWithdrawnLocation (Tmp_win);
	}
      XRemoveFromSaveSet (dpy, Event.xunmap.window);
      XSelectInput (dpy, Event.xunmap.window, NoEventMask);
      HandleDestroyNotify ();		/* do not need to mash event before */
      /*
       * Flush any pending events for the window.
       */
      while(XCheckWindowEvent(dpy, Event.xunmap.window,
			      StructureNotifyMask | PropertyChangeMask |
			      ColormapChangeMask | VisibilityChangeMask |
			      EnterWindowMask | LeaveWindowMask, &dummy));
      
    } /* else window no longer exists and we'll get a destroy notify */
  XUngrabServer (dpy);

  XFlush (dpy);
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleMotionNotify - MotionNotify event handler
 *
 **********************************************************************/
void HandleMotionNotify()
{
  int delta_x, delta_y;
  int warp_x,warp_y;

  while(XCheckTypedEvent(dpy,MotionNotify,&Event));
  delta_x = delta_y = warp_x = warp_y = 0;

  /* here is the code for dragging the viewport around within the pager.*/
#ifndef NO_PAGER
  if((Tmp_win)&&(Tmp_win->w == Pager_w))
    {
      if(Event.xmotion.state == Button3MotionMask)
	SwitchPages(FALSE,FALSE);
    }
#endif

#ifndef NON_VIRTUAL
  if(Event.xmotion.x_root >= Scr.MyDisplayWidth -2)
    {
      delta_x = Scr.EdgeScrollX;
      warp_x = Scr.EdgeScrollX - 4;
    }
  if(Event.xmotion.y_root >= Scr.MyDisplayHeight -2)
    {
      delta_y = Scr.EdgeScrollY;
      warp_y = Scr.EdgeScrollY - 4;      
    }
  if(Event.xmotion.x_root < 2)
    {
      delta_x = -Scr.EdgeScrollX;
      warp_x =  -Scr.EdgeScrollX + 4;
    }
  if(Event.xmotion.y_root < 2)
    {
      delta_y = -Scr.EdgeScrollY;
      warp_y =  -Scr.EdgeScrollY + 4;
    }
  if(Scr.Vx + delta_x < 0)
    delta_x = -Scr.Vx;
  if(Scr.Vy + delta_y < 0)
    delta_y = -Scr.Vy;
  if(Scr.Vx + delta_x > Scr.VxMax)
    delta_x = Scr.VxMax - Scr.Vx;
  if(Scr.Vy + delta_y > Scr.VyMax)
    delta_y = Scr.VyMax - Scr.Vy;
  if((delta_x!=0)||(delta_y!=0))
    {
      XGrabServer(dpy);
      XWarpPointer(dpy, Scr.Root, Scr.Root, 0, 0, Scr.MyDisplayWidth, 
		   Scr.MyDisplayHeight, 
		   Event.xmotion.x_root - warp_x,
		   Event.xmotion.y_root - warp_y);
      MoveViewport(Scr.Vx + delta_x,Scr.Vy+delta_y,False);

      XSync (dpy, 0);
      XUngrabServer(dpy);
    }
#endif
  while(XCheckTypedEvent(dpy,MotionNotify,&Event));
}

/***********************************************************************
 *
 *  Procedure:
 *	HandleButtonPress - ButtonPress event handler
 *
 ***********************************************************************/
void HandleButtonPress()
{
  unsigned int modifier;
  MouseButton *MouseEntry;
#ifndef COHERENT
  struct itimerval value;
#endif

  /* click to focus stuff goes here */
  if((Scr.ClickToFocus)&&(Tmp_win != Scr.Focus))
    {
      /* ClickToFocus focus queue manipulation */
      if (Scr.Focus && Tmp_win && Tmp_win != Scr.Focus &&
          Tmp_win != &Scr.FvwmRoot && Scr.Focus != &Scr.FvwmRoot)
        {
          FvwmWindow *tmp_win1, *tmp_win2;
	  
          tmp_win1 = Tmp_win->prev;
          tmp_win2 = Tmp_win->next;
	  
          if (tmp_win1) tmp_win1->next = tmp_win2;
          if (tmp_win2) tmp_win2->prev = tmp_win1;
	  
	  Tmp_win->next = Scr.FvwmRoot.next;
	  Scr.FvwmRoot.next->prev = Tmp_win;
	  Scr.FvwmRoot.next = Tmp_win;
	  Tmp_win->prev = &Scr.FvwmRoot;
        }

      if(Tmp_win)
	{
	  SetBorder (Tmp_win, True,False,True);
#ifndef COHERENT
	  if(Scr.AutoRaiseDelay > 0)
	    {
	      value.it_value.tv_usec = 1000*(Scr.AutoRaiseDelay%1000);
	      value.it_value.tv_sec = Scr.AutoRaiseDelay/1000;
	      value.it_interval.tv_usec = 0;
	      value.it_interval.tv_sec = 0;
	      setitimer(ITIMER_REAL,&value,NULL);
	    }
	  else 
#endif
	    if(Event.xany.window != Tmp_win->w)
	      RaiseWindow(Tmp_win);
	  XAllowEvents(dpy,ReplayPointer,CurrentTime);

	  KeepOnTop();
	  return;
	}
    }

  /* here is the code for moving windows in the pager.*/
#ifndef NO_PAGER
  if((Tmp_win)&&(Tmp_win->w == Event.xbutton.window)&&
     (Tmp_win->w == Pager_w)&&(Event.xbutton.button == Button2))
    {
      PagerMoveWindow();
      return;
    }
#endif


  Context = GetContext(Tmp_win,&Event);

  ButtonWindow = Tmp_win;
  
  /* we have to execute a function or pop up a menu
   */
  modifier = (Event.xbutton.state & mods_used);
  /* need to search for an appropriate mouse binding */
  MouseEntry = Scr.MouseButtonRoot;
  while(MouseEntry != (MouseButton *)0)
    {
      if(((MouseEntry->Button == Event.xbutton.button)||(MouseEntry->Button == 0))&&
	 (MouseEntry->Context & Context)&&
	 ((MouseEntry->Modifier == AnyModifier)||
	  (MouseEntry->Modifier == modifier)))
	{
	  /* got a match, now process it */
	  if (MouseEntry->func != (int)NULL)
	    {
	      Action = MouseEntry->item ? MouseEntry->item->action : NULL;
	      ExecuteFunction(MouseEntry->func, Action, Event.xany.window, 
			      Tmp_win, &Event, Context,MouseEntry->val1,
			      MouseEntry->val2,MouseEntry->menu);
	      break;
	    }
	}
      MouseEntry = MouseEntry->NextButton;
    }
}

/***********************************************************************
 *
 *  Procedure:
 *	HandleButtonRelease - ButtonRelease event handler
 *
 ***********************************************************************/
void HandleButtonRelease()
{
#ifndef NO_PAGER
  if((Tmp_win)&&(Event.xany.window == Pager_w))
    {
      switch(Event.xbutton.button)
	{
	default:
	case Button1:
	  SwitchPages(TRUE,TRUE);
	  break;
	case Button3:
	  SwitchPages(FALSE,FALSE);
	  break;
	}
    }
#endif
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleEnterNotify - EnterNotify event handler
 *
 ************************************************************************/
void HandleEnterNotify()
{
  XEnterWindowEvent *ewp = &Event.xcrossing;
  XEvent d;
#ifndef COHERENT
  struct itimerval value;
#endif

  /* look for a matching leaveNotify which would nullify this enterNotify */
  if(XCheckTypedWindowEvent (dpy, ewp->window, LeaveNotify, &d))
    if((d.xcrossing.mode==NotifyNormal)&&(d.xcrossing.detail!=NotifyInferior))
      return;

  /* make sure its for one of our windows */
  if (!Tmp_win) 
    return;
      
  if(!Scr.ClickToFocus)
    {
      SetBorder (Tmp_win, True,False,True);
#ifndef COHERENT
      if(Scr.AutoRaiseDelay > 0)
	{
	  value.it_value.tv_usec = 1000*(Scr.AutoRaiseDelay%1000);
	  value.it_value.tv_sec = Scr.AutoRaiseDelay/1000;
	  value.it_interval.tv_usec = 0;
	  value.it_interval.tv_sec = 0;
	  setitimer(ITIMER_REAL,&value,NULL);
	}
#endif
    }
  if(!(Tmp_win->flags & ICON))
    if(Tmp_win->w == Event.xcrossing.window)
      InstallWindowColormaps(EnterNotify, Tmp_win);
  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleLeaveNotify - LeaveNotify event handler
 *
 ************************************************************************/
void HandleLeaveNotify()
{
  XEvent dummy;
#ifndef COHERENT
  struct itimerval value;
#endif

  /* look for a matching EnterNotify which would nullify this LeaveNotify*/
  if(XCheckTypedWindowEvent (dpy, Event.xcrossing.window, EnterNotify, &dummy))
    return;
  
  if (!Tmp_win) 
    return;
  /*
   * We're not interested in pseudo Enter/Leave events generated
   * from grab initiations and terminations.
   */
  if (Event.xcrossing.detail != NotifyInferior) 
    {
      if((Event.xcrossing.window==Tmp_win->frame)||
	 (Event.xcrossing.window==Tmp_win->icon_w)||
	 (Event.xcrossing.window==Tmp_win->icon_pixmap_w)||
	 ((Event.xcrossing.window==Tmp_win->w)&&
	  !(Tmp_win->flags & TITLE)&& !(Tmp_win->flags & BORDER)))
	{
	  if(!Scr.ClickToFocus)
	    {
#ifndef COHERENT
	      if(Scr.AutoRaiseDelay > 0)
		{
		  value.it_value.tv_usec = 0;
		  value.it_value.tv_sec = 0;
		  value.it_interval.tv_usec = 0;
		  value.it_interval.tv_sec = 0;
		  setitimer(ITIMER_REAL,&value,NULL);
		}
#endif
	      SetBorder (Tmp_win, False,False,True);
	    }
	  InstallWindowColormaps (LeaveNotify, &Scr.FvwmRoot);
	}
      else if(Event.xcrossing.window == Tmp_win->w)
	{
	  InstallWindowColormaps (LeaveNotify, &Scr.FvwmRoot);
	}
    }
  XSync (dpy, 0);
  return;
}


/**************************************************************************
 * 
 * For auto-raising windows, this routine is called
 *
 *************************************************************************/
volatile int alarmed;
void enterAlarm()
{
  alarmed = True;
  signal(SIGALRM, enterAlarm);
}

/***********************************************************************
 *
 *  Procedure:
 *	HandleConfigureRequest - ConfigureRequest event handler
 *
 ************************************************************************/
void HandleConfigureRequest()
{
  XWindowChanges xwc;
  unsigned long xwcm;
  int x, y, width, height;
  int gravx, gravy;
  XConfigureRequestEvent *cre = &Event.xconfigurerequest;
  
  /*
   * Event.xany.window is Event.xconfigurerequest.parent, so Tmp_win will
   * be wrong
   */
  Event.xany.window = cre->window;	/* mash parent field */
  if (XFindContext (dpy, cre->window, FvwmContext, (caddr_t *) &Tmp_win) ==
      XCNOENT)
    Tmp_win = NULL;
  
  /*
   * According to the July 27, 1988 ICCCM draft, we should ignore size and
   * position fields in the WM_NORMAL_HINTS property when we map a window.
   * Instead, we'll read the current geometry.  Therefore, we should respond
   * to configuration requests for windows which have never been mapped.
   */
  if (!Tmp_win || (Tmp_win->icon_w == cre->window))
    {
      xwcm = cre->value_mask & 
	(CWX | CWY | CWWidth | CWHeight | CWBorderWidth);
      xwc.x = cre->x;
      xwc.y = cre->y;
      if((Tmp_win)&&((Tmp_win->icon_w == cre->window)))
	{
	  Tmp_win->icon_xl_loc = cre->x;
	  Tmp_win->icon_x_loc = cre->x + 
	    (Tmp_win->icon_w_width - Tmp_win->icon_p_width)/2;
	  Tmp_win->icon_y_loc = cre->y - Tmp_win->icon_p_height;
	}
      xwc.width = cre->width;
      xwc.height = cre->height;
      xwc.border_width = cre->border_width;
      XConfigureWindow(dpy, Event.xany.window, xwcm, &xwc);

      if(Tmp_win)
	{
	  xwc.x = Tmp_win->icon_x_loc;
	  xwc.y = Tmp_win->icon_y_loc - Tmp_win->icon_p_height;
	  xwcm = cre->value_mask & (CWX | CWY);
	  if(Tmp_win->icon_pixmap_w != None)
	    XConfigureWindow(dpy, Tmp_win->icon_pixmap_w, xwcm, &xwc);
	}
      return;
    }
  
  if (cre->value_mask & CWStackMode) 
    {
      FvwmWindow *otherwin;
      
      xwc.sibling = (((cre->value_mask & CWSibling) &&
		      (XFindContext (dpy, cre->above, FvwmContext,
				     (caddr_t *) &otherwin) == XCSUCCESS))
		     ? otherwin->frame : cre->above);
      xwc.stack_mode = cre->detail;
      XConfigureWindow (dpy, Tmp_win->lead_w,
			cre->value_mask & (CWSibling | CWStackMode), &xwc);
    }

#ifdef SHAPE
  {
    int xws, yws, xbs, ybs;
    unsigned wws, hws, wbs, hbs;
    int boundingShaped, clipShaped;
    
    XShapeQueryExtents (dpy, Tmp_win->w,&boundingShaped, &xws, &yws, &wws,
			&hws,&clipShaped, &xbs, &ybs, &wbs, &hbs);
    Tmp_win->wShaped = boundingShaped;
  }
#endif /* SHAPE */
  
  /* Don't modify frame_XXX fields before calling SetupWindow! */
  x = Tmp_win->frame_x;
  y = Tmp_win->frame_y;
  width = Tmp_win->frame_width;
  height = Tmp_win->frame_height;
  
  /*
   * Section 4.1.5 of the ICCCM states that the (x,y) coordinates in the
   * configure request are for the upper-left outer corner of the window.
   * This means that we need to adjust for the additional title height as
   * well as for any border width changes that we decide to allow.  The
   * current window gravity is to be used in computing the adjustments, just
   * as when initially locating the window.  Note that if we do decide to 
   * allow border width changes, we will need to send the synthetic 
   * ConfigureNotify event.
   */
  GetGravityOffsets (Tmp_win, &gravx, &gravy);

 /* for restoring */  
  if (cre->value_mask & CWBorderWidth) 
    Tmp_win->old_bw = cre->border_width; 
  /* override even if border change */
  if (cre->value_mask & CWX)
    x = cre->x + Tmp_win->old_bw - 2*Tmp_win->frame_bw;
  if (cre->value_mask & CWY) 
    y = cre->y - ((gravy < 0) ? 0 : Tmp_win->title_height) - Tmp_win->old_bw +
      2*Tmp_win->frame_bw;
  if (cre->value_mask & CWWidth) 
    width = cre->width + 2*Tmp_win->boundary_width;
  if (cre->value_mask & CWHeight) 
    height = cre->height+Tmp_win->title_height+2*Tmp_win->boundary_width;
  
  /*
   * SetupWindow (x,y) are the location of the upper-left outer corner and
   * are passed directly to XMoveResizeWindow (frame).  The (width,height)
   * are the inner size of the frame.  The inner width is the same as the 
   * requested client window width; the inner height is the same as the
   * requested client window height plus any title bar slop.
   */
  SetupFrame (Tmp_win, x, y, width, height,FALSE);
  KeepOnTop();

#ifndef NO_PAGER
  RedrawPager();
#endif
}

/***********************************************************************
 *
 *  Procedure:
 *      HandleShapeNotify - shape notification event handler
 *
 ***********************************************************************/
#ifdef SHAPE
void HandleShapeNotify ()
{
  XShapeEvent *sev = (XShapeEvent *) &Event;

  if (!Tmp_win)
    return;
  if (sev->kind != ShapeBounding)
    return;
  Tmp_win->wShaped = sev->shaped;
    
  SetupFrame(Tmp_win, Tmp_win->frame_x,Tmp_win->frame_y,
	     Tmp_win->frame_width, Tmp_win->frame_height,False);
}
#endif SHAPE

/***********************************************************************
 *
 *  Procedure:
 *	HandleVisibilityNotify - record fully visible windows for
 *      use in the RaiseLower function and the OnTop type windows.
 *
 ************************************************************************/
void HandleVisibilityNotify()
{
  XVisibilityEvent *vevent = (XVisibilityEvent *) &Event;

  if(Tmp_win)
    {
      if(vevent->state == VisibilityUnobscured)
	Tmp_win->flags |= VISIBLE;
      else
	Tmp_win->flags &= ~VISIBLE;

      /* For the most part, we'll raised partially obscured ONTOP windows
       * here. The exception is ONTOP windows that are obscured by
       * other ONTOP windows, which are raised in KeepOnTop(). This
       * complicated set-up saves us from continually re-raising
       * every on top window */
      if(((vevent->state == VisibilityPartiallyObscured)||
	  (vevent->state == VisibilityFullyObscured))&&
	 (Tmp_win->flags&ONTOP)&&(Tmp_win->flags & RAISED))
	{
	  RaiseWindow(Tmp_win);
	  Tmp_win->flags &= ~RAISED;
	}
    }
}
      
/**************************************************************************
 *
 * Removes expose events for a specific window from the queue 
 *
 *************************************************************************/
void flush_expose (Window w)
{
  XEvent dummy;
  while (XCheckTypedWindowEvent (dpy, w, Expose, &dummy)) ;
}


/**************************************************************************
 *
 * Updates the colormap when a window changes its colormap
 *
 *************************************************************************/
FvwmWindow *ColormapWindow = &Scr.FvwmRoot;
void HandleColormapNotify()
{
  InstallWindowColormaps(0,ColormapWindow);
}

/***********************************************************************
 *
 *  Procedure:
 *	InstallWindowColormaps - install the colormaps for one fvwm window
 *
 *  Inputs:
 *	type	- type of event that caused the installation
 *	tmp	- for a subset of event types, the address of the
 *		  window structure, whose colormaps are to be installed.
 *
 ************************************************************************/
void InstallWindowColormaps (int type, FvwmWindow *tmp)
{
  XWindowAttributes attributes;
  static Colormap last_cmap;

  /*
   * Make sure the client window still exists. 
   */
  if (XGetGeometry(dpy, tmp->w, &JunkRoot, &JunkX, &JunkY,
		   &JunkWidth, &JunkHeight, &JunkBW, &JunkDepth) == 0)
    {
      return;
    }

  ColormapWindow = tmp;
  /* Save the colormap to be loaded for when force loading of
   * root colormap(s) ends.
   */
  Scr.pushed_window = tmp;
  /* Don't load any new colormap if root colormap(s) has been
   * force loaded.
   */
  if (Scr.root_pushes)
    {
      return;
    }

  if(tmp == &Scr.FvwmRoot)
    {
      XGetWindowAttributes(dpy,Scr.Root,&attributes);
    }
  else
    {
      XGetWindowAttributes(dpy,tmp->w,&attributes);
    }
  
  if(last_cmap != attributes.colormap)
    {
      last_cmap = attributes.colormap;
      XInstallColormap(dpy,attributes.colormap);    
    }
}


/***********************************************************************
 *
 *  Procedures:
 *	<Uni/I>nstallRootColormap - Force (un)loads root colormap(s)
 *
 *	   These matching routines provide a mechanism to insure that
 *	   the root colormap(s) is installed during operations like
 *	   rubber banding or menu display that require colors from
 *	   that colormap.  Calls may be nested arbitrarily deeply,
 *	   as long as there is one UninstallRootColormap call per
 *	   InstallRootColormap call.
 *
 *	   The final UninstallRootColormap will cause the colormap list
 *	   which would otherwise have be loaded to be loaded, unless
 *	   Enter or Leave Notify events are queued, indicating some
 *	   other colormap list would potentially be loaded anyway.
 ***********************************************************************/
void InstallRootColormap()
{
    FvwmWindow *tmp;
    if (Scr.root_pushes == 0) 
      {
	/*
	 * The saving and restoring of cmapInfo.pushed_window here
	 * is a slimy way to remember the actual pushed list and
	 * not that of the root window.
	 */
	tmp = Scr.pushed_window;
	InstallWindowColormaps(0, &Scr.FvwmRoot);
	Scr.pushed_window = tmp;
      }
    Scr.root_pushes++;
    return;
}

/***************************************************************************
 * 
 * Unstacks one layer of root colormap pushing 
 * If we peel off the last layer, re-install th e application colormap
 * 
 ***************************************************************************/
void UninstallRootColormap()
{
  if (Scr.root_pushes)
    Scr.root_pushes--;
  
  if (!Scr.root_pushes) 
    {
     XSync (dpy, 0);
     InstallWindowColormaps(0, Scr.pushed_window);
     
    }
  return;
}

  
/***********************************************************************
 *
 *  Procedure:
 *	RestoreWithdrawnLocation
 * 
 *  Puts windows back where they were before fvwm took over 
 *
 ************************************************************************/
void RestoreWithdrawnLocation (FvwmWindow *tmp)
{
  int a,b;
  unsigned int bw,mask;
  XWindowChanges xwc;

  if(!tmp)
    return;

  if (XGetGeometry (dpy, tmp->w, &JunkRoot, &xwc.x, &xwc.y, 
		    &JunkWidth, &JunkHeight, &bw, &JunkDepth)) 
    {
      if((tmp->flags & TITLE)||(tmp->flags & BORDER))
	{
	  XTranslateCoordinates(dpy,tmp->lead_w,Scr.Root,xwc.x,xwc.y,
				&a,&b,&JunkChild);
	  xwc.x = a;
	  xwc.y = b;
	}
      xwc.x += tmp->xdiff;
      xwc.y += tmp->ydiff;
      xwc.border_width = tmp->old_bw;
      mask = (CWX | CWY|CWBorderWidth);

      if((tmp->flags & TITLE)||(tmp->flags & BORDER))
	{
	  XReparentWindow (dpy, tmp->w,Scr.Root,xwc.x,xwc.y);
	  XDestroyWindow(dpy,tmp->frame);
	}

      XConfigureWindow (dpy, tmp->w, mask, &xwc);

      XSync(dpy,0);
    }
}


/***************************************************************************
 *
 * ICCCM Client Messages - Section 4.2.8 of the ICCCM dictates that all
 * client messages will have the following form:
 *
 *     event type	ClientMessage
 *     message type	_XA_WM_PROTOCOLS
 *     window		tmp->w
 *     format		32
 *     data[0]		message atom
 *     data[1]		time stamp
 *
 ****************************************************************************/
void send_clientmessage (Window w, Atom a, Time timestamp)
{
  XClientMessageEvent ev;
  
  ev.type = ClientMessage;
  ev.window = w;
  ev.message_type = _XA_WM_PROTOCOLS;
  ev.format = 32;
  ev.data.l[0] = a;
  ev.data.l[1] = timestamp;
  XSendEvent (dpy, w, False, 0L, (XEvent *) &ev);
}

void My_XNextEvent(Display *dpy, XEvent *event)
{
#ifndef COHERENT
  struct itimerval value;
#endif
  fd_set in_fdset;

  if(XPending(dpy))
    {
      XNextEvent(dpy,event);
      return;
    }

#ifndef COHERENT
  getitimer(ITIMER_REAL,&value);
#endif

  FD_ZERO(&in_fdset);
  FD_SET(x_fd,&in_fdset);
#ifndef COHERENT
  if((value.it_value.tv_usec != 0)||
     (value.it_value.tv_sec != 0))
    select(fd_width,&in_fdset, 0, 0, &value.it_value);
  else
#endif
    select(fd_width,&in_fdset, 0, 0, NULL);

  if(alarmed)
    {
      alarmed = False;
      if(Scr.Focus != NULL)
      RaiseWindow(Scr.Focus);
      KeepOnTop();
    }

  if(XPending(dpy))
    {
      XNextEvent(dpy,event);
      return;
    }
}
